/**
 * Session Monitor JavaScript
 * Monitors user activity and handles session timeout
 */

class SessionMonitor {
    constructor(options = {}) {
        this.timeoutDuration = options.timeoutDuration || 60; // 60 seconds
        this.warningDuration = options.warningDuration || 30; // 30 seconds
        this.checkInterval = options.checkInterval || 5000; // Check every 5 seconds
        this.userType = options.userType || 'admin';
        this.loginUrl = options.loginUrl || 'authentication/login.php';
        
        this.warningShown = false;
        this.checkTimer = null;
        this.activityTimer = null;
        this.lastActivity = Date.now();
        
        this.init();
    }
    
    init() {
        this.startActivityTracking();
        this.startSessionChecking();
        this.showSessionInfo();
    }
    
    startActivityTracking() {
        // Track user activity
        const events = ['mousedown', 'mousemove', 'keypress', 'scroll', 'touchstart', 'click'];
        
        events.forEach(event => {
            document.addEventListener(event, () => {
                this.updateActivity();
            }, true);
        });
        
        // Update activity on page visibility change
        document.addEventListener('visibilitychange', () => {
            if (!document.hidden) {
                this.updateActivity();
            }
        });
    }
    
    startSessionChecking() {
        this.checkTimer = setInterval(() => {
            this.checkSession();
        }, this.checkInterval);
    }
    
    updateActivity() {
        this.lastActivity = Date.now();
        
        // Send activity update to server
        this.sendActivityUpdate();
    }
    
    async sendActivityUpdate() {
        try {
            const response = await fetch('config/session_manager.php?action=extend_session&user_type=' + this.userType, {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json',
                }
            });
            
            if (response.ok) {
                const data = await response.json();
                if (data.success) {
                    this.warningShown = false;
                    this.hideWarning();
                }
            }
        } catch (error) {
            console.error('Session update failed:', error);
        }
    }
    
    async checkSession() {
        try {
            const response = await fetch('config/session_manager.php?action=check_session&user_type=' + this.userType, {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json',
                }
            });
            
            if (response.ok) {
                const data = await response.json();
                
                if (!data.valid) {
                    this.handleSessionExpired();
                    return;
                }
                
                if (data.expiring_soon && !this.warningShown) {
                    this.showWarning(data.remaining_time);
                }
                
                this.updateSessionInfo(data.remaining_time);
            }
        } catch (error) {
            console.error('Session check failed:', error);
        }
    }
    
    showWarning(remainingTime) {
        this.warningShown = true;
        
        // Create warning modal
        const warningModal = document.createElement('div');
        warningModal.id = 'session-warning-modal';
        warningModal.innerHTML = `
            <div class="session-warning-overlay">
                <div class="session-warning-content">
                    <div class="session-warning-header">
                        <i class="fas fa-exclamation-triangle"></i>
                        <h3>Session Timeout Warning</h3>
                    </div>
                    <div class="session-warning-body">
                        <p>Your session will expire in <span id="session-countdown">${remainingTime}</span> seconds.</p>
                        <p>Click anywhere to extend your session.</p>
                    </div>
                    <div class="session-warning-footer">
                        <button id="extend-session-btn" class="btn btn-primary">Extend Session</button>
                        <button id="logout-now-btn" class="btn btn-secondary">Logout Now</button>
                    </div>
                </div>
            </div>
        `;
        
        document.body.appendChild(warningModal);
        
        // Add event listeners
        document.getElementById('extend-session-btn').addEventListener('click', () => {
            this.updateActivity();
        });
        
        document.getElementById('logout-now-btn').addEventListener('click', () => {
            this.logout();
        });
        
        // Start countdown
        this.startCountdown(remainingTime);
        
        // Add styles
        this.addWarningStyles();
    }
    
    hideWarning() {
        const warningModal = document.getElementById('session-warning-modal');
        if (warningModal) {
            warningModal.remove();
        }
        this.warningShown = false;
    }
    
    startCountdown(remainingTime) {
        let timeLeft = remainingTime;
        const countdownElement = document.getElementById('session-countdown');
        
        const countdownTimer = setInterval(() => {
            timeLeft--;
            if (countdownElement) {
                countdownElement.textContent = timeLeft;
            }
            
            if (timeLeft <= 0) {
                clearInterval(countdownTimer);
                this.handleSessionExpired();
            }
        }, 1000);
    }
    
    handleSessionExpired() {
        // Clear timers
        if (this.checkTimer) {
            clearInterval(this.checkTimer);
        }
        
        // Show expired message
        this.showSessionExpired();
        
        // Redirect to login after delay
        setTimeout(() => {
            this.logout();
        }, 2000);
    }
    
    showSessionExpired() {
        const expiredModal = document.createElement('div');
        expiredModal.innerHTML = `
            <div class="session-expired-overlay">
                <div class="session-expired-content">
                    <div class="session-expired-header">
                        <i class="fas fa-clock"></i>
                        <h3>Session Expired</h3>
                    </div>
                    <div class="session-expired-body">
                        <p>Your session has expired due to inactivity.</p>
                        <p>You will be redirected to the login page.</p>
                    </div>
                </div>
            </div>
        `;
        
        document.body.appendChild(expiredModal);
        this.addExpiredStyles();
    }
    
    logout() {
        window.location.href = this.loginUrl;
    }
    
    showSessionInfo() {
        // Create session info element
        const sessionInfo = document.createElement('div');
        sessionInfo.id = 'session-info';
        sessionInfo.innerHTML = `
            <div class="session-info-content">
                <i class="fas fa-user-clock"></i>
                <span id="session-time-remaining">Active</span>
            </div>
        `;
        
        document.body.appendChild(sessionInfo);
        this.addSessionInfoStyles();
    }
    
    updateSessionInfo(remainingTime) {
        const timeElement = document.getElementById('session-time-remaining');
        if (timeElement) {
            if (remainingTime <= this.warningDuration) {
                timeElement.textContent = `Expires in ${remainingTime}s`;
                timeElement.style.color = '#dc3545';
            } else {
                timeElement.textContent = 'Active';
                timeElement.style.color = '#28a745';
            }
        }
    }
    
    addWarningStyles() {
        if (document.getElementById('session-warning-styles')) return;
        
        const styles = document.createElement('style');
        styles.id = 'session-warning-styles';
        styles.textContent = `
            .session-warning-overlay {
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: rgba(0, 0, 0, 0.8);
                z-index: 9999;
                display: flex;
                align-items: center;
                justify-content: center;
            }
            
            .session-warning-content {
                background: white;
                border-radius: 12px;
                padding: 30px;
                max-width: 400px;
                text-align: center;
                box-shadow: 0 10px 30px rgba(0, 0, 0, 0.3);
            }
            
            .session-warning-header {
                margin-bottom: 20px;
            }
            
            .session-warning-header i {
                font-size: 48px;
                color: #ffc107;
                margin-bottom: 15px;
            }
            
            .session-warning-header h3 {
                margin: 0;
                color: #333;
            }
            
            .session-warning-body {
                margin-bottom: 25px;
                color: #666;
            }
            
            .session-warning-body p {
                margin: 10px 0;
            }
            
            #session-countdown {
                font-weight: bold;
                color: #dc3545;
                font-size: 1.2em;
            }
            
            .session-warning-footer {
                display: flex;
                gap: 10px;
                justify-content: center;
            }
            
            .session-warning-footer .btn {
                padding: 10px 20px;
                border: none;
                border-radius: 6px;
                cursor: pointer;
                font-weight: 500;
            }
            
            .btn-primary {
                background: #007bff;
                color: white;
            }
            
            .btn-secondary {
                background: #6c757d;
                color: white;
            }
        `;
        
        document.head.appendChild(styles);
    }
    
    addExpiredStyles() {
        if (document.getElementById('session-expired-styles')) return;
        
        const styles = document.createElement('style');
        styles.id = 'session-expired-styles';
        styles.textContent = `
            .session-expired-overlay {
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: rgba(0, 0, 0, 0.9);
                z-index: 10000;
                display: flex;
                align-items: center;
                justify-content: center;
            }
            
            .session-expired-content {
                background: white;
                border-radius: 12px;
                padding: 30px;
                max-width: 400px;
                text-align: center;
                box-shadow: 0 10px 30px rgba(0, 0, 0, 0.3);
            }
            
            .session-expired-header {
                margin-bottom: 20px;
            }
            
            .session-expired-header i {
                font-size: 48px;
                color: #dc3545;
                margin-bottom: 15px;
            }
            
            .session-expired-header h3 {
                margin: 0;
                color: #333;
            }
            
            .session-expired-body {
                color: #666;
            }
            
            .session-expired-body p {
                margin: 10px 0;
            }
        `;
        
        document.head.appendChild(styles);
    }
    
    addSessionInfoStyles() {
        if (document.getElementById('session-info-styles')) return;
        
        const styles = document.createElement('style');
        styles.id = 'session-info-styles';
        styles.textContent = `
            #session-info {
                position: fixed;
                top: 20px;
                right: 20px;
                z-index: 1000;
                background: rgba(255, 255, 255, 0.9);
                backdrop-filter: blur(10px);
                border-radius: 25px;
                padding: 8px 15px;
                box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
                font-size: 12px;
                font-weight: 500;
            }
            
            .session-info-content {
                display: flex;
                align-items: center;
                gap: 8px;
            }
            
            .session-info-content i {
                color: #28a745;
            }
        `;
        
        document.head.appendChild(styles);
    }
    
    destroy() {
        if (this.checkTimer) {
            clearInterval(this.checkTimer);
        }
        
        this.hideWarning();
        
        const sessionInfo = document.getElementById('session-info');
        if (sessionInfo) {
            sessionInfo.remove();
        }
    }
}

// Auto-initialize if options are provided via data attributes
document.addEventListener('DOMContentLoaded', function() {
    const sessionData = document.querySelector('[data-session-monitor]');
    if (sessionData) {
        const options = {
            timeoutDuration: parseInt(sessionData.dataset.timeoutDuration) || 60,
            warningDuration: parseInt(sessionData.dataset.warningDuration) || 30,
            checkInterval: parseInt(sessionData.dataset.checkInterval) || 5000,
            userType: sessionData.dataset.userType || 'admin',
            loginUrl: sessionData.dataset.loginUrl || 'authentication/login.php'
        };
        
        window.sessionMonitor = new SessionMonitor(options);
    }
});
