<?php
/**
 * Branding Configuration Functions
 * Handles logo, favicon, and other branding elements
 */

/**
 * Get favicon HTML
 * @return string HTML for favicon
 */
function getFaviconHtml() {
    // Get favicon from settings if available
    if (function_exists('getSetting')) {
        $faviconPath = getSetting('favicon', 'assets/img/favicon.png');
        
        // Determine file type based on extension
        $extension = strtolower(pathinfo($faviconPath, PATHINFO_EXTENSION));
        $mimeType = 'image/x-icon'; // Default
        
        switch ($extension) {
            case 'ico':
                $mimeType = 'image/x-icon';
                break;
            case 'png':
                $mimeType = 'image/png';
                break;
            case 'jpg':
            case 'jpeg':
                $mimeType = 'image/jpeg';
                break;
            case 'svg':
                $mimeType = 'image/svg+xml';
                break;
        }
        
        // Check if we're in a subdirectory and adjust path accordingly
        $requestUri = $_SERVER['REQUEST_URI'] ?? '';
        if (strpos($requestUri, '/admin-dashboard/') !== false || strpos($requestUri, '/users/') !== false) {
            // We're in a subdirectory, need to go up one directory
            if (strpos($faviconPath, 'uploads/') === 0) {
                $faviconPath = '../' . $faviconPath;
            } elseif (strpos($faviconPath, 'assets/') === 0) {
                $faviconPath = '../' . $faviconPath;
            }
        }
        
        return '<link rel="icon" type="' . $mimeType . '" href="' . $faviconPath . '?v=' . time() . '">' . 
               '<link rel="apple-touch-icon" sizes="180x180" href="' . $faviconPath . '?v=' . time() . '">';
    }
    
    // Default favicon - check if we're in a subdirectory
    $defaultPath = 'assets/img/favicon.png';
    $requestUri = $_SERVER['REQUEST_URI'] ?? '';
    if (strpos($requestUri, '/admin-dashboard/') !== false || strpos($requestUri, '/users/') !== false) {
        $defaultPath = '../' . $defaultPath;
    }
    
    return '<link rel="icon" type="image/png" href="' . $defaultPath . '?v=' . time() . '">' . 
           '<link rel="apple-touch-icon" sizes="180x180" href="' . $defaultPath . '?v=' . time() . '">';
}

/**
 * Get logo HTML
 * @param string $type Logo type (white, black)
 * @param string $class CSS class for the logo
 * @param string $alt Alt text for the logo
 * @return string HTML for logo
 */
function getLogoHtml($type = 'white', $class = 'logo', $alt = 'Logo') {
    // Get logo from settings if available
    if (function_exists('getSetting')) {
        $mainLogo = getSetting('main_logo', '');
        $miniLogo = getSetting('mini_logo', '');
        
        // For white logo type, return empty (hidden)
        if ($type === 'white') {
            return '<img src="' . ($mainLogo ?: 'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMSIgaGVpZ2h0PSIxIiB2aWV3Qm94PSIwIDAgMSAxIiBmaWxsPSJub25lIiB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciPjxyZWN0IHdpZHRoPSIxIiBoZWlnaHQ9IjEiIGZpbGw9InRyYW5zcGFyZW50Ii8+PC9zdmc+') . '" alt="' . $alt . '" class="' . $class . '" style="display: none;">';
        }
        
        // For black logo type, use main logo (colored logo)
        if ($type === 'black' && !empty($mainLogo)) {
            return '<img src="' . $mainLogo . '" alt="' . $alt . '" class="' . $class . '">';
        }
        
        // Fallback to mini logo if main logo not available
        if (!empty($miniLogo)) {
            return '<img src="' . $miniLogo . '" alt="' . $alt . '" class="' . $class . '">';
        }
    }
    
    // Define logo paths based on type
    $logoPaths = [
        'white' => 'admin-dashboard/assets/images/white-logo.png',
        'black' => 'admin-dashboard/assets/images/black-logo.png'
    ];
    
    // Check if custom logo exists
    if (isset($logoPaths[$type]) && file_exists($logoPaths[$type])) {
        return '<img src="' . $logoPaths[$type] . '" alt="' . $alt . '" class="' . $class . '">';
    }
    
    // Default logo paths
    $defaultPaths = [
        'white' => 'assets/img/logo-white.png',
        'black' => 'assets/img/logo-black.png'
    ];
    
    if (isset($defaultPaths[$type])) {
        return '<img src="' . $defaultPaths[$type] . '" alt="' . $alt . '" class="' . $class . '">';
    }
    
    // Fallback to text logo
    return '<span class="' . $class . '">Omateq Digitals</span>';
}

/**
 * Get site branding color
 * @return string Brand color hex code
 */
function getBrandColor() {
    return '#007bff'; // Default blue color
}

/**
 * Get site name
 * @return string Site name
 */
function getSiteName() {
    // Try to get from settings first
    if (function_exists('getSetting')) {
        return getSetting('site_name', 'Omateq Digitals');
    }
    return 'Omateq Digitals';
}

/**
 * Get site tagline
 * @return string Site tagline
 */
function getSiteTagline() {
    // Try to get from settings first
    if (function_exists('getSetting')) {
        return getSetting('site_tagline', 'IT Solutions & Business Services');
    }
    return 'IT Solutions & Business Services';
}

/**
 * Get company contact info
 * @return array Contact information
 */
function getCompanyContact() {
    return [
        'phone' => '+2347069378861',
        'email' => 'info@omateqdigitals.com',
        'address' => 'Suite 8, Jehovah Bueze Plaza, Opp. Berger Paint, Suleja'
    ];
}

/**
 * Get social media links
 * @return array Social media URLs
 */
function getSocialLinks() {
    return [
        'facebook' => 'https://facebook.com/omateqdigitals',
        'youtube' => 'https://youtube.com/@learnwithomateq',
        'linkedin' => 'https://www.linkedin.com/omateqdigitals'
    ];
}
?>
