<?php
/**
 * Portfolio Data Functions
 * Handles portfolio items and statistics
 */

/**
 * Get all active portfolio items from database
 * @return array Array of portfolio items
 */
function getPortfolioItems() {
    try {
        // Include database connection
        if (!function_exists('getConnection')) {
            require_once __DIR__ . '/database.php';
        }
        
        $pdo = getConnection();
        $stmt = $pdo->prepare("SELECT * FROM portfolio_items WHERE status = 'active' ORDER BY sort_order ASC, created_at DESC");
        $stmt->execute();
        $items = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Process items to add computed fields
        foreach ($items as &$item) {
            // Set default image if none provided
            if (empty($item['image_url'])) {
                $item['image_url'] = 'assets/img/portfolio/portfolio-default.jpg';
            }
            
            // Parse technologies into array if stored as JSON
            if (!empty($item['technologies'])) {
                $technologies = json_decode($item['technologies'], true);
                if (is_array($technologies)) {
                    $item['technologies_array'] = $technologies;
                } else {
                    $item['technologies_array'] = explode(',', $item['technologies']);
                }
            } else {
                $item['technologies_array'] = [];
            }
            
            // Parse features into array if stored as JSON
            if (!empty($item['features'])) {
                $features = json_decode($item['features'], true);
                if (is_array($features)) {
                    $item['features_array'] = $features;
                } else {
                    $item['features_array'] = explode(',', $item['features']);
                }
            } else {
                $item['features_array'] = [];
            }
            
            // Generate category class for filtering
            $item['category_class'] = strtolower(str_replace(' ', '-', $item['category']));
            
            // Generate short description (first 100 characters)
            $item['short_description'] = strlen($item['description']) > 100 
                ? substr($item['description'], 0, 100) . '...' 
                : $item['description'];
        }
        
        return $items;
        
    } catch (Exception $e) {
        error_log("Error fetching portfolio items: " . $e->getMessage());
        return [];
    }
}

/**
 * Get portfolio statistics
 * @return array Portfolio statistics
 */
function getPortfolioStats() {
    try {
        if (!function_exists('getConnection')) {
            require_once __DIR__ . '/database.php';
        }
        
        $pdo = getConnection();
        
        // Get total projects
        $stmt = $pdo->query("SELECT COUNT(*) as total FROM portfolio_items WHERE status = 'active'");
        $totalProjects = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
        
        // Get projects by category
        $stmt = $pdo->query("SELECT category, COUNT(*) as count FROM portfolio_items WHERE status = 'active' GROUP BY category");
        $categoryStats = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get recent projects (last 6 months)
        $stmt = $pdo->query("SELECT COUNT(*) as recent FROM portfolio_items WHERE status = 'active' AND created_at >= DATE_SUB(NOW(), INTERVAL 6 MONTH)");
        $recentProjects = $stmt->fetch(PDO::FETCH_ASSOC)['recent'];
        
        return [
            'total_projects' => $totalProjects,
            'category_stats' => $categoryStats,
            'recent_projects' => $recentProjects
        ];
        
    } catch (Exception $e) {
        error_log("Error fetching portfolio stats: " . $e->getMessage());
        return [
            'total_projects' => 0,
            'category_stats' => [],
            'recent_projects' => 0
        ];
    }
}

/**
 * Get portfolio items by category
 * @param string $category Category name
 * @return array Array of portfolio items in the specified category
 */
function getPortfolioItemsByCategory($category) {
    try {
        if (!function_exists('getConnection')) {
            require_once __DIR__ . '/database.php';
        }
        
        $pdo = getConnection();
        $stmt = $pdo->prepare("SELECT * FROM portfolio_items WHERE category = ? AND status = 'active' ORDER BY sort_order ASC, created_at DESC");
        $stmt->execute([$category]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
        
    } catch (Exception $e) {
        error_log("Error fetching portfolio items by category: " . $e->getMessage());
        return [];
    }
}

/**
 * Get featured portfolio items
 * @param int $limit Number of items to return
 * @return array Array of featured portfolio items
 */
function getFeaturedPortfolioItems($limit = 6) {
    try {
        if (!function_exists('getConnection')) {
            require_once __DIR__ . '/database.php';
        }
        
        $pdo = getConnection();
        $stmt = $pdo->prepare("SELECT * FROM portfolio_items WHERE status = 'active' ORDER BY sort_order ASC, created_at DESC LIMIT ?");
        $stmt->execute([$limit]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
        
    } catch (Exception $e) {
        error_log("Error fetching featured portfolio items: " . $e->getMessage());
        return [];
    }
}

/**
 * Get portfolio item by ID
 * @param int $id Portfolio item ID
 * @return array|null Portfolio item data or null if not found
 */
function getPortfolioItemById($id) {
    try {
        if (!function_exists('getConnection')) {
            require_once __DIR__ . '/database.php';
        }
        
        $pdo = getConnection();
        $stmt = $pdo->prepare("SELECT * FROM portfolio_items WHERE id = ? AND status = 'active'");
        $stmt->execute([$id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
        
    } catch (Exception $e) {
        error_log("Error fetching portfolio item by ID: " . $e->getMessage());
        return null;
    }
}

/**
 * Get all unique categories
 * @return array Array of unique categories
 */
function getPortfolioCategories() {
    try {
        if (!function_exists('getConnection')) {
            require_once __DIR__ . '/database.php';
        }
        
        $pdo = getConnection();
        $stmt = $pdo->query("SELECT DISTINCT category FROM portfolio_items WHERE status = 'active' ORDER BY category ASC");
        return $stmt->fetchAll(PDO::FETCH_COLUMN);
        
    } catch (Exception $e) {
        error_log("Error fetching portfolio categories: " . $e->getMessage());
        return [];
    }
}
?>
