<?php
/**
 * SEO Configuration for Omateq Digitals
 * This file contains SEO-related functions and configurations
 */

// Prevent direct access
if (!defined('ABSPATH') && !defined('INCLUDED')) {
    die('Direct access not allowed');
}

/**
 * Generate SEO meta tags for a page
 * @param string $page_title - The page title
 * @param string $page_description - The page description
 * @param string $page_keywords - The page keywords
 * @param string $page_type - The page type (website, article, etc.)
 * @param string $og_image - Open Graph image
 * @return string - HTML meta tags
 */
function generateSEOMetaTags($page_title, $page_description, $page_keywords, $page_type = 'website', $og_image = 'og-image.jpg') {
    $base_url = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'];
    $current_url = $base_url . $_SERVER['REQUEST_URI'];
    $og_image_url = $base_url . '/assets/img/' . $og_image;
    
    $meta_tags = '
    <!-- SEO Meta Tags -->
    <title>' . htmlspecialchars($page_title) . '</title>
    <meta name="description" content="' . htmlspecialchars($page_description) . '">
    <meta name="keywords" content="' . htmlspecialchars($page_keywords) . '">
    <meta name="author" content="Omateq Digitals">
    <meta name="robots" content="index, follow">
    <meta name="language" content="English">
    <meta name="revisit-after" content="7 days">
    
    <!-- Open Graph Meta Tags -->
    <meta property="og:title" content="' . htmlspecialchars($page_title) . '">
    <meta property="og:description" content="' . htmlspecialchars($page_description) . '">
    <meta property="og:type" content="' . $page_type . '">
    <meta property="og:url" content="' . $current_url . '">
    <meta property="og:image" content="' . $og_image_url . '">
    <meta property="og:image:width" content="1200">
    <meta property="og:image:height" content="630">
    <meta property="og:site_name" content="Omateq Digitals">
    <meta property="og:locale" content="en_NG">
    
    <!-- Twitter Card Meta Tags -->
    <meta name="twitter:card" content="summary_large_image">
    <meta name="twitter:title" content="' . htmlspecialchars($page_title) . '">
    <meta name="twitter:description" content="' . htmlspecialchars($page_description) . '">
    <meta name="twitter:image" content="' . $og_image_url . '">
    
    <!-- Canonical URL -->
    <link rel="canonical" href="' . $current_url . '">
    
    <!-- Geo Tags -->
    <meta name="geo.region" content="NG">
    <meta name="geo.placename" content="Suleja, Niger State, Nigeria">
    <meta name="geo.position" content="9.1689706;7.2096018">
    <meta name="ICBM" content="9.1689706, 7.2096018">
    
    <!-- Business Contact Info -->
    <meta name="contact" content="info@omateqdigitals.com">
    <meta name="phone" content="+2347069378861">
    <meta name="address" content="Suite 8, Jehovah Bueze Plaza, Opp. Berger Paint, Suleja, Niger State, Nigeria">';
    
    return $meta_tags;
}

/**
 * Generate Organization structured data
 * @return string - JSON-LD structured data
 */
function generateOrganizationSchema() {
    $base_url = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'];
    
    return '
    <script type="application/ld+json">
    {
        "@context": "https://schema.org",
        "@type": "Organization",
        "name": "Omateq Digitals",
        "alternateName": "Omateq Digitals Limited",
        "url": "' . $base_url . '",
        "logo": "' . $base_url . '/assets/img/logo.png",
        "description": "Leading web development company in Nigeria specializing in school portals, inventory software, custom web applications, and digital skills training.",
        "foundingDate": "2018",
        "founder": {
            "@type": "Person",
            "name": "David Omateq"
        },
        "address": {
            "@type": "PostalAddress",
            "streetAddress": "Suite 8, Jehovah Bueze Plaza, Opp. Berger Paint",
            "addressLocality": "Suleja",
            "addressRegion": "Niger State",
            "addressCountry": "NG"
        },
        "contactPoint": {
            "@type": "ContactPoint",
            "telephone": "+2347069378861",
            "contactType": "customer service",
            "email": "info@omateqdigitals.com",
            "availableLanguage": "English"
        },
        "sameAs": [
            "https://facebook.com/omateqdigitals",
            "https://youtube.com/@learnwithomateq",
            "https://www.linkedin.com/company/omateqdigitals"
        ],
        "aggregateRating": {
            "@type": "AggregateRating",
            "ratingValue": "4.8",
            "reviewCount": "200"
        }
    }
    </script>';
}

/**
 * Generate Local Business structured data
 * @return string - JSON-LD structured data
 */
function generateLocalBusinessSchema() {
    $base_url = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'];
    
    return '
    <script type="application/ld+json">
    {
        "@context": "https://schema.org",
        "@type": "LocalBusiness",
        "@id": "' . $base_url . '/#business",
        "name": "Omateq Digitals",
        "image": "' . $base_url . '/assets/img/logo.png",
        "telephone": "+2347069378861",
        "email": "info@omateqdigitals.com",
        "address": {
            "@type": "PostalAddress",
            "streetAddress": "Suite 8, Jehovah Bueze Plaza, Opp. Berger Paint",
            "addressLocality": "Suleja",
            "addressRegion": "Niger State",
            "postalCode": "923101",
            "addressCountry": "NG"
        },
        "geo": {
            "@type": "GeoCoordinates",
            "latitude": "9.1689706",
            "longitude": "7.2096018"
        },
        "url": "' . $base_url . '",
        "openingHours": "Mo-Fr 09:00-18:00",
        "priceRange": "$$"
    }
    </script>';
}

/**
 * Generate Breadcrumb structured data
 * @param array $breadcrumbs - Array of breadcrumb items
 * @return string - JSON-LD structured data
 */
function generateBreadcrumbSchema($breadcrumbs) {
    $base_url = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'];
    
    $items = [];
    $position = 1;
    
    foreach ($breadcrumbs as $breadcrumb) {
        $items[] = '{
            "@type": "ListItem",
            "position": ' . $position . ',
            "name": "' . htmlspecialchars($breadcrumb['name']) . '",
            "item": "' . $base_url . $breadcrumb['url'] . '"
        }';
        $position++;
    }
    
    return '
    <script type="application/ld+json">
    {
        "@context": "https://schema.org",
        "@type": "BreadcrumbList",
        "itemListElement": [' . implode(',', $items) . ']
    }
    </script>';
}

/**
 * Generate FAQ structured data
 * @param array $faqs - Array of FAQ items
 * @return string - JSON-LD structured data
 */
function generateFAQSchema($faqs) {
    $items = [];
    
    foreach ($faqs as $faq) {
        $items[] = '{
            "@type": "Question",
            "name": "' . htmlspecialchars($faq['question']) . '",
            "acceptedAnswer": {
                "@type": "Answer",
                "text": "' . htmlspecialchars($faq['answer']) . '"
            }
        }';
    }
    
    return '
    <script type="application/ld+json">
    {
        "@context": "https://schema.org",
        "@type": "FAQPage",
        "mainEntity": [' . implode(',', $items) . ']
    }
    </script>';
}

/**
 * Generate Course structured data
 * @param array $course - Course data
 * @return string - JSON-LD structured data
 */
function generateCourseSchema($course) {
    $base_url = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'];
    
    return '
    <script type="application/ld+json">
    {
        "@context": "https://schema.org",
        "@type": "Course",
        "name": "' . htmlspecialchars($course['title']) . '",
        "description": "' . htmlspecialchars($course['description']) . '",
        "provider": {
            "@type": "Organization",
            "name": "Omateq Digitals",
            "url": "' . $base_url . '"
        },
        "courseMode": "online",
        "educationalLevel": "' . htmlspecialchars($course['level']) . '",
        "timeRequired": "' . htmlspecialchars($course['duration']) . '",
        "offers": {
            "@type": "Offer",
            "price": "' . $course['price'] . '",
            "priceCurrency": "USD"
        }
    }
    </script>';
}

/**
 * Generate Product structured data
 * @param array $product - Product data
 * @return string - JSON-LD structured data
 */
function generateProductSchema($product) {
    $base_url = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'];
    
    return '
    <script type="application/ld+json">
    {
        "@context": "https://schema.org",
        "@type": "SoftwareApplication",
        "name": "' . htmlspecialchars($product['product_name']) . '",
        "description": "' . htmlspecialchars($product['product_description']) . '",
        "applicationCategory": "BusinessApplication",
        "operatingSystem": "Web Browser",
        "offers": {
            "@type": "Offer",
            "price": "' . $product['product_amount'] . '",
            "priceCurrency": "USD"
        },
        "provider": {
            "@type": "Organization",
            "name": "Omateq Digitals",
            "url": "' . $base_url . '"
        }
    }
    </script>';
}

/**
 * Get SEO-friendly URL slug
 * @param string $text - Text to convert to slug
 * @return string - SEO-friendly slug
 */
function generateSlug($text) {
    $text = strtolower($text);
    $text = preg_replace('/[^a-z0-9\s-]/', '', $text);
    $text = preg_replace('/[\s-]+/', '-', $text);
    $text = trim($text, '-');
    return $text;
}

/**
 * Generate meta robots tag based on page type
 * @param string $page_type - Type of page (index, noindex, follow, nofollow)
 * @return string - Meta robots tag
 */
function generateRobotsTag($page_type = 'index,follow') {
    return '<meta name="robots" content="' . $page_type . '">';
}

/**
 * Generate hreflang tags for multilingual support
 * @param array $languages - Array of language codes and URLs
 * @return string - Hreflang tags
 */
function generateHreflangTags($languages) {
    $tags = '';
    foreach ($languages as $lang => $url) {
        $tags .= '<link rel="alternate" hreflang="' . $lang . '" href="' . $url . '">' . "\n    ";
    }
    return $tags;
}

// Define common SEO constants
define('SEO_SITE_NAME', 'Omateq Digitals');
define('SEO_DEFAULT_DESCRIPTION', 'Leading web development company in Nigeria specializing in school portals, inventory software, custom web applications, and digital skills training.');
define('SEO_DEFAULT_KEYWORDS', 'web development Nigeria, school portal development, inventory software, custom web applications, digital skills training, software development Abuja, IT solutions Nigeria');
define('SEO_COMPANY_PHONE', '+2347069378861');
define('SEO_COMPANY_EMAIL', 'info@omateqdigitals.com');
define('SEO_COMPANY_ADDRESS', 'Suite 8, Jehovah Bueze Plaza, Opp. Berger Paint, Suleja, Niger State, Nigeria');
define('SEO_COMPANY_LATITUDE', '9.1689706');
define('SEO_COMPANY_LONGITUDE', '7.2096018');

?>
