<?php
/**
 * Session Cleanup Utility
 * Cleans up expired sessions and provides session management functions
 */

require_once 'database.php';
require_once 'session_manager.php';

class SessionCleanup {
    
    /**
     * Clean up expired sessions from the database
     */
    public static function cleanupExpiredSessions() {
        try {
            $pdo = getConnection();
            
            // Clean up expired password reset tokens
            $stmt = $pdo->prepare("UPDATE admin_users SET reset_token = NULL, reset_token_expires = NULL WHERE reset_token_expires < NOW()");
            $stmt->execute();
            
            // Clean up expired user sessions (if you have a sessions table)
            // $stmt = $pdo->prepare("DELETE FROM user_sessions WHERE expires_at < NOW()");
            // $stmt->execute();
            
            return true;
        } catch (Exception $e) {
            error_log("Session cleanup failed: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get session statistics
     */
    public static function getSessionStats() {
        try {
            $pdo = getConnection();
            
            $stats = [];
            
            // Count active admin sessions (users with recent activity)
            $stmt = $pdo->query("SELECT COUNT(*) as count FROM admin_users WHERE last_activity > DATE_SUB(NOW(), INTERVAL 1 HOUR)");
            $stats['active_admin_sessions'] = $stmt->fetch()['count'];
            
            // Count active user sessions (users with recent activity)
            $stmt = $pdo->query("SELECT COUNT(*) as count FROM users WHERE last_activity > DATE_SUB(NOW(), INTERVAL 1 HOUR)");
            $stats['active_user_sessions'] = $stmt->fetch()['count'];
            
            // Count expired reset tokens
            $stmt = $pdo->query("SELECT COUNT(*) as count FROM admin_users WHERE reset_token IS NOT NULL AND reset_token_expires < NOW()");
            $stats['expired_reset_tokens'] = $stmt->fetch()['count'];
            
            return $stats;
        } catch (Exception $e) {
            error_log("Session stats failed: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Force logout all users (emergency function)
     */
    public static function forceLogoutAllUsers() {
        try {
            $pdo = getConnection();
            
            // Clear all reset tokens
            $stmt = $pdo->prepare("UPDATE admin_users SET reset_token = NULL, reset_token_expires = NULL");
            $stmt->execute();
            
            // Clear all user sessions (if you have a sessions table)
            // $stmt = $pdo->prepare("DELETE FROM user_sessions");
            // $stmt->execute();
            
            return true;
        } catch (Exception $e) {
            error_log("Force logout failed: " . $e->getMessage());
            return false;
        }
    }
}

// CLI usage for cron jobs
if (php_sapi_name() === 'cli') {
    echo "Starting session cleanup...\n";
    
    $result = SessionCleanup::cleanupExpiredSessions();
    if ($result) {
        echo "Session cleanup completed successfully.\n";
        
        $stats = SessionCleanup::getSessionStats();
        if ($stats) {
            echo "Session Statistics:\n";
            echo "- Active Admin Sessions: " . $stats['active_admin_sessions'] . "\n";
            echo "- Active User Sessions: " . $stats['active_user_sessions'] . "\n";
            echo "- Expired Reset Tokens: " . $stats['expired_reset_tokens'] . "\n";
        }
    } else {
        echo "Session cleanup failed.\n";
        exit(1);
    }
}
?>
