<?php
/**
 * Session Management Utility
 * Handles session timeout and cleanup for both admin and user areas
 */

class SessionManager {
    private static $timeout_duration = 60; // 1 minute timeout (60 seconds)
    private static $warning_duration = 30; // Warning at 30 seconds
    
    /**
     * Initialize session with timeout tracking
     */
    public static function initSession() {
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
        
        // Set initial last activity time if not set
        if (!isset($_SESSION['last_activity'])) {
            $_SESSION['last_activity'] = time();
        }
        
        // Check for session timeout
        self::checkSessionTimeout();
    }
    
    /**
     * Update last activity timestamp
     */
    public static function updateActivity() {
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
        $_SESSION['last_activity'] = time();
    }
    
    /**
     * Check if session has timed out
     */
    public static function checkSessionTimeout() {
        if (!isset($_SESSION['last_activity'])) {
            return false;
        }
        
        $current_time = time();
        $last_activity = $_SESSION['last_activity'];
        $inactive_time = $current_time - $last_activity;
        
        if ($inactive_time > self::$timeout_duration) {
            self::destroySession();
            return true;
        }
        
        return false;
    }
    
    /**
     * Get remaining session time in seconds
     */
    public static function getRemainingTime() {
        if (!isset($_SESSION['last_activity'])) {
            return 0;
        }
        
        $current_time = time();
        $last_activity = $_SESSION['last_activity'];
        $inactive_time = $current_time - $last_activity;
        $remaining_time = self::$timeout_duration - $inactive_time;
        
        return max(0, $remaining_time);
    }
    
    /**
     * Check if session is about to expire (warning period)
     */
    public static function isSessionExpiringSoon() {
        $remaining_time = self::getRemainingTime();
        return $remaining_time <= self::$warning_duration && $remaining_time > 0;
    }
    
    /**
     * Destroy session and redirect to login
     */
    public static function destroySession() {
        if (session_status() === PHP_SESSION_ACTIVE) {
            // Clear session data
            $_SESSION = array();
            
            // Destroy session cookie
            if (ini_get("session.use_cookies")) {
                $params = session_get_cookie_params();
                setcookie(session_name(), '', time() - 42000,
                    $params["path"], $params["domain"],
                    $params["secure"], $params["httponly"]
                );
            }
            
            // Destroy session
            session_destroy();
        }
    }
    
    /**
     * Logout user and redirect
     */
    public static function logout($redirect_url = null) {
        self::destroySession();
        
        if ($redirect_url) {
            header("Location: " . $redirect_url);
            exit();
        }
    }
    
    /**
     * Get session timeout configuration
     */
    public static function getTimeoutDuration() {
        return self::$timeout_duration;
    }
    
    public static function getWarningDuration() {
        return self::$warning_duration;
    }
    
    /**
     * Set custom timeout duration
     */
    public static function setTimeoutDuration($seconds) {
        self::$timeout_duration = $seconds;
    }
    
    /**
     * Check if user is logged in and session is valid
     */
    public static function isLoggedIn($user_type = 'admin') {
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
        
        $session_key = ($user_type === 'admin') ? 'admin_id' : 'user_id';
        
        if (!isset($_SESSION[$session_key]) || !isset($_SESSION['last_activity'])) {
            return false;
        }
        
        // Check for timeout
        if (self::checkSessionTimeout()) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Force logout if session is invalid
     */
    public static function requireLogin($user_type = 'admin', $login_url = null) {
        if (!self::isLoggedIn($user_type)) {
            if (!$login_url) {
                $login_url = ($user_type === 'admin') ? 'authentication/login.php' : '../users/authentication/login.php';
            }
            header("Location: " . $login_url);
            exit();
        }
        
        // Update activity on each page load
        self::updateActivity();
    }
}

/**
 * AJAX endpoint for session check
 */
if (isset($_GET['action']) && $_GET['action'] === 'check_session') {
    header('Content-Type: application/json');
    
    SessionManager::initSession();
    
    $response = [
        'valid' => SessionManager::isLoggedIn($_GET['user_type'] ?? 'admin'),
        'remaining_time' => SessionManager::getRemainingTime(),
        'expiring_soon' => SessionManager::isSessionExpiringSoon()
    ];
    
    echo json_encode($response);
    exit();
}

/**
 * AJAX endpoint for session extension
 */
if (isset($_GET['action']) && $_GET['action'] === 'extend_session') {
    header('Content-Type: application/json');
    
    SessionManager::initSession();
    SessionManager::updateActivity();
    
    $response = [
        'success' => true,
        'remaining_time' => SessionManager::getRemainingTime()
    ];
    
    echo json_encode($response);
    exit();
}
?>
