<?php
/**
 * Settings Helper Functions
 * Provides easy access to system settings across the application
 */

require_once 'database.php';

class SettingsManager {
    private static $settings = null;
    private static $pdo = null;
    
    /**
     * Get database connection
     */
    private static function getConnection() {
        if (self::$pdo === null) {
            self::$pdo = getConnection();
        }
        return self::$pdo;
    }
    
    /**
     * Load all settings from database
     */
    private static function loadSettings() {
        if (self::$settings === null) {
            try {
                $pdo = self::getConnection();
                $stmt = $pdo->query("SELECT setting_key, setting_value, setting_type FROM settings");
                $results = $stmt->fetchAll();
                
                self::$settings = [];
                foreach ($results as $row) {
                    $value = $row['setting_value'];
                    
                    // Convert value based on type
                    switch ($row['setting_type']) {
                        case 'boolean':
                            $value = (bool) $value;
                            break;
                        case 'number':
                            $value = is_numeric($value) ? (float) $value : 0;
                            break;
                        case 'json':
                            $value = json_decode($value, true);
                            break;
                        default:
                            // Keep as string
                            break;
                    }
                    
                    self::$settings[$row['setting_key']] = $value;
                }
            } catch (Exception $e) {
                error_log("Settings loading error: " . $e->getMessage());
                self::$settings = [];
            }
        }
        
        return self::$settings;
    }
    
    /**
     * Get a specific setting value
     */
    public static function get($key, $default = null) {
        $settings = self::loadSettings();
        return isset($settings[$key]) ? $settings[$key] : $default;
    }
    
    /**
     * Get all settings for a specific category
     */
    public static function getCategory($category) {
        try {
            $pdo = self::getConnection();
            $stmt = $pdo->prepare("SELECT setting_key, setting_value, setting_type FROM settings WHERE category = ?");
            $stmt->execute([$category]);
            $results = $stmt->fetchAll();
            
            $categorySettings = [];
            foreach ($results as $row) {
                $value = $row['setting_value'];
                
                // Convert value based on type
                switch ($row['setting_type']) {
                    case 'boolean':
                        $value = (bool) $value;
                        break;
                    case 'number':
                        $value = is_numeric($value) ? (float) $value : 0;
                        break;
                    case 'json':
                        $value = json_decode($value, true);
                        break;
                    default:
                        // Keep as string
                        break;
                }
                
                $categorySettings[$row['setting_key']] = $value;
            }
            
            return $categorySettings;
        } catch (Exception $e) {
            error_log("Category settings loading error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Clear cached settings (call this after updating settings)
     */
    public static function clearCache() {
        self::$settings = null;
    }
    
    /**
     * Get crypto settings
     */
    public static function getCryptoSettings() {
        return [
            'bitcoin_address' => self::get('bitcoin_address', ''),
            'ethereum_address' => self::get('ethereum_address', ''),
            'usdt_address' => self::get('usdt_address', ''),
            'usdc_address' => self::get('usdc_address', ''),
            'instructions' => self::get('crypto_instructions', '')
        ];
    }
    
    /**
     * Get bank settings
     */
    public static function getBankSettings() {
        return [
            'bank_name' => self::get('bank_name', ''),
            'account_name' => self::get('account_name', ''),
            'account_number' => self::get('account_number', ''),
            'bank_code' => self::get('bank_code', ''),
            'instructions' => self::get('bank_instructions', '')
        ];
    }
    
    /**
     * Get Paystack settings
     */
    public static function getPaystackSettings() {
        return [
            'public_key' => self::get('pk_live_899e1eaefbc652e8cd6edcb7f0b93a88076b6e40', ''),
            'secret_key' => self::get('sk_live_608f039297d2402cbc490acdb644023241edc5ac', ''),
            'webhook_secret' => self::get('https://omateqdigitals.com/wc-api/Tbz_WC_Paystack_Webhook/', ''),
            'currency' => self::get('paystack_currency', 'NGN'),
            'instructions' => self::get('paystack_instructions', '')
        ];
    }
    
    /**
     * Get general settings
     */
    public static function getGeneralSettings() {
        return [
            'site_name' => self::get('site_name', 'Omateq Digitals'),
            'site_email' => self::get('site_email', 'info@omateqdigitals.com'),
            'site_phone' => self::get('site_phone', ''),
            'site_address' => self::get('site_address', ''),
            'currency' => self::get('currency', 'USD'),
            'timezone' => self::get('timezone', 'Africa/Lagos')
        ];
    }
    
    /**
     * Get system settings
     */
    public static function getSystemSettings() {
        return [
            'maintenance_mode' => self::get('maintenance_mode', false),
            'registration_enabled' => self::get('registration_enabled', true),
            'email_verification' => self::get('email_verification', true),
            'max_downloads' => self::get('max_downloads', 3),
            'download_expiry_days' => self::get('download_expiry_days', 30)
        ];
    }
    
    /**
     * Get logo settings
     */
    public static function getLogoSettings() {
        return [
            'main_logo' => self::get('main_logo', 'assets/images/logo.svg'),
            'mini_logo' => self::get('mini_logo', 'assets/images/logo-text-white.svg')
        ];
    }
}

// Convenience functions
function getSetting($key, $default = null) {
    return SettingsManager::get($key, $default);
}

function getCryptoSettings() {
    return SettingsManager::getCryptoSettings();
}

function getBankSettings() {
    return SettingsManager::getBankSettings();
}

function getPaystackSettings() {
    return SettingsManager::getPaystackSettings();
}

function getGeneralSettings() {
    return SettingsManager::getGeneralSettings();
}

function getSystemSettings() {
    return SettingsManager::getSystemSettings();
}

function getLogoSettings() {
    return SettingsManager::getLogoSettings();
}
?>
