<?php
require_once 'config/database.php';
require_once 'config/settings.php';
require_once 'config/branding.php';

$error = '';
$success = '';

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: authentication/login.php');
    exit();
}

// Handle course actions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $action = $_POST['action'] ?? '';
    $course_id = $_POST['course_id'] ?? '';
    
    try {
        $pdo = getConnection();
        
        switch ($action) {
            case 'create':
                $title = trim($_POST['course_title'] ?? '');
                $description = trim($_POST['course_description'] ?? '');
                $curriculum = trim($_POST['course_features'] ?? '');
                $instructor = trim($_POST['course_instructor'] ?? '');
                $price = (float)($_POST['course_price'] ?? 0);
                $duration = trim($_POST['course_duration'] ?? '');
                $level = $_POST['level'] ?? 'beginner';
                $status = $_POST['status'] ?? 'active';
                
                if (empty($title) || empty($description) || empty($instructor) || $price <= 0) {
                    $error = "Course title, description, instructor, and price are required";
                } else {
                    // Handle image upload
                    $thumbnail = '';
                    if (isset($_FILES['thumbnail']) && $_FILES['thumbnail']['error'] == 0) {
                        $upload_dir = '../assets/uploads/courses/';
                        if (!is_dir($upload_dir)) {
                            mkdir($upload_dir, 0755, true);
                        }
                        
                        $file_extension = pathinfo($_FILES['thumbnail']['name'], PATHINFO_EXTENSION);
                        $file_name = 'course_' . time() . '_' . uniqid() . '.' . $file_extension;
                        $file_path = $upload_dir . $file_name;
                        
                        if (move_uploaded_file($_FILES['thumbnail']['tmp_name'], $file_path)) {
                            $thumbnail = 'assets/uploads/courses/' . $file_name;
                        }
                    }
                    
                    // Handle course video upload
                    $course_video = '';
                    if (isset($_FILES['course_video']) && $_FILES['course_video']['error'] == 0) {
                        // Validate file type
                        $allowed_extensions = ['mp4', 'webm', 'ogg', 'avi', 'mov'];
                        $allowed_mime_types = [
                            'video/mp4', 'video/webm', 'video/ogg', 'video/avi', 'video/mov',
                            'video/quicktime', 'video/x-msvideo', 'video/x-ms-wmv'
                        ];
                        
                        $file_extension = strtolower(pathinfo($_FILES['course_video']['name'], PATHINFO_EXTENSION));
                        $file_mime_type = $_FILES['course_video']['type'];
                        
                        if (in_array($file_extension, $allowed_extensions) && in_array($file_mime_type, $allowed_mime_types)) {
                            // Validate file size (max 500MB)
                            $max_size = 500 * 1024 * 1024; // 500MB
                            if ($_FILES['course_video']['size'] <= $max_size) {
                                $upload_dir = '../uploads/course_videos/';
                                if (!is_dir($upload_dir)) {
                                    mkdir($upload_dir, 0755, true);
                                }
                                
                                $file_name = 'course_preview_' . time() . '_' . uniqid() . '.' . $file_extension;
                                $file_path = $upload_dir . $file_name;
                                
                                if (move_uploaded_file($_FILES['course_video']['tmp_name'], $file_path)) {
                                    $course_video = $file_name;
                                }
                            } else {
                                $error = "Course video file too large. Maximum size is 500MB";
                            }
                        } else {
                            $error = "Invalid course video file type. Only MP4, WebM, OGG, AVI, and MOV files are allowed";
                        }
                    }
                    
                    $stmt = $pdo->prepare("
                        INSERT INTO courses (title, description, price, duration, level, thumbnail, course_video, curriculum, instructor, status) 
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                    ");
                    $stmt->execute([$title, $description, $price, $duration, $level, $thumbnail, $course_video, $curriculum, $instructor, $status]);
                    $success = "Course created successfully";
                }
                break;
                
            case 'update':
                $title = trim($_POST['course_title'] ?? '');
                $description = trim($_POST['course_description'] ?? '');
                $curriculum = trim($_POST['course_features'] ?? '');
                $instructor = trim($_POST['course_instructor'] ?? '');
                $price = (float)($_POST['course_price'] ?? 0);
                $duration = trim($_POST['course_duration'] ?? '');
                $level = $_POST['level'] ?? 'beginner';
                $status = $_POST['status'] ?? 'active';
                
                if (empty($title) || empty($description) || empty($instructor) || $price <= 0) {
                    $error = "Course title, description, instructor, and price are required";
                } else {
                    // Handle image upload
                    $thumbnail = $_POST['current_thumbnail'] ?? '';
                    if (isset($_FILES['thumbnail']) && $_FILES['thumbnail']['error'] == 0) {
                        $upload_dir = '../assets/uploads/courses/';
                        if (!is_dir($upload_dir)) {
                            mkdir($upload_dir, 0755, true);
                        }
                        
                        $file_extension = pathinfo($_FILES['thumbnail']['name'], PATHINFO_EXTENSION);
                        $file_name = 'course_' . time() . '_' . uniqid() . '.' . $file_extension;
                        $file_path = $upload_dir . $file_name;
                        
                        if (move_uploaded_file($_FILES['thumbnail']['tmp_name'], $file_path)) {
                            $thumbnail = 'assets/uploads/courses/' . $file_name;
                        }
                    }
                    
                    // Handle course video upload
                    $current_course = $pdo->prepare("SELECT course_video FROM courses WHERE id = ?");
                    $current_course->execute([$course_id]);
                    $currentCourseData = $current_course->fetch();
                    $course_video = $currentCourseData['course_video'] ?? '';
                    
                    if (isset($_FILES['course_video']) && $_FILES['course_video']['error'] == 0) {
                        // Validate file type
                        $allowed_extensions = ['mp4', 'webm', 'ogg', 'avi', 'mov'];
                        $allowed_mime_types = [
                            'video/mp4', 'video/webm', 'video/ogg', 'video/avi', 'video/mov',
                            'video/quicktime', 'video/x-msvideo', 'video/x-ms-wmv'
                        ];
                        
                        $file_extension = strtolower(pathinfo($_FILES['course_video']['name'], PATHINFO_EXTENSION));
                        $file_mime_type = $_FILES['course_video']['type'];
                        
                        if (in_array($file_extension, $allowed_extensions) && in_array($file_mime_type, $allowed_mime_types)) {
                            // Validate file size (max 500MB)
                            $max_size = 500 * 1024 * 1024; // 500MB
                            if ($_FILES['course_video']['size'] <= $max_size) {
                                $upload_dir = '../uploads/course_videos/';
                                if (!is_dir($upload_dir)) {
                                    mkdir($upload_dir, 0755, true);
                                }
                                
                                $file_name = 'course_preview_' . time() . '_' . uniqid() . '.' . $file_extension;
                                $file_path = $upload_dir . $file_name;
                                
                                if (move_uploaded_file($_FILES['course_video']['tmp_name'], $file_path)) {
                                    // Delete old course video if exists
                                    if (!empty($currentCourseData['course_video'])) {
                                        $old_video_path = '../uploads/course_videos/' . $currentCourseData['course_video'];
                                        if (file_exists($old_video_path)) {
                                            unlink($old_video_path);
                                        }
                                    }
                                    $course_video = $file_name;
                                }
                            } else {
                                $error = "Course video file too large. Maximum size is 500MB";
                            }
                        } else {
                            $error = "Invalid course video file type. Only MP4, WebM, OGG, AVI, and MOV files are allowed";
                        }
                    }
                    
                    $stmt = $pdo->prepare("
                        UPDATE courses 
                        SET title = ?, description = ?, price = ?, duration = ?, level = ?, thumbnail = ?, course_video = ?, curriculum = ?, instructor = ?, status = ?, updated_at = NOW()
                        WHERE id = ?
                    ");
                    $stmt->execute([$title, $description, $price, $duration, $level, $thumbnail, $course_video, $curriculum, $instructor, $status, $course_id]);
                    $success = "Course updated successfully";
                }
                break;
                
            case 'delete':
                $stmt = $pdo->prepare("DELETE FROM courses WHERE id = ?");
                $stmt->execute([$course_id]);
                $success = "Course deleted successfully";
                break;
                
            case 'toggle_status':
                $stmt = $pdo->prepare("UPDATE courses SET status = CASE WHEN status = 'active' THEN 'inactive' ELSE 'active' END WHERE id = ?");
                $stmt->execute([$course_id]);
                $success = "Course status updated successfully";
                break;
                
            case 'add_lesson':
                $course_id = $_POST['course_id'] ?? '';
                $lesson_title = trim($_POST['lesson_title'] ?? '');
                $lesson_description = trim($_POST['lesson_description'] ?? '');
                $duration = trim($_POST['duration'] ?? '');
                $lesson_number = (int)($_POST['lesson_number'] ?? 1);
                
                if (empty($lesson_title)) {
                    $error = "Lesson title is required";
                } else {
                    // Handle video upload
                    $video_file = '';
                    $video_type = 'youtube'; // Default
                    $video_url = '';
                    
                    if (isset($_FILES['lesson_video']) && $_FILES['lesson_video']['error'] == 0) {
                        // Validate file type by extension and MIME type
                        $allowed_extensions = ['mp4', 'webm', 'ogg', 'avi', 'mov'];
                        $allowed_mime_types = [
                            'video/mp4', 'video/webm', 'video/ogg', 'video/avi', 'video/mov',
                            'video/quicktime', 'video/x-msvideo', 'video/x-ms-wmv'
                        ];
                        
                        $file_extension = strtolower(pathinfo($_FILES['lesson_video']['name'], PATHINFO_EXTENSION));
                        $file_mime_type = $_FILES['lesson_video']['type'];
                        
                        if (!in_array($file_extension, $allowed_extensions) || !in_array($file_mime_type, $allowed_mime_types)) {
                            $error = "Invalid file type. Only MP4, WebM, OGG, AVI, and MOV files are allowed";
                        } else {
                            // Validate file size (max 500MB)
                            $max_size = 500 * 1024 * 1024; // 500MB
                            if ($_FILES['lesson_video']['size'] > $max_size) {
                                $error = "File too large. Maximum size is 500MB";
                            } else {
                                // Create upload directory
                                $upload_dir = '../uploads/course_videos/';
                                if (!is_dir($upload_dir)) {
                                    mkdir($upload_dir, 0755, true);
                                }
                                
                                // Generate unique filename
                                $file_extension = pathinfo($_FILES['lesson_video']['name'], PATHINFO_EXTENSION);
                                $filename = 'lesson_' . $course_id . '_' . time() . '_' . uniqid() . '.' . $file_extension;
                                $file_path = $upload_dir . $filename;
                                
                                if (move_uploaded_file($_FILES['lesson_video']['tmp_name'], $file_path)) {
                                    $video_file = $filename;
                                    $video_type = 'upload';
                                    
                                    // Get video duration using FFmpeg if available
                                    if (function_exists('shell_exec')) {
                                        $command = "ffprobe -v quiet -show_entries format=duration -of csv=p=0 " . escapeshellarg($file_path) . " 2>/dev/null";
                                        $output = shell_exec($command);
                                        if ($output && is_numeric(trim($output))) {
                                            $seconds = (int)trim($output);
                                            $minutes = floor($seconds / 60);
                                            $remaining_seconds = $seconds % 60;
                                            $duration = sprintf('%d:%02d', $minutes, $remaining_seconds);
                                        }
                                    }
                                } else {
                                    $error = "Failed to upload video file";
                                }
                            }
                        }
                    } else {
                        $error = "Video file is required";
                    }
                    
                    if (empty($error)) {
                        $stmt = $pdo->prepare("
                            INSERT INTO course_lessons (course_id, lesson_number, lesson_title, lesson_description, video_url, video_file, video_type, duration) 
                            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
                        ");
                        $stmt->execute([$course_id, $lesson_number, $lesson_title, $lesson_description, $video_url, $video_file, $video_type, $duration]);
                        $success = "Lesson added successfully";
                    }
                }
                break;
                
            case 'update_lesson':
                $lesson_id = $_POST['lesson_id'] ?? '';
                $lesson_title = trim($_POST['lesson_title'] ?? '');
                $lesson_description = trim($_POST['lesson_description'] ?? '');
                $duration = trim($_POST['duration'] ?? '');
                $lesson_number = (int)($_POST['lesson_number'] ?? 1);
                $is_active = $_POST['is_active'] ?? 'yes';
                
                if (empty($lesson_title)) {
                    $error = "Lesson title is required";
                } else {
                    // Get current lesson data
                    $currentStmt = $pdo->prepare("SELECT video_file, video_type FROM course_lessons WHERE id = ?");
                    $currentStmt->execute([$lesson_id]);
                    $currentLesson = $currentStmt->fetch(PDO::FETCH_ASSOC);
                    
                    $video_file = $currentLesson['video_file'] ?? '';
                    $video_type = $currentLesson['video_type'] ?? 'youtube';
                    $video_url = '';
                    
                    if (isset($_FILES['lesson_video']) && $_FILES['lesson_video']['error'] == 0) {
                        // Validate file type by extension and MIME type
                        $allowed_extensions = ['mp4', 'webm', 'ogg', 'avi', 'mov'];
                        $allowed_mime_types = [
                            'video/mp4', 'video/webm', 'video/ogg', 'video/avi', 'video/mov',
                            'video/quicktime', 'video/x-msvideo', 'video/x-ms-wmv'
                        ];
                        
                        $file_extension = strtolower(pathinfo($_FILES['lesson_video']['name'], PATHINFO_EXTENSION));
                        $file_mime_type = $_FILES['lesson_video']['type'];
                        
                        if (!in_array($file_extension, $allowed_extensions) || !in_array($file_mime_type, $allowed_mime_types)) {
                            $error = "Invalid file type. Only MP4, WebM, OGG, AVI, and MOV files are allowed";
                        } else {
                            // Validate file size (max 500MB)
                            $max_size = 500 * 1024 * 1024; // 500MB
                            if ($_FILES['lesson_video']['size'] > $max_size) {
                                $error = "File too large. Maximum size is 500MB";
                            } else {
                                // Create upload directory
                                $upload_dir = '../uploads/course_videos/';
                                if (!is_dir($upload_dir)) {
                                    mkdir($upload_dir, 0755, true);
                                }
                                
                                // Delete old video file if exists
                                if (!empty($video_file) && file_exists($upload_dir . $video_file)) {
                                    unlink($upload_dir . $video_file);
                                }
                                
                                // Generate unique filename
                                $file_extension = pathinfo($_FILES['lesson_video']['name'], PATHINFO_EXTENSION);
                                $filename = 'lesson_' . $lesson_id . '_' . time() . '_' . uniqid() . '.' . $file_extension;
                                $file_path = $upload_dir . $filename;
                                
                                if (move_uploaded_file($_FILES['lesson_video']['tmp_name'], $file_path)) {
                                    $video_file = $filename;
                                    $video_type = 'upload';
                                    $video_url = ''; // Clear YouTube URL when uploading file
                                    
                                    // Get video duration using FFmpeg if available
                                    if (function_exists('shell_exec')) {
                                        $command = "ffprobe -v quiet -show_entries format=duration -of csv=p=0 " . escapeshellarg($file_path) . " 2>/dev/null";
                                        $output = shell_exec($command);
                                        if ($output && is_numeric(trim($output))) {
                                            $seconds = (int)trim($output);
                                            $minutes = floor($seconds / 60);
                                            $remaining_seconds = $seconds % 60;
                                            $duration = sprintf('%d:%02d', $minutes, $remaining_seconds);
                                        }
                                    }
                                } else {
                                    $error = "Failed to upload video file";
                                }
                            }
                        }
                    } else {
                        // Keep existing video if no new file uploaded
                        // Keep current video_file and video_type values
                    }
                    
                    if (empty($error)) {
                        $stmt = $pdo->prepare("
                            UPDATE course_lessons 
                            SET lesson_number = ?, lesson_title = ?, lesson_description = ?, video_url = ?, video_file = ?, video_type = ?, duration = ?, is_active = ?, updated_at = NOW()
                            WHERE id = ?
                        ");
                        $stmt->execute([$lesson_number, $lesson_title, $lesson_description, $video_url, $video_file, $video_type, $duration, $is_active, $lesson_id]);
                        $success = "Lesson updated successfully";
                    }
                }
                break;
                
            case 'delete_lesson':
                $lesson_id = $_POST['lesson_id'] ?? '';
                $stmt = $pdo->prepare("DELETE FROM course_lessons WHERE id = ?");
                $stmt->execute([$lesson_id]);
                $success = "Lesson deleted successfully";
                break;
                
            case 'toggle_lesson_status':
                $lesson_id = $_POST['lesson_id'] ?? '';
                $stmt = $pdo->prepare("UPDATE course_lessons SET is_active = CASE WHEN is_active = 'yes' THEN 'no' ELSE 'yes' END WHERE id = ?");
                $stmt->execute([$lesson_id]);
                $success = "Lesson status updated successfully";
                break;
        }
    } catch (Exception $e) {
        $error = "Action failed: " . $e->getMessage();
    }
}

// Get courses with pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 9; // 3x3 grid for course cards
$offset = ($page - 1) * $limit;

try {
    $pdo = getConnection();
    
    // Get total count
    $countStmt = $pdo->query("SELECT COUNT(*) FROM courses");
    $totalCourses = $countStmt->fetchColumn();
    $totalPages = ceil($totalCourses / $limit);
    
    // Get courses
    $stmt = $pdo->prepare("
        SELECT c.*, 
               COUNT(ce.id) as enrollment_count
        FROM courses c
        LEFT JOIN course_enrollments ce ON c.id = ce.course_id
        GROUP BY c.id
        ORDER BY c.created_at DESC 
        LIMIT :limit OFFSET :offset
    ");
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $courses = $stmt->fetchAll();
    
    // Get course statistics
    $activeCourses = $pdo->query("SELECT COUNT(*) FROM courses WHERE status = 'active'")->fetchColumn();
    $inactiveCourses = $pdo->query("SELECT COUNT(*) FROM courses WHERE status = 'inactive'")->fetchColumn();
    
    // Get enrollment statistics
    $totalEnrollments = $pdo->query("SELECT COUNT(*) FROM course_enrollments")->fetchColumn();
    $completedCourses = $pdo->query("SELECT COUNT(*) FROM course_enrollments WHERE status = 'completed'")->fetchColumn();
    
    // Calculate total revenue from course enrollments
    $revenueStmt = $pdo->query("SELECT SUM(payment_amount) FROM course_enrollments WHERE payment_status = 'paid'");
    $totalRevenue = $revenueStmt->fetchColumn() ?: 0;
    
} catch (Exception $e) {
    $error = "Database error: " . $e->getMessage();
    $courses = [];
    $totalPages = 0;
    $totalCourses = 0;
    $activeCourses = 0;
    $inactiveCourses = 0;
    $totalEnrollments = 0;
    $completedCourses = 0;
    $totalRevenue = 0;
}
?>
<!DOCTYPE html>
<html lang="en">

<head>

  <base href="">

  <!-- begin::Edura Meta Basic -->
  <meta charset="utf-8">
  <meta name="theme-color" content="#0D9488">
  <meta name="robots" content="index, follow">
  <meta name="author" content="Omateq Digitals">
  <meta name="format-detection" content="telephone=no">
  <meta name="keywords" content="Admin Dashboard, Omateq Digitals, Course Management, Admin Panel, Online Learning">
  <meta name="description" content="Omateq Digitals Admin Dashboard - Manage courses, create new courses, and update course information for frontend display.">
  <!-- end::Edura Meta Basic -->

  <!-- begin::Edura Meta Social -->
  <meta property="og:url" content="">
  <meta property="og:site_name" content="Manage Courses | Omateq Digitals">
  <meta property="og:type" content="website">
  <meta property="og:locale" content="en_us">
  <meta property="og:title" content="Manage Courses | Omateq Digitals">
  <meta property="og:description" content="Omateq Digitals Admin Dashboard - Manage courses, create new courses, and update course information for frontend display.">
  <!-- end::Edura Meta Social -->

  <!-- begin::Edura Meta Twitter -->
  <meta name="twitter:card" content="summary">
  <meta name="twitter:url" content="">
  <meta name="twitter:creator" content="@omateqdigitals">
  <meta name="twitter:title" content="Manage Courses | Omateq Digitals">
  <meta name="twitter:description" content="Omateq Digitals Admin Dashboard - Manage courses, create new courses, and update course information for frontend display.">
  <!-- end::Edura Meta Twitter -->

  <!-- begin::Edura Website Page Title -->
  <title>Manage Courses | Omateq Digitals</title>
  <!-- end::Edura Website Page Title -->

  <!-- begin::Edura Mobile Specific -->
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <!-- end::Edura Mobile Specific -->

  <!-- begin::Edura Favicon Tags -->
  <?php echo getFaviconHtml(); ?>
  <!-- end::Edura Favicon Tags -->

  <!-- begin::Edura Google Fonts -->
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Plus+Jakarta+Sans:ital,wght@0,200..800;1,200..800&display=swap" rel="stylesheet">
  <!-- end::Edura Google Fonts -->

  <!-- begin::Edura Required Stylesheet -->
  <link rel="stylesheet" href="assets/libs/flaticon/css/all/all.css">
  <link rel="stylesheet" href="assets/libs/lucide/lucide.css">
  <link rel="stylesheet" href="assets/libs/fontawesome/css/all.min.css">
  <link rel="stylesheet" href="assets/libs/simplebar/simplebar.css">
  <link rel="stylesheet" href="assets/libs/node-waves/waves.css">
  <link rel="stylesheet" href="assets/libs/bootstrap-select/css/bootstrap-select.min.css">
  <!-- end::Edura Required Stylesheet -->

  <!-- begin::Edura CSS Stylesheet -->
  <link rel="stylesheet" href="assets/css/styles.css">
  <!-- end::Edura CSS Stylesheet -->

  <!-- begin::Edura Googletagmanager -->
  <script async src="https://www.googletagmanager.com/gtag/js?id=G-XWVQM68HHQ"></script>
  <script>
		window.dataLayer = window.dataLayer || [];
		function gtag(){dataLayer.push(arguments);}
		gtag('js', new Date());

		gtag('config', 'G-XWVQM68HHQ', {
			'cookie_flags': 'SameSite=None;Secure',
			'send_page_view': true
		});
	</script>
  <!-- end::Edura Googletagmanager -->

</head>

<body>
  <div class="page-layout">

    <?php include 'includes/header.php'; ?>

     <!-- begin::Edura Sidebar Menu -->
     <aside class="app-menubar" id="menubar">
      <div class="app-navbar-brand">
        <a class="navbar-brand-logo" href="index.php">
          <img src="../<?php echo getSetting('main_logo', 'assets/images/logo.svg'); ?>" alt="logo">
        </a>
        <a class="navbar-brand-mini" href="index.php">
          <img src="../<?php echo getSetting('mini_logo', 'assets/images/logo-text-white.svg'); ?>" alt="logo">
        </a>
      </div>
      <nav class="app-navbar" data-simplebar>
        <ul class="menubar">
          <li class="menu-item">
            <a class="menu-link" href="index.php">
              <i class="fi fi-rr-apps"></i>
              <span class="menu-label">Dashboard</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/users.php">
              <i class="fi fi-rr-users"></i>
              <span class="menu-label">Manage Users</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/portfolio.php">
              <i class="fi fi-rr-portfolio"></i>
              <span class="menu-label">Manage Portfolio</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/products.php">
              <i class="fi fi-rr-shopping-cart"></i>
              <span class="menu-label">Manage Products</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/orders.php">
              <i class="fi fi-rr-receipt"></i>
              <span class="menu-label">Manage Orders</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/courses.php">
              <i class="fi fi-rr-book-alt"></i>
              <span class="menu-label">Manage Courses</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/script_requests.php">
              <i class="fi fi-rr-file-code"></i>
              <span class="menu-label">Manage Script Request</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/settings.php">
              <i class="fi fi-rr-settings"></i>
              <span class="menu-label">Settings</span>
            </a>
          </li>
        </ul>
      </nav>
      <div class="app-footer">
        <a href="pages/faq.html" class="btn btn-outline-white waves-effect btn-app-nav w-100">
          <i class="fi fi-rs-interrogation"></i>
          <span class="nav-text">Help and Support</span>
        </a>
      </div>
    </aside>
    <!-- end::Edura Sidebar Menu -->

    <main class="app-wrapper">

      <div class="container-fluid">

        <div class="app-page-head d-flex flex-wrap gap-3 align-items-center justify-content-between">
          <div class="clearfix">
            <h1 class="app-page-title">Manage Courses</h1>
            <nav aria-label="breadcrumb">
              <ol class="breadcrumb mb-0">
                <li class="breadcrumb-item">
                  <a href="index.php">Dashboard</a>
                </li>
                <li class="breadcrumb-item active" aria-current="page">Manage Courses</li>
              </ol>
            </nav>
          </div>
        </div>

        <?php if ($success): ?>
          <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fi fi-rr-check-circle me-2"></i><?php echo htmlspecialchars($success); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
          </div>
        <?php endif; ?>
        
        <?php if ($error): ?>
          <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fi fi-rr-exclamation-circle me-2"></i><?php echo htmlspecialchars($error); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
          </div>
        <?php endif; ?>


        <!-- Statistics Cards -->
        <div class="row mb-4">
          <!-- First Row - 2 cards on mobile, 4 on desktop -->
          <div class="col-lg-3 col-md-6 col-6 mb-3">
            <div class="card">
              <div class="card-body">
                <div class="d-flex align-items-center">
                  <div class="flex-shrink-0">
                    <div class="avatar avatar-sm bg-primary text-white rounded">
                      <i class="fi fi-rr-book-alt"></i>
                    </div>
                  </div>
                  <div class="flex-grow-1 ms-3">
                    <div class="text-muted small">Total Courses</div>
                    <div class="h5 mb-0"><?php echo $totalCourses; ?></div>
                  </div>
                </div>
              </div>
            </div>
          </div>
          <div class="col-lg-3 col-md-6 col-6 mb-3">
            <div class="card">
              <div class="card-body">
                <div class="d-flex align-items-center">
                  <div class="flex-shrink-0">
                    <div class="avatar avatar-sm bg-success text-white rounded">
                      <i class="fi fi-rr-check"></i>
                    </div>
                  </div>
                  <div class="flex-grow-1 ms-3">
                    <div class="text-muted small">Active Courses</div>
                    <div class="h5 mb-0"><?php echo $activeCourses; ?></div>
                  </div>
                </div>
              </div>
            </div>
          </div>
          <!-- Second Row - 2 cards on mobile, 4 on desktop -->
          <div class="col-lg-3 col-md-6 col-6 mb-3">
            <div class="card">
              <div class="card-body">
                <div class="d-flex align-items-center">
                  <div class="flex-shrink-0">
                    <div class="avatar avatar-sm bg-info text-white rounded">
                      <i class="fi fi-rr-users"></i>
                    </div>
                  </div>
                  <div class="flex-grow-1 ms-3">
                    <div class="text-muted small">Total Enrollments</div>
                    <div class="h5 mb-0"><?php echo $totalEnrollments; ?></div>
                  </div>
                </div>
              </div>
            </div>
          </div>
          <div class="col-lg-3 col-md-6 col-6 mb-3">
            <div class="card">
              <div class="card-body">
                <div class="d-flex align-items-center">
                  <div class="flex-shrink-0">
                    <div class="avatar avatar-sm bg-warning text-white rounded">
                      <i class="fi fi-rr-dollar"></i>
                    </div>
                  </div>
                  <div class="flex-grow-1 ms-3">
                    <div class="text-muted small">Course Revenue</div>
                    <div class="h5 mb-0">$<?php echo number_format($totalRevenue, 2); ?></div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>

        <div class="row">

          <div class="col-lg-12">
            <div class="card overflow-hidden">
              <div class="card-header d-flex align-items-center justify-content-between border-0">
                <h6 class="card-title mb-0">Courses List (<?php echo $totalCourses; ?> total courses)</h6>
                <div class="d-flex align-items-center gap-2">
                  <button class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#addCourseModal">
                    <i class="fi fi-rr-plus"></i> Add New Course
                  </button>
                  </div>
                </div>
              <div class="card-body p-4">
                <div class="row">
                  <?php if (empty($courses)): ?>
                    <div class="col-12">
                      <div class="text-center py-5">
                        <i class="fi fi-rr-book-alt text-muted mb-3" style="font-size: 4rem;"></i>
                        <h5 class="text-muted mb-3">No courses found</h5>
                        <p class="text-muted mb-4">Start by creating your first course to get students enrolled.</p>
                        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addCourseModal">
                          <i class="fi fi-rr-plus"></i> Create First Course
                              </button>
                            </div>
                    </div>
                      <?php else: ?>
                    <?php foreach ($courses as $course): ?>
                      <div class="col-lg-4 col-md-6 mb-4">
                        <div class="card h-100 course-card">
                          <!-- Course Image -->
                          <div class="course-image-container" style="height: 200px; overflow: hidden; position: relative;">
                            <?php if (!empty($course['thumbnail'])): ?>
                              <img src="../<?php echo htmlspecialchars($course['thumbnail']); ?>" 
                                   alt="<?php echo htmlspecialchars($course['title'] ?? 'Course'); ?>" 
                                   class="card-img-top" 
                                   style="width: 100%; height: 100%; object-fit: cover;">
                            <?php else: ?>
                              <div class="bg-gradient-primary d-flex align-items-center justify-content-center h-100">
                                <i class="fi fi-rr-book-alt text-white" style="font-size: 3rem;"></i>
                              </div>
                            <?php endif; ?>
                            
                            <!-- Status Badge -->
                            <div class="position-absolute top-0 end-0 m-2">
                              <?php
                              $status = $course['status'] ?? 'active';
                              $statusClass = $status === 'active' ? 'bg-success' : 'bg-warning';
                              ?>
                              <span class="badge <?php echo $statusClass; ?>"><?php echo ucfirst($status); ?></span>
                            </div>
                          </div>

                          <div class="card-body d-flex flex-column">
                            <!-- Course Title -->
                            <h5 class="card-title mb-2"><?php echo htmlspecialchars($course['title'] ?? 'Untitled Course'); ?></h5>
                            
                            <!-- Course Description -->
                            <p class="card-text text-muted small mb-3">
                              <?php 
                              $description = $course['description'] ?? 'No description available';
                              echo htmlspecialchars(substr($description, 0, 100)) . (strlen($description) > 100 ? '...' : ''); 
                              ?>
                            </p>

                            <!-- Rating and Students -->
                            <div class="d-flex align-items-center mb-3">
                              <div class="d-flex align-items-center me-3">
                                <div class="text-warning me-1">
                                  <?php for ($i = 1; $i <= 5; $i++): ?>
                                    <i class="fi fi-rr-star-fill" style="font-size: 0.8rem;"></i>
                                  <?php endfor; ?>
                                </div>
                                <small class="text-muted">(4.5/<?php echo number_format($course['enrollment_count'] ?? 0); ?> ratings)</small>
                              </div>
                            </div>

                            <!-- Course Info -->
                            <div class="d-flex justify-content-between align-items-center mb-3">
                              <div class="d-flex align-items-center text-muted small">
                                <i class="fi fi-rr-users me-1"></i>
                                <span><?php echo number_format($course['enrollment_count'] ?? 0); ?> Students</span>
                              </div>
                              <div class="d-flex align-items-center text-muted small">
                                <i class="fi fi-rr-clock me-1"></i>
                                <span><?php echo htmlspecialchars($course['duration'] ?? 'N/A'); ?></span>
                              </div>
                            </div>

                            <!-- Price -->
                            <div class="d-flex justify-content-between align-items-center mb-3">
                              <div class="d-flex align-items-center">
                                <span class="h5 text-success mb-0">$<?php echo number_format($course['price'] ?? 0, 0); ?></span>
                              </div>
                            </div>

                            <!-- Action Buttons -->
                            <div class="mt-auto">
                              <div class="d-flex gap-1 mb-2">
                                <button class="btn btn-outline-primary btn-sm flex-fill" onclick="editCourse(<?php echo htmlspecialchars(json_encode($course)); ?>)">
                                  <i class="fi fi-rr-edit"></i> Edit
                                </button>
                                
                                <button class="btn btn-outline-info btn-sm flex-fill" onclick="manageLessons(<?php echo $course['id'] ?? 0; ?>, '<?php echo htmlspecialchars($course['title'] ?? 'Course'); ?>')">
                                  <i class="fi fi-rr-book"></i> Lessons
                                </button>
                              </div>
                              <div class="d-flex gap-1">
                                <form method="POST" style="display: inline; flex: 1;">
                                  <input type="hidden" name="action" value="toggle_status">
                                  <input type="hidden" name="course_id" value="<?php echo $course['id'] ?? 0; ?>">
                                  <button type="submit" class="btn <?php echo ($course['status'] ?? 'active') === 'active' ? 'btn-warning' : 'btn-success'; ?> btn-sm w-100" onclick="return confirm('Are you sure you want to change the status of this course?')">
                                    <i class="fi fi-rr-<?php echo ($course['status'] ?? 'active') === 'active' ? 'pause' : 'play'; ?>"></i>
                                  </button>
                                </form>
                                
                                <form method="POST" style="display: inline; flex: 1;">
                                  <input type="hidden" name="action" value="delete">
                                  <input type="hidden" name="course_id" value="<?php echo $course['id'] ?? 0; ?>">
                                  <button type="submit" class="btn btn-danger btn-sm w-100" onclick="return confirm('Are you sure you want to delete this course? This action cannot be undone.')">
                                    <i class="fi fi-rr-trash"></i>
                                  </button>
                                </form>
                              </div>
                            </div>
                          </div>
                        </div>
                      </div>
                        <?php endforeach; ?>
                      <?php endif; ?>
                </div>
              </div>
              <?php if ($totalPages > 1): ?>
              <div class="card-footer border-0 p-3">
                <nav aria-label="Page navigation example">
                    <ul class="pagination pagination-sm justify-content-center mb-0">
                    <?php if ($page > 1): ?>
                      <li class="page-item">
                        <a class="page-link" href="?page=<?php echo $page - 1; ?>" aria-label="Previous">
                          <i class="fi fi-rr-angle-double-left"></i>
                        </a>
                      </li>
                    <?php endif; ?>
                    
                      <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                      <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                        <a class="page-link" href="?page=<?php echo $i; ?>"><?php echo $i; ?></a>
                      </li>
                    <?php endfor; ?>
                    
                    <?php if ($page < $totalPages): ?>
                      <li class="page-item">
                        <a class="page-link" href="?page=<?php echo $page + 1; ?>" aria-label="Next">
                          <i class="fi fi-rr-angle-double-right"></i>
                        </a>
                      </li>
                    <?php endif; ?>
                  </ul>
                </nav>
              </div>
              <?php endif; ?>
            </div>
          </div>

        </div>

      </div>

    </main>

    <!-- begin::Edura Footer -->
    <footer class="footer-wrapper bg-body text-center">
      <div class="container-fluid">
        <p class="mb-0">© <span class="currentYear">2025</span> Edura. Proudly powered by <a href="javascript:void(0);">LayoutDrop</a>.</p>
      </div>
    </footer>
    <!-- end::Edura Footer -->

  </div>

  <!-- begin::Edura Page Scripts -->
  <script src="assets/libs/global/global.min.js"></script>
  <script src="assets/js/appSettings.js"></script>
  <script src="assets/js/main.js"></script>
  <!-- end::Edura Page Scripts -->
  <!-- Add Course Modal -->
  <div class="modal fade" id="addCourseModal" tabindex="-1" aria-labelledby="addCourseModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-xl">
      <div class="modal-content">
        <form method="POST" enctype="multipart/form-data">
          <input type="hidden" name="action" value="create">
          <div class="modal-header">
            <h5 class="modal-title" id="addCourseModalLabel">Add New Course</h5>
            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
          </div>
          <div class="modal-body">
            <div class="row">
              <div class="col-md-8">
                <div class="mb-3">
                  <label for="course_title" class="form-label">Course Title *</label>
                  <input type="text" class="form-control" id="course_title" name="course_title" required>
                </div>
              </div>
              <div class="col-md-4">
                <div class="mb-3">
                  <label for="status" class="form-label">Status</label>
                  <select class="form-select" id="status" name="status">
                    <option value="active">Active</option>
                    <option value="inactive">Inactive</option>
                  </select>
                </div>
              </div>
            </div>
            <div class="mb-3">
              <label for="course_description" class="form-label">Course Description *</label>
              <textarea class="form-control" id="course_description" name="course_description" rows="4" required></textarea>
            </div>
            <div class="row">
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="course_price" class="form-label">Course Price *</label>
                  <div class="input-group">
                    <span class="input-group-text">$</span>
                    <input type="number" class="form-control" id="course_price" name="course_price" step="0.01" min="0" required>
                  </div>
                </div>
              </div>
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="course_duration" class="form-label">Course Duration</label>
                  <input type="text" class="form-control" id="course_duration" name="course_duration" placeholder="e.g., 4 weeks, 20 hours">
                </div>
              </div>
            </div>
            <div class="row">
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="level" class="form-label">Course Level</label>
                  <select class="form-select" id="level" name="level">
                    <option value="beginner">Beginner</option>
                    <option value="intermediate">Intermediate</option>
                    <option value="advanced">Advanced</option>
                  </select>
                </div>
              </div>
            </div>
                <div class="mb-3">
              <label for="course_features" class="form-label">Course Features</label>
              <textarea class="form-control" id="course_features" name="course_features" rows="3" placeholder="What makes this course special? Key features, benefits, etc."></textarea>
            </div>
            <div class="mb-3">
              <label for="course_instructor" class="form-label">Course Instructor *</label>
              <input type="text" class="form-control" id="course_instructor" name="course_instructor" required placeholder="Enter the instructor's name">
            </div>
            <div class="mb-3">
              <label for="what_you_learn" class="form-label">What You'll Learn</label>
              <textarea class="form-control" id="what_you_learn" name="what_you_learn" rows="4" placeholder="List the key learning outcomes and skills students will gain"></textarea>
            </div>
            <div class="mb-3">
              <label for="course_video" class="form-label">Course Preview Video</label>
              <div class="video-upload-container">
                <!-- Video Upload Area -->
                <div class="video-upload-area border border-2 border-dashed rounded p-4 text-center" id="courseVideoUploadArea" style="cursor: pointer;">
                  <i class="fi fi-rr-cloud-upload text-muted" style="font-size: 2rem;"></i>
                  <h6 class="mt-2 mb-1">Upload Course Preview Video</h6>
                  <p class="text-muted small mb-0">
                    Click or drag & drop video file<br>
                    <small>MP4, WebM, OGG, AVI, MOV (Max 500MB)</small>
                  </p>
                  <input type="file" class="d-none" id="course_video" name="course_video" accept="video/*">
                </div>
                
                <!-- Upload Progress -->
                <div class="upload-progress mt-3" id="courseUploadProgress" style="display: none;">
                  <div class="progress">
                    <div class="progress-bar" role="progressbar" style="width: 0%"></div>
                  </div>
                  <small class="text-muted mt-1">Uploading video...</small>
                </div>
                
                <!-- Video Preview -->
                <div class="video-preview mt-3" id="courseVideoPreview" style="display: none;">
                  <video class="w-100 rounded" controls style="max-height: 200px;">
                    <source src="" type="video/mp4">
                    Your browser does not support the video tag.
                  </video>
                  <div class="mt-2">
                    <button type="button" class="btn btn-sm btn-outline-danger" onclick="removeCourseVideo()">
                      <i class="fi fi-rr-trash"></i> Remove Video
                    </button>
                  </div>
                </div>
              </div>
            </div>
            
            <div class="mb-3">
              <label for="thumbnail" class="form-label">Course Thumbnail</label>
              <input type="file" class="form-control" id="thumbnail" name="thumbnail" accept="image/*">
              <div class="form-text">Recommended size: 800x600px or similar aspect ratio</div>
            </div>
          </div>
          <div class="modal-footer">
            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
            <button type="submit" class="btn btn-primary">Add Course</button>
          </div>
        </form>
      </div>
    </div>
  </div>

  <!-- Edit Course Modal -->
  <div class="modal fade" id="editCourseModal" tabindex="-1" aria-labelledby="editCourseModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-xl">
      <div class="modal-content">
        <form method="POST" enctype="multipart/form-data">
          <input type="hidden" name="action" value="update">
          <input type="hidden" name="course_id" id="edit_course_id">
          <div class="modal-header">
            <h5 class="modal-title" id="editCourseModalLabel">Edit Course</h5>
            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
          </div>
          <div class="modal-body">
            <div class="row">
              <div class="col-md-8">
                <div class="mb-3">
                  <label for="edit_course_title" class="form-label">Course Title *</label>
                  <input type="text" class="form-control" id="edit_course_title" name="course_title" required>
                </div>
              </div>
              <div class="col-md-4">
                <div class="mb-3">
                  <label for="edit_status" class="form-label">Status</label>
                  <select class="form-select" id="edit_status" name="status">
                    <option value="active">Active</option>
                    <option value="inactive">Inactive</option>
                  </select>
                </div>
              </div>
            </div>
            <div class="mb-3">
              <label for="edit_course_description" class="form-label">Course Description *</label>
              <textarea class="form-control" id="edit_course_description" name="course_description" rows="4" required></textarea>
            </div>
            <div class="row">
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="edit_course_price" class="form-label">Course Price *</label>
                  <div class="input-group">
                    <span class="input-group-text">$</span>
                    <input type="number" class="form-control" id="edit_course_price" name="course_price" step="0.01" min="0" required>
                  </div>
                </div>
              </div>
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="edit_course_duration" class="form-label">Course Duration</label>
                  <input type="text" class="form-control" id="edit_course_duration" name="course_duration" placeholder="e.g., 4 weeks, 20 hours">
                </div>
              </div>
            </div>
            <div class="row">
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="edit_level" class="form-label">Course Level</label>
                  <select class="form-select" id="edit_level" name="level">
                    <option value="beginner">Beginner</option>
                    <option value="intermediate">Intermediate</option>
                    <option value="advanced">Advanced</option>
                  </select>
                </div>
              </div>
            </div>
                <div class="mb-3">
              <label for="edit_course_features" class="form-label">Course Features</label>
              <textarea class="form-control" id="edit_course_features" name="course_features" rows="3" placeholder="What makes this course special? Key features, benefits, etc."></textarea>
            </div>
            <div class="mb-3">
              <label for="edit_course_instructor" class="form-label">Course Instructor *</label>
              <input type="text" class="form-control" id="edit_course_instructor" name="course_instructor" required placeholder="Enter the instructor's name">
            </div>
            <div class="mb-3">
              <label for="edit_what_you_learn" class="form-label">What You'll Learn</label>
              <textarea class="form-control" id="edit_what_you_learn" name="what_you_learn" rows="4" placeholder="List the key learning outcomes and skills students will gain"></textarea>
            </div>
            <div class="mb-3">
              <label for="edit_course_video" class="form-label">Course Preview Video</label>
              <div class="video-upload-container">
                <!-- Video Upload Area -->
                <div class="video-upload-area border border-2 border-dashed rounded p-4 text-center" id="editCourseVideoUploadArea" style="cursor: pointer;">
                  <i class="fi fi-rr-cloud-upload text-muted" style="font-size: 2rem;"></i>
                  <h6 class="mt-2 mb-1">Upload Course Preview Video</h6>
                  <p class="text-muted small mb-0">
                    Click or drag & drop video file<br>
                    <small>MP4, WebM, OGG, AVI, MOV (Max 500MB)</small>
                  </p>
                  <input type="file" class="d-none" id="edit_course_video" name="course_video" accept="video/*">
                </div>
                
                <!-- Upload Progress -->
                <div class="upload-progress mt-3" id="editCourseUploadProgress" style="display: none;">
                  <div class="progress">
                    <div class="progress-bar" role="progressbar" style="width: 0%"></div>
                  </div>
                  <small class="text-muted mt-1">Uploading video...</small>
                </div>
                
                <!-- Video Preview -->
                <div class="video-preview mt-3" id="editCourseVideoPreview" style="display: none;">
                  <video class="w-100 rounded" controls style="max-height: 200px;">
                    <source src="" type="video/mp4">
                    Your browser does not support the video tag.
                  </video>
                  <div class="mt-2">
                    <button type="button" class="btn btn-sm btn-outline-danger" onclick="removeEditCourseVideo()">
                      <i class="fi fi-rr-trash"></i> Remove Video
                    </button>
                  </div>
                </div>
                
                <!-- Current Video Display -->
                <div id="current_course_video_preview" class="mt-2"></div>
              </div>
            </div>
            
            <div class="mb-3">
              <label for="edit_thumbnail" class="form-label">Course Thumbnail</label>
              <input type="file" class="form-control" id="edit_thumbnail" name="thumbnail" accept="image/*">
              <input type="hidden" id="edit_current_thumbnail" name="current_thumbnail">
              <div class="form-text">Leave empty to keep current image. Recommended size: 800x600px or similar aspect ratio</div>
              <div id="current_thumbnail_preview" class="mt-2"></div>
            </div>
          </div>
          <div class="modal-footer">
            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
            <button type="submit" class="btn btn-primary">Update Course</button>
          </div>
        </form>
      </div>
    </div>
  </div>

  <!-- Manage Lessons Modal -->
  <div class="modal fade" id="manageLessonsModal" tabindex="-1" aria-labelledby="manageLessonsModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-xl">
      <div class="modal-content">
        <div class="modal-header">
          <h5 class="modal-title" id="manageLessonsModalLabel">Manage Lessons</h5>
          <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
        </div>
        <div class="modal-body">
          <div class="d-flex justify-content-between align-items-center mb-3">
            <h6 id="courseTitle">Course Lessons</h6>
            <button class="btn btn-primary btn-sm" onclick="showAddLessonForm()">
              <i class="fi fi-rr-plus"></i> Add New Lesson
            </button>
          </div>
          
          <!-- Add/Edit Lesson Form (Hidden by default) -->
          <div id="lessonForm" class="mb-4" style="display: none;">
            <div class="card">
              <div class="card-header">
                <h6 id="lessonFormTitle">Add New Lesson</h6>
              </div>
              <div class="card-body">
                <form method="POST" id="lessonFormElement" enctype="multipart/form-data">
                  <input type="hidden" name="action" id="lessonAction" value="add_lesson">
                  <input type="hidden" name="course_id" id="lessonCourseId">
                  <input type="hidden" name="lesson_id" id="lessonId">
                  
                  <div class="row">
                    <div class="col-md-6">
                      <div class="mb-3">
                        <label for="lesson_number" class="form-label">Lesson Number *</label>
                        <input type="number" class="form-control" id="lesson_number" name="lesson_number" min="1" required>
                      </div>
                    </div>
                    <div class="col-md-6">
                      <div class="mb-3">
                        <label for="duration" class="form-label">Duration</label>
                        <input type="text" class="form-control" id="duration" name="duration" placeholder="e.g., 15 minutes, 1 hour">
                      </div>
                    </div>
                  </div>
                  
                  <div class="mb-3">
                    <label for="lesson_title" class="form-label">Lesson Title *</label>
                    <input type="text" class="form-control" id="lesson_title" name="lesson_title" required>
                  </div>
                  
                  <div class="mb-3">
                    <label for="lesson_description" class="form-label">Lesson Description</label>
                    <textarea class="form-control" id="lesson_description" name="lesson_description" rows="3"></textarea>
                  </div>
                  
                  <!-- Video Upload Section -->
                  <div class="mb-3">
                    <label class="form-label">Lesson Video *</label>
                    <div class="video-upload-container">
                      <!-- Video Upload Area -->
                      <div class="video-upload-area border border-2 border-dashed rounded p-4 text-center" id="videoUploadArea" style="cursor: pointer;">
                        <i class="fi fi-rr-cloud-upload text-muted" style="font-size: 2rem;"></i>
                        <h6 class="mt-2 mb-1">Upload Video File</h6>
                        <p class="text-muted small mb-0">
                          Click or drag & drop video file<br>
                          <small>MP4, WebM, OGG, AVI, MOV (Max 500MB)</small>
                        </p>
                        <input type="file" class="d-none" id="lesson_video" name="lesson_video" accept="video/*">
                      </div>
                      
                      <!-- Upload Progress -->
                      <div class="upload-progress mt-3" id="uploadProgress" style="display: none;">
                        <div class="progress">
                          <div class="progress-bar" role="progressbar" style="width: 0%"></div>
                        </div>
                        <small class="text-muted mt-1">Uploading video...</small>
                      </div>
                      
                      <!-- Video Preview -->
                      <div class="video-preview mt-3" id="videoPreview" style="display: none;">
                        <video class="w-100 rounded" controls style="max-height: 200px;">
                          <source src="" type="video/mp4">
                          Your browser does not support the video tag.
                        </video>
                        <div class="mt-2">
                          <button type="button" class="btn btn-sm btn-outline-danger" onclick="removeVideo()">
                            <i class="fi fi-rr-trash"></i> Remove Video
                          </button>
                        </div>
                      </div>
                    </div>
                  </div>
                  
                  
                  <div class="mb-3" id="lessonStatusField" style="display: none;">
                    <label for="is_active" class="form-label">Status</label>
                    <select class="form-select" id="is_active" name="is_active">
                      <option value="yes">Active</option>
                      <option value="no">Inactive</option>
                    </select>
                  </div>
                  
                  <div class="d-flex gap-2">
                    <button type="submit" class="btn btn-primary" id="lessonSubmitBtn">Add Lesson</button>
                    <button type="button" class="btn btn-secondary" onclick="hideLessonForm()">Cancel</button>
                  </div>
                </form>
              </div>
            </div>
          </div>
          
          <!-- Lessons List -->
          <div id="lessonsList">
            <div class="text-center py-4">
              <div class="spinner-border text-primary" role="status">
                <span class="visually-hidden">Loading lessons...</span>
              </div>
            </div>
          </div>
        </div>
        <div class="modal-footer">
          <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
        </div>
      </div>
    </div>
  </div>

  <script>
    // Video Upload Functionality
    let selectedVideoFile = null;
    let selectedCourseVideoFile = null;
    let selectedEditCourseVideoFile = null;
    
    document.addEventListener('DOMContentLoaded', function() {
      // Lesson video upload area click handler
      document.getElementById('videoUploadArea').addEventListener('click', function() {
        document.getElementById('lesson_video').click();
      });
      
      // Course video upload area click handler
      document.getElementById('courseVideoUploadArea').addEventListener('click', function() {
        document.getElementById('course_video').click();
      });
      
      // Edit course video upload area click handler
      document.getElementById('editCourseVideoUploadArea').addEventListener('click', function() {
        document.getElementById('edit_course_video').click();
      });
      
      // Lesson file input change handler
      document.getElementById('lesson_video').addEventListener('change', function(e) {
        const file = e.target.files[0];
        if (file) {
          handleVideoSelection(file);
        }
      });
      
      // Course file input change handler
      document.getElementById('course_video').addEventListener('change', function(e) {
        const file = e.target.files[0];
        if (file) {
          handleCourseVideoSelection(file);
        }
      });
      
      // Edit course file input change handler
      document.getElementById('edit_course_video').addEventListener('change', function(e) {
        const file = e.target.files[0];
        if (file) {
          handleEditCourseVideoSelection(file);
        }
      });
      
      // Drag and drop handlers
      const uploadArea = document.getElementById('videoUploadArea');
      
      uploadArea.addEventListener('dragover', function(e) {
        e.preventDefault();
        uploadArea.classList.add('border-primary', 'bg-light');
      });
      
      uploadArea.addEventListener('dragleave', function(e) {
        e.preventDefault();
        uploadArea.classList.remove('border-primary', 'bg-light');
      });
      
      uploadArea.addEventListener('drop', function(e) {
        e.preventDefault();
        uploadArea.classList.remove('border-primary', 'bg-light');
        
        const files = e.dataTransfer.files;
        if (files.length > 0) {
          handleVideoSelection(files[0]);
        }
      });
    });
    
    function handleVideoSelection(file) {
      // Validate file type by extension and MIME type
      const allowedExtensions = ['mp4', 'webm', 'ogg', 'avi', 'mov'];
      const allowedMimeTypes = [
        'video/mp4', 'video/webm', 'video/ogg', 'video/avi', 'video/mov',
        'video/quicktime', 'video/x-msvideo', 'video/x-ms-wmv'
      ];
      
      const fileExtension = file.name.split('.').pop().toLowerCase();
      const fileMimeType = file.type;
      
      if (!allowedExtensions.includes(fileExtension) || !allowedMimeTypes.includes(fileMimeType)) {
        alert('Invalid file type. Only MP4, WebM, OGG, AVI, and MOV files are allowed.');
        return;
      }
      
      // Validate file size (500MB max)
      const maxSize = 500 * 1024 * 1024; // 500MB
      if (file.size > maxSize) {
        alert('File too large. Maximum size is 500MB.');
        return;
      }
      
      selectedVideoFile = file;
      
      // Show video preview
      const preview = document.getElementById('videoPreview');
      const video = preview.querySelector('video');
      const url = URL.createObjectURL(file);
      
      video.src = url;
      preview.style.display = 'block';
      
      // Hide upload area
      document.getElementById('videoUploadArea').style.display = 'none';
    }
    
    function removeVideo() {
      selectedVideoFile = null;
      
      // Hide preview
      document.getElementById('videoPreview').style.display = 'none';
      
      // Show upload area
      document.getElementById('videoUploadArea').style.display = 'block';
      
      // Clear file input
      document.getElementById('lesson_video').value = '';
    }
    
    function handleCourseVideoSelection(file) {
      // Validate file type
      const allowedExtensions = ['mp4', 'webm', 'ogg', 'avi', 'mov'];
      const allowedMimeTypes = [
        'video/mp4', 'video/webm', 'video/ogg', 'video/avi', 'video/mov',
        'video/quicktime', 'video/x-msvideo', 'video/x-ms-wmv'
      ];
      
      const fileExtension = file.name.split('.').pop().toLowerCase();
      const fileMimeType = file.type;
      
      if (!allowedExtensions.includes(fileExtension) || !allowedMimeTypes.includes(fileMimeType)) {
        alert('Invalid file type. Only MP4, WebM, OGG, AVI, and MOV files are allowed.');
        return;
      }
      
      // Validate file size (500MB max)
      const maxSize = 500 * 1024 * 1024; // 500MB
      if (file.size > maxSize) {
        alert('File too large. Maximum size is 500MB.');
        return;
      }
      
      selectedCourseVideoFile = file;
      
      // Show video preview
      const preview = document.getElementById('courseVideoPreview');
      const video = preview.querySelector('video');
      const url = URL.createObjectURL(file);
      
      video.src = url;
      preview.style.display = 'block';
      
      // Hide upload area
      document.getElementById('courseVideoUploadArea').style.display = 'none';
    }
    
    function removeCourseVideo() {
      selectedCourseVideoFile = null;
      
      // Hide preview
      document.getElementById('courseVideoPreview').style.display = 'none';
      
      // Show upload area
      document.getElementById('courseVideoUploadArea').style.display = 'block';
      
      // Clear file input
      document.getElementById('course_video').value = '';
    }
    
    function handleEditCourseVideoSelection(file) {
      // Validate file type
      const allowedExtensions = ['mp4', 'webm', 'ogg', 'avi', 'mov'];
      const allowedMimeTypes = [
        'video/mp4', 'video/webm', 'video/ogg', 'video/avi', 'video/mov',
        'video/quicktime', 'video/x-msvideo', 'video/x-ms-wmv'
      ];
      
      const fileExtension = file.name.split('.').pop().toLowerCase();
      const fileMimeType = file.type;
      
      if (!allowedExtensions.includes(fileExtension) || !allowedMimeTypes.includes(fileMimeType)) {
        alert('Invalid file type. Only MP4, WebM, OGG, AVI, and MOV files are allowed.');
        return;
      }
      
      // Validate file size (500MB max)
      const maxSize = 500 * 1024 * 1024; // 500MB
      if (file.size > maxSize) {
        alert('File too large. Maximum size is 500MB.');
        return;
      }
      
      selectedEditCourseVideoFile = file;
      
      // Show video preview
      const preview = document.getElementById('editCourseVideoPreview');
      const video = preview.querySelector('video');
      const url = URL.createObjectURL(file);
      
      video.src = url;
      preview.style.display = 'block';
      
      // Hide upload area
      document.getElementById('editCourseVideoUploadArea').style.display = 'none';
      
      // Hide current video preview
      document.getElementById('current_course_video_preview').style.display = 'none';
    }
    
    function removeEditCourseVideo() {
      selectedEditCourseVideoFile = null;
      
      // Hide preview
      document.getElementById('editCourseVideoPreview').style.display = 'none';
      
      // Show upload area
      document.getElementById('editCourseVideoUploadArea').style.display = 'block';
      
      // Show current video preview
      document.getElementById('current_course_video_preview').style.display = 'block';
      
      // Clear file input
      document.getElementById('edit_course_video').value = '';
    }

    function editCourse(course) {
      document.getElementById('edit_course_id').value = course.id;
      document.getElementById('edit_course_title').value = course.title;
      document.getElementById('edit_course_description').value = course.description || '';
      document.getElementById('edit_course_price').value = course.price || 0;
      document.getElementById('edit_course_duration').value = course.duration || '';
      document.getElementById('edit_course_features').value = course.curriculum || '';
      document.getElementById('edit_course_instructor').value = course.instructor || '';
      document.getElementById('edit_level').value = course.level || 'beginner';
      document.getElementById('edit_status').value = course.status;
      document.getElementById('edit_current_thumbnail').value = course.thumbnail || '';
      
      // Show current thumbnail preview
      const preview = document.getElementById('current_thumbnail_preview');
      if (course.thumbnail) {
        preview.innerHTML = '<img src="../' + course.thumbnail + '" alt="Current thumbnail" class="img-thumbnail" style="max-width: 200px;">';
      } else {
        preview.innerHTML = '<span class="text-muted">No current thumbnail</span>';
      }
      
      // Show current course video preview
      const videoPreview = document.getElementById('current_course_video_preview');
      if (course.course_video) {
        videoPreview.innerHTML = `
          <div class="alert alert-info">
            <h6><i class="fi fi-rr-play me-2"></i>Current Course Video</h6>
            <video class="w-100 rounded" controls style="max-height: 200px;">
              <source src="../uploads/course_videos/${course.course_video}" type="video/mp4">
              Your browser does not support the video tag.
            </video>
            <div class="mt-2">
              <small class="text-muted">Upload a new video to replace this one</small>
            </div>
          </div>
        `;
      } else {
        videoPreview.innerHTML = `
          <div class="alert alert-light">
            <i class="fi fi-rr-video-camera me-2"></i>
            <small class="text-muted">No course preview video uploaded yet</small>
          </div>
        `;
      }
      
      // Show the modal
      new bootstrap.Modal(document.getElementById('editCourseModal')).show();
    }

    // Lesson Management Functions
    function manageLessons(courseId, courseTitle) {
      document.getElementById('courseTitle').textContent = courseTitle + ' - Lessons';
      document.getElementById('lessonCourseId').value = courseId;
      hideLessonForm();
      loadLessons(courseId);
      new bootstrap.Modal(document.getElementById('manageLessonsModal')).show();
    }

    function loadLessons(courseId) {
      const lessonsList = document.getElementById('lessonsList');
      lessonsList.innerHTML = '<div class="text-center py-4"><div class="spinner-border text-primary" role="status"><span class="visually-hidden">Loading lessons...</span></div></div>';
      
      // Fetch lessons via AJAX
      fetch(`get_lessons.php?course_id=${courseId}`)
        .then(response => response.text())
        .then(html => {
          lessonsList.innerHTML = html;
        })
        .catch(error => {
          lessonsList.innerHTML = '<div class="alert alert-danger">Error loading lessons. Please try again.</div>';
          console.error('Error:', error);
        });
    }

    function showAddLessonForm() {
      document.getElementById('lessonForm').style.display = 'block';
      document.getElementById('lessonFormTitle').textContent = 'Add New Lesson';
      document.getElementById('lessonAction').value = 'add_lesson';
      document.getElementById('lessonSubmitBtn').textContent = 'Add Lesson';
      document.getElementById('lessonStatusField').style.display = 'none';
      document.getElementById('lessonFormElement').reset();
      document.getElementById('lessonId').value = '';
      
      // Reset video upload interface
      document.getElementById('videoUploadArea').style.display = 'block';
      document.getElementById('videoPreview').style.display = 'none';
      document.getElementById('lesson_video').value = '';
      selectedVideoFile = null;
    }

    function hideLessonForm() {
      document.getElementById('lessonForm').style.display = 'none';
    }

    function editLesson(lesson) {
      document.getElementById('lessonForm').style.display = 'block';
      document.getElementById('lessonFormTitle').textContent = 'Edit Lesson';
      document.getElementById('lessonAction').value = 'update_lesson';
      document.getElementById('lessonSubmitBtn').textContent = 'Update Lesson';
      document.getElementById('lessonStatusField').style.display = 'block';
      
      document.getElementById('lessonId').value = lesson.id;
      document.getElementById('lesson_number').value = lesson.lesson_number;
      document.getElementById('lesson_title').value = lesson.lesson_title;
      document.getElementById('lesson_description').value = lesson.lesson_description || '';
      document.getElementById('duration').value = lesson.duration || '';
      document.getElementById('is_active').value = lesson.is_active;
      
      // Handle video display based on type
      const uploadArea = document.getElementById('videoUploadArea');
      const preview = document.getElementById('videoPreview');
      
      if (lesson.video_file) {
        // Show uploaded video
        const video = preview.querySelector('video');
        video.src = '../uploads/course_videos/' + lesson.video_file;
        preview.style.display = 'block';
        uploadArea.style.display = 'none';
      } else {
        // Show upload area for new video
        preview.style.display = 'none';
        uploadArea.style.display = 'block';
      }
      
      // Reset video file input
      document.getElementById('lesson_video').value = '';
      selectedVideoFile = null;
    }

    function deleteLesson(lessonId) {
      if (confirm('Are you sure you want to delete this lesson? This action cannot be undone.')) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.innerHTML = `
          <input type="hidden" name="action" value="delete_lesson">
          <input type="hidden" name="lesson_id" value="${lessonId}">
        `;
        document.body.appendChild(form);
        form.submit();
      }
    }

    function toggleLessonStatus(lessonId) {
      const form = document.createElement('form');
      form.method = 'POST';
      form.innerHTML = `
        <input type="hidden" name="action" value="toggle_lesson_status">
        <input type="hidden" name="lesson_id" value="${lessonId}">
      `;
      document.body.appendChild(form);
      form.submit();
    }
  </script>

  </body>

</html>
