<?php
require_once 'config/database.php';
require_once 'config/settings.php';
require_once 'config/branding.php';
require_once 'config/session_manager.php';

// Initialize session and check login with timeout
SessionManager::initSession();
SessionManager::requireLogin('admin', 'authentication/login.php');

// Fetch dashboard statistics and admin data
try {
    $pdo = getConnection();
    
    // Get admin details
    $adminStmt = $pdo->prepare("SELECT * FROM admin_users WHERE id = ?");
    $adminStmt->execute([$_SESSION['admin_id']]);
    $adminData = $adminStmt->fetch();
    
    // Get user statistics
    $totalUsers = $pdo->query("SELECT COUNT(*) FROM users")->fetchColumn();
    $activeUsers = $pdo->query("SELECT COUNT(*) FROM users WHERE status = 'active'")->fetchColumn();
    $suspendedUsers = $pdo->query("SELECT COUNT(*) FROM users WHERE status = 'suspended'")->fetchColumn();
    
    // Get product statistics
    $totalProducts = $pdo->query("SELECT COUNT(*) FROM products")->fetchColumn();
    $activeProducts = $pdo->query("SELECT COUNT(*) FROM products WHERE status = 'active'")->fetchColumn();
    $totalRevenue = $pdo->query("SELECT SUM(product_amount) FROM products WHERE status = 'active'")->fetchColumn() ?: 0;
    
    // Get course statistics
    $totalCourses = $pdo->query("SELECT COUNT(*) FROM courses")->fetchColumn();
    $activeCourses = $pdo->query("SELECT COUNT(*) FROM courses WHERE status = 'active'")->fetchColumn();
    $totalEnrollments = $pdo->query("SELECT COUNT(*) FROM course_enrollments")->fetchColumn();
    
    // Get script request statistics
    $totalRequests = $pdo->query("SELECT COUNT(*) FROM script_requests")->fetchColumn();
    $pendingRequests = $pdo->query("SELECT COUNT(*) FROM script_requests WHERE status = 'pending'")->fetchColumn();
    $completedRequests = $pdo->query("SELECT COUNT(*) FROM script_requests WHERE status = 'completed'")->fetchColumn();
    
    // Get order statistics
    $totalOrders = $pdo->query("SELECT COUNT(*) FROM orders")->fetchColumn();
    $paidOrders = $pdo->query("SELECT COUNT(*) FROM orders WHERE payment_status = 'paid'")->fetchColumn();
    $pendingOrders = $pdo->query("SELECT COUNT(*) FROM orders WHERE payment_status = 'pending'")->fetchColumn();
    $orderRevenue = $pdo->query("SELECT SUM(total_amount) FROM orders WHERE payment_status = 'paid'")->fetchColumn() ?: 0;
    
    // Get recent users (last 10)
    $recentUsers = $pdo->query("
        SELECT id, username, full_name, email, status, created_at 
        FROM users 
        ORDER BY created_at DESC 
        LIMIT 10
    ")->fetchAll();
    
    // Get recent orders (last 10)
    $recentOrders = $pdo->query("
        SELECT o.*, u.username, u.full_name 
        FROM orders o 
        LEFT JOIN users u ON o.user_id = u.id 
        ORDER BY o.created_at DESC 
        LIMIT 10
    ")->fetchAll();
    
    // Get recent notifications (product/course purchases and script requests)
    $recentNotifications = $pdo->query("
        (SELECT 'order' as type, o.id, CAST(o.order_number as CHAR) as title, CAST(o.customer_name as CHAR) as user_name, o.total_amount as amount, o.created_at, 'Product Purchase' as description
         FROM orders o 
         WHERE o.payment_status = 'paid' 
         ORDER BY o.created_at DESC 
         LIMIT 5)
        UNION ALL
        (SELECT 'course' as type, ce.id, CAST(c.title as CHAR) as title, CAST(u.full_name as CHAR) as user_name, c.price as amount, ce.enrollment_date as created_at, 'Course Enrollment' as description
         FROM course_enrollments ce 
         JOIN courses c ON ce.course_id = c.id 
         JOIN users u ON ce.user_id = u.id 
         ORDER BY ce.enrollment_date DESC 
         LIMIT 5)
        UNION ALL
        (SELECT 'script_request' as type, sr.id, CAST(CONCAT('Script Request #', sr.id) as CHAR) as title, CAST(sr.full_name as CHAR) as user_name, 0 as amount, sr.created_at, 'New Script Request' as description
         FROM script_requests sr 
         WHERE sr.status = 'pending' 
         ORDER BY sr.created_at DESC 
         LIMIT 5)
        ORDER BY created_at DESC 
        LIMIT 10
    ")->fetchAll();
    
    // Get recent script requests for messages
    $recentMessages = $pdo->query("
        SELECT sr.*, u.username, u.full_name as user_full_name
        FROM script_requests sr
        LEFT JOIN users u ON sr.user_id = u.id
        ORDER BY sr.created_at DESC 
        LIMIT 5
    ")->fetchAll();
    
    // Count unread notifications
    $unreadNotifications = count($recentNotifications);
    $unreadMessages = count($recentMessages);
    
} catch (Exception $e) {
    $error = "Database error: " . $e->getMessage();
    $adminData = null;
    $totalUsers = $activeUsers = $suspendedUsers = 0;
    $totalProducts = $activeProducts = $totalRevenue = 0;
    $totalCourses = $activeCourses = $totalEnrollments = 0;
    $totalRequests = $pendingRequests = $completedRequests = 0;
    $totalOrders = $paidOrders = $pendingOrders = $orderRevenue = 0;
    $recentUsers = $recentOrders = [];
    $recentNotifications = $recentMessages = [];
    $unreadNotifications = $unreadMessages = 0;
}
?>
<!DOCTYPE html>
<html lang="en">

<head>


  <!-- begin::Edura Meta Basic -->
  <meta charset="utf-8">
  <meta name="theme-color" content="#0D9488">
  <meta name="robots" content="index, follow">
  <meta name="author" content="Omateq Digitals">
  <meta name="format-detection" content="telephone=no">
  <meta name="keywords" content="Admin Dashboard, Omateq Digitals, User Management, Product Management, Course Management, Order Management, Script Requests, Admin Panel">
  <meta name="description" content="Omateq Digitals Admin Dashboard - Comprehensive management system for users, products, courses, orders, and script requests.">
  <!-- end::Edura Meta Basic -->

  <!-- begin::Edura Meta Social -->
  <meta property="og:url" content="">
  <meta property="og:site_name" content="Admin Dashboard | Omateq Digitals">
  <meta property="og:type" content="website">
  <meta property="og:locale" content="en_us">
  <meta property="og:title" content="Admin Dashboard | Omateq Digitals">
  <meta property="og:description" content="Omateq Digitals Admin Dashboard - Comprehensive management system for users, products, courses, orders, and script requests.">
  <!-- end::Edura Meta Social -->

  <!-- begin::Edura Meta Twitter -->
  <meta name="twitter:card" content="summary">
  <meta name="twitter:url" content="">
  <meta name="twitter:creator" content="@omateqdigitals">
  <meta name="twitter:title" content="Admin Dashboard | Omateq Digitals">
  <meta name="twitter:description" content="Omateq Digitals Admin Dashboard - Comprehensive management system for users, products, courses, orders, and script requests.">
  <!-- end::Edura Meta Twitter -->

  <!-- begin::Edura Website Page Title -->
  <title>Admin Dashboard | Omateq Digitals</title>
  <!-- end::Edura Website Page Title -->

  <!-- begin::Edura Mobile Specific -->
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <!-- end::Edura Mobile Specific -->

  <!-- begin::Edura Favicon Tags -->
  <?php echo getFaviconHtml(); ?>
  <!-- end::Edura Favicon Tags -->

  <!-- begin::Edura Google Fonts -->
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Plus+Jakarta+Sans:ital,wght@0,200..800;1,200..800&display=swap" rel="stylesheet">
  <!-- end::Edura Google Fonts -->

  <!-- begin::Edura Required Stylesheet -->
  <link rel="stylesheet" href="assets/libs/flaticon/css/all/all.css">
  <link rel="stylesheet" href="assets/libs/lucide/lucide.css">
  <link rel="stylesheet" href="assets/libs/fontawesome/css/all.min.css">
  <link rel="stylesheet" href="assets/libs/simplebar/simplebar.css">
  <link rel="stylesheet" href="assets/libs/node-waves/waves.css">
  <link rel="stylesheet" href="assets/libs/bootstrap-select/css/bootstrap-select.min.css">
  <!-- end::Edura Required Stylesheet -->

  <!-- begin::Edura CSS Stylesheet -->
  <link rel="stylesheet" href="assets/libs/flatpickr/flatpickr.min.css">
  <link rel="stylesheet" href="assets/libs/datatables/datatables.min.css">
  <link rel="stylesheet" href="assets/css/styles.css">
  <!-- end::Edura CSS Stylesheet -->
  
  <style>
    /* Logo sizing for better fit */
    .app-navbar-brand .navbar-brand-logo img {
      max-height: 50px;
      max-width: 180px;
      width: auto;
      height: auto;
      object-fit: contain;
    }
    
    .app-navbar-brand .navbar-brand-mini img {
      max-height: 35px;
      max-width: 35px;
      width: auto;
      height: auto;
      object-fit: contain;
    }
  </style>

  <!-- begin::Edura Googletagmanager -->
  <script async src="https://www.googletagmanager.com/gtag/js?id=G-XWVQM68HHQ"></script>
  <script>
		window.dataLayer = window.dataLayer || [];
		function gtag(){dataLayer.push(arguments);}
		gtag('js', new Date());

		gtag('config', 'G-XWVQM68HHQ', {
			'cookie_flags': 'SameSite=None;Secure',
			'send_page_view': true
		});
	</script>
  <!-- end::Edura Googletagmanager -->

</head>

<body>
  <div class="page-layout">

    <?php include 'includes/header.php'; ?>

    <!-- begin::Edura Sidebar Menu -->
    <aside class="app-menubar" id="menubar">
      <div class="app-navbar-brand">
        <a class="navbar-brand-logo" href="index.php">
          <img src="../<?php echo getSetting('main_logo', 'assets/images/logo.svg'); ?>" alt="logo">
        </a>
        <a class="navbar-brand-mini" href="index.php">
          <img src="../<?php echo getSetting('mini_logo', 'assets/images/logo-text-white.svg'); ?>" alt="logo">
        </a>
      </div>
      <nav class="app-navbar" data-simplebar>
        <ul class="menubar">
              <li class="menu-item">
                <a class="menu-link" href="index.php">
              <i class="fi fi-rr-apps"></i>
                  <span class="menu-label">Dashboard</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/users.php">
              <i class="fi fi-rr-users"></i>
              <span class="menu-label">Manage Users</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/portfolio.php">
              <i class="fi fi-rr-portfolio"></i>
              <span class="menu-label">Manage Portfolio</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/products.php">
              <i class="fi fi-rr-shopping-cart"></i>
              <span class="menu-label">Manage Products</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/orders.php">
              <i class="fi fi-rr-receipt"></i>
              <span class="menu-label">Manage Orders</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/courses.php">
              <i class="fi fi-rr-book-alt"></i>
              <span class="menu-label">Manage Courses</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/script_requests.php">
              <i class="fi fi-rr-file-code"></i>
              <span class="menu-label">Manage Script Request</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/settings.php">
              <i class="fi fi-rr-settings"></i>
              <span class="menu-label">Settings</span>
            </a>
          </li>
        </ul>
      </nav>
      <div class="app-footer">
        <a href="pages/faq.html" class="btn btn-outline-white waves-effect btn-app-nav w-100">
          <i class="fi fi-rs-interrogation"></i>
          <span class="nav-text">Help and Support</span>
        </a>
      </div>
    </aside>
    <!-- end::Edura Sidebar Menu -->

    <main class="app-wrapper">

      <div class="container-fluid">

        <div class="app-page-head d-flex flex-wrap gap-3 align-items-center justify-content-between">
          <div class="clearfix">
            <h1 class="app-page-title">Dashboard</h1>
            <span><?php echo date('M d, Y'); ?></span>
          </div>
        </div>

        <?php if (isset($error)): ?>
          <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fi fi-rr-exclamation-circle me-2"></i><?php echo htmlspecialchars($error); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
          </div>
        <?php endif; ?>

        <!-- Statistics Widgets -->
        <div class="row mb-4">
          <!-- First Row - 2 cards on mobile, 4 on desktop -->
          <div class="col-lg-3 col-md-6 col-6 mb-3">
                <div class="card">
                  <div class="card-body">
                <div class="d-flex align-items-center">
                  <div class="flex-shrink-0">
                    <div class="avatar avatar-sm bg-primary text-white rounded">
                      <i class="fi fi-rr-users"></i>
                    </div>
                    </div>
                  <div class="flex-grow-1 ms-3">
                    <div class="text-muted small">Total Users</div>
                    <div class="h5 mb-0"><?php echo $totalUsers; ?></div>
                  </div>
                  </div>
                </div>
              </div>
          </div>
          <div class="col-lg-3 col-md-6 col-6 mb-3">
                <div class="card">
                  <div class="card-body">
                <div class="d-flex align-items-center">
                  <div class="flex-shrink-0">
                    <div class="avatar avatar-sm bg-success text-white rounded">
                      <i class="fi fi-rr-shopping-cart"></i>
                    </div>
                    </div>
                  <div class="flex-grow-1 ms-3">
                    <div class="text-muted small">Total Products</div>
                    <div class="h5 mb-0"><?php echo $totalProducts; ?></div>
                  </div>
                  </div>
                </div>
              </div>
          </div>
          <!-- Second Row - 2 cards on mobile, 4 on desktop -->
          <div class="col-lg-3 col-md-6 col-6 mb-3">
                <div class="card">
                  <div class="card-body">
                <div class="d-flex align-items-center">
                  <div class="flex-shrink-0">
                    <div class="avatar avatar-sm bg-info text-white rounded">
                      <i class="fi fi-rr-book-alt"></i>
                    </div>
                    </div>
                  <div class="flex-grow-1 ms-3">
                    <div class="text-muted small">Total Courses</div>
                    <div class="h5 mb-0"><?php echo $totalCourses; ?></div>
                  </div>
                  </div>
                </div>
              </div>
          </div>
          <div class="col-lg-3 col-md-6 col-6 mb-3">
                <div class="card">
                  <div class="card-body">
                <div class="d-flex align-items-center">
                  <div class="flex-shrink-0">
                    <div class="avatar avatar-sm bg-warning text-white rounded">
                      <i class="fi fi-rr-file-code"></i>
                    </div>
                    </div>
                  <div class="flex-grow-1 ms-3">
                    <div class="text-muted small">Script Requests</div>
                    <div class="h5 mb-0"><?php echo $totalRequests; ?></div>
                  </div>
                </div>
              </div>
                  </div>
                </div>
              </div>

        <!-- Additional Statistics Row -->
        <div class="row mb-4">
          <div class="col-lg-3 col-md-6 col-6 mb-3">
                <div class="card">
              <div class="card-body">
                <div class="d-flex align-items-center">
                  <div class="flex-shrink-0">
                    <div class="avatar avatar-sm bg-success text-white rounded">
                      <i class="fi fi-rr-check"></i>
                  </div>
                  </div>
                  <div class="flex-grow-1 ms-3">
                    <div class="text-muted small">Active Users</div>
                    <div class="h5 mb-0"><?php echo $activeUsers; ?></div>
                </div>
              </div>
                  </div>
            </div>
          </div>
          <div class="col-lg-3 col-md-6 col-6 mb-3">
            <div class="card">
                  <div class="card-body">
                <div class="d-flex align-items-center">
                  <div class="flex-shrink-0">
                    <div class="avatar avatar-sm bg-info text-white rounded">
                      <i class="fi fi-rr-receipt"></i>
                      </div>
                    </div>
                  <div class="flex-grow-1 ms-3">
                    <div class="text-muted small">Total Orders</div>
                    <div class="h5 mb-0"><?php echo $totalOrders; ?></div>
                  </div>
                  </div>
                </div>
              </div>
                  </div>
          <div class="col-lg-3 col-md-6 col-6 mb-3">
            <div class="card">
                  <div class="card-body">
                <div class="d-flex align-items-center">
                  <div class="flex-shrink-0">
                    <div class="avatar avatar-sm bg-success text-white rounded">
                      <i class="fi fi-rr-dollar"></i>
                      </div>
                    </div>
                  <div class="flex-grow-1 ms-3">
                    <div class="text-muted small">Revenue</div>
                    <div class="h5 mb-0">$<?php echo number_format($orderRevenue, 2); ?></div>
                  </div>
                  </div>
                </div>
              </div>
                  </div>
          <div class="col-lg-3 col-md-6 col-6 mb-3">
            <div class="card">
                  <div class="card-body">
                <div class="d-flex align-items-center">
                  <div class="flex-shrink-0">
                    <div class="avatar avatar-sm bg-warning text-white rounded">
                      <i class="fi fi-rr-clock"></i>
                      </div>
                    </div>
                  <div class="flex-grow-1 ms-3">
                    <div class="text-muted small">Pending Requests</div>
                    <div class="h5 mb-0"><?php echo $pendingRequests; ?></div>
                  </div>
                  </div>
                </div>
              </div>
            </div>
          </div>

            <div class="row">
          <!-- Users Table -->
          <div class="col-12 mb-4">
                <div class="card">
              <div class="card-header d-flex align-items-center justify-content-between border-0">
                <h6 class="card-title mb-0">Recent Users</h6>
                <a href="users.php" class="btn btn-sm btn-outline-primary">View All</a>
                  </div>
              <div class="card-body p-0">
                <div class="table-responsive">
                  <table class="table table-hover mb-0">
                    <thead class="table-light">
                      <tr>
                        <th>User</th>
                        <th>Status</th>
                        <th>Joined</th>
                      </tr>
                    </thead>
                    <tbody>
                      <?php if (empty($recentUsers)): ?>
                        <tr>
                          <td colspan="3" class="text-center py-3">
                            <div class="d-flex flex-column align-items-center">
                              <i class="fi fi-rr-users text-muted mb-2" style="font-size: 2rem;"></i>
                              <p class="text-muted mb-0">No users found</p>
                  </div>
                          </td>
                        </tr>
                      <?php else: ?>
                        <?php foreach ($recentUsers as $user): ?>
                          <tr>
                            <td>
                              <div class="d-flex align-items-center">
                                <div class="avatar avatar-sm bg-primary text-white rounded-circle me-2">
                                  <?php echo strtoupper(substr($user['username'], 0, 1)); ?>
                </div>
                                <div>
                                  <div class="fw-semibold"><?php echo htmlspecialchars($user['full_name'] ?: $user['username']); ?></div>
                                  <small class="text-muted"><?php echo htmlspecialchars($user['email']); ?></small>
              </div>
                  </div>
                            </td>
                            <td>
                              <?php
                              $statusClass = '';
                              switch ($user['status']) {
                                case 'active':
                                  $statusClass = 'bg-success-subtle text-success';
                                  break;
                                case 'suspended':
                                  $statusClass = 'bg-warning-subtle text-warning';
                                  break;
                                case 'inactive':
                                  $statusClass = 'bg-secondary-subtle text-secondary';
                                  break;
                                default:
                                  $statusClass = 'bg-secondary-subtle text-secondary';
                              }
                              ?>
                              <span class="badge <?php echo $statusClass; ?>"><?php echo ucfirst($user['status']); ?></span>
                            </td>
                            <td>
                              <small class="text-muted"><?php echo date('M d, Y', strtotime($user['created_at'])); ?></small>
                            </td>
                          </tr>
                        <?php endforeach; ?>
                      <?php endif; ?>
                    </tbody>
                  </table>
                      </div>
                  </div>
                </div>
              </div>

          <!-- Orders Table -->
          <div class="col-12 mb-4">
                <div class="card">
              <div class="card-header d-flex align-items-center justify-content-between border-0">
                <h6 class="card-title mb-0">Recent Orders</h6>
                <a href="orders.php" class="btn btn-sm btn-outline-primary">View All</a>
                    </div>
              <div class="card-body p-0">
                <div class="table-responsive">
                  <table class="table table-hover mb-0">
                    <thead class="table-light">
                      <tr>
                        <th>Order #</th>
                        <th>Customer</th>
                        <th>Amount</th>
                        <th>Status</th>
                      </tr>
                    </thead>
                    <tbody>
                      <?php if (empty($recentOrders)): ?>
                        <tr>
                          <td colspan="4" class="text-center py-3">
                            <div class="d-flex flex-column align-items-center">
                              <i class="fi fi-rr-receipt text-muted mb-2" style="font-size: 2rem;"></i>
                              <p class="text-muted mb-0">No orders found</p>
                  </div>
                          </td>
                        </tr>
                      <?php else: ?>
                        <?php foreach ($recentOrders as $order): ?>
                          <tr>
                            <td>
                              <span class="text-primary">#<?php echo $order['order_number']; ?></span>
                            </td>
                            <td>
                              <div>
                                <div class="fw-semibold"><?php echo htmlspecialchars($order['customer_name']); ?></div>
                                <small class="text-muted"><?php echo htmlspecialchars($order['customer_email']); ?></small>
                        </div>
                            </td>
                            <td>
                              <span class="fw-semibold text-success">$<?php echo number_format($order['total_amount'], 2); ?></span>
                            </td>
                            <td>
                              <?php
                              $statusClass = '';
                              switch ($order['status']) {
                                case 'pending':
                                  $statusClass = 'bg-warning-subtle text-warning';
                                  break;
                                case 'processing':
                                  $statusClass = 'bg-info-subtle text-info';
                                  break;
                                case 'delivered':
                                  $statusClass = 'bg-success-subtle text-success';
                                  break;
                                case 'cancelled':
                                  $statusClass = 'bg-danger-subtle text-danger';
                                  break;
                                default:
                                  $statusClass = 'bg-secondary-subtle text-secondary';
                              }
                              ?>
                              <span class="badge <?php echo $statusClass; ?>"><?php echo ucfirst($order['status']); ?></span>
                            </td>
                          </tr>
                        <?php endforeach; ?>
                      <?php endif; ?>
                    </tbody>
                  </table>
                        </div>
                        </div>
                        </div>
            </div>
          </div>

        </div>

      </div>

    </main>

    <!-- begin::Edura Footer -->
    <footer class="footer-wrapper bg-body text-center">
      <div class="container-fluid">
        <p class="mb-0">© <span class="currentYear">2025</span> Omateq Digitals. All rights reserved.</p>
      </div>
    </footer>
    <!-- end::Edura Footer -->

  </div>
  <!-- begin::Edura Page Scripts -->
  <script src="assets/libs/global/global.min.js"></script>
  <script src="assets/libs/sortable/Sortable.min.js"></script>
  <script src="assets/libs/chartjs/chart.js"></script>
  <script src="assets/libs/flatpickr/flatpickr.min.js"></script>
  <script src="assets/libs/apexcharts/apexcharts.min.js"></script>
  <script src="assets/libs/datatables/datatables.min.js"></script>
  <script src="assets/js/dashboard.js"></script>
  <script src="assets/js/appSettings.js"></script>
  <script src="assets/js/main.js"></script>
  <!-- end::Edura Page Scripts -->
</body>

</html>