<?php
require_once 'config/database.php';
require_once 'config/settings.php';
require_once 'config/branding.php';

$error = '';
$success = '';

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: authentication/login.php');
    exit();
}

// Handle order actions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $action = $_POST['action'] ?? '';
    $order_id = $_POST['order_id'] ?? '';
    
    try {
        $pdo = getConnection();
        
        switch ($action) {
            case 'update_status':
                $status = $_POST['status'] ?? '';
                $admin_notes = trim($_POST['admin_notes'] ?? '');
                
                if (empty($status)) {
                    $error = "Status is required";
                } else {
                    $stmt = $pdo->prepare("
                        UPDATE orders 
                        SET status = ?, admin_notes = ?, updated_at = NOW()
                        WHERE id = ?
                    ");
                    $stmt->execute([$status, $admin_notes, $order_id]);
                    $success = "Order status updated successfully";
                }
                break;
                
            case 'update_payment_status':
                $payment_status = $_POST['payment_status'] ?? '';
                $payment_method = trim($_POST['payment_method'] ?? '');
                $payment_reference = trim($_POST['payment_reference'] ?? '');
                
                if (empty($payment_status)) {
                    $error = "Payment status is required";
                } else {
                    $stmt = $pdo->prepare("
                        UPDATE orders 
                        SET payment_status = ?, payment_method = ?, payment_reference = ?, updated_at = NOW()
                        WHERE id = ?
                    ");
                    $stmt->execute([$payment_status, $payment_method, $payment_reference, $order_id]);
                    $success = "Payment status updated successfully";
                }
                break;
                
            case 'add_notes':
                $admin_notes = trim($_POST['admin_notes'] ?? '');
                
                if (empty($admin_notes)) {
                    $error = "Admin notes cannot be empty";
                } else {
                    $stmt = $pdo->prepare("
                        UPDATE orders 
                        SET admin_notes = ?, updated_at = NOW()
                        WHERE id = ?
                    ");
                    $stmt->execute([$admin_notes, $order_id]);
                    $success = "Admin notes updated successfully";
                }
                break;
                
            case 'delete':
                $stmt = $pdo->prepare("DELETE FROM orders WHERE id = ?");
                $stmt->execute([$order_id]);
                $success = "Order deleted successfully";
                break;
        }
    } catch (Exception $e) {
        $error = "Action failed: " . $e->getMessage();
    }
}

// Get filter parameters
$status_filter = $_GET['status'] ?? '';
$payment_status_filter = $_GET['payment_status'] ?? '';
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';
$search = $_GET['search'] ?? '';

// Get orders with pagination and filtering
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 10;
$offset = ($page - 1) * $limit;

try {
    $pdo = getConnection();
    
    // Build WHERE clause for filtering
    $where_conditions = [];
    $params = [];
    
    if (!empty($status_filter)) {
        $where_conditions[] = "status = ?";
        $params[] = $status_filter;
    }
    
    if (!empty($payment_status_filter)) {
        $where_conditions[] = "payment_status = ?";
        $params[] = $payment_status_filter;
    }
    
    if (!empty($date_from)) {
        $where_conditions[] = "DATE(created_at) >= ?";
        $params[] = $date_from;
    }
    
    if (!empty($date_to)) {
        $where_conditions[] = "DATE(created_at) <= ?";
        $params[] = $date_to;
    }
    
    if (!empty($search)) {
        $where_conditions[] = "(order_number LIKE ? OR customer_name LIKE ? OR customer_email LIKE ? OR product_name LIKE ?)";
        $search_term = "%$search%";
        $params[] = $search_term;
        $params[] = $search_term;
        $params[] = $search_term;
        $params[] = $search_term;
    }
    
    $where_clause = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";
    
    // Get total count
    $count_sql = "SELECT COUNT(*) FROM orders $where_clause";
    $countStmt = $pdo->prepare($count_sql);
    $countStmt->execute($params);
    $totalOrders = $countStmt->fetchColumn();
    $totalPages = ceil($totalOrders / $limit);
    
    // Get orders
    $orders_sql = "
        SELECT o.*, u.username, u.full_name as user_full_name, p.product_name as actual_product_name
        FROM orders o
        LEFT JOIN users u ON o.user_id = u.id
        LEFT JOIN products p ON o.product_id = p.id
        $where_clause
        ORDER BY o.created_at DESC 
        LIMIT :limit OFFSET :offset
    ";
    $stmt = $pdo->prepare($orders_sql);
    
    // Bind filter parameters
    foreach ($params as $i => $param) {
        $stmt->bindValue($i + 1, $param);
    }
    
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $orders = $stmt->fetchAll();
    
    // Get order statistics
    $pendingOrders = $pdo->query("SELECT COUNT(*) FROM orders WHERE status = 'pending'")->fetchColumn();
    $processingOrders = $pdo->query("SELECT COUNT(*) FROM orders WHERE status = 'processing'")->fetchColumn();
    $deliveredOrders = $pdo->query("SELECT COUNT(*) FROM orders WHERE status = 'delivered'")->fetchColumn();
    $cancelledOrders = $pdo->query("SELECT COUNT(*) FROM orders WHERE status = 'cancelled'")->fetchColumn();
    
    // Get payment statistics
    $paidOrders = $pdo->query("SELECT COUNT(*) FROM orders WHERE payment_status = 'paid'")->fetchColumn();
    $pendingPayments = $pdo->query("SELECT COUNT(*) FROM orders WHERE payment_status = 'pending'")->fetchColumn();
    $refundedOrders = $pdo->query("SELECT COUNT(*) FROM orders WHERE payment_status = 'refunded'")->fetchColumn();
    
    // Calculate total revenue
    $revenueStmt = $pdo->query("SELECT SUM(total_amount) FROM orders WHERE payment_status = 'paid'");
    $totalRevenue = $revenueStmt->fetchColumn() ?: 0;
    
    // Calculate today's revenue
    $todayRevenueStmt = $pdo->query("SELECT SUM(total_amount) FROM orders WHERE payment_status = 'paid' AND DATE(created_at) = CURDATE()");
    $todayRevenue = $todayRevenueStmt->fetchColumn() ?: 0;
    
} catch (Exception $e) {
    $error = "Database error: " . $e->getMessage();
    $orders = [];
    $totalPages = 0;
    $totalOrders = 0;
    $pendingOrders = 0;
    $processingOrders = 0;
    $deliveredOrders = 0;
    $cancelledOrders = 0;
    $paidOrders = 0;
    $pendingPayments = 0;
    $refundedOrders = 0;
    $totalRevenue = 0;
    $todayRevenue = 0;
}
?>
<!DOCTYPE html>
<html lang="en">

<head>

  <base href="">

  <!-- begin::Edura Meta Basic -->
  <meta charset="utf-8">
  <meta name="theme-color" content="#0D9488">
  <meta name="robots" content="index, follow">
  <meta name="author" content="Omateq Digitals">
  <meta name="format-detection" content="telephone=no">
  <meta name="keywords" content="Admin Dashboard, Omateq Digitals, Order Management, Admin Panel, E-commerce Orders">
  <meta name="description" content="Omateq Digitals Admin Dashboard - Manage customer orders, track order status, and process payments.">
  <!-- end::Edura Meta Basic -->

  <!-- begin::Edura Meta Social -->
  <meta property="og:url" content="">
  <meta property="og:site_name" content="Manage Orders | Omateq Digitals">
  <meta property="og:type" content="website">
  <meta property="og:locale" content="en_us">
  <meta property="og:title" content="Manage Orders | Omateq Digitals">
  <meta property="og:description" content="Omateq Digitals Admin Dashboard - Manage customer orders, track order status, and process payments.">
  <!-- end::Edura Meta Social -->

  <!-- begin::Edura Meta Twitter -->
  <meta name="twitter:card" content="summary">
  <meta name="twitter:url" content="">
  <meta name="twitter:creator" content="@omateqdigitals">
  <meta name="twitter:title" content="Manage Orders | Omateq Digitals">
  <meta name="twitter:description" content="Omateq Digitals Admin Dashboard - Manage customer orders, track order status, and process payments.">
  <!-- end::Edura Meta Twitter -->

  <!-- begin::Edura Website Page Title -->
  <title>Manage Orders | Omateq Digitals</title>
  <!-- end::Edura Website Page Title -->

  <!-- begin::Edura Mobile Specific -->
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <!-- end::Edura Mobile Specific -->

  <!-- begin::Edura Favicon Tags -->
  <?php echo getFaviconHtml(); ?>
  <!-- end::Edura Favicon Tags -->

  <!-- begin::Edura Google Fonts -->
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Plus+Jakarta+Sans:ital,wght@0,200..800;1,200..800&display=swap" rel="stylesheet">
  <!-- end::Edura Google Fonts -->

  <!-- begin::Edura Required Stylesheet -->
  <link rel="stylesheet" href="assets/libs/flaticon/css/all/all.css">
  <link rel="stylesheet" href="assets/libs/lucide/lucide.css">
  <link rel="stylesheet" href="assets/libs/fontawesome/css/all.min.css">
  <link rel="stylesheet" href="assets/libs/simplebar/simplebar.css">
  <link rel="stylesheet" href="assets/libs/node-waves/waves.css">
  <link rel="stylesheet" href="assets/libs/bootstrap-select/css/bootstrap-select.min.css">
  <!-- end::Edura Required Stylesheet -->

  <!-- begin::Edura CSS Stylesheet -->
  <link rel="stylesheet" href="assets/css/styles.css">
  <!-- end::Edura CSS Stylesheet -->

  <!-- begin::Edura Googletagmanager -->
  <script async src="https://www.googletagmanager.com/gtag/js?id=G-XWVQM68HHQ"></script>
  <script>
		window.dataLayer = window.dataLayer || [];
		function gtag(){dataLayer.push(arguments);}
		gtag('js', new Date());

		gtag('config', 'G-XWVQM68HHQ', {
			'cookie_flags': 'SameSite=None;Secure',
			'send_page_view': true
		});
	</script>
  <!-- end::Edura Googletagmanager -->

</head>

<body>
  <div class="page-layout">

    <?php include 'includes/header.php'; ?>

     <!-- begin::Edura Sidebar Menu -->
     <aside class="app-menubar" id="menubar">
      <div class="app-navbar-brand">
        <a class="navbar-brand-logo" href="index.php">
          <img src="../<?php echo getSetting('main_logo', 'assets/images/logo.svg'); ?>" alt="logo">
        </a>
        <a class="navbar-brand-mini" href="index.php">
          <img src="../<?php echo getSetting('mini_logo', 'assets/images/logo-text-white.svg'); ?>" alt="logo">
        </a>
      </div>
      <nav class="app-navbar" data-simplebar>
        <ul class="menubar">
          <li class="menu-item">
            <a class="menu-link" href="index.php">
              <i class="fi fi-rr-apps"></i>
              <span class="menu-label">Dashboard</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/users.php">
              <i class="fi fi-rr-users"></i>
              <span class="menu-label">Manage Users</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/portfolio.php">
              <i class="fi fi-rr-portfolio"></i>
              <span class="menu-label">Manage Portfolio</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/products.php">
              <i class="fi fi-rr-shopping-cart"></i>
              <span class="menu-label">Manage Products</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/orders.php">
              <i class="fi fi-rr-receipt"></i>
              <span class="menu-label">Manage Orders</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/courses.php">
              <i class="fi fi-rr-book-alt"></i>
              <span class="menu-label">Manage Courses</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/script_requests.php">
              <i class="fi fi-rr-file-code"></i>
              <span class="menu-label">Manage Script Request</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/settings.php">
              <i class="fi fi-rr-settings"></i>
              <span class="menu-label">Settings</span>
            </a>
          </li>
        </ul>
      </nav>
      <div class="app-footer">
        <a href="pages/faq.html" class="btn btn-outline-white waves-effect btn-app-nav w-100">
          <i class="fi fi-rs-interrogation"></i>
          <span class="nav-text">Help and Support</span>
        </a>
      </div>
    </aside>
    <!-- end::Edura Sidebar Menu -->

    <main class="app-wrapper">

      <div class="container-fluid">

        <div class="app-page-head d-flex flex-wrap gap-3 align-items-center justify-content-between">
          <div class="clearfix">
            <h1 class="app-page-title">Manage Orders</h1>
            <nav aria-label="breadcrumb">
              <ol class="breadcrumb mb-0">
                <li class="breadcrumb-item">
                  <a href="index.php">Dashboard</a>
                </li>
                <li class="breadcrumb-item active" aria-current="page">Manage Orders</li>
              </ol>
            </nav>
          </div>
        </div>

        <?php if ($success): ?>
          <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fi fi-rr-check-circle me-2"></i><?php echo htmlspecialchars($success); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
          </div>
        <?php endif; ?>
        
        <?php if ($error): ?>
          <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fi fi-rr-exclamation-circle me-2"></i><?php echo htmlspecialchars($error); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
          </div>
        <?php endif; ?>

        <!-- Statistics Cards -->
        <div class="row mb-4">
          <!-- First Row - 2 cards on mobile, 4 on desktop -->
          <div class="col-lg-3 col-md-6 col-6 mb-3">
            <div class="card">
              <div class="card-body">
                <div class="d-flex align-items-center">
                  <div class="flex-shrink-0">
                    <div class="avatar avatar-sm bg-primary text-white rounded">
                      <i class="fi fi-rr-receipt"></i>
                    </div>
                  </div>
                  <div class="flex-grow-1 ms-3">
                    <div class="text-muted small">Total Orders</div>
                    <div class="h5 mb-0"><?php echo $totalOrders; ?></div>
                  </div>
                </div>
              </div>
            </div>
          </div>
          <div class="col-lg-3 col-md-6 col-6 mb-3">
            <div class="card">
              <div class="card-body">
                <div class="d-flex align-items-center">
                  <div class="flex-shrink-0">
                    <div class="avatar avatar-sm bg-warning text-white rounded">
                      <i class="fi fi-rr-clock"></i>
                    </div>
                  </div>
                  <div class="flex-grow-1 ms-3">
                    <div class="text-muted small">Pending Orders</div>
                    <div class="h5 mb-0"><?php echo $pendingOrders; ?></div>
                  </div>
                </div>
              </div>
            </div>
          </div>
          <!-- Second Row - 2 cards on mobile, 4 on desktop -->
          <div class="col-lg-3 col-md-6 col-6 mb-3">
            <div class="card">
              <div class="card-body">
                <div class="d-flex align-items-center">
                  <div class="flex-shrink-0">
                    <div class="avatar avatar-sm bg-success text-white rounded">
                      <i class="fi fi-rr-check"></i>
                    </div>
                  </div>
                  <div class="flex-grow-1 ms-3">
                    <div class="text-muted small">Delivered</div>
                    <div class="h5 mb-0"><?php echo $deliveredOrders; ?></div>
                  </div>
                </div>
              </div>
            </div>
          </div>
          <div class="col-lg-3 col-md-6 col-6 mb-3">
            <div class="card">
              <div class="card-body">
                <div class="d-flex align-items-center">
                  <div class="flex-shrink-0">
                    <div class="avatar avatar-sm bg-info text-white rounded">
                      <i class="fi fi-rr-dollar"></i>
                    </div>
                  </div>
                  <div class="flex-grow-1 ms-3">
                    <div class="text-muted small">Total Revenue</div>
                    <div class="h5 mb-0">$<?php echo number_format($totalRevenue, 2); ?></div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>

        <!-- Filters -->
        <div class="row mb-4">
          <div class="col-lg-12">
            <div class="card">
              <div class="card-body">
                <form method="GET" class="row g-3">
                  <div class="col-md-3">
                    <label for="status" class="form-label">Order Status</label>
                    <select class="form-select" id="status" name="status">
                      <option value="">All Statuses</option>
                      <option value="pending" <?php echo $status_filter === 'pending' ? 'selected' : ''; ?>>Pending</option>
                      <option value="processing" <?php echo $status_filter === 'processing' ? 'selected' : ''; ?>>Processing</option>
                      <option value="shipped" <?php echo $status_filter === 'shipped' ? 'selected' : ''; ?>>Shipped</option>
                      <option value="delivered" <?php echo $status_filter === 'delivered' ? 'selected' : ''; ?>>Delivered</option>
                      <option value="cancelled" <?php echo $status_filter === 'cancelled' ? 'selected' : ''; ?>>Cancelled</option>
                      <option value="refunded" <?php echo $status_filter === 'refunded' ? 'selected' : ''; ?>>Refunded</option>
                    </select>
                  </div>
                  <div class="col-md-3">
                    <label for="payment_status" class="form-label">Payment Status</label>
                    <select class="form-select" id="payment_status" name="payment_status">
                      <option value="">All Payments</option>
                      <option value="pending" <?php echo $payment_status_filter === 'pending' ? 'selected' : ''; ?>>Pending</option>
                      <option value="paid" <?php echo $payment_status_filter === 'paid' ? 'selected' : ''; ?>>Paid</option>
                      <option value="failed" <?php echo $payment_status_filter === 'failed' ? 'selected' : ''; ?>>Failed</option>
                      <option value="refunded" <?php echo $payment_status_filter === 'refunded' ? 'selected' : ''; ?>>Refunded</option>
                    </select>
                  </div>
                  <div class="col-md-2">
                    <label for="date_from" class="form-label">From Date</label>
                    <input type="date" class="form-control" id="date_from" name="date_from" value="<?php echo htmlspecialchars($date_from); ?>">
                  </div>
                  <div class="col-md-2">
                    <label for="date_to" class="form-label">To Date</label>
                    <input type="date" class="form-control" id="date_to" name="date_to" value="<?php echo htmlspecialchars($date_to); ?>">
                  </div>
                  <div class="col-md-2">
                    <label for="search" class="form-label">Search</label>
                    <input type="text" class="form-control" id="search" name="search" placeholder="Order #, Name, Email..." value="<?php echo htmlspecialchars($search); ?>">
                  </div>
                  <div class="col-md-12">
                    <button type="submit" class="btn btn-primary">
                      <i class="fi fi-rr-search"></i> Filter Orders
                    </button>
                    <a href="orders.php" class="btn btn-outline-secondary">
                      <i class="fi fi-rr-refresh"></i> Clear Filters
                    </a>
                  </div>
                </form>
              </div>
            </div>
          </div>
        </div>

        <div class="row">

          <div class="col-lg-12">
            <div class="card overflow-hidden">
              <div class="card-header d-flex align-items-center justify-content-between border-0">
                <h6 class="card-title mb-0">Orders List (<?php echo $totalOrders; ?> total orders)</h6>
                <div class="d-flex align-items-center gap-2">
                  <div class="mail-header">
                    <form class="d-flex align-items-center h-100 w-200px w-md-250px position-relative me-auto" action="orders.php" method="GET">
                      <button type="button" class="btn btn-sm border-0 position-absolute start-0 ms-3 p-0">
                        <i class="fi fi-rr-search"></i>
                      </button>
                      <input type="text" class="form-control ps-5" placeholder="Search orders..." name="search" value="<?php echo htmlspecialchars($search); ?>">
                    </form>
                  </div>
                </div>
              </div>
              <div class="card-body p-0">
                <div class="table table-responsive mb-0">
                  <table class="table mb-0">
                    <thead class="table-light">
                      <tr>
                        <th class="minw-100px">Order #</th>
                        <th class="minw-150px">Customer</th>
                        <th class="minw-200px">Product</th>
                        <th class="minw-100px">Amount</th>
                        <th class="minw-100px">Status</th>
                        <th class="minw-100px">Payment</th>
                        <th class="minw-150px">Date</th>
                        <th class="minw-200px">Actions</th>
                      </tr>
                    </thead>
                    <tbody>
                      <?php if (empty($orders)): ?>
                        <tr>
                          <td colspan="8" class="text-center py-4">
                            <div class="d-flex flex-column align-items-center">
                              <i class="fi fi-rr-receipt text-muted mb-2" style="font-size: 2rem;"></i>
                              <p class="text-muted mb-0">No orders found</p>
                            </div>
                          </td>
                        </tr>
                      <?php else: ?>
                        <?php foreach ($orders as $order): ?>
                          <tr>
                            <td>
                              <span class="text-primary fw-semibold"><?php echo htmlspecialchars($order['order_number']); ?></span>
                            </td>
                            <td>
                              <div class="mw-150px">
                                <div class="fw-semibold"><?php echo htmlspecialchars($order['customer_name']); ?></div>
                                <small class="text-muted"><?php echo htmlspecialchars($order['customer_email']); ?></small>
                                <?php if ($order['customer_phone']): ?>
                                  <br><small class="text-muted"><?php echo htmlspecialchars($order['customer_phone']); ?></small>
                                <?php endif; ?>
                              </div>
                            </td>
                            <td>
                              <div class="mw-200px">
                                <div class="fw-semibold"><?php echo htmlspecialchars($order['product_name']); ?></div>
                                <small class="text-muted">Qty: <?php echo $order['quantity']; ?></small>
                              </div>
                            </td>
                            <td>
                              <span class="fw-semibold text-success">$<?php echo number_format($order['total_amount'], 2); ?></span>
                            </td>
                            <td>
                              <?php
                              $statusClass = '';
                              switch ($order['status']) {
                                case 'pending':
                                  $statusClass = 'bg-warning-subtle text-warning';
                                  break;
                                case 'processing':
                                  $statusClass = 'bg-info-subtle text-info';
                                  break;
                                case 'shipped':
                                  $statusClass = 'bg-primary-subtle text-primary';
                                  break;
                                case 'delivered':
                                  $statusClass = 'bg-success-subtle text-success';
                                  break;
                                case 'cancelled':
                                  $statusClass = 'bg-danger-subtle text-danger';
                                  break;
                                case 'refunded':
                                  $statusClass = 'bg-secondary-subtle text-secondary';
                                  break;
                                default:
                                  $statusClass = 'bg-secondary-subtle text-secondary';
                              }
                              ?>
                              <span class="badge <?php echo $statusClass; ?>"><?php echo ucfirst($order['status']); ?></span>
                            </td>
                            <td>
                              <?php
                              $paymentClass = '';
                              switch ($order['payment_status']) {
                                case 'pending':
                                  $paymentClass = 'bg-warning-subtle text-warning';
                                  break;
                                case 'paid':
                                  $paymentClass = 'bg-success-subtle text-success';
                                  break;
                                case 'failed':
                                  $paymentClass = 'bg-danger-subtle text-danger';
                                  break;
                                case 'refunded':
                                  $paymentClass = 'bg-secondary-subtle text-secondary';
                                  break;
                                default:
                                  $paymentClass = 'bg-secondary-subtle text-secondary';
                              }
                              ?>
                              <span class="badge <?php echo $paymentClass; ?>"><?php echo ucfirst($order['payment_status']); ?></span>
                            </td>
                            <td><?php echo date('M d, Y', strtotime($order['created_at'])); ?></td>
                            <td>
                              <div class="d-flex gap-1 align-items-center">
                                <button class="btn btn-sm btn-outline-primary" onclick="viewOrder(<?php echo htmlspecialchars(json_encode($order)); ?>)">
                                  <i class="fi fi-rr-eye"></i>
                                </button>
                                
                                <button class="btn btn-sm btn-outline-warning" onclick="updateOrderStatus(<?php echo htmlspecialchars(json_encode($order)); ?>)">
                                  <i class="fi fi-rr-edit"></i>
                                </button>
                                
                                <button class="btn btn-sm btn-outline-info" onclick="updatePaymentStatus(<?php echo htmlspecialchars(json_encode($order)); ?>)">
                                  <i class="fi fi-rr-credit-card"></i>
                                </button>
                                
                                <form method="POST" style="display: inline;">
                                  <input type="hidden" name="action" value="delete">
                                  <input type="hidden" name="order_id" value="<?php echo $order['id']; ?>">
                                  <button type="submit" class="btn btn-sm btn-danger" onclick="return confirm('Are you sure you want to delete this order? This action cannot be undone.')">
                                    <i class="fi fi-rr-trash"></i>
                                  </button>
                                </form>
                              </div>
                            </td>
                          </tr>
                        <?php endforeach; ?>
                      <?php endif; ?>
                    </tbody>
                  </table>
                </div>
              </div>
              <div class="card-footer border-0 p-3">
                <nav aria-label="Page navigation example">
                  <ul class="pagination pagination-sm mb-0">
                    <?php if ($page > 1): ?>
                      <li class="page-item">
                        <a class="page-link" href="?page=<?php echo $page - 1; ?>" aria-label="Previous">
                          <i class="fi fi-rr-angle-double-left"></i>
                        </a>
                      </li>
                    <?php endif; ?>
                    
                    <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                      <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                        <a class="page-link" href="?page=<?php echo $i; ?>"><?php echo $i; ?></a>
                      </li>
                    <?php endfor; ?>
                    
                    <?php if ($page < $totalPages): ?>
                      <li class="page-item">
                        <a class="page-link" href="?page=<?php echo $page + 1; ?>" aria-label="Next">
                          <i class="fi fi-rr-angle-double-right"></i>
                        </a>
                      </li>
                    <?php endif; ?>
                  </ul>
                </nav>
              </div>
            </div>
          </div>

        </div>

      </div>

    </main>

    <!-- begin::Edura Footer -->
    <footer class="footer-wrapper bg-body text-center">
      <div class="container-fluid">
        <p class="mb-0">© <span class="currentYear">2025</span> Edura. Proudly powered by <a href="javascript:void(0);">LayoutDrop</a>.</p>
      </div>
    </footer>
    <!-- end::Edura Footer -->

  </div>

  <!-- begin::Edura Page Scripts -->
  <script src="assets/libs/global/global.min.js"></script>
  <script src="assets/js/appSettings.js"></script>
  <script src="assets/js/main.js"></script>
  <!-- end::Edura Page Scripts -->
  <!-- Add Portfolio Modal -->
  <div class="modal fade" id="addPortfolioModal" tabindex="-1" aria-labelledby="addPortfolioModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
      <div class="modal-content">
        <form method="POST" enctype="multipart/form-data">
          <input type="hidden" name="action" value="create">
          <div class="modal-header">
            <h5 class="modal-title" id="addPortfolioModalLabel">Add New Portfolio Item</h5>
            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
          </div>
          <div class="modal-body">
            <div class="row">
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="title" class="form-label">Title *</label>
                  <input type="text" class="form-control" id="title" name="title" required>
                </div>
              </div>
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="category" class="form-label">Category *</label>
                  <select class="form-select" id="category" name="category" required>
                    <option value="">Select Category</option>
                    <option value="website">Website</option>
                    <option value="webapplication">Web Application</option>
                    <option value="software">Software</option>
                  </select>
                </div>
              </div>
            </div>
            <div class="mb-3">
              <label for="description" class="form-label">Description</label>
              <textarea class="form-control" id="description" name="description" rows="3"></textarea>
            </div>
            <div class="row">
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="project_url" class="form-label">Project URL</label>
                  <input type="url" class="form-control" id="project_url" name="project_url">
                </div>
              </div>
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="sort_order" class="form-label">Sort Order</label>
                  <input type="number" class="form-control" id="sort_order" name="sort_order" value="0">
                </div>
              </div>
            </div>
            <div class="mb-3">
              <label for="technologies" class="form-label">Technologies Used</label>
              <input type="text" class="form-control" id="technologies" name="technologies" placeholder="e.g., PHP, JavaScript, MySQL, Bootstrap">
            </div>
            <div class="mb-3">
              <label for="features" class="form-label">Key Features</label>
              <textarea class="form-control" id="features" name="features" rows="2" placeholder="e.g., Responsive Design, User Authentication, Payment Integration"></textarea>
            </div>
            <div class="mb-3">
              <label for="image" class="form-label">Project Image</label>
              <input type="file" class="form-control" id="image" name="image" accept="image/*">
              <div class="form-text">Recommended size: 800x600px or similar aspect ratio</div>
            </div>
            <div class="mb-3">
              <label for="status" class="form-label">Status</label>
              <select class="form-select" id="status" name="status">
                <option value="active">Active</option>
                <option value="inactive">Inactive</option>
              </select>
            </div>
          </div>
          <div class="modal-footer">
            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
            <button type="submit" class="btn btn-primary">Add Portfolio Item</button>
          </div>
        </form>
      </div>
    </div>
  </div>

  <!-- Edit Portfolio Modal -->
  <div class="modal fade" id="editPortfolioModal" tabindex="-1" aria-labelledby="editPortfolioModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
      <div class="modal-content">
        <form method="POST" enctype="multipart/form-data">
          <input type="hidden" name="action" value="update">
          <input type="hidden" name="portfolio_id" id="edit_portfolio_id">
          <div class="modal-header">
            <h5 class="modal-title" id="editPortfolioModalLabel">Edit Portfolio Item</h5>
            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
          </div>
          <div class="modal-body">
            <div class="row">
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="edit_title" class="form-label">Title *</label>
                  <input type="text" class="form-control" id="edit_title" name="title" required>
                </div>
              </div>
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="edit_category" class="form-label">Category *</label>
                  <select class="form-select" id="edit_category" name="category" required>
                    <option value="">Select Category</option>
                    <option value="website">Website</option>
                    <option value="webapplication">Web Application</option>
                    <option value="software">Software</option>
                  </select>
                </div>
              </div>
            </div>
            <div class="mb-3">
              <label for="edit_description" class="form-label">Description</label>
              <textarea class="form-control" id="edit_description" name="description" rows="3"></textarea>
            </div>
            <div class="row">
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="edit_project_url" class="form-label">Project URL</label>
                  <input type="url" class="form-control" id="edit_project_url" name="project_url">
                </div>
              </div>
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="edit_sort_order" class="form-label">Sort Order</label>
                  <input type="number" class="form-control" id="edit_sort_order" name="sort_order" value="0">
                </div>
              </div>
            </div>
            <div class="mb-3">
              <label for="edit_technologies" class="form-label">Technologies Used</label>
              <input type="text" class="form-control" id="edit_technologies" name="technologies" placeholder="e.g., PHP, JavaScript, MySQL, Bootstrap">
            </div>
            <div class="mb-3">
              <label for="edit_features" class="form-label">Key Features</label>
              <textarea class="form-control" id="edit_features" name="features" rows="2" placeholder="e.g., Responsive Design, User Authentication, Payment Integration"></textarea>
            </div>
            <div class="mb-3">
              <label for="edit_image" class="form-label">Project Image</label>
              <input type="file" class="form-control" id="edit_image" name="image" accept="image/*">
              <input type="hidden" id="edit_current_image" name="current_image">
              <div class="form-text">Leave empty to keep current image. Recommended size: 800x600px or similar aspect ratio</div>
              <div id="current_image_preview" class="mt-2"></div>
            </div>
            <div class="mb-3">
              <label for="edit_status" class="form-label">Status</label>
              <select class="form-select" id="edit_status" name="status">
                <option value="active">Active</option>
                <option value="inactive">Inactive</option>
              </select>
            </div>
          </div>
          <div class="modal-footer">
            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
            <button type="submit" class="btn btn-primary">Update Portfolio Item</button>
          </div>
        </form>
      </div>
    </div>
  </div>

  <script>
    function editPortfolio(item) {
      document.getElementById('edit_portfolio_id').value = item.id;
      document.getElementById('edit_title').value = item.title;
      document.getElementById('edit_description').value = item.description || '';
      document.getElementById('edit_category').value = item.category;
      document.getElementById('edit_project_url').value = item.project_url || '';
      document.getElementById('edit_technologies').value = item.technologies || '';
      document.getElementById('edit_features').value = item.features || '';
      document.getElementById('edit_sort_order').value = item.sort_order || 0;
      document.getElementById('edit_status').value = item.status;
      document.getElementById('edit_current_image').value = item.image_url || '';
      
      // Show current image preview
      const preview = document.getElementById('current_image_preview');
      if (item.image_url) {
        preview.innerHTML = '<img src="../' + item.image_url + '" alt="Current image" class="img-thumbnail" style="max-width: 200px;">';
      } else {
        preview.innerHTML = '<span class="text-muted">No current image</span>';
      }
      
      // Show the modal
      new bootstrap.Modal(document.getElementById('editPortfolioModal')).show();
    }
  </script>

  <!-- View Order Modal -->
  <div class="modal fade" id="viewOrderModal" tabindex="-1" aria-labelledby="viewOrderModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
      <div class="modal-content">
        <div class="modal-header">
          <h5 class="modal-title" id="viewOrderModalLabel">Order Details</h5>
          <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
        </div>
        <div class="modal-body" id="viewOrderContent">
          <!-- Order details will be populated here -->
        </div>
        <div class="modal-footer">
          <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
        </div>
      </div>
    </div>
  </div>

  <!-- Update Order Status Modal -->
  <div class="modal fade" id="updateOrderStatusModal" tabindex="-1" aria-labelledby="updateOrderStatusModalLabel" aria-hidden="true">
    <div class="modal-dialog">
      <div class="modal-content">
        <form method="POST">
          <input type="hidden" name="action" value="update_status">
          <input type="hidden" name="order_id" id="update_order_id">
          <div class="modal-header">
            <h5 class="modal-title" id="updateOrderStatusModalLabel">Update Order Status</h5>
            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
          </div>
          <div class="modal-body">
            <div class="mb-3">
              <label for="order_status" class="form-label">Order Status *</label>
              <select class="form-select" id="order_status" name="status" required>
                <option value="pending">Pending</option>
                <option value="processing">Processing</option>
                <option value="shipped">Shipped</option>
                <option value="delivered">Delivered</option>
                <option value="cancelled">Cancelled</option>
                <option value="refunded">Refunded</option>
              </select>
            </div>
            <div class="mb-3">
              <label for="admin_notes" class="form-label">Admin Notes</label>
              <textarea class="form-control" id="admin_notes" name="admin_notes" rows="3" placeholder="Add any notes about this order..."></textarea>
            </div>
          </div>
          <div class="modal-footer">
            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
            <button type="submit" class="btn btn-primary">Update Status</button>
          </div>
        </form>
      </div>
    </div>
  </div>

  <!-- Update Payment Status Modal -->
  <div class="modal fade" id="updatePaymentStatusModal" tabindex="-1" aria-labelledby="updatePaymentStatusModalLabel" aria-hidden="true">
    <div class="modal-dialog">
      <div class="modal-content">
        <form method="POST">
          <input type="hidden" name="action" value="update_payment_status">
          <input type="hidden" name="order_id" id="update_payment_order_id">
          <div class="modal-header">
            <h5 class="modal-title" id="updatePaymentStatusModalLabel">Update Payment Status</h5>
            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
          </div>
          <div class="modal-body">
            <div class="mb-3">
              <label for="payment_status" class="form-label">Payment Status *</label>
              <select class="form-select" id="payment_status" name="payment_status" required>
                <option value="pending">Pending</option>
                <option value="paid">Paid</option>
                <option value="failed">Failed</option>
                <option value="refunded">Refunded</option>
              </select>
            </div>
            <div class="mb-3">
              <label for="payment_method" class="form-label">Payment Method</label>
              <input type="text" class="form-control" id="payment_method" name="payment_method" placeholder="e.g., Bank Transfer, PayPal, Credit Card">
            </div>
            <div class="mb-3">
              <label for="payment_reference" class="form-label">Payment Reference</label>
              <input type="text" class="form-control" id="payment_reference" name="payment_reference" placeholder="Transaction ID or reference number">
            </div>
          </div>
          <div class="modal-footer">
            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
            <button type="submit" class="btn btn-primary">Update Payment Status</button>
          </div>
        </form>
      </div>
    </div>
  </div>

  <script>
    function viewOrder(order) {
      const content = `
        <div class="row">
          <div class="col-md-6">
            <h6>Order Information</h6>
            <p><strong>Order Number:</strong> ${order.order_number}</p>
            <p><strong>Product:</strong> ${order.product_name}</p>
            <p><strong>Quantity:</strong> ${order.quantity}</p>
            <p><strong>Total Amount:</strong> $${parseFloat(order.total_amount).toFixed(2)}</p>
            <p><strong>Status:</strong> <span class="badge bg-primary">${order.status}</span></p>
            <p><strong>Payment Status:</strong> <span class="badge bg-success">${order.payment_status}</span></p>
          </div>
          <div class="col-md-6">
            <h6>Customer Information</h6>
            <p><strong>Name:</strong> ${order.customer_name}</p>
            <p><strong>Email:</strong> ${order.customer_email}</p>
            <p><strong>Phone:</strong> ${order.customer_phone || 'N/A'}</p>
            <p><strong>Payment Method:</strong> ${order.payment_method || 'N/A'}</p>
            <p><strong>Payment Reference:</strong> ${order.payment_reference || 'N/A'}</p>
          </div>
        </div>
        <div class="row mt-3">
          <div class="col-12">
            <h6>Addresses</h6>
            <div class="row">
              <div class="col-md-6">
                <strong>Shipping Address:</strong>
                <p>${order.shipping_address || 'N/A'}</p>
              </div>
              <div class="col-md-6">
                <strong>Billing Address:</strong>
                <p>${order.billing_address || 'N/A'}</p>
              </div>
            </div>
          </div>
        </div>
        ${order.notes ? `
        <div class="row mt-3">
          <div class="col-12">
            <h6>Customer Notes</h6>
            <p>${order.notes}</p>
          </div>
        </div>
        ` : ''}
        ${order.admin_notes ? `
        <div class="row mt-3">
          <div class="col-12">
            <h6>Admin Notes</h6>
            <p>${order.admin_notes}</p>
          </div>
        </div>
        ` : ''}
        <div class="row mt-3">
          <div class="col-12">
            <h6>Timestamps</h6>
            <p><strong>Created:</strong> ${new Date(order.created_at).toLocaleString()}</p>
            <p><strong>Updated:</strong> ${new Date(order.updated_at).toLocaleString()}</p>
          </div>
        </div>
      `;
      
      document.getElementById('viewOrderContent').innerHTML = content;
      new bootstrap.Modal(document.getElementById('viewOrderModal')).show();
    }

    function updateOrderStatus(order) {
      document.getElementById('update_order_id').value = order.id;
      document.getElementById('order_status').value = order.status;
      document.getElementById('admin_notes').value = order.admin_notes || '';
      new bootstrap.Modal(document.getElementById('updateOrderStatusModal')).show();
    }

    function updatePaymentStatus(order) {
      document.getElementById('update_payment_order_id').value = order.id;
      document.getElementById('payment_status').value = order.payment_status;
      document.getElementById('payment_method').value = order.payment_method || '';
      document.getElementById('payment_reference').value = order.payment_reference || '';
      new bootstrap.Modal(document.getElementById('updatePaymentStatusModal')).show();
    }
  </script>

  </body>

</html>