<?php
require_once 'config/database.php';
require_once 'config/settings.php';
require_once 'config/branding.php';

$error = '';
$success = '';

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: authentication/login.php');
    exit();
}

// Handle portfolio actions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $action = $_POST['action'] ?? '';
    $portfolio_id = $_POST['portfolio_id'] ?? '';
    
    try {
        $pdo = getConnection();
        
        switch ($action) {
            case 'create':
                $title = trim($_POST['title'] ?? '');
                $description = trim($_POST['description'] ?? '');
                $category = $_POST['category'] ?? '';
                $project_url = trim($_POST['project_url'] ?? '');
                $technologies = trim($_POST['technologies'] ?? '');
                $features = trim($_POST['features'] ?? '');
                $status = $_POST['status'] ?? 'active';
                $sort_order = (int)($_POST['sort_order'] ?? 0);
                
                if (empty($title) || empty($category)) {
                    $error = "Title and category are required";
                } else {
                    // Handle image upload
                    $image_url = '';
                    if (isset($_FILES['image']) && $_FILES['image']['error'] == 0) {
                        $upload_dir = '../assets/uploads/portfolio/';
                        if (!is_dir($upload_dir)) {
                            mkdir($upload_dir, 0755, true);
                        }
                        
                        $file_extension = pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION);
                        $file_name = 'portfolio_' . time() . '_' . uniqid() . '.' . $file_extension;
                        $file_path = $upload_dir . $file_name;
                        
                        if (move_uploaded_file($_FILES['image']['tmp_name'], $file_path)) {
                            $image_url = 'assets/uploads/portfolio/' . $file_name;
                        }
                    }
                    
                    $stmt = $pdo->prepare("
                        INSERT INTO portfolio_items (title, description, category, image_url, project_url, technologies, features, status, sort_order) 
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                    ");
                    $stmt->execute([$title, $description, $category, $image_url, $project_url, $technologies, $features, $status, $sort_order]);
                    $success = "Portfolio item created successfully";
                }
                break;
                
            case 'update':
                $title = trim($_POST['title'] ?? '');
                $description = trim($_POST['description'] ?? '');
                $category = $_POST['category'] ?? '';
                $project_url = trim($_POST['project_url'] ?? '');
                $technologies = trim($_POST['technologies'] ?? '');
                $features = trim($_POST['features'] ?? '');
                $status = $_POST['status'] ?? 'active';
                $sort_order = (int)($_POST['sort_order'] ?? 0);
                
                if (empty($title) || empty($category)) {
                    $error = "Title and category are required";
                } else {
                    // Handle image upload
                    $image_url = $_POST['current_image'] ?? '';
                    if (isset($_FILES['image']) && $_FILES['image']['error'] == 0) {
                        $upload_dir = '../assets/uploads/portfolio/';
                        if (!is_dir($upload_dir)) {
                            mkdir($upload_dir, 0755, true);
                        }
                        
                        $file_extension = pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION);
                        $file_name = 'portfolio_' . time() . '_' . uniqid() . '.' . $file_extension;
                        $file_path = $upload_dir . $file_name;
                        
                        if (move_uploaded_file($_FILES['image']['tmp_name'], $file_path)) {
                            $image_url = 'assets/uploads/portfolio/' . $file_name;
                        }
                    }
                    
                    $stmt = $pdo->prepare("
                        UPDATE portfolio_items 
                        SET title = ?, description = ?, category = ?, image_url = ?, project_url = ?, technologies = ?, features = ?, status = ?, sort_order = ?, updated_at = NOW()
                        WHERE id = ?
                    ");
                    $stmt->execute([$title, $description, $category, $image_url, $project_url, $technologies, $features, $status, $sort_order, $portfolio_id]);
                    $success = "Portfolio item updated successfully";
                }
                break;
                
            case 'delete':
                $stmt = $pdo->prepare("DELETE FROM portfolio_items WHERE id = ?");
                $stmt->execute([$portfolio_id]);
                $success = "Portfolio item deleted successfully";
                break;
                
            case 'toggle_status':
                $stmt = $pdo->prepare("UPDATE portfolio_items SET status = CASE WHEN status = 'active' THEN 'inactive' ELSE 'active' END WHERE id = ?");
                $stmt->execute([$portfolio_id]);
                $success = "Portfolio item status updated successfully";
                break;
        }
    } catch (Exception $e) {
        $error = "Action failed: " . $e->getMessage();
    }
}

// Get portfolio items with pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 10;
$offset = ($page - 1) * $limit;

try {
    $pdo = getConnection();
    
    // Get total count
    $countStmt = $pdo->query("SELECT COUNT(*) FROM portfolio_items");
    $totalPortfolio = $countStmt->fetchColumn();
    $totalPages = ceil($totalPortfolio / $limit);
    
    // Get portfolio items
    $stmt = $pdo->prepare("
        SELECT id, title, description, category, image_url, project_url, technologies, features, status, sort_order, created_at, updated_at
        FROM portfolio_items 
        ORDER BY sort_order ASC, created_at DESC 
        LIMIT :limit OFFSET :offset
    ");
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $portfolioItems = $stmt->fetchAll();
    
    // Get portfolio statistics
    $activePortfolio = $pdo->query("SELECT COUNT(*) FROM portfolio_items WHERE status = 'active'")->fetchColumn();
    $inactivePortfolio = $pdo->query("SELECT COUNT(*) FROM portfolio_items WHERE status = 'inactive'")->fetchColumn();
    
    // Get category statistics
    $websiteCount = $pdo->query("SELECT COUNT(*) FROM portfolio_items WHERE category = 'website' AND status = 'active'")->fetchColumn();
    $webappCount = $pdo->query("SELECT COUNT(*) FROM portfolio_items WHERE category = 'webapplication' AND status = 'active'")->fetchColumn();
    $softwareCount = $pdo->query("SELECT COUNT(*) FROM portfolio_items WHERE category = 'software' AND status = 'active'")->fetchColumn();
    
} catch (Exception $e) {
    $error = "Database error: " . $e->getMessage();
    $portfolioItems = [];
    $totalPages = 0;
    $totalPortfolio = 0;
    $activePortfolio = 0;
    $inactivePortfolio = 0;
    $websiteCount = 0;
    $webappCount = 0;
    $softwareCount = 0;
}
?>
<!DOCTYPE html>
<html lang="en">

<head>

  <base href="">

  <!-- begin::Edura Meta Basic -->
  <meta charset="utf-8">
  <meta name="theme-color" content="#0D9488">
  <meta name="robots" content="index, follow">
  <meta name="author" content="Omateq Digitals">
  <meta name="format-detection" content="telephone=no">
  <meta name="keywords" content="Admin Dashboard, Omateq Digitals, Portfolio Management, Admin Panel, Project Management">
  <meta name="description" content="Omateq Digitals Admin Dashboard - Manage portfolio items, create and update projects for frontend display.">
  <!-- end::Edura Meta Basic -->

  <!-- begin::Edura Meta Social -->
  <meta property="og:url" content="">
  <meta property="og:site_name" content="Manage Portfolio | Omateq Digitals">
  <meta property="og:type" content="website">
  <meta property="og:locale" content="en_us">
  <meta property="og:title" content="Manage Portfolio | Omateq Digitals">
  <meta property="og:description" content="Omateq Digitals Admin Dashboard - Manage portfolio items, create and update projects for frontend display.">
  <!-- end::Edura Meta Social -->

  <!-- begin::Edura Meta Twitter -->
  <meta name="twitter:card" content="summary">
  <meta name="twitter:url" content="">
  <meta name="twitter:creator" content="@omateqdigitals">
  <meta name="twitter:title" content="Manage Portfolio | Omateq Digitals">
  <meta name="twitter:description" content="Omateq Digitals Admin Dashboard - Manage portfolio items, create and update projects for frontend display.">
  <!-- end::Edura Meta Twitter -->

  <!-- begin::Edura Website Page Title -->
  <title>Manage Portfolio | Omateq Digitals</title>
  <!-- end::Edura Website Page Title -->

  <!-- begin::Edura Mobile Specific -->
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <!-- end::Edura Mobile Specific -->

  <!-- begin::Edura Favicon Tags -->
  <?php echo getFaviconHtml(); ?>
  <!-- end::Edura Favicon Tags -->

  <!-- begin::Edura Google Fonts -->
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Plus+Jakarta+Sans:ital,wght@0,200..800;1,200..800&display=swap" rel="stylesheet">
  <!-- end::Edura Google Fonts -->

  <!-- begin::Edura Required Stylesheet -->
  <link rel="stylesheet" href="assets/libs/flaticon/css/all/all.css">
  <link rel="stylesheet" href="assets/libs/lucide/lucide.css">
  <link rel="stylesheet" href="assets/libs/fontawesome/css/all.min.css">
  <link rel="stylesheet" href="assets/libs/simplebar/simplebar.css">
  <link rel="stylesheet" href="assets/libs/node-waves/waves.css">
  <link rel="stylesheet" href="assets/libs/bootstrap-select/css/bootstrap-select.min.css">
  <!-- end::Edura Required Stylesheet -->

  <!-- begin::Edura CSS Stylesheet -->
  <link rel="stylesheet" href="assets/css/styles.css">
  <!-- end::Edura CSS Stylesheet -->

  <!-- begin::Edura Googletagmanager -->
  <script async src="https://www.googletagmanager.com/gtag/js?id=G-XWVQM68HHQ"></script>
  <script>
		window.dataLayer = window.dataLayer || [];
		function gtag(){dataLayer.push(arguments);}
		gtag('js', new Date());

		gtag('config', 'G-XWVQM68HHQ', {
			'cookie_flags': 'SameSite=None;Secure',
			'send_page_view': true
		});
	</script>
  <!-- end::Edura Googletagmanager -->

</head>

<body>
  <div class="page-layout">

    <?php include 'includes/header.php'; ?>

     <!-- begin::Edura Sidebar Menu -->
     <aside class="app-menubar" id="menubar">
      <div class="app-navbar-brand">
        <a class="navbar-brand-logo" href="index.php">
          <img src="../<?php echo getSetting('main_logo', 'assets/images/logo.svg'); ?>" alt="logo">
        </a>
        <a class="navbar-brand-mini" href="index.php">
          <img src="../<?php echo getSetting('mini_logo', 'assets/images/logo-text-white.svg'); ?>" alt="logo">
        </a>
      </div>
      <nav class="app-navbar" data-simplebar>
        <ul class="menubar">
          <li class="menu-item">
            <a class="menu-link" href="index.php">
              <i class="fi fi-rr-apps"></i>
              <span class="menu-label">Dashboard</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/users.php">
              <i class="fi fi-rr-users"></i>
              <span class="menu-label">Manage Users</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/portfolio.php">
              <i class="fi fi-rr-portfolio"></i>
              <span class="menu-label">Manage Portfolio</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/products.php">
              <i class="fi fi-rr-shopping-cart"></i>
              <span class="menu-label">Manage Products</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/orders.php">
              <i class="fi fi-rr-receipt"></i>
              <span class="menu-label">Manage Orders</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/courses.php">
              <i class="fi fi-rr-book-alt"></i>
              <span class="menu-label">Manage Courses</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/script_requests.php">
              <i class="fi fi-rr-file-code"></i>
              <span class="menu-label">Manage Script Request</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/settings.php">
              <i class="fi fi-rr-settings"></i>
              <span class="menu-label">Settings</span>
            </a>
          </li>
        </ul>
      </nav>
      <div class="app-footer">
        <a href="pages/faq.html" class="btn btn-outline-white waves-effect btn-app-nav w-100">
          <i class="fi fi-rs-interrogation"></i>
          <span class="nav-text">Help and Support</span>
        </a>
      </div>
    </aside>
    <!-- end::Edura Sidebar Menu -->

    <main class="app-wrapper">

      <div class="container-fluid">

        <div class="app-page-head d-flex flex-wrap gap-3 align-items-center justify-content-between">
          <div class="clearfix">
            <h1 class="app-page-title">Manage Portfolio</h1>
            <nav aria-label="breadcrumb">
              <ol class="breadcrumb mb-0">
                <li class="breadcrumb-item">
                  <a href="index.php">Dashboard</a>
                </li>
                <li class="breadcrumb-item active" aria-current="page">Manage Portfolio</li>
              </ol>
            </nav>
          </div>
        </div>

        <?php if ($success): ?>
          <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fi fi-rr-check-circle me-2"></i><?php echo htmlspecialchars($success); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
          </div>
        <?php endif; ?>
        
        <?php if ($error): ?>
          <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fi fi-rr-exclamation-circle me-2"></i><?php echo htmlspecialchars($error); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
          </div>
        <?php endif; ?>

        <!-- Statistics Cards -->
        <div class="row mb-4">
          <!-- First Row - 2 cards on mobile, 4 on desktop -->
          <div class="col-lg-3 col-md-6 col-6 mb-3">
            <div class="card">
              <div class="card-body">
                <div class="d-flex align-items-center">
                  <div class="flex-shrink-0">
                    <div class="avatar avatar-sm bg-primary text-white rounded">
                      <i class="fi fi-rr-portfolio"></i>
                    </div>
                  </div>
                  <div class="flex-grow-1 ms-3">
                    <div class="text-muted small">Total Portfolio</div>
                    <div class="h5 mb-0"><?php echo $totalPortfolio; ?></div>
                  </div>
                </div>
              </div>
            </div>
          </div>
          <div class="col-lg-3 col-md-6 col-6 mb-3">
            <div class="card">
              <div class="card-body">
                <div class="d-flex align-items-center">
                  <div class="flex-shrink-0">
                    <div class="avatar avatar-sm bg-success text-white rounded">
                      <i class="fi fi-rr-check"></i>
                    </div>
                  </div>
                  <div class="flex-grow-1 ms-3">
                    <div class="text-muted small">Active Projects</div>
                    <div class="h5 mb-0"><?php echo $activePortfolio; ?></div>
                  </div>
                </div>
              </div>
            </div>
          </div>
          <!-- Second Row - 2 cards on mobile, 4 on desktop -->
          <div class="col-lg-3 col-md-6 col-6 mb-3">
            <div class="card">
              <div class="card-body">
                <div class="d-flex align-items-center">
                  <div class="flex-shrink-0">
                    <div class="avatar avatar-sm bg-warning text-white rounded">
                      <i class="fi fi-rr-pause"></i>
                    </div>
                  </div>
                  <div class="flex-grow-1 ms-3">
                    <div class="text-muted small">Inactive Projects</div>
                    <div class="h5 mb-0"><?php echo $inactivePortfolio; ?></div>
                  </div>
                </div>
              </div>
            </div>
          </div>
          <div class="col-lg-3 col-md-6 col-6 mb-3">
            <div class="card">
              <div class="card-body">
                <div class="d-flex align-items-center">
                  <div class="flex-shrink-0">
                    <div class="avatar avatar-sm bg-info text-white rounded">
                      <i class="fi fi-rr-globe"></i>
                    </div>
                  </div>
                  <div class="flex-grow-1 ms-3">
                    <div class="text-muted small">Websites</div>
                    <div class="h5 mb-0"><?php echo $websiteCount; ?></div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>

        <div class="row">

          <div class="col-lg-12">
            <div class="card overflow-hidden">
              <div class="card-header d-flex align-items-center justify-content-between border-0">
                <h6 class="card-title mb-0">Portfolio Items (<?php echo $totalPortfolio; ?> total items)</h6>
                <div class="d-flex align-items-center gap-2">
                  <button class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#addPortfolioModal">
                    <i class="fi fi-rr-plus"></i> Add New Portfolio
                  </button>
                  <div class="mail-header">
                    <form class="d-flex align-items-center h-100 w-200px w-md-250px position-relative me-auto" action="portfolio.php" method="GET">
                      <button type="button" class="btn btn-sm border-0 position-absolute start-0 ms-3 p-0">
                        <i class="fi fi-rr-search"></i>
                      </button>
                      <input type="text" class="form-control ps-5" placeholder="Search portfolio..." name="search">
                    </form>
                  </div>
                </div>
              </div>
              <div class="card-body p-0">
                <div class="table table-responsive mb-0">
                  <table class="table mb-0">
                    <thead class="table-light">
                      <tr>
                        <th class="minw-100px">ID</th>
                        <th class="minw-100px">Image</th>
                        <th class="minw-200px">Title</th>
                        <th class="minw-120px">Category</th>
                        <th class="minw-100px">Status</th>
                        <th class="minw-100px">Sort Order</th>
                        <th class="minw-150px">Created</th>
                        <th class="minw-200px">Actions</th>
                      </tr>
                    </thead>
                    <tbody>
                      <?php if (empty($portfolioItems)): ?>
                        <tr>
                          <td colspan="8" class="text-center py-4">
                            <div class="d-flex flex-column align-items-center">
                              <i class="fi fi-rr-portfolio text-muted mb-2" style="font-size: 2rem;"></i>
                              <p class="text-muted mb-0">No portfolio items found</p>
                              <button class="btn btn-primary btn-sm mt-2" data-bs-toggle="modal" data-bs-target="#addPortfolioModal">
                                <i class="fi fi-rr-plus"></i> Add First Portfolio Item
                              </button>
                            </div>
                          </td>
                        </tr>
                      <?php else: ?>
                        <?php foreach ($portfolioItems as $item): ?>
                          <tr>
                            <td>
                              <span class="text-primary">#<?php echo $item['id']; ?></span>
                            </td>
                            <td>
                              <?php if ($item['image_url']): ?>
                                <img src="../<?php echo htmlspecialchars($item['image_url']); ?>" alt="<?php echo htmlspecialchars($item['title']); ?>" class="rounded" style="width: 50px; height: 50px; object-fit: cover;">
                              <?php else: ?>
                                <div class="bg-light rounded d-flex align-items-center justify-content-center" style="width: 50px; height: 50px;">
                                  <i class="fi fi-rr-image text-muted"></i>
                                </div>
                              <?php endif; ?>
                            </td>
                            <td>
                              <div class="mw-200px">
                                <div class="fw-semibold"><?php echo htmlspecialchars($item['title']); ?></div>
                                <small class="text-muted"><?php echo htmlspecialchars(substr($item['description'], 0, 50)) . (strlen($item['description']) > 50 ? '...' : ''); ?></small>
                              </div>
                            </td>
                            <td>
                              <span class="badge bg-info-subtle text-info">
                                <?php echo ucfirst(str_replace('application', 'App', $item['category'])); ?>
                              </span>
                            </td>
                            <td>
                              <?php
                              $statusClass = $item['status'] === 'active' ? 'bg-success-subtle text-success' : 'bg-warning-subtle text-warning';
                              ?>
                              <span class="badge <?php echo $statusClass; ?>"><?php echo ucfirst($item['status']); ?></span>
                            </td>
                            <td>
                              <span class="badge bg-secondary-subtle text-secondary"><?php echo $item['sort_order']; ?></span>
                            </td>
                            <td><?php echo date('M d, Y', strtotime($item['created_at'])); ?></td>
                            <td>
                              <div class="d-flex gap-1 align-items-center">
                                <button class="btn btn-sm btn-outline-primary" onclick="editPortfolio(<?php echo htmlspecialchars(json_encode($item)); ?>)">
                                  <i class="fi fi-rr-edit"></i>
                                </button>
                                
                                <form method="POST" style="display: inline;">
                                  <input type="hidden" name="action" value="toggle_status">
                                  <input type="hidden" name="portfolio_id" value="<?php echo $item['id']; ?>">
                                  <button type="submit" class="btn btn-sm <?php echo $item['status'] === 'active' ? 'btn-warning' : 'btn-success'; ?>" onclick="return confirm('Are you sure you want to change the status of this portfolio item?')">
                                    <i class="fi fi-rr-<?php echo $item['status'] === 'active' ? 'pause' : 'play'; ?>"></i>
                                  </button>
                                </form>
                                
                                <form method="POST" style="display: inline;">
                                  <input type="hidden" name="action" value="delete">
                                  <input type="hidden" name="portfolio_id" value="<?php echo $item['id']; ?>">
                                  <button type="submit" class="btn btn-sm btn-danger" onclick="return confirm('Are you sure you want to delete this portfolio item? This action cannot be undone.')">
                                    <i class="fi fi-rr-trash"></i>
                                  </button>
                                </form>
                              </div>
                            </td>
                          </tr>
                        <?php endforeach; ?>
                      <?php endif; ?>
                    </tbody>
                  </table>
                </div>
              </div>
              <div class="card-footer border-0 p-3">
                <nav aria-label="Page navigation example">
                  <ul class="pagination pagination-sm mb-0">
                    <?php if ($page > 1): ?>
                      <li class="page-item">
                        <a class="page-link" href="?page=<?php echo $page - 1; ?>" aria-label="Previous">
                          <i class="fi fi-rr-angle-double-left"></i>
                        </a>
                      </li>
                    <?php endif; ?>
                    
                    <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                      <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                        <a class="page-link" href="?page=<?php echo $i; ?>"><?php echo $i; ?></a>
                      </li>
                    <?php endfor; ?>
                    
                    <?php if ($page < $totalPages): ?>
                      <li class="page-item">
                        <a class="page-link" href="?page=<?php echo $page + 1; ?>" aria-label="Next">
                          <i class="fi fi-rr-angle-double-right"></i>
                        </a>
                      </li>
                    <?php endif; ?>
                  </ul>
                </nav>
              </div>
            </div>
          </div>

        </div>

      </div>

    </main>

    <!-- begin::Edura Footer -->
    <footer class="footer-wrapper bg-body text-center">
      <div class="container-fluid">
        <p class="mb-0">© <span class="currentYear">2025</span> Edura. Proudly powered by <a href="javascript:void(0);">LayoutDrop</a>.</p>
      </div>
    </footer>
    <!-- end::Edura Footer -->

  </div>

  <!-- begin::Edura Page Scripts -->
  <script src="assets/libs/global/global.min.js"></script>
  <script src="assets/js/appSettings.js"></script>
  <script src="assets/js/main.js"></script>
  <!-- end::Edura Page Scripts -->
  <!-- Add Portfolio Modal -->
  <div class="modal fade" id="addPortfolioModal" tabindex="-1" aria-labelledby="addPortfolioModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
      <div class="modal-content">
        <form method="POST" enctype="multipart/form-data">
          <input type="hidden" name="action" value="create">
          <div class="modal-header">
            <h5 class="modal-title" id="addPortfolioModalLabel">Add New Portfolio Item</h5>
            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
          </div>
          <div class="modal-body">
            <div class="row">
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="title" class="form-label">Title *</label>
                  <input type="text" class="form-control" id="title" name="title" required>
                </div>
              </div>
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="category" class="form-label">Category *</label>
                  <select class="form-select" id="category" name="category" required>
                    <option value="">Select Category</option>
                    <option value="website">Website</option>
                    <option value="webapplication">Web Application</option>
                    <option value="software">Software</option>
                  </select>
                </div>
              </div>
            </div>
            <div class="mb-3">
              <label for="description" class="form-label">Description</label>
              <textarea class="form-control" id="description" name="description" rows="3"></textarea>
            </div>
            <div class="row">
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="project_url" class="form-label">Project URL</label>
                  <input type="url" class="form-control" id="project_url" name="project_url">
                </div>
              </div>
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="sort_order" class="form-label">Sort Order</label>
                  <input type="number" class="form-control" id="sort_order" name="sort_order" value="0">
                </div>
              </div>
            </div>
            <div class="mb-3">
              <label for="technologies" class="form-label">Technologies Used</label>
              <input type="text" class="form-control" id="technologies" name="technologies" placeholder="e.g., PHP, JavaScript, MySQL, Bootstrap">
            </div>
            <div class="mb-3">
              <label for="features" class="form-label">Key Features</label>
              <textarea class="form-control" id="features" name="features" rows="2" placeholder="e.g., Responsive Design, User Authentication, Payment Integration"></textarea>
            </div>
            <div class="mb-3">
              <label for="image" class="form-label">Project Image</label>
              <input type="file" class="form-control" id="image" name="image" accept="image/*">
              <div class="form-text">Recommended size: 800x600px or similar aspect ratio</div>
            </div>
            <div class="mb-3">
              <label for="status" class="form-label">Status</label>
              <select class="form-select" id="status" name="status">
                <option value="active">Active</option>
                <option value="inactive">Inactive</option>
              </select>
            </div>
          </div>
          <div class="modal-footer">
            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
            <button type="submit" class="btn btn-primary">Add Portfolio Item</button>
          </div>
        </form>
      </div>
    </div>
  </div>

  <!-- Edit Portfolio Modal -->
  <div class="modal fade" id="editPortfolioModal" tabindex="-1" aria-labelledby="editPortfolioModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
      <div class="modal-content">
        <form method="POST" enctype="multipart/form-data">
          <input type="hidden" name="action" value="update">
          <input type="hidden" name="portfolio_id" id="edit_portfolio_id">
          <div class="modal-header">
            <h5 class="modal-title" id="editPortfolioModalLabel">Edit Portfolio Item</h5>
            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
          </div>
          <div class="modal-body">
            <div class="row">
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="edit_title" class="form-label">Title *</label>
                  <input type="text" class="form-control" id="edit_title" name="title" required>
                </div>
              </div>
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="edit_category" class="form-label">Category *</label>
                  <select class="form-select" id="edit_category" name="category" required>
                    <option value="">Select Category</option>
                    <option value="website">Website</option>
                    <option value="webapplication">Web Application</option>
                    <option value="software">Software</option>
                  </select>
                </div>
              </div>
            </div>
            <div class="mb-3">
              <label for="edit_description" class="form-label">Description</label>
              <textarea class="form-control" id="edit_description" name="description" rows="3"></textarea>
            </div>
            <div class="row">
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="edit_project_url" class="form-label">Project URL</label>
                  <input type="url" class="form-control" id="edit_project_url" name="project_url">
                </div>
              </div>
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="edit_sort_order" class="form-label">Sort Order</label>
                  <input type="number" class="form-control" id="edit_sort_order" name="sort_order" value="0">
                </div>
              </div>
            </div>
            <div class="mb-3">
              <label for="edit_technologies" class="form-label">Technologies Used</label>
              <input type="text" class="form-control" id="edit_technologies" name="technologies" placeholder="e.g., PHP, JavaScript, MySQL, Bootstrap">
            </div>
            <div class="mb-3">
              <label for="edit_features" class="form-label">Key Features</label>
              <textarea class="form-control" id="edit_features" name="features" rows="2" placeholder="e.g., Responsive Design, User Authentication, Payment Integration"></textarea>
            </div>
            <div class="mb-3">
              <label for="edit_image" class="form-label">Project Image</label>
              <input type="file" class="form-control" id="edit_image" name="image" accept="image/*">
              <input type="hidden" id="edit_current_image" name="current_image">
              <div class="form-text">Leave empty to keep current image. Recommended size: 800x600px or similar aspect ratio</div>
              <div id="current_image_preview" class="mt-2"></div>
            </div>
            <div class="mb-3">
              <label for="edit_status" class="form-label">Status</label>
              <select class="form-select" id="edit_status" name="status">
                <option value="active">Active</option>
                <option value="inactive">Inactive</option>
              </select>
            </div>
          </div>
          <div class="modal-footer">
            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
            <button type="submit" class="btn btn-primary">Update Portfolio Item</button>
          </div>
        </form>
      </div>
    </div>
  </div>

  <script>
    function editPortfolio(item) {
      document.getElementById('edit_portfolio_id').value = item.id;
      document.getElementById('edit_title').value = item.title;
      document.getElementById('edit_description').value = item.description || '';
      document.getElementById('edit_category').value = item.category;
      document.getElementById('edit_project_url').value = item.project_url || '';
      document.getElementById('edit_technologies').value = item.technologies || '';
      document.getElementById('edit_features').value = item.features || '';
      document.getElementById('edit_sort_order').value = item.sort_order || 0;
      document.getElementById('edit_status').value = item.status;
      document.getElementById('edit_current_image').value = item.image_url || '';
      
      // Show current image preview
      const preview = document.getElementById('current_image_preview');
      if (item.image_url) {
        preview.innerHTML = '<img src="../' + item.image_url + '" alt="Current image" class="img-thumbnail" style="max-width: 200px;">';
      } else {
        preview.innerHTML = '<span class="text-muted">No current image</span>';
      }
      
      // Show the modal
      new bootstrap.Modal(document.getElementById('editPortfolioModal')).show();
    }
  </script>

  </body>

</html>