<?php
require_once 'config/database.php';
require_once 'config/settings.php';
require_once 'config/branding.php';

$error = '';
$success = '';

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: authentication/login.php');
    exit();
}

// Handle product actions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $action = $_POST['action'] ?? '';
    $product_id = $_POST['product_id'] ?? '';
    
    try {
        $pdo = getConnection();
        
        switch ($action) {
            case 'create':
                $product_name = trim($_POST['product_name'] ?? '');
                $product_description = trim($_POST['product_description'] ?? '');
                $category = $_POST['category'] ?? '';
                $product_features = trim($_POST['product_features'] ?? '');
                $demo_link = trim($_POST['demo_link'] ?? '');
                $demo_login_details = trim($_POST['demo_login_details'] ?? '');
                $product_specifications = trim($_POST['product_specifications'] ?? '');
                $product_amount = (float)($_POST['product_amount'] ?? 0);
                $status = $_POST['status'] ?? 'active';
                $sort_order = (int)($_POST['sort_order'] ?? 0);
                $is_downloadable = isset($_POST['is_downloadable']) ? 1 : 0;
                $download_limit = (int)($_POST['download_limit'] ?? 3);
                $download_access_duration = (int)($_POST['download_access_duration'] ?? 365); // days
                $download_instructions = trim($_POST['download_instructions'] ?? '');
                
                if (empty($product_name) || empty($category) || $product_amount <= 0) {
                    $error = "Product name, category, and amount are required";
                } else {
                    // Handle image upload
                    $thumbnail_url = '';
                    if (isset($_FILES['thumbnail']) && $_FILES['thumbnail']['error'] == 0) {
                        $upload_dir = '../assets/uploads/products/';
                        if (!is_dir($upload_dir)) {
                            mkdir($upload_dir, 0755, true);
                        }
                        
                        $file_extension = pathinfo($_FILES['thumbnail']['name'], PATHINFO_EXTENSION);
                        $file_name = 'product_' . time() . '_' . uniqid() . '.' . $file_extension;
                        $file_path = $upload_dir . $file_name;
                        
                        if (move_uploaded_file($_FILES['thumbnail']['tmp_name'], $file_path)) {
                            $thumbnail_url = 'assets/uploads/products/' . $file_name;
                        }
                    }
                    
                    // Handle download file upload
                    $download_file_url = '';
                    $download_file_name = '';
                    $download_file_size = 0;
                    $download_file_type = '';
                    
                    if ($is_downloadable && isset($_FILES['download_file']) && $_FILES['download_file']['error'] == 0) {
                        $download_dir = '../assets/downloads/products/';
                        if (!is_dir($download_dir)) {
                            mkdir($download_dir, 0755, true);
                        }
                        
                        $original_name = $_FILES['download_file']['name'];
                        $file_extension = pathinfo($original_name, PATHINFO_EXTENSION);
                        $file_name = 'download_' . time() . '_' . uniqid() . '.' . $file_extension;
                        $file_path = $download_dir . $file_name;
                        
                        if (move_uploaded_file($_FILES['download_file']['tmp_name'], $file_path)) {
                            $download_file_url = 'assets/downloads/products/' . $file_name;
                            $download_file_name = $original_name;
                            $download_file_size = $_FILES['download_file']['size'];
                            $download_file_type = $_FILES['download_file']['type'];
                        }
                    }
                    
                    $stmt = $pdo->prepare("
                        INSERT INTO products (product_name, product_description, thumbnail_url, category, product_features, demo_link, demo_login_details, product_specifications, product_amount, status, sort_order, is_downloadable, download_file_url, download_file_name, download_file_size, download_file_type, download_access_duration, download_limit, download_instructions) 
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                    ");
                    $stmt->execute([$product_name, $product_description, $thumbnail_url, $category, $product_features, $demo_link, $demo_login_details, $product_specifications, $product_amount, $status, $sort_order, $is_downloadable, $download_file_url, $download_file_name, $download_file_size, $download_file_type, $download_access_duration, $download_limit, $download_instructions]);
                    $success = "Product created successfully";
                }
                break;
                
            case 'update':
                $product_name = trim($_POST['product_name'] ?? '');
                $product_description = trim($_POST['product_description'] ?? '');
                $category = $_POST['category'] ?? '';
                $product_features = trim($_POST['product_features'] ?? '');
                $demo_link = trim($_POST['demo_link'] ?? '');
                $demo_login_details = trim($_POST['demo_login_details'] ?? '');
                $product_specifications = trim($_POST['product_specifications'] ?? '');
                $product_amount = (float)($_POST['product_amount'] ?? 0);
                $status = $_POST['status'] ?? 'active';
                $sort_order = (int)($_POST['sort_order'] ?? 0);
                $is_downloadable = isset($_POST['is_downloadable']) ? 1 : 0;
                $download_limit = (int)($_POST['download_limit'] ?? 3);
                $download_access_duration = (int)($_POST['download_access_duration'] ?? 365); // days
                $download_instructions = trim($_POST['download_instructions'] ?? '');
                
                if (empty($product_name) || empty($category) || $product_amount <= 0) {
                    $error = "Product name, category, and amount are required";
                } else {
                    // Handle image upload
                    $thumbnail_url = $_POST['current_thumbnail'] ?? '';
                    if (isset($_FILES['thumbnail']) && $_FILES['thumbnail']['error'] == 0) {
                        $upload_dir = '../assets/uploads/products/';
                        if (!is_dir($upload_dir)) {
                            mkdir($upload_dir, 0755, true);
                        }
                        
                        $file_extension = pathinfo($_FILES['thumbnail']['name'], PATHINFO_EXTENSION);
                        $file_name = 'product_' . time() . '_' . uniqid() . '.' . $file_extension;
                        $file_path = $upload_dir . $file_name;
                        
                        if (move_uploaded_file($_FILES['thumbnail']['tmp_name'], $file_path)) {
                            $thumbnail_url = 'assets/uploads/products/' . $file_name;
                        }
                    }
                    
                    // Handle download file upload
                    $download_file_url = $_POST['current_download_file'] ?? '';
                    $download_file_name = $_POST['current_download_file_name'] ?? '';
                    $download_file_size = (int)($_POST['current_download_file_size'] ?? 0);
                    $download_file_type = $_POST['current_download_file_type'] ?? '';
                    
                    if ($is_downloadable && isset($_FILES['download_file']) && $_FILES['download_file']['error'] == 0) {
                        $download_dir = '../assets/downloads/products/';
                        if (!is_dir($download_dir)) {
                            mkdir($download_dir, 0755, true);
                        }
                        
                        $original_name = $_FILES['download_file']['name'];
                        $file_extension = pathinfo($original_name, PATHINFO_EXTENSION);
                        $file_name = 'download_' . time() . '_' . uniqid() . '.' . $file_extension;
                        $file_path = $download_dir . $file_name;
                        
                        if (move_uploaded_file($_FILES['download_file']['tmp_name'], $file_path)) {
                            $download_file_url = 'assets/downloads/products/' . $file_name;
                            $download_file_name = $original_name;
                            $download_file_size = $_FILES['download_file']['size'];
                            $download_file_type = $_FILES['download_file']['type'];
                        }
                    }
                    
                    $stmt = $pdo->prepare("
                        UPDATE products 
                        SET product_name = ?, product_description = ?, thumbnail_url = ?, category = ?, product_features = ?, demo_link = ?, demo_login_details = ?, product_specifications = ?, product_amount = ?, status = ?, sort_order = ?, is_downloadable = ?, download_file_url = ?, download_file_name = ?, download_file_size = ?, download_file_type = ?, download_access_duration = ?, download_limit = ?, download_instructions = ?, updated_at = NOW()
                        WHERE id = ?
                    ");
                    $stmt->execute([$product_name, $product_description, $thumbnail_url, $category, $product_features, $demo_link, $demo_login_details, $product_specifications, $product_amount, $status, $sort_order, $is_downloadable, $download_file_url, $download_file_name, $download_file_size, $download_file_type, $download_access_duration, $download_limit, $download_instructions, $product_id]);
                    $success = "Product updated successfully";
                }
                break;
                
            case 'delete':
                $stmt = $pdo->prepare("DELETE FROM products WHERE id = ?");
                $stmt->execute([$product_id]);
                $success = "Product deleted successfully";
                break;
                
            case 'toggle_status':
                $stmt = $pdo->prepare("UPDATE products SET status = CASE WHEN status = 'active' THEN 'inactive' ELSE 'active' END WHERE id = ?");
                $stmt->execute([$product_id]);
                $success = "Product status updated successfully";
                break;
        }
    } catch (Exception $e) {
        $error = "Action failed: " . $e->getMessage();
    }
}

// Get products with pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 10;
$offset = ($page - 1) * $limit;

try {
    $pdo = getConnection();
    
    // Get total count
    $countStmt = $pdo->query("SELECT COUNT(*) FROM products");
    $totalProducts = $countStmt->fetchColumn();
    $totalPages = ceil($totalProducts / $limit);
    
    // Get products
    $stmt = $pdo->prepare("
        SELECT id, product_name, product_description, thumbnail_url, category, product_features, demo_link, demo_login_details, product_specifications, product_amount, status, sort_order, is_downloadable, download_file_url, download_file_name, download_file_size, download_file_type, download_access_duration, download_limit, download_instructions, created_at, updated_at
        FROM products 
        ORDER BY sort_order ASC, created_at DESC 
        LIMIT :limit OFFSET :offset
    ");
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $products = $stmt->fetchAll();
    
    // Get product statistics
    $activeProducts = $pdo->query("SELECT COUNT(*) FROM products WHERE status = 'active'")->fetchColumn();
    $inactiveProducts = $pdo->query("SELECT COUNT(*) FROM products WHERE status = 'inactive'")->fetchColumn();
    $draftProducts = $pdo->query("SELECT COUNT(*) FROM products WHERE status = 'draft'")->fetchColumn();
    
    // Get category statistics
    $phpCount = $pdo->query("SELECT COUNT(*) FROM products WHERE category = 'php' AND status = 'active'")->fetchColumn();
    $laravelCount = $pdo->query("SELECT COUNT(*) FROM products WHERE category = 'laravel' AND status = 'active'")->fetchColumn();
    $reactCount = $pdo->query("SELECT COUNT(*) FROM products WHERE category = 'react' AND status = 'active'")->fetchColumn();
    
    // Calculate total revenue (sum of all active products)
    $revenueStmt = $pdo->query("SELECT SUM(product_amount) FROM products WHERE status = 'active'");
    $totalRevenue = $revenueStmt->fetchColumn() ?: 0;
    
} catch (Exception $e) {
    $error = "Database error: " . $e->getMessage();
    $products = [];
    $totalPages = 0;
    $totalProducts = 0;
    $activeProducts = 0;
    $inactiveProducts = 0;
    $draftProducts = 0;
    $phpCount = 0;
    $laravelCount = 0;
    $reactCount = 0;
    $totalRevenue = 0;
}
?>
<!DOCTYPE html>
<html lang="en">

<head>

  <base href="">

  <!-- begin::Edura Meta Basic -->
  <meta charset="utf-8">
  <meta name="theme-color" content="#0D9488">
  <meta name="robots" content="index, follow">
  <meta name="author" content="Omateq Digitals">
  <meta name="format-detection" content="telephone=no">
  <meta name="keywords" content="Admin Dashboard, Omateq Digitals, Product Management, Admin Panel, E-commerce Management">
  <meta name="description" content="Omateq Digitals Admin Dashboard - Manage products, pricing, categories, and product information for frontend display.">
  <!-- end::Edura Meta Basic -->

  <!-- begin::Edura Meta Social -->
  <meta property="og:url" content="">
  <meta property="og:site_name" content="Manage Products | Omateq Digitals">
  <meta property="og:type" content="website">
  <meta property="og:locale" content="en_us">
  <meta property="og:title" content="Manage Products | Omateq Digitals">
  <meta property="og:description" content="Omateq Digitals Admin Dashboard - Manage products, pricing, categories, and product information for frontend display.">
  <!-- end::Edura Meta Social -->

  <!-- begin::Edura Meta Twitter -->
  <meta name="twitter:card" content="summary">
  <meta name="twitter:url" content="">
  <meta name="twitter:creator" content="@omateqdigitals">
  <meta name="twitter:title" content="Manage Products | Omateq Digitals">
  <meta name="twitter:description" content="Omateq Digitals Admin Dashboard - Manage products, pricing, categories, and product information for frontend display.">
  <!-- end::Edura Meta Twitter -->

  <!-- begin::Edura Website Page Title -->
  <title>Manage Products | Omateq Digitals</title>
  <!-- end::Edura Website Page Title -->

  <!-- begin::Edura Mobile Specific -->
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <!-- end::Edura Mobile Specific -->

  <!-- begin::Edura Favicon Tags -->
  <?php echo getFaviconHtml(); ?>
  <!-- end::Edura Favicon Tags -->

  <!-- begin::Edura Google Fonts -->
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Plus+Jakarta+Sans:ital,wght@0,200..800;1,200..800&display=swap" rel="stylesheet">
  <!-- end::Edura Google Fonts -->

  <!-- begin::Edura Required Stylesheet -->
  <link rel="stylesheet" href="assets/libs/flaticon/css/all/all.css">
  <link rel="stylesheet" href="assets/libs/lucide/lucide.css">
  <link rel="stylesheet" href="assets/libs/fontawesome/css/all.min.css">
  <link rel="stylesheet" href="assets/libs/simplebar/simplebar.css">
  <link rel="stylesheet" href="assets/libs/node-waves/waves.css">
  <link rel="stylesheet" href="assets/libs/bootstrap-select/css/bootstrap-select.min.css">
  <!-- end::Edura Required Stylesheet -->

  <!-- begin::Edura CSS Stylesheet -->
  <link rel="stylesheet" href="assets/css/styles.css">
  <!-- end::Edura CSS Stylesheet -->

  <!-- begin::Edura Googletagmanager -->
  <script async src="https://www.googletagmanager.com/gtag/js?id=G-XWVQM68HHQ"></script>
  <script>
		window.dataLayer = window.dataLayer || [];
		function gtag(){dataLayer.push(arguments);}
		gtag('js', new Date());

		gtag('config', 'G-XWVQM68HHQ', {
			'cookie_flags': 'SameSite=None;Secure',
			'send_page_view': true
		});
	</script>
  <!-- end::Edura Googletagmanager -->

</head>

<body>
  <div class="page-layout">

    <?php include 'includes/header.php'; ?>

     <!-- begin::Edura Sidebar Menu -->
     <aside class="app-menubar" id="menubar">
      <div class="app-navbar-brand">
        <a class="navbar-brand-logo" href="index.php">
          <img src="../<?php echo getSetting('main_logo', 'assets/images/logo.svg'); ?>" alt="logo">
        </a>
        <a class="navbar-brand-mini" href="index.php">
          <img src="../<?php echo getSetting('mini_logo', 'assets/images/logo-text-white.svg'); ?>" alt="logo">
        </a>
      </div>
      <nav class="app-navbar" data-simplebar>
        <ul class="menubar">
          <li class="menu-item">
            <a class="menu-link" href="index.php">
              <i class="fi fi-rr-apps"></i>
              <span class="menu-label">Dashboard</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/users.php">
              <i class="fi fi-rr-users"></i>
              <span class="menu-label">Manage Users</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/portfolio.php">
              <i class="fi fi-rr-portfolio"></i>
              <span class="menu-label">Manage Portfolio</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/products.php">
              <i class="fi fi-rr-shopping-cart"></i>
              <span class="menu-label">Manage Products</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/orders.php">
              <i class="fi fi-rr-receipt"></i>
              <span class="menu-label">Manage Orders</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/courses.php">
              <i class="fi fi-rr-book-alt"></i>
              <span class="menu-label">Manage Courses</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/script_requests.php">
              <i class="fi fi-rr-file-code"></i>
              <span class="menu-label">Manage Script Request</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/settings.php">
              <i class="fi fi-rr-settings"></i>
              <span class="menu-label">Settings</span>
            </a>
          </li>
        </ul>
      </nav>
      <div class="app-footer">
        <a href="pages/faq.html" class="btn btn-outline-white waves-effect btn-app-nav w-100">
          <i class="fi fi-rs-interrogation"></i>
          <span class="nav-text">Help and Support</span>
        </a>
      </div>
    </aside>
    <!-- end::Edura Sidebar Menu -->

    <main class="app-wrapper">

      <div class="container-fluid">

        <div class="app-page-head d-flex flex-wrap gap-3 align-items-center justify-content-between">
          <div class="clearfix">
            <h1 class="app-page-title">Manage Products</h1>
            <nav aria-label="breadcrumb">
              <ol class="breadcrumb mb-0">
                <li class="breadcrumb-item">
                  <a href="index.php">Dashboard</a>
                </li>
                <li class="breadcrumb-item active" aria-current="page">Manage Products</li>
              </ol>
            </nav>
          </div>
        </div>

        <?php if ($success): ?>
          <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fi fi-rr-check-circle me-2"></i><?php echo htmlspecialchars($success); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
          </div>
        <?php endif; ?>
        
        <?php if ($error): ?>
          <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fi fi-rr-exclamation-circle me-2"></i><?php echo htmlspecialchars($error); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
          </div>
        <?php endif; ?>

        <!-- Statistics Cards -->
        <div class="row mb-4">
          <!-- First Row - 2 cards on mobile, 4 on desktop -->
          <div class="col-lg-3 col-md-6 col-6 mb-3">
            <div class="card">
              <div class="card-body">
                <div class="d-flex align-items-center">
                  <div class="flex-shrink-0">
                    <div class="avatar avatar-sm bg-primary text-white rounded">
                      <i class="fi fi-rr-shopping-cart"></i>
                    </div>
                  </div>
                  <div class="flex-grow-1 ms-3">
                    <div class="text-muted small">Total Products</div>
                    <div class="h5 mb-0"><?php echo $totalProducts; ?></div>
                  </div>
                </div>
              </div>
            </div>
          </div>
          <div class="col-lg-3 col-md-6 col-6 mb-3">
            <div class="card">
              <div class="card-body">
                <div class="d-flex align-items-center">
                  <div class="flex-shrink-0">
                    <div class="avatar avatar-sm bg-success text-white rounded">
                      <i class="fi fi-rr-check"></i>
                    </div>
                  </div>
                  <div class="flex-grow-1 ms-3">
                    <div class="text-muted small">Active Products</div>
                    <div class="h5 mb-0"><?php echo $activeProducts; ?></div>
                  </div>
                </div>
              </div>
            </div>
          </div>
          <!-- Second Row - 2 cards on mobile, 4 on desktop -->
          <div class="col-lg-3 col-md-6 col-6 mb-3">
            <div class="card">
              <div class="card-body">
                <div class="d-flex align-items-center">
                  <div class="flex-shrink-0">
                    <div class="avatar avatar-sm bg-warning text-white rounded">
                      <i class="fi fi-rr-pause"></i>
                    </div>
                  </div>
                  <div class="flex-grow-1 ms-3">
                    <div class="text-muted small">Inactive Products</div>
                    <div class="h5 mb-0"><?php echo $inactiveProducts; ?></div>
                  </div>
                </div>
              </div>
            </div>
          </div>
          <div class="col-lg-3 col-md-6 col-6 mb-3">
            <div class="card">
              <div class="card-body">
                <div class="d-flex align-items-center">
                  <div class="flex-shrink-0">
                    <div class="avatar avatar-sm bg-info text-white rounded">
                      <i class="fi fi-rr-dollar"></i>
                    </div>
                  </div>
                  <div class="flex-grow-1 ms-3">
                    <div class="text-muted small">Total Revenue</div>
                    <div class="h5 mb-0">$<?php echo number_format($totalRevenue, 2); ?></div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>

        <div class="row">

          <div class="col-lg-12">
            <div class="card overflow-hidden">
              <div class="card-header d-flex align-items-center justify-content-between border-0">
                <h6 class="card-title mb-0">Products List (<?php echo $totalProducts; ?> total products)</h6>
                <div class="d-flex align-items-center gap-2">
                  <button class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#addProductModal">
                    <i class="fi fi-rr-plus"></i> Add New Product
                  </button>
                  <div class="mail-header">
                    <form class="d-flex align-items-center h-100 w-200px w-md-250px position-relative me-auto" action="products.php" method="GET">
                      <button type="button" class="btn btn-sm border-0 position-absolute start-0 ms-3 p-0">
                        <i class="fi fi-rr-search"></i>
                      </button>
                      <input type="text" class="form-control ps-5" placeholder="Search products..." name="search">
                    </form>
                  </div>
                </div>
              </div>
              <div class="card-body p-0">
                <div class="table table-responsive mb-0">
                  <table class="table mb-0">
                    <thead class="table-light">
                      <tr>
                        <th class="minw-100px">ID</th>
                        <th class="minw-100px">Image</th>
                        <th class="minw-200px">Product Name</th>
                        <th class="minw-120px">Category</th>
                        <th class="minw-100px">Price</th>
                        <th class="minw-100px">Status</th>
                        <th class="minw-120px">Download</th>
                        <th class="minw-100px">Sort Order</th>
                        <th class="minw-150px">Created</th>
                        <th class="minw-200px">Actions</th>
                      </tr>
                    </thead>
                    <tbody>
                      <?php if (empty($products)): ?>
                        <tr>
                          <td colspan="10" class="text-center py-4">
                            <div class="d-flex flex-column align-items-center">
                              <i class="fi fi-rr-shopping-cart text-muted mb-2" style="font-size: 2rem;"></i>
                              <p class="text-muted mb-0">No products found</p>
                              <button class="btn btn-primary btn-sm mt-2" data-bs-toggle="modal" data-bs-target="#addProductModal">
                                <i class="fi fi-rr-plus"></i> Add First Product
                              </button>
                            </div>
                          </td>
                        </tr>
                      <?php else: ?>
                        <?php foreach ($products as $product): ?>
                          <tr>
                            <td>
                              <span class="text-primary">#<?php echo $product['id']; ?></span>
                            </td>
                            <td>
                              <?php if ($product['thumbnail_url']): ?>
                                <img src="../<?php echo htmlspecialchars($product['thumbnail_url']); ?>" alt="<?php echo htmlspecialchars($product['product_name']); ?>" class="rounded" style="width: 50px; height: 50px; object-fit: cover;">
                              <?php else: ?>
                                <div class="bg-light rounded d-flex align-items-center justify-content-center" style="width: 50px; height: 50px;">
                                  <i class="fi fi-rr-image text-muted"></i>
                                </div>
                              <?php endif; ?>
                            </td>
                            <td>
                              <div class="mw-200px">
                                <div class="fw-semibold"><?php echo htmlspecialchars($product['product_name']); ?></div>
                                <small class="text-muted"><?php echo htmlspecialchars(substr($product['product_description'], 0, 50)) . (strlen($product['product_description']) > 50 ? '...' : ''); ?></small>
                              </div>
                            </td>
                            <td>
                              <span class="badge bg-info-subtle text-info">
                                <?php echo ucfirst($product['category']); ?>
                              </span>
                            </td>
                            <td>
                              <span class="fw-semibold text-success">$<?php echo number_format($product['product_amount'], 2); ?></span>
                            </td>
                            <td>
                              <?php
                              $statusClass = '';
                              switch ($product['status']) {
                                case 'active':
                                  $statusClass = 'bg-success-subtle text-success';
                                  break;
                                case 'inactive':
                                  $statusClass = 'bg-warning-subtle text-warning';
                                  break;
                                case 'draft':
                                  $statusClass = 'bg-secondary-subtle text-secondary';
                                  break;
                                default:
                                  $statusClass = 'bg-secondary-subtle text-secondary';
                              }
                              ?>
                              <span class="badge <?php echo $statusClass; ?>"><?php echo ucfirst($product['status']); ?></span>
                            </td>
                            <td>
                              <?php if ($product['is_downloadable'] == 1): ?>
                                <div class="d-flex flex-column">
                                  <span class="badge bg-success-subtle text-success mb-1">
                                    <i class="fi fi-rr-download"></i> Downloadable
                                  </span>
                                  <?php if ($product['download_file_name']): ?>
                                    <small class="text-muted">
                                      <i class="fi fi-rr-file"></i> <?php echo htmlspecialchars(substr($product['download_file_name'], 0, 15)) . (strlen($product['download_file_name']) > 15 ? '...' : ''); ?>
                                    </small>
                                    <small class="text-muted">
                                      <i class="fi fi-rr-clock"></i> <?php echo $product['download_limit']; ?> downloads
                                    </small>
                                  <?php else: ?>
                                    <small class="text-warning">
                                      <i class="fi fi-rr-exclamation"></i> No file uploaded
                                    </small>
                                  <?php endif; ?>
                                </div>
                              <?php else: ?>
                                <span class="badge bg-secondary-subtle text-secondary">
                                  <i class="fi fi-rr-ban"></i> Not Downloadable
                                </span>
                              <?php endif; ?>
                            </td>
                            <td>
                              <span class="badge bg-secondary-subtle text-secondary"><?php echo $product['sort_order']; ?></span>
                            </td>
                            <td><?php echo date('M d, Y', strtotime($product['created_at'])); ?></td>
                            <td>
                              <div class="d-flex gap-1 align-items-center">
                                <button class="btn btn-sm btn-outline-primary" onclick="editProduct(<?php echo htmlspecialchars(json_encode($product)); ?>)">
                                  <i class="fi fi-rr-edit"></i>
                                </button>
                                
                                <form method="POST" style="display: inline;">
                                  <input type="hidden" name="action" value="toggle_status">
                                  <input type="hidden" name="product_id" value="<?php echo $product['id']; ?>">
                                  <button type="submit" class="btn btn-sm <?php echo $product['status'] === 'active' ? 'btn-warning' : 'btn-success'; ?>" onclick="return confirm('Are you sure you want to change the status of this product?')">
                                    <i class="fi fi-rr-<?php echo $product['status'] === 'active' ? 'pause' : 'play'; ?>"></i>
                                  </button>
                                </form>
                                
                                <form method="POST" style="display: inline;">
                                  <input type="hidden" name="action" value="delete">
                                  <input type="hidden" name="product_id" value="<?php echo $product['id']; ?>">
                                  <button type="submit" class="btn btn-sm btn-danger" onclick="return confirm('Are you sure you want to delete this product? This action cannot be undone.')">
                                    <i class="fi fi-rr-trash"></i>
                                  </button>
                                </form>
                              </div>
                            </td>
                          </tr>
                        <?php endforeach; ?>
                      <?php endif; ?>
                    </tbody>
                  </table>
                </div>
              </div>
              <div class="card-footer border-0 p-3">
                <nav aria-label="Page navigation example">
                  <ul class="pagination pagination-sm mb-0">
                    <?php if ($page > 1): ?>
                      <li class="page-item">
                        <a class="page-link" href="?page=<?php echo $page - 1; ?>" aria-label="Previous">
                          <i class="fi fi-rr-angle-double-left"></i>
                        </a>
                      </li>
                    <?php endif; ?>
                    
                    <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                      <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                        <a class="page-link" href="?page=<?php echo $i; ?>"><?php echo $i; ?></a>
                      </li>
                    <?php endfor; ?>
                    
                    <?php if ($page < $totalPages): ?>
                      <li class="page-item">
                        <a class="page-link" href="?page=<?php echo $page + 1; ?>" aria-label="Next">
                          <i class="fi fi-rr-angle-double-right"></i>
                        </a>
                      </li>
                    <?php endif; ?>
                  </ul>
                </nav>
              </div>
            </div>
          </div>

        </div>

      </div>

    </main>

    <!-- begin::Edura Footer -->
    <footer class="footer-wrapper bg-body text-center">
      <div class="container-fluid">
        <p class="mb-0">© <span class="currentYear">2025</span> Edura. Proudly powered by <a href="javascript:void(0);">LayoutDrop</a>.</p>
      </div>
    </footer>
    <!-- end::Edura Footer -->

  </div>

  <!-- begin::Edura Page Scripts -->
  <script src="assets/libs/global/global.min.js"></script>
  <script src="assets/js/appSettings.js"></script>
  <script src="assets/js/main.js"></script>
  <!-- end::Edura Page Scripts -->
  <!-- Add Product Modal -->
  <div class="modal fade" id="addProductModal" tabindex="-1" aria-labelledby="addProductModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-xl">
      <div class="modal-content">
        <form method="POST" enctype="multipart/form-data">
          <input type="hidden" name="action" value="create">
          <div class="modal-header">
            <h5 class="modal-title" id="addProductModalLabel">Add New Product</h5>
            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
          </div>
          <div class="modal-body">
            <div class="row">
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="product_name" class="form-label">Product Name *</label>
                  <input type="text" class="form-control" id="product_name" name="product_name" required>
                </div>
              </div>
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="category" class="form-label">Category *</label>
                  <select class="form-select" id="category" name="category" required>
                    <option value="">Select Category</option>
                    <option value="php">PHP</option>
                    <option value="laravel">Laravel</option>
                    <option value="javascript">JavaScript</option>
                    <option value="react">React</option>
                    <option value="vue">Vue.js</option>
                    <option value="angular">Angular</option>
                    <option value="nodejs">Node.js</option>
                    <option value="python">Python</option>
                    <option value="django">Django</option>
                    <option value="flask">Flask</option>
                    <option value="other">Other</option>
                  </select>
                </div>
              </div>
            </div>
            <div class="mb-3">
              <label for="product_description" class="form-label">Product Description *</label>
              <textarea class="form-control" id="product_description" name="product_description" rows="4" required></textarea>
            </div>
            <div class="row">
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="product_amount" class="form-label">Price *</label>
                  <div class="input-group">
                    <span class="input-group-text">$</span>
                    <input type="number" class="form-control" id="product_amount" name="product_amount" step="0.01" min="0" required>
                  </div>
                </div>
              </div>
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="sort_order" class="form-label">Sort Order</label>
                  <input type="number" class="form-control" id="sort_order" name="sort_order" value="0">
                </div>
              </div>
            </div>
            <div class="row">
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="demo_link" class="form-label">Demo Link</label>
                  <input type="url" class="form-control" id="demo_link" name="demo_link" placeholder="https://demo.example.com">
                </div>
              </div>
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="status" class="form-label">Status</label>
                  <select class="form-select" id="status" name="status">
                    <option value="active">Active</option>
                    <option value="inactive">Inactive</option>
                    <option value="draft">Draft</option>
                  </select>
                </div>
              </div>
            </div>
            <div class="mb-3">
              <label for="product_features" class="form-label">Product Features</label>
              <textarea class="form-control" id="product_features" name="product_features" rows="3" placeholder="Feature 1, Feature 2, Feature 3..."></textarea>
            </div>
            <div class="mb-3">
              <label for="demo_login_details" class="form-label">Demo Login Details</label>
              <textarea class="form-control" id="demo_login_details" name="demo_login_details" rows="2" placeholder="Username: admin, Password: admin123"></textarea>
            </div>
            <div class="mb-3">
              <label for="product_specifications" class="form-label">Product Specifications</label>
              <textarea class="form-control" id="product_specifications" name="product_specifications" rows="3" placeholder="PHP 7.4+, MySQL 5.7+, Apache/Nginx, 100MB Storage, SSL Certificate"></textarea>
            </div>
            
            <!-- Download File Section -->
            <div class="card mb-3">
              <div class="card-header">
                <h6 class="mb-0">Download File Settings</h6>
              </div>
              <div class="card-body">
                <div class="mb-3">
                  <div class="form-check">
                    <input class="form-check-input" type="checkbox" id="is_downloadable" name="is_downloadable" onchange="toggleDownloadFields()">
                    <label class="form-check-label" for="is_downloadable">
                      <strong>Make this product downloadable</strong>
                    </label>
                  </div>
                </div>
                
                <div id="downloadFields" style="display: none;">
                  <div class="row">
                    <div class="col-md-6">
                      <div class="mb-3">
                        <label for="download_file" class="form-label">Download File *</label>
                        <input type="file" class="form-control" id="download_file" name="download_file" accept=".zip,.rar,.tar.gz,.pdf,.doc,.docx,.txt">
                        <div class="form-text">Upload the file users will download after purchase (ZIP, RAR, PDF, etc.)</div>
                      </div>
                    </div>
                    <div class="col-md-6">
                      <div class="mb-3">
                        <label for="download_limit" class="form-label">Download Limit</label>
                        <input type="number" class="form-control" id="download_limit" name="download_limit" value="3" min="1" max="100">
                        <div class="form-text">Number of times user can download (default: 3)</div>
                      </div>
                    </div>
                  </div>
                  <div class="row">
                    <div class="col-md-6">
                      <div class="mb-3">
                        <label for="download_access_duration" class="form-label">Access Duration (Days)</label>
                        <input type="number" class="form-control" id="download_access_duration" name="download_access_duration" value="365" min="1" max="3650">
                        <div class="form-text">How long users can access downloads (default: 365 days)</div>
                      </div>
                    </div>
                    <div class="col-md-6">
                      <div class="mb-3">
                        <label for="download_instructions" class="form-label">Download Instructions</label>
                        <textarea class="form-control" id="download_instructions" name="download_instructions" rows="3" placeholder="Instructions for users on how to use the downloaded files..."></textarea>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            </div>
            
            <div class="mb-3">
              <label for="thumbnail" class="form-label">Product Thumbnail</label>
              <input type="file" class="form-control" id="thumbnail" name="thumbnail" accept="image/*">
              <div class="form-text">Recommended size: 800x600px or similar aspect ratio</div>
            </div>
          </div>
          <div class="modal-footer">
            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
            <button type="submit" class="btn btn-primary">Add Product</button>
          </div>
        </form>
      </div>
    </div>
  </div>

  <!-- Edit Product Modal -->
  <div class="modal fade" id="editProductModal" tabindex="-1" aria-labelledby="editProductModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-xl">
      <div class="modal-content">
        <form method="POST" enctype="multipart/form-data">
          <input type="hidden" name="action" value="update">
          <input type="hidden" name="product_id" id="edit_product_id">
          <div class="modal-header">
            <h5 class="modal-title" id="editProductModalLabel">Edit Product</h5>
            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
          </div>
          <div class="modal-body">
            <div class="row">
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="edit_product_name" class="form-label">Product Name *</label>
                  <input type="text" class="form-control" id="edit_product_name" name="product_name" required>
                </div>
              </div>
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="edit_category" class="form-label">Category *</label>
                  <select class="form-select" id="edit_category" name="category" required>
                    <option value="">Select Category</option>
                    <option value="php">PHP</option>
                    <option value="laravel">Laravel</option>
                    <option value="javascript">JavaScript</option>
                    <option value="react">React</option>
                    <option value="vue">Vue.js</option>
                    <option value="angular">Angular</option>
                    <option value="nodejs">Node.js</option>
                    <option value="python">Python</option>
                    <option value="django">Django</option>
                    <option value="flask">Flask</option>
                    <option value="other">Other</option>
                  </select>
                </div>
              </div>
            </div>
            <div class="mb-3">
              <label for="edit_product_description" class="form-label">Product Description *</label>
              <textarea class="form-control" id="edit_product_description" name="product_description" rows="4" required></textarea>
            </div>
            <div class="row">
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="edit_product_amount" class="form-label">Price *</label>
                  <div class="input-group">
                    <span class="input-group-text">$</span>
                    <input type="number" class="form-control" id="edit_product_amount" name="product_amount" step="0.01" min="0" required>
                  </div>
                </div>
              </div>
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="edit_sort_order" class="form-label">Sort Order</label>
                  <input type="number" class="form-control" id="edit_sort_order" name="sort_order" value="0">
                </div>
              </div>
            </div>
            <div class="row">
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="edit_demo_link" class="form-label">Demo Link</label>
                  <input type="url" class="form-control" id="edit_demo_link" name="demo_link" placeholder="https://demo.example.com">
                </div>
              </div>
              <div class="col-md-6">
                <div class="mb-3">
                  <label for="edit_status" class="form-label">Status</label>
                  <select class="form-select" id="edit_status" name="status">
                    <option value="active">Active</option>
                    <option value="inactive">Inactive</option>
                    <option value="draft">Draft</option>
                  </select>
                </div>
              </div>
            </div>
            <div class="mb-3">
              <label for="edit_product_features" class="form-label">Product Features</label>
              <textarea class="form-control" id="edit_product_features" name="product_features" rows="3" placeholder="Feature 1, Feature 2, Feature 3..."></textarea>
            </div>
            <div class="mb-3">
              <label for="edit_demo_login_details" class="form-label">Demo Login Details</label>
              <textarea class="form-control" id="edit_demo_login_details" name="demo_login_details" rows="2" placeholder="Username: admin, Password: admin123"></textarea>
            </div>
            <div class="mb-3">
              <label for="edit_product_specifications" class="form-label">Product Specifications</label>
              <textarea class="form-control" id="edit_product_specifications" name="product_specifications" rows="3" placeholder="PHP 7.4+, MySQL 5.7+, Apache/Nginx, 100MB Storage, SSL Certificate"></textarea>
            </div>
            
            <!-- Download File Section -->
            <div class="card mb-3">
              <div class="card-header">
                <h6 class="mb-0">Download File Settings</h6>
              </div>
              <div class="card-body">
                <div class="mb-3">
                  <div class="form-check">
                    <input class="form-check-input" type="checkbox" id="edit_is_downloadable" name="is_downloadable" onchange="toggleEditDownloadFields()">
                    <label class="form-check-label" for="edit_is_downloadable">
                      <strong>Make this product downloadable</strong>
                    </label>
                  </div>
                </div>
                
                <div id="editDownloadFields" style="display: none;">
                  <div class="row">
                    <div class="col-md-6">
                      <div class="mb-3">
                        <label for="edit_download_file" class="form-label">Download File</label>
                        <input type="file" class="form-control" id="edit_download_file" name="download_file" accept=".zip,.rar,.tar.gz,.pdf,.doc,.docx,.txt">
                        <input type="hidden" id="edit_current_download_file" name="current_download_file">
                        <input type="hidden" id="edit_current_download_file_name" name="current_download_file_name">
                        <input type="hidden" id="edit_current_download_file_size" name="current_download_file_size">
                        <input type="hidden" id="edit_current_download_file_type" name="current_download_file_type">
                        <div class="form-text">Leave empty to keep current file. Upload new file to replace.</div>
                        <div id="current_download_file_preview" class="mt-2"></div>
                      </div>
                    </div>
                    <div class="col-md-6">
                      <div class="mb-3">
                        <label for="edit_download_limit" class="form-label">Download Limit</label>
                        <input type="number" class="form-control" id="edit_download_limit" name="download_limit" value="3" min="1" max="100">
                        <div class="form-text">Number of times user can download (default: 3)</div>
                      </div>
                    </div>
                  </div>
                  <div class="row">
                    <div class="col-md-6">
                      <div class="mb-3">
                        <label for="edit_download_access_duration" class="form-label">Access Duration (Days)</label>
                        <input type="number" class="form-control" id="edit_download_access_duration" name="download_access_duration" value="365" min="1" max="3650">
                        <div class="form-text">How long users can access downloads (default: 365 days)</div>
                      </div>
                    </div>
                    <div class="col-md-6">
                      <div class="mb-3">
                        <label for="edit_download_instructions" class="form-label">Download Instructions</label>
                        <textarea class="form-control" id="edit_download_instructions" name="download_instructions" rows="3" placeholder="Instructions for users on how to use the downloaded files..."></textarea>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            </div>
            
            <div class="mb-3">
              <label for="edit_thumbnail" class="form-label">Product Thumbnail</label>
              <input type="file" class="form-control" id="edit_thumbnail" name="thumbnail" accept="image/*">
              <input type="hidden" id="edit_current_thumbnail" name="current_thumbnail">
              <div class="form-text">Leave empty to keep current image. Recommended size: 800x600px or similar aspect ratio</div>
              <div id="current_thumbnail_preview" class="mt-2"></div>
            </div>
          </div>
          <div class="modal-footer">
            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
            <button type="submit" class="btn btn-primary">Update Product</button>
          </div>
        </form>
      </div>
    </div>
  </div>

  <script>
    function editProduct(product) {
      document.getElementById('edit_product_id').value = product.id;
      document.getElementById('edit_product_name').value = product.product_name;
      document.getElementById('edit_product_description').value = product.product_description || '';
      document.getElementById('edit_category').value = product.category;
      document.getElementById('edit_product_amount').value = product.product_amount || 0;
      document.getElementById('edit_sort_order').value = product.sort_order || 0;
      document.getElementById('edit_demo_link').value = product.demo_link || '';
      document.getElementById('edit_status').value = product.status;
      document.getElementById('edit_product_features').value = product.product_features || '';
      document.getElementById('edit_demo_login_details').value = product.demo_login_details || '';
      document.getElementById('edit_product_specifications').value = product.product_specifications || '';
      document.getElementById('edit_current_thumbnail').value = product.thumbnail_url || '';
      
      // Populate download file fields
      document.getElementById('edit_is_downloadable').checked = product.is_downloadable == 1;
      document.getElementById('edit_download_limit').value = product.download_limit || 3;
      document.getElementById('edit_download_access_duration').value = product.download_access_duration || 365;
      document.getElementById('edit_download_instructions').value = product.download_instructions || '';
      document.getElementById('edit_current_download_file').value = product.download_file_url || '';
      document.getElementById('edit_current_download_file_name').value = product.download_file_name || '';
      document.getElementById('edit_current_download_file_size').value = product.download_file_size || 0;
      document.getElementById('edit_current_download_file_type').value = product.download_file_type || '';
      
      // Toggle download fields visibility
      toggleEditDownloadFields();
      
      // Show current thumbnail preview
      const preview = document.getElementById('current_thumbnail_preview');
      if (product.thumbnail_url) {
        preview.innerHTML = '<img src="../' + product.thumbnail_url + '" alt="Current thumbnail" class="img-thumbnail" style="max-width: 200px;">';
      } else {
        preview.innerHTML = '<span class="text-muted">No current thumbnail</span>';
      }
      
      // Show current download file preview
      const downloadPreview = document.getElementById('current_download_file_preview');
      if (product.download_file_url && product.download_file_name) {
        const fileSize = product.download_file_size ? formatFileSize(product.download_file_size) : 'Unknown size';
        downloadPreview.innerHTML = '<div class="alert alert-info"><strong>Current File:</strong> ' + product.download_file_name + ' (' + fileSize + ')</div>';
      } else {
        downloadPreview.innerHTML = '<span class="text-muted">No current download file</span>';
      }
      
      // Show the modal
      new bootstrap.Modal(document.getElementById('editProductModal')).show();
    }
    
    function toggleDownloadFields() {
      const checkbox = document.getElementById('is_downloadable');
      const fields = document.getElementById('downloadFields');
      if (checkbox.checked) {
        fields.style.display = 'block';
      } else {
        fields.style.display = 'none';
      }
    }
    
    function toggleEditDownloadFields() {
      const checkbox = document.getElementById('edit_is_downloadable');
      const fields = document.getElementById('editDownloadFields');
      if (checkbox.checked) {
        fields.style.display = 'block';
      } else {
        fields.style.display = 'none';
      }
    }
    
    function formatFileSize(bytes) {
      if (bytes === 0) return '0 Bytes';
      const k = 1024;
      const sizes = ['Bytes', 'KB', 'MB', 'GB'];
      const i = Math.floor(Math.log(bytes) / Math.log(k));
      return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }
  </script>

  </body>

</html>