<?php
session_start();

// Check if admin is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: authentication/login.php');
    exit();
}

require_once 'config/database.php';
require_once 'config/settings.php';
require_once 'config/branding.php';

$success_message = '';
$error_message = '';
$adminData = null;

try {
    $pdo = getConnection();
    
    // Get admin details
    $adminStmt = $pdo->prepare("SELECT * FROM admin_users WHERE id = ?");
    $adminStmt->execute([$_SESSION['admin_id']]);
    $adminData = $adminStmt->fetch();
    
    if (!$adminData) {
        header('Location: authentication/login.php');
        exit();
    }
    
    // Handle form submissions
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        if (isset($_POST['action'])) {
            switch ($_POST['action']) {
                case 'update_profile':
                    $full_name = trim($_POST['full_name']);
                    $email = trim($_POST['email']);
                    $phone = trim($_POST['phone']);
                    $bio = trim($_POST['bio']);
                    
                    if (empty($full_name) || empty($email)) {
                        $error_message = "Full name and email are required.";
                    } else {
                        $updateStmt = $pdo->prepare("UPDATE admin_users SET full_name = ?, email = ?, phone = ?, bio = ? WHERE id = ?");
                        if ($updateStmt->execute([$full_name, $email, $phone, $bio, $_SESSION['admin_id']])) {
                            $success_message = "Profile updated successfully!";
                            // Refresh admin data
                            $adminStmt->execute([$_SESSION['admin_id']]);
                            $adminData = $adminStmt->fetch();
                        } else {
                            $error_message = "Failed to update profile.";
                        }
                    }
                    break;
                    
                case 'change_password':
                    $current_password = $_POST['current_password'];
                    $new_password = $_POST['new_password'];
                    $confirm_password = $_POST['confirm_password'];
                    
                    if (empty($current_password) || empty($new_password) || empty($confirm_password)) {
                        $error_message = "All password fields are required.";
                    } elseif ($new_password !== $confirm_password) {
                        $error_message = "New passwords do not match.";
                    } elseif (strlen($new_password) < 6) {
                        $error_message = "New password must be at least 6 characters long.";
                    } elseif (!password_verify($current_password, $adminData['password'])) {
                        $error_message = "Current password is incorrect.";
                    } else {
                        $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
                        $passwordStmt = $pdo->prepare("UPDATE admin_users SET password = ? WHERE id = ?");
                        if ($passwordStmt->execute([$hashed_password, $_SESSION['admin_id']])) {
                            $success_message = "Password changed successfully!";
                        } else {
                            $error_message = "Failed to change password.";
                        }
                    }
                    break;
                    
                case 'upload_avatar':
                    if (isset($_FILES['avatar']) && $_FILES['avatar']['error'] === UPLOAD_ERR_OK) {
                        $upload_dir = '../assets/uploads/admin/';
                        if (!file_exists($upload_dir)) {
                            mkdir($upload_dir, 0777, true);
                        }
                        
                        $file_extension = strtolower(pathinfo($_FILES['avatar']['name'], PATHINFO_EXTENSION));
                        $allowed_extensions = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
                        
                        if (!in_array($file_extension, $allowed_extensions)) {
                            $error_message = "Invalid file type. Only JPG, PNG, GIF, and WebP are allowed.";
                        } elseif ($_FILES['avatar']['size'] > 5 * 1024 * 1024) { // 5MB limit
                            $error_message = "File size too large. Maximum 5MB allowed.";
                        } else {
                            $new_filename = 'admin_' . $_SESSION['admin_id'] . '_' . time() . '.' . $file_extension;
                            $upload_path = $upload_dir . $new_filename;
                            
                            if (move_uploaded_file($_FILES['avatar']['tmp_name'], $upload_path)) {
                                // Delete old avatar if exists
                                if (!empty($adminData['profile_image']) && file_exists('../' . $adminData['profile_image'])) {
                                    unlink('../' . $adminData['profile_image']);
                                }
                                
                                $avatar_path = 'assets/uploads/admin/' . $new_filename;
                                $avatarStmt = $pdo->prepare("UPDATE admin_users SET profile_image = ? WHERE id = ?");
                                if ($avatarStmt->execute([$avatar_path, $_SESSION['admin_id']])) {
                                    $success_message = "Profile image updated successfully!";
                                    // Refresh admin data
                                    $adminStmt->execute([$_SESSION['admin_id']]);
                                    $adminData = $adminStmt->fetch();
                                } else {
                                    $error_message = "Failed to update profile image in database.";
                                }
                            } else {
                                $error_message = "Failed to upload file.";
                            }
                        }
                    } else {
                        $error_message = "No file uploaded or upload error occurred.";
                    }
                    break;
            }
        }
    }
    
} catch (Exception $e) {
    $error_message = "Database error: " . $e->getMessage();
}
?>
<!DOCTYPE html>
<html lang="en">

<head>

  <!-- begin::Edura Meta Basic -->
  <meta charset="utf-8">
  <meta name="theme-color" content="#0D9488">
  <meta name="robots" content="index, follow">
  <meta name="author" content="LayoutDrop">
  <meta name="format-detection" content="telephone=no">
  <meta name="keywords" content="School Management, Education Admin Dashboard, Student Information System, University Dashboard, College Admin Template, Learning Management System, Education CRM, Teacher Management, Student Attendance Dashboard, Online Education Admin, Responsive Admin Template, Bootstrap 5 Education Admin, eLearning Dashboard, School ERP, Class Management Dashboard, Academic Dashboard, Education Web Application, Modern Admin Template, Admin UI Kit, ThemeForest Admin Template">
  <meta name="description" content="Edura is a professional and modern School Management & Education Admin Dashboard Template built with Bootstrap. It includes light and dark modes, and is perfect for managing students, teachers, courses, attendance, fees, and academic activities — ideal for schools, colleges, universities, and eLearning platforms.">
  <!-- end::Edura Meta Basic -->

  <!-- begin::Edura Meta Social -->
  <meta property="og:url" content="https://demos.layoutdrop.com/edura/demo/">
  <meta property="og:site_name" content="Profile | Edura School Management & Education Admin Dashboard Template">
  <meta property="og:type" content="website">
  <meta property="og:locale" content="en_us">
  <meta property="og:title" content="Profile | Edura School Management & Education Admin Dashboard Template">
  <meta property="og:description" content="Edura is a professional and modern School Management & Education Admin Dashboard Template built with Bootstrap. It includes light and dark modes, and is perfect for managing students, teachers, courses, attendance, fees, and academic activities — ideal for schools, colleges, universities, and eLearning platforms.">
  <meta property="og:image" content="https://demos.layoutdrop.com/edura/demo/preview.png">
  <!-- end::Edura Meta Social -->

  <!-- begin::Edura Meta Twitter -->
  <meta name="twitter:card" content="summary">
  <meta name="twitter:url" content="https://demos.layoutdrop.com/edura/demo/">
  <meta name="twitter:creator" content="@layoutdrop">
  <meta name="twitter:title" content="Profile | Edura School Management & Education Admin Dashboard Template">
  <meta name="twitter:description" content="Edura is a professional and modern School Management & Education Admin Dashboard Template built with Bootstrap. It includes light and dark modes, and is perfect for managing students, teachers, courses, attendance, fees, and academic activities — ideal for schools, colleges, universities, and eLearning platforms.">
  <!-- end::Edura Meta Twitter -->

  <!-- begin::Edura Website Page Title -->
  <title>Profile | Edura School Management & Education Admin Dashboard Template</title>
  <!-- end::Edura Website Page Title -->

  <!-- begin::Edura Mobile Specific -->
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <!-- end::Edura Mobile Specific -->

  <!-- begin::Edura Favicon Tags -->
  <?php echo getFaviconHtml(); ?>
  <!-- end::Edura Favicon Tags -->

  <!-- begin::Edura Google Fonts -->
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Plus+Jakarta+Sans:ital,wght@0,200..800;1,200..800&display=swap" rel="stylesheet">
  <!-- end::Edura Google Fonts -->

  <!-- begin::Edura Required Stylesheet -->
  <link rel="stylesheet" href="assets/libs/flaticon/css/all/all.css">
  <link rel="stylesheet" href="assets/libs/lucide/lucide.css">
  <link rel="stylesheet" href="assets/libs/fontawesome/css/all.min.css">
  <link rel="stylesheet" href="assets/libs/simplebar/simplebar.css">
  <link rel="stylesheet" href="assets/libs/node-waves/waves.css">
  <link rel="stylesheet" href="assets/libs/bootstrap-select/css/bootstrap-select.min.css">
  <!-- end::Edura Required Stylesheet -->

  <!-- begin::Edura CSS Stylesheet -->
  <link rel="stylesheet" href="assets/css/styles.css">
  <!-- end::Edura CSS Stylesheet -->

  <!-- begin::Edura Googletagmanager -->
  <script async src="https://www.googletagmanager.com/gtag/js?id=G-XWVQM68HHQ"></script>
  <script>
		window.dataLayer = window.dataLayer || [];
		function gtag(){dataLayer.push(arguments);}
		gtag('js', new Date());

		gtag('config', 'G-XWVQM68HHQ', {
			'cookie_flags': 'SameSite=None;Secure',
			'send_page_view': true
		});
	</script>
  <!-- end::Edura Googletagmanager -->

</head>

<body>
  <div class="page-layout">

    <?php include 'includes/header.php'; ?>

    <!-- begin::Edura Sidebar Menu -->
    <aside class="app-menubar" id="menubar">
      <div class="app-navbar-brand">
        <a class="navbar-brand-logo" href="index.php">
          <img src="../<?php echo getSetting('main_logo', 'assets/images/logo.svg'); ?>" alt="logo">
        </a>
        <a class="navbar-brand-mini" href="index.php">
          <img src="../<?php echo getSetting('mini_logo', 'assets/images/logo-text-white.svg'); ?>" alt="logo">
        </a>
      </div>
      <nav class="app-navbar" data-simplebar>
        <ul class="menubar">
          <li class="menu-item">
            <a class="menu-link" href="index.php">
              <i class="fi fi-rr-apps"></i>
              <span class="menu-label">Dashboard</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/users.php">
              <i class="fi fi-rr-users"></i>
              <span class="menu-label">Manage Users</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/portfolio.php">
              <i class="fi fi-rr-portfolio"></i>
              <span class="menu-label">Manage Portfolio</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/products.php">
              <i class="fi fi-rr-shopping-cart"></i>
              <span class="menu-label">Manage Products</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/orders.php">
              <i class="fi fi-rr-receipt"></i>
              <span class="menu-label">Manage Orders</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/courses.php">
              <i class="fi fi-rr-book-alt"></i>
              <span class="menu-label">Manage Courses</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/script_requests.php">
              <i class="fi fi-rr-file-code"></i>
              <span class="menu-label">Manage Script Request</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/settings.php">
              <i class="fi fi-rr-settings"></i>
              <span class="menu-label">Settings</span>
            </a>
          </li>
        </ul>
      </nav>
      <div class="app-footer">
        <a href="pages/faq.html" class="btn btn-outline-white waves-effect btn-app-nav w-100">
          <i class="fi fi-rs-interrogation"></i>
          <span class="nav-text">Help and Support</span>
        </a>
      </div>
    </aside>
    <!-- end::Edura Sidebar Menu -->

    <main class="app-wrapper">

      <div class="container-fluid">

        <div class="app-page-head d-flex flex-wrap gap-3 align-items-center justify-content-between">
          <div class="clearfix">
            <h1 class="app-page-title">Profile</h1>
            <nav aria-label="breadcrumb">
              <ol class="breadcrumb mb-0">
                <li class="breadcrumb-item">
                  <a href="index.php">Dashboard</a>
                </li>
                <li class="breadcrumb-item active" aria-current="page">Profile</li>
              </ol>
            </nav>
          </div>
        </div>

        <!-- Success/Error Messages -->
        <?php if (!empty($success_message)): ?>
          <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fi fi-rr-check-circle me-2"></i>
            <?php echo htmlspecialchars($success_message); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
          </div>
        <?php endif; ?>

        <?php if (!empty($error_message)): ?>
          <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fi fi-rr-exclamation-triangle me-2"></i>
            <?php echo htmlspecialchars($error_message); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
          </div>
        <?php endif; ?>

        <div class="row">

          <!-- Profile Header Card -->
          <div class="col-lg-12">
            <div class="card">
              <div class="card-body">
                <div class="d-flex flex-wrap gap-4 align-items-center">
                  <div class="d-flex align-items-center">
                    <div class="position-relative">
                      <div class="avatar avatar-xxl rounded-circle">
                        <?php if ($adminData && !empty($adminData['profile_image'])): ?>
                          <img src="../<?php echo htmlspecialchars($adminData['profile_image']); ?>" alt="Admin Avatar">
                        <?php else: ?>
                          <div class="bg-primary text-white d-flex align-items-center justify-content-center h-100 w-100 rounded-circle" style="font-size: 2rem;">
                            <?php echo strtoupper(substr($adminData['username'] ?? 'A', 0, 1)); ?>
                          </div>
                        <?php endif; ?>
                      </div>
                      <button type="button" class="avatar avatar-xxs bg-primary rounded-circle text-white position-absolute top-0 end-0" data-bs-toggle="modal" data-bs-target="#avatarModal">
                        <i class="fi fi-rr-camera"></i>
                      </button>
                    </div>
                    <div class="ms-3">
                      <h4 class="fw-bold mb-0"><?php echo htmlspecialchars($adminData['full_name'] ?? $adminData['username'] ?? 'Admin'); ?></h4>
                      <small class="mb-2"><?php echo htmlspecialchars($adminData['bio'] ?? 'Administrator'); ?></small>
                      <div class="d-flex flex-wrap gap-1 mt-2">
                        <span class="badge badge-sm px-2 rounded-pill text-bg-primary">Administrator</span>
                        <span class="badge badge-sm px-2 rounded-pill text-bg-success">Active</span>
                      </div>
                    </div>
                  </div>
                  <div class="d-flex gap-2 ms-md-auto">
                    <span class="text-muted">Member since <?php echo date('M Y', strtotime($adminData['created_at'] ?? 'now')); ?></span>
                  </div>
                </div>
              </div>
            </div>
          </div>

          <!-- Profile Information Card -->
          <div class="col-lg-4 col-sm-12">
            <div class="card">
              <div class="card-header">
                <h4 class="card-title mb-0">Profile Information</h4>
              </div>
              <div class="card-body">
                <div class="mb-3">
                  <span class="mb-1 text-muted">Username</span>
                  <p class="text-dark fw-semibold mb-0"><?php echo htmlspecialchars($adminData['username'] ?? 'N/A'); ?></p>
                </div>
                <div class="mb-3">
                  <span class="mb-1 text-muted">Full Name</span>
                  <p class="text-dark fw-semibold mb-0"><?php echo htmlspecialchars($adminData['full_name'] ?? 'N/A'); ?></p>
                </div>
                <div class="mb-3">
                  <span class="mb-1 text-muted">Email</span>
                  <p class="text-dark fw-semibold mb-0"><?php echo htmlspecialchars($adminData['email'] ?? 'N/A'); ?></p>
                </div>
                <div class="mb-3">
                  <span class="mb-1 text-muted">Phone</span>
                  <p class="text-dark fw-semibold mb-0"><?php echo htmlspecialchars($adminData['phone'] ?? 'N/A'); ?></p>
                </div>
                <div class="mb-2">
                  <span class="mb-1 text-muted">Joined Date</span>
                  <p class="text-dark fw-semibold mb-0"><?php echo date('M d, Y', strtotime($adminData['created_at'] ?? 'now')); ?></p>
                </div>
              </div>
            </div>
          </div>

          <!-- Profile Update Forms -->
          <div class="col-lg-8 col-sm-12">
            <div class="row">
              <!-- Update Profile Form -->
              <div class="col-12 mb-4">
                <div class="card">
                  <div class="card-header">
                    <h4 class="card-title">Update Profile Information</h4>
                  </div>
                  <div class="card-body">
                    <form method="POST" action="">
                      <input type="hidden" name="action" value="update_profile">
                      <div class="row mb-3">
                        <div class="col-md-6">
                          <label class="form-label">Full Name <span class="text-danger">*</span></label>
                          <input type="text" class="form-control" name="full_name" value="<?php echo htmlspecialchars($adminData['full_name'] ?? ''); ?>" required>
                        </div>
                        <div class="col-md-6">
                          <label class="form-label">Email <span class="text-danger">*</span></label>
                          <input type="email" class="form-control" name="email" value="<?php echo htmlspecialchars($adminData['email'] ?? ''); ?>" required>
                        </div>
                      </div>
                      <div class="row mb-3">
                        <div class="col-md-6">
                          <label class="form-label">Phone</label>
                          <input type="tel" class="form-control" name="phone" value="<?php echo htmlspecialchars($adminData['phone'] ?? ''); ?>">
                        </div>
                        <div class="col-md-6">
                          <label class="form-label">Username</label>
                          <input type="text" class="form-control" value="<?php echo htmlspecialchars($adminData['username'] ?? ''); ?>" readonly>
                          <small class="text-muted">Username cannot be changed</small>
                        </div>
                      </div>
                      <div class="mb-3">
                        <label class="form-label">Bio</label>
                        <textarea class="form-control" name="bio" rows="4" placeholder="Tell us about yourself..."><?php echo htmlspecialchars($adminData['bio'] ?? ''); ?></textarea>
                      </div>
                      <div class="text-end">
                        <button type="submit" class="btn btn-primary waves-effect waves-light">
                          <i class="fi fi-rr-disk me-2"></i>Update Profile
                        </button>
                      </div>
                    </form>
                  </div>
                </div>
              </div>

              <!-- Change Password Form -->
              <div class="col-12">
                <div class="card">
                  <div class="card-header">
                    <h4 class="card-title">Change Password</h4>
                  </div>
                  <div class="card-body">
                    <form method="POST" action="">
                      <input type="hidden" name="action" value="change_password">
                      <div class="row mb-3">
                        <div class="col-md-6">
                          <label class="form-label">Current Password <span class="text-danger">*</span></label>
                          <input type="password" class="form-control" name="current_password" required>
                        </div>
                        <div class="col-md-6">
                          <label class="form-label">New Password <span class="text-danger">*</span></label>
                          <input type="password" class="form-control" name="new_password" required minlength="6">
                        </div>
                      </div>
                      <div class="row mb-3">
                        <div class="col-md-6">
                          <label class="form-label">Confirm New Password <span class="text-danger">*</span></label>
                          <input type="password" class="form-control" name="confirm_password" required minlength="6">
                        </div>
                        <div class="col-md-6">
                          <div class="mt-4">
                            <small class="text-muted">
                              <i class="fi fi-rr-info me-1"></i>
                              Password must be at least 6 characters long
                            </small>
                          </div>
                        </div>
                      </div>
                      <div class="text-end">
                        <button type="submit" class="btn btn-warning waves-effect waves-light">
                          <i class="fi fi-rr-lock me-2"></i>Change Password
                        </button>
                      </div>
                    </form>
                  </div>
                </div>
              </div>
            </div>
          </div>

        </div>

      </div>

    </main>

    <!-- begin::Edura Footer -->
    <footer class="footer-wrapper bg-body text-center">
      <div class="container-fluid">
        <p class="mb-0">© <span class="currentYear">2025</span> Edura. Proudly powered by <a href="javascript:void(0);">LayoutDrop</a>.</p>
      </div>
    </footer>
    <!-- end::Edura Footer -->

  </div>

  <!-- Avatar Upload Modal -->
  <div class="modal fade" id="avatarModal" tabindex="-1" aria-labelledby="avatarModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
      <div class="modal-content">
        <div class="modal-header">
          <h5 class="modal-title" id="avatarModalLabel">Update Profile Picture</h5>
          <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
        </div>
        <form method="POST" action="" enctype="multipart/form-data">
          <input type="hidden" name="action" value="upload_avatar">
          <div class="modal-body">
            <div class="text-center mb-3">
              <div class="avatar avatar-xxl rounded-circle mx-auto mb-3">
                <?php if ($adminData && !empty($adminData['profile_image'])): ?>
                  <img src="../<?php echo htmlspecialchars($adminData['profile_image']); ?>" alt="Current Avatar" id="currentAvatar">
                <?php else: ?>
                  <div class="bg-primary text-white d-flex align-items-center justify-content-center h-100 w-100 rounded-circle" style="font-size: 2rem;" id="currentAvatar">
                    <?php echo strtoupper(substr($adminData['username'] ?? 'A', 0, 1)); ?>
                  </div>
                <?php endif; ?>
              </div>
              <p class="text-muted">Click to select a new profile picture</p>
            </div>
            <div class="mb-3">
              <input type="file" class="form-control" name="avatar" id="avatarInput" accept="image/*" required>
              <small class="text-muted">Supported formats: JPG, PNG, GIF, WebP. Max size: 5MB</small>
            </div>
            <div class="text-center">
              <img id="avatarPreview" src="" alt="Preview" class="img-thumbnail" style="max-width: 150px; max-height: 150px; display: none;">
            </div>
          </div>
          <div class="modal-footer">
            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
            <button type="submit" class="btn btn-primary">Update Picture</button>
          </div>
        </form>
      </div>
    </div>
  </div>

  <!-- begin::Edura Page Scripts -->
  <script src="assets/libs/global/global.min.js"></script>
  <script src="assets/js/appSettings.js"></script>
  <script src="assets/js/main.js"></script>
  
  <!-- Avatar Preview Script -->
  <script>
    document.getElementById('avatarInput').addEventListener('change', function(e) {
      const file = e.target.files[0];
      if (file) {
        const reader = new FileReader();
        reader.onload = function(e) {
          const preview = document.getElementById('avatarPreview');
          preview.src = e.target.result;
          preview.style.display = 'block';
        };
        reader.readAsDataURL(file);
      }
    });
  </script>
  <!-- end::Edura Page Scripts -->
</body>

</html>