<?php
require_once 'config/database.php';
require_once 'config/settings.php';

$error = '';
$success = '';

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: authentication/login.php');
    exit();
}

// Handle settings update
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $action = $_POST['action'] ?? '';
    
    try {
        $pdo = getConnection();
        
        if ($action === 'update_settings') {
            $category = $_POST['category'] ?? '';
            $settings = $_POST['settings'] ?? [];
            
            if (empty($category)) {
                $error = "Category is required";
            } else {
                // Update settings for the category
                foreach ($settings as $key => $value) {
                    $stmt = $pdo->prepare("
                        UPDATE settings 
                        SET setting_value = ?, updated_at = NOW() 
                        WHERE setting_key = ? AND category = ?
                    ");
                    $stmt->execute([$value, $key, $category]);
                }
                
                // Clear settings cache
                SettingsManager::clearCache();
                
                // If updating Paystack settings, also clear payment config cache
                if ($category === 'paystack') {
                    // Clear any cached payment configurations
                    if (function_exists('opcache_invalidate')) {
                        opcache_invalidate(__DIR__ . '/../users/config/payment_config.php');
                    }
                    
                    // Log the update
                    error_log("Paystack settings updated by admin: " . $_SESSION['admin_id']);
                }
                
                $success = ucfirst($category) . " settings updated successfully";
            }
        } elseif ($action === 'update_logo') {
            // Handle logo uploads
            $uploadDir = '../uploads/logos/';
            
            // Create upload directory if it doesn't exist
            if (!file_exists($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }
            
            $logoUpdated = false;
            
            // Handle main logo upload
            if (isset($_FILES['main_logo']) && $_FILES['main_logo']['error'] === UPLOAD_ERR_OK) {
                $mainLogoFile = $_FILES['main_logo'];
                $mainLogoExtension = strtolower(pathinfo($mainLogoFile['name'], PATHINFO_EXTENSION));
                
                if (in_array($mainLogoExtension, ['svg', 'png', 'jpg', 'jpeg'])) {
                    $mainLogoName = 'main_logo_' . time() . '.' . $mainLogoExtension;
                    $mainLogoPath = $uploadDir . $mainLogoName;
                    
                    if (move_uploaded_file($mainLogoFile['tmp_name'], $mainLogoPath)) {
                        // Update setting in database
                        $stmt = $pdo->prepare("
                            INSERT INTO settings (setting_key, setting_value, setting_type, category, created_at, updated_at) 
                            VALUES ('main_logo', ?, 'string', 'logo', NOW(), NOW())
                            ON DUPLICATE KEY UPDATE setting_value = ?, updated_at = NOW()
                        ");
                        $stmt->execute(['uploads/logos/' . $mainLogoName, 'uploads/logos/' . $mainLogoName]);
                        $logoUpdated = true;
                    }
                }
            }
            
            // Handle mini logo upload
            if (isset($_FILES['mini_logo']) && $_FILES['mini_logo']['error'] === UPLOAD_ERR_OK) {
                $miniLogoFile = $_FILES['mini_logo'];
                $miniLogoExtension = strtolower(pathinfo($miniLogoFile['name'], PATHINFO_EXTENSION));
                
                if (in_array($miniLogoExtension, ['svg', 'png', 'jpg', 'jpeg'])) {
                    $miniLogoName = 'mini_logo_' . time() . '.' . $miniLogoExtension;
                    $miniLogoPath = $uploadDir . $miniLogoName;
                    
                    if (move_uploaded_file($miniLogoFile['tmp_name'], $miniLogoPath)) {
                        // Update setting in database
                        $stmt = $pdo->prepare("
                            INSERT INTO settings (setting_key, setting_value, setting_type, category, created_at, updated_at) 
                            VALUES ('mini_logo', ?, 'string', 'logo', NOW(), NOW())
                            ON DUPLICATE KEY UPDATE setting_value = ?, updated_at = NOW()
                        ");
                        $stmt->execute(['uploads/logos/' . $miniLogoName, 'uploads/logos/' . $miniLogoName]);
                        $logoUpdated = true;
                    }
                }
            }
            
            // Handle reset to default logos
            if (isset($_POST['reset_logos']) && $_POST['reset_logos'] == '1') {
                $stmt = $pdo->prepare("
                    INSERT INTO settings (setting_key, setting_value, setting_type, category, created_at, updated_at) 
                    VALUES ('main_logo', 'assets/images/logo.svg', 'string', 'logo', NOW(), NOW())
                    ON DUPLICATE KEY UPDATE setting_value = 'assets/images/logo.svg', updated_at = NOW()
                ");
                $stmt->execute();
                
                $stmt = $pdo->prepare("
                    INSERT INTO settings (setting_key, setting_value, setting_type, category, created_at, updated_at) 
                    VALUES ('mini_logo', 'assets/images/logo-text-white.svg', 'string', 'logo', NOW(), NOW())
                    ON DUPLICATE KEY UPDATE setting_value = 'assets/images/logo-text-white.svg', updated_at = NOW()
                ");
                $stmt->execute();
                
                $logoUpdated = true;
            }
            
            if ($logoUpdated) {
                SettingsManager::clearCache();
                $success = "Logo settings updated successfully";
            } else {
                $error = "No logo files were uploaded or updated";
            }
        } elseif ($action === 'update_favicon') {
            // Handle favicon upload
            $uploadDir = '../uploads/logos/';
            
            // Create upload directory if it doesn't exist
            if (!file_exists($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }
            
            $faviconUpdated = false;
            
            // Handle favicon upload
            if (isset($_FILES['favicon']) && $_FILES['favicon']['error'] === UPLOAD_ERR_OK) {
                $faviconFile = $_FILES['favicon'];
                $faviconExtension = strtolower(pathinfo($faviconFile['name'], PATHINFO_EXTENSION));
                
                if (in_array($faviconExtension, ['ico', 'png', 'jpg', 'jpeg', 'svg'])) {
                    $faviconName = 'favicon_' . time() . '.' . $faviconExtension;
                    $faviconPath = $uploadDir . $faviconName;
                    
                    if (move_uploaded_file($faviconFile['tmp_name'], $faviconPath)) {
                        // Update setting in database
                        $stmt = $pdo->prepare("
                            INSERT INTO settings (setting_key, setting_value, setting_type, category, created_at, updated_at) 
                            VALUES ('favicon', ?, 'string', 'logo', NOW(), NOW())
                            ON DUPLICATE KEY UPDATE setting_value = ?, updated_at = NOW()
                        ");
                        $stmt->execute(['uploads/logos/' . $faviconName, 'uploads/logos/' . $faviconName]);
                        $faviconUpdated = true;
                    }
                }
            }
            
            // Handle reset to default favicon
            if (isset($_POST['reset_favicon']) && $_POST['reset_favicon'] == '1') {
                $stmt = $pdo->prepare("
                    INSERT INTO settings (setting_key, setting_value, setting_type, category, created_at, updated_at) 
                    VALUES ('favicon', 'assets/images/favicon.png', 'string', 'logo', NOW(), NOW())
                    ON DUPLICATE KEY UPDATE setting_value = 'assets/images/favicon.png', updated_at = NOW()
                ");
                $stmt->execute();
                $faviconUpdated = true;
            }
            
            if ($faviconUpdated) {
                SettingsManager::clearCache();
                $success = "Favicon updated successfully";
            } else {
                $error = "No favicon file was uploaded or updated";
            }
        }
    } catch (Exception $e) {
        $error = "Action failed: " . $e->getMessage();
    }
}

// Get current settings for all categories
$generalSettings = SettingsManager::getGeneralSettings();
$cryptoSettings = SettingsManager::getCryptoSettings();
$bankSettings = SettingsManager::getBankSettings();
$paystackSettings = SettingsManager::getPaystackSettings();
$systemSettings = SettingsManager::getSystemSettings();

// Get email settings
$emailSettings = SettingsManager::getCategory('email');
?>
<!DOCTYPE html>
<html lang="en">

<head>

  <base href="">

  <!-- begin::Edura Meta Basic -->
  <meta charset="utf-8">
  <meta name="theme-color" content="#0D9488">
  <meta name="robots" content="index, follow">
  <meta name="author" content="Omateq Digitals">
  <meta name="format-detection" content="telephone=no">
  <meta name="keywords" content="Admin Dashboard, Omateq Digitals, Settings, Configuration, Admin Panel, System Settings">
  <meta name="description" content="Omateq Digitals Admin Dashboard - Manage system settings, payment configurations, and general preferences.">
  <!-- end::Edura Meta Basic -->

  <!-- begin::Edura Meta Social -->
  <meta property="og:url" content="">
  <meta property="og:site_name" content="Settings | Omateq Digitals">
  <meta property="og:type" content="website">
  <meta property="og:locale" content="en_us">
  <meta property="og:title" content="Settings | Omateq Digitals">
  <meta property="og:description" content="Omateq Digitals Admin Dashboard - Manage system settings, payment configurations, and general preferences.">
  <!-- end::Edura Meta Social -->

  <!-- begin::Edura Meta Twitter -->
  <meta name="twitter:card" content="summary">
  <meta name="twitter:url" content="">
  <meta name="twitter:creator" content="@omateqdigitals">
  <meta name="twitter:title" content="Settings | Omateq Digitals">
  <meta name="twitter:description" content="Omateq Digitals Admin Dashboard - Manage system settings, payment configurations, and general preferences.">
  <!-- end::Edura Meta Twitter -->

  <!-- begin::Edura Website Page Title -->
  <title>Settings | Omateq Digitals</title>
  <!-- end::Edura Website Page Title -->

  <!-- begin::Edura Mobile Specific -->
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <!-- end::Edura Mobile Specific -->

  <!-- begin::Edura Favicon Tags -->
  <link rel="icon" type="image/png" href="../<?php echo getSetting('favicon', 'assets/images/favicon.png'); ?>">
  <link rel="apple-touch-icon" sizes="180x180" href="../<?php echo getSetting('favicon', 'assets/images/favicon.png'); ?>">
  <!-- end::Edura Favicon Tags -->

  <!-- begin::Edura Google Fonts -->
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Plus+Jakarta+Sans:ital,wght@0,200..800;1,200..800&display=swap" rel="stylesheet">
  <!-- end::Edura Google Fonts -->

  <!-- begin::Edura Required Stylesheet -->
  <link rel="stylesheet" href="assets/libs/flaticon/css/all/all.css">
  <link rel="stylesheet" href="assets/libs/lucide/lucide.css">
  <link rel="stylesheet" href="assets/libs/fontawesome/css/all.min.css">
  <link rel="stylesheet" href="assets/libs/simplebar/simplebar.css">
  <link rel="stylesheet" href="assets/libs/node-waves/waves.css">
  <link rel="stylesheet" href="assets/libs/bootstrap-select/css/bootstrap-select.min.css">
  <!-- end::Edura Required Stylesheet -->

  <!-- begin::Edura CSS Stylesheet -->
  <link rel="stylesheet" href="assets/css/styles.css">
  <!-- end::Edura CSS Stylesheet -->
  
  <style>
    /* Logo sizing for better fit */
    .app-navbar-brand .navbar-brand-logo img {
      max-height: 50px;
      max-width: 180px;
      width: auto;
      height: auto;
      object-fit: contain;
    }
    
    .app-navbar-brand .navbar-brand-mini img {
      max-height: 35px;
      max-width: 35px;
      width: auto;
      height: auto;
      object-fit: contain;
    }
    
    /* Logo preview sizing */
    #current_main_logo {
      max-height: 50px !important;
      max-width: 180px !important;
      width: auto !important;
      height: auto !important;
      object-fit: contain;
    }
    
    #current_mini_logo {
      max-height: 35px !important;
      max-width: 35px !important;
      width: auto !important;
      height: auto !important;
      object-fit: contain;
    }
  </style>

  <!-- begin::Edura Googletagmanager -->
  <script async src="https://www.googletagmanager.com/gtag/js?id=G-XWVQM68HHQ"></script>
  <script>
		window.dataLayer = window.dataLayer || [];
		function gtag(){dataLayer.push(arguments);}
		gtag('js', new Date());

		gtag('config', 'G-XWVQM68HHQ', {
			'cookie_flags': 'SameSite=None;Secure',
			'send_page_view': true
		});
	</script>
  <!-- end::Edura Googletagmanager -->

</head>

<body>
  <div class="page-layout">

    <?php include 'includes/header.php'; ?>

     <!-- begin::Edura Sidebar Menu -->
     <aside class="app-menubar" id="menubar">
      <div class="app-navbar-brand">
        <a class="navbar-brand-logo" href="index.php">
          <img src="../<?php echo getSetting('main_logo', 'assets/images/logo.svg'); ?>" alt="logo">
        </a>
        <a class="navbar-brand-mini" href="index.php">
          <img src="../<?php echo getSetting('mini_logo', 'assets/images/logo-text-white.svg'); ?>" alt="logo">
        </a>
      </div>
      <nav class="app-navbar" data-simplebar>
        <ul class="menubar">
          <li class="menu-item">
            <a class="menu-link" href="index.php">
              <i class="fi fi-rr-apps"></i>
              <span class="menu-label">Dashboard</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/users.php">
              <i class="fi fi-rr-users"></i>
              <span class="menu-label">Manage Users</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/portfolio.php">
              <i class="fi fi-rr-portfolio"></i>
              <span class="menu-label">Manage Portfolio</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/products.php">
              <i class="fi fi-rr-shopping-cart"></i>
              <span class="menu-label">Manage Products</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/orders.php">
              <i class="fi fi-rr-receipt"></i>
              <span class="menu-label">Manage Orders</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/courses.php">
              <i class="fi fi-rr-book-alt"></i>
              <span class="menu-label">Manage Courses</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/script_requests.php">
              <i class="fi fi-rr-file-code"></i>
              <span class="menu-label">Manage Script Request</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/settings.php">
              <i class="fi fi-rr-settings"></i>
              <span class="menu-label">Settings</span>
            </a>
          </li>
        </ul>
      </nav>
      <div class="app-footer">
        <a href="pages/faq.html" class="btn btn-outline-white waves-effect btn-app-nav w-100">
          <i class="fi fi-rs-interrogation"></i>
          <span class="nav-text">Help and Support</span>
        </a>
      </div>
    </aside>
    <!-- end::Edura Sidebar Menu -->

    <main class="app-wrapper">

      <div class="container-fluid">

        <div class="app-page-head d-flex flex-wrap gap-3 align-items-center justify-content-between">
          <div class="clearfix">
            <h1 class="app-page-title">Settings</h1>
            <nav aria-label="breadcrumb">
              <ol class="breadcrumb mb-0">
                <li class="breadcrumb-item">
                  <a href="index.php">Dashboard</a>
                </li>
                <li class="breadcrumb-item active" aria-current="page">Settings</li>
              </ol>
            </nav>
          </div>
        </div>

        <?php if ($success): ?>
          <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fi fi-rr-check-circle me-2"></i><?php echo htmlspecialchars($success); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
          </div>
        <?php endif; ?>
        
        <?php if ($error): ?>
          <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fi fi-rr-exclamation-circle me-2"></i><?php echo htmlspecialchars($error); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
          </div>
        <?php endif; ?>

        <!-- Settings Tabs -->
        <div class="row">
          <div class="col-12">
            <div class="card">
              <div class="card-body">
                <!-- Tab Navigation -->
                <ul class="nav nav-tabs nav-fill mb-4" id="settingsTabs" role="tablist">
                  <li class="nav-item" role="presentation">
                    <button class="nav-link active" id="general-tab" data-bs-toggle="tab" data-bs-target="#general" type="button" role="tab" aria-controls="general" aria-selected="true">
                      <i class="fi fi-rr-settings me-2"></i>General
                    </button>
                  </li>
                  <li class="nav-item" role="presentation">
                    <button class="nav-link" id="crypto-tab" data-bs-toggle="tab" data-bs-target="#crypto" type="button" role="tab" aria-controls="crypto" aria-selected="false">
                      <i class="fi fi-rr-bitcoin me-2"></i>Crypto
                    </button>
                  </li>
                  <li class="nav-item" role="presentation">
                    <button class="nav-link" id="bank-tab" data-bs-toggle="tab" data-bs-target="#bank" type="button" role="tab" aria-controls="bank" aria-selected="false">
                      <i class="fi fi-rr-bank me-2"></i>Bank
                    </button>
                  </li>
                  <li class="nav-item" role="presentation">
                    <button class="nav-link" id="paystack-tab" data-bs-toggle="tab" data-bs-target="#paystack" type="button" role="tab" aria-controls="paystack" aria-selected="false">
                      <i class="fi fi-rr-credit-card me-2"></i>Paystack
                    </button>
                  </li>
                  <li class="nav-item" role="presentation">
                    <button class="nav-link" id="email-tab" data-bs-toggle="tab" data-bs-target="#email" type="button" role="tab" aria-controls="email" aria-selected="false">
                      <i class="fi fi-rr-envelope me-2"></i>Email
                    </button>
                  </li>
                  <li class="nav-item" role="presentation">
                    <button class="nav-link" id="logo-tab" data-bs-toggle="tab" data-bs-target="#logo" type="button" role="tab" aria-controls="logo" aria-selected="false">
                      <i class="fi fi-rr-picture me-2"></i>Logo
                    </button>
                  </li>
                  <li class="nav-item" role="presentation">
                    <button class="nav-link" id="favicon-tab" data-bs-toggle="tab" data-bs-target="#favicon" type="button" role="tab" aria-controls="favicon" aria-selected="false">
                      <i class="fi fi-rr-star me-2"></i>Favicon
                    </button>
                  </li>
                  <li class="nav-item" role="presentation">
                    <button class="nav-link" id="system-tab" data-bs-toggle="tab" data-bs-target="#system" type="button" role="tab" aria-controls="system" aria-selected="false">
                      <i class="fi fi-rr-gear me-2"></i>System
                    </button>
                  </li>
                </ul>

                <!-- Tab Content -->
                <div class="tab-content" id="settingsTabsContent">
                  <!-- General Settings Tab -->
                  <div class="tab-pane fade show active" id="general" role="tabpanel" aria-labelledby="general-tab">
                    <form method="POST">
                      <input type="hidden" name="action" value="update_settings">
                      <input type="hidden" name="category" value="general">
                      <div class="row">
                        <div class="col-md-6">
                          <div class="mb-3">
                            <label for="site_name" class="form-label">Site Name</label>
                            <input type="text" class="form-control" id="site_name" name="settings[site_name]" value="<?php echo htmlspecialchars($generalSettings['site_name']); ?>">
                          </div>
                        </div>
                        <div class="col-md-6">
                          <div class="mb-3">
                            <label for="site_email" class="form-label">Site Email</label>
                            <input type="email" class="form-control" id="site_email" name="settings[site_email]" value="<?php echo htmlspecialchars($generalSettings['site_email']); ?>">
                          </div>
                        </div>
                      </div>
                      <div class="row">
                        <div class="col-md-6">
                          <div class="mb-3">
                            <label for="site_phone" class="form-label">Phone Number</label>
                            <input type="text" class="form-control" id="site_phone" name="settings[site_phone]" value="<?php echo htmlspecialchars($generalSettings['site_phone']); ?>">
                          </div>
                        </div>
                        <div class="col-md-6">
                          <div class="mb-3">
                            <label for="currency" class="form-label">Currency</label>
                            <select class="form-select" id="currency" name="settings[currency]">
                              <option value="USD" <?php echo $generalSettings['currency'] === 'USD' ? 'selected' : ''; ?>>USD</option>
                              <option value="NGN" <?php echo $generalSettings['currency'] === 'NGN' ? 'selected' : ''; ?>>NGN</option>
                              <option value="EUR" <?php echo $generalSettings['currency'] === 'EUR' ? 'selected' : ''; ?>>EUR</option>
                              <option value="GBP" <?php echo $generalSettings['currency'] === 'GBP' ? 'selected' : ''; ?>>GBP</option>
                            </select>
                          </div>
                        </div>
                      </div>
                      <div class="mb-3">
                        <label for="site_address" class="form-label">Address</label>
                        <textarea class="form-control" id="site_address" name="settings[site_address]" rows="3"><?php echo htmlspecialchars($generalSettings['site_address']); ?></textarea>
                      </div>
                      <div class="mb-3">
                        <label for="timezone" class="form-label">Timezone</label>
                        <select class="form-select" id="timezone" name="settings[timezone]">
                          <option value="Africa/Lagos" <?php echo $generalSettings['timezone'] === 'Africa/Lagos' ? 'selected' : ''; ?>>Africa/Lagos</option>
                          <option value="America/New_York" <?php echo $generalSettings['timezone'] === 'America/New_York' ? 'selected' : ''; ?>>America/New_York</option>
                          <option value="Europe/London" <?php echo $generalSettings['timezone'] === 'Europe/London' ? 'selected' : ''; ?>>Europe/London</option>
                          <option value="Asia/Tokyo" <?php echo $generalSettings['timezone'] === 'Asia/Tokyo' ? 'selected' : ''; ?>>Asia/Tokyo</option>
                        </select>
                      </div>
                      <button type="submit" class="btn btn-primary">Update General Settings</button>
                    </form>
                  </div>

                  <!-- Crypto Settings Tab -->
                  <div class="tab-pane fade" id="crypto" role="tabpanel" aria-labelledby="crypto-tab">
                    <form method="POST">
                      <input type="hidden" name="action" value="update_settings">
                      <input type="hidden" name="category" value="crypto">
                      <div class="row">
                        <div class="col-md-6">
                          <div class="mb-3">
                            <label for="bitcoin_address" class="form-label">Bitcoin Address</label>
                            <input type="text" class="form-control" id="bitcoin_address" name="settings[bitcoin_address]" value="<?php echo htmlspecialchars($cryptoSettings['bitcoin_address']); ?>">
                          </div>
                        </div>
                        <div class="col-md-6">
                          <div class="mb-3">
                            <label for="ethereum_address" class="form-label">Ethereum Address</label>
                            <input type="text" class="form-control" id="ethereum_address" name="settings[ethereum_address]" value="<?php echo htmlspecialchars($cryptoSettings['ethereum_address']); ?>">
                          </div>
                        </div>
                      </div>
                      <div class="row">
                        <div class="col-md-6">
                          <div class="mb-3">
                            <label for="usdt_address" class="form-label">USDT Address</label>
                            <input type="text" class="form-control" id="usdt_address" name="settings[usdt_address]" value="<?php echo htmlspecialchars($cryptoSettings['usdt_address']); ?>">
                          </div>
                        </div>
                        <div class="col-md-6">
                          <div class="mb-3">
                            <label for="usdc_address" class="form-label">USDC Address</label>
                            <input type="text" class="form-control" id="usdc_address" name="settings[usdc_address]" value="<?php echo htmlspecialchars($cryptoSettings['usdc_address']); ?>">
                          </div>
                        </div>
                      </div>
                      <div class="mb-3">
                        <label for="crypto_instructions" class="form-label">Payment Instructions</label>
                        <textarea class="form-control" id="crypto_instructions" name="settings[crypto_instructions]" rows="3"><?php echo htmlspecialchars($cryptoSettings['instructions']); ?></textarea>
                      </div>
                      <button type="submit" class="btn btn-primary">Update Crypto Settings</button>
                    </form>
                  </div>

                  <!-- Bank Settings Tab -->
                  <div class="tab-pane fade" id="bank" role="tabpanel" aria-labelledby="bank-tab">
                    <form method="POST">
                      <input type="hidden" name="action" value="update_settings">
                      <input type="hidden" name="category" value="bank">
                      <div class="row">
                        <div class="col-md-6">
                          <div class="mb-3">
                            <label for="bank_name" class="form-label">Bank Name</label>
                            <input type="text" class="form-control" id="bank_name" name="settings[bank_name]" value="<?php echo htmlspecialchars($bankSettings['bank_name']); ?>">
                          </div>
                        </div>
                        <div class="col-md-6">
                          <div class="mb-3">
                            <label for="account_name" class="form-label">Account Name</label>
                            <input type="text" class="form-control" id="account_name" name="settings[account_name]" value="<?php echo htmlspecialchars($bankSettings['account_name']); ?>">
                          </div>
                        </div>
                      </div>
                      <div class="row">
                        <div class="col-md-6">
                          <div class="mb-3">
                            <label for="account_number" class="form-label">Account Number</label>
                            <input type="text" class="form-control" id="account_number" name="settings[account_number]" value="<?php echo htmlspecialchars($bankSettings['account_number']); ?>">
                          </div>
                        </div>
                        <div class="col-md-6">
                          <div class="mb-3">
                            <label for="bank_code" class="form-label">Bank Code</label>
                            <input type="text" class="form-control" id="bank_code" name="settings[bank_code]" value="<?php echo htmlspecialchars($bankSettings['bank_code']); ?>">
                          </div>
                        </div>
                      </div>
                      <div class="mb-3">
                        <label for="bank_instructions" class="form-label">Payment Instructions</label>
                        <textarea class="form-control" id="bank_instructions" name="settings[bank_instructions]" rows="3"><?php echo htmlspecialchars($bankSettings['instructions']); ?></textarea>
                      </div>
                      <button type="submit" class="btn btn-primary">Update Bank Settings</button>
                    </form>
                  </div>

                  <!-- Paystack Settings Tab -->
                  <div class="tab-pane fade" id="paystack" role="tabpanel" aria-labelledby="paystack-tab">
                    <form method="POST">
                      <input type="hidden" name="action" value="update_settings">
                      <input type="hidden" name="category" value="paystack">
                      <div class="row">
                        <div class="col-md-6">
                          <div class="mb-3">
                            <label for="paystack_public_key" class="form-label">Public Key</label>
                            <input type="text" class="form-control" id="paystack_public_key" name="settings[paystack_public_key]" value="<?php echo htmlspecialchars($paystackSettings['public_key']); ?>">
                          </div>
                        </div>
                        <div class="col-md-6">
                          <div class="mb-3">
                            <label for="paystack_secret_key" class="form-label">Secret Key</label>
                            <input type="password" class="form-control" id="paystack_secret_key" name="settings[paystack_secret_key]" value="<?php echo htmlspecialchars($paystackSettings['secret_key']); ?>">
                          </div>
                        </div>
                      </div>
                      <div class="row">
                        <div class="col-md-6">
                          <div class="mb-3">
                            <label for="paystack_webhook_secret" class="form-label">Webhook Secret</label>
                            <input type="password" class="form-control" id="paystack_webhook_secret" name="settings[paystack_webhook_secret]" value="<?php echo htmlspecialchars($paystackSettings['webhook_secret']); ?>">
                          </div>
                        </div>
                        <div class="col-md-6">
                          <div class="mb-3">
                            <label for="paystack_currency" class="form-label">Currency</label>
                            <select class="form-select" id="paystack_currency" name="settings[paystack_currency]">
                              <option value="NGN" <?php echo $paystackSettings['currency'] === 'NGN' ? 'selected' : ''; ?>>NGN</option>
                              <option value="USD" <?php echo $paystackSettings['currency'] === 'USD' ? 'selected' : ''; ?>>USD</option>
                              <option value="GHS" <?php echo $paystackSettings['currency'] === 'GHS' ? 'selected' : ''; ?>>GHS</option>
                              <option value="ZAR" <?php echo $paystackSettings['currency'] === 'ZAR' ? 'selected' : ''; ?>>ZAR</option>
                            </select>
                          </div>
                        </div>
                      </div>
                      <div class="mb-3">
                        <label for="paystack_instructions" class="form-label">Payment Instructions</label>
                        <textarea class="form-control" id="paystack_instructions" name="settings[paystack_instructions]" rows="3"><?php echo htmlspecialchars($paystackSettings['instructions']); ?></textarea>
                      </div>
                      <div class="d-flex gap-2">
                        <button type="submit" class="btn btn-primary">Update Paystack Settings</button>
                        <button type="button" class="btn btn-outline-info" onclick="testPaystackConnection()">
                          <i class="fi fi-rr-check me-1"></i>Test Connection
                        </button>
                      </div>
                    </form>
                    
                    <!-- Test Results -->
                    <div id="paystack-test-result" class="mt-3" style="display: none;">
                      <div class="alert" role="alert">
                        <div id="test-message"></div>
                      </div>
                    </div>
                  </div>

                  <!-- Email Settings Tab -->
                  <div class="tab-pane fade" id="email" role="tabpanel" aria-labelledby="email-tab">
                    <form method="POST">
                      <input type="hidden" name="action" value="update_settings">
                      <input type="hidden" name="category" value="email">
                      <div class="row">
                        <div class="col-md-6">
                          <div class="mb-3">
                            <label for="smtp_host" class="form-label">SMTP Host</label>
                            <input type="text" class="form-control" id="smtp_host" name="settings[smtp_host]" value="<?php echo htmlspecialchars($emailSettings['smtp_host'] ?? ''); ?>">
                          </div>
                        </div>
                        <div class="col-md-6">
                          <div class="mb-3">
                            <label for="smtp_port" class="form-label">SMTP Port</label>
                            <input type="number" class="form-control" id="smtp_port" name="settings[smtp_port]" value="<?php echo htmlspecialchars($emailSettings['smtp_port'] ?? ''); ?>">
                          </div>
                        </div>
                      </div>
                      <div class="row">
                        <div class="col-md-6">
                          <div class="mb-3">
                            <label for="smtp_username" class="form-label">SMTP Username</label>
                            <input type="email" class="form-control" id="smtp_username" name="settings[smtp_username]" value="<?php echo htmlspecialchars($emailSettings['smtp_username'] ?? ''); ?>">
                          </div>
                        </div>
                        <div class="col-md-6">
                          <div class="mb-3">
                            <label for="smtp_password" class="form-label">SMTP Password</label>
                            <input type="password" class="form-control" id="smtp_password" name="settings[smtp_password]" value="<?php echo htmlspecialchars($emailSettings['smtp_password'] ?? ''); ?>">
                          </div>
                        </div>
                      </div>
                      <div class="mb-3">
                        <label for="smtp_encryption" class="form-label">Encryption</label>
                        <select class="form-select" id="smtp_encryption" name="settings[smtp_encryption]">
                          <option value="tls" <?php echo ($emailSettings['smtp_encryption'] ?? '') === 'tls' ? 'selected' : ''; ?>>TLS</option>
                          <option value="ssl" <?php echo ($emailSettings['smtp_encryption'] ?? '') === 'ssl' ? 'selected' : ''; ?>>SSL</option>
                          <option value="none" <?php echo ($emailSettings['smtp_encryption'] ?? '') === 'none' ? 'selected' : ''; ?>>None</option>
                        </select>
                      </div>
                      <button type="submit" class="btn btn-primary">Update Email Settings</button>
                    </form>
                  </div>

                  <!-- Logo Settings Tab -->
                  <div class="tab-pane fade" id="logo" role="tabpanel" aria-labelledby="logo-tab">
                    <form method="POST" enctype="multipart/form-data">
                      <input type="hidden" name="action" value="update_logo">
                      <div class="row">
                        <div class="col-md-6">
                          <div class="mb-3">
                            <label for="main_logo" class="form-label">Main Logo (SVG/PNG)</label>
                            <input type="file" class="form-control" id="main_logo" name="main_logo" accept=".svg,.png,.jpg,.jpeg">
                            <div class="form-text">Recommended size: 180x50px or similar aspect ratio. SVG preferred for scalability.</div>
                          </div>
                          <div class="mb-3">
                            <label for="mini_logo" class="form-label">Mini Logo (SVG/PNG)</label>
                            <input type="file" class="form-control" id="mini_logo" name="mini_logo" accept=".svg,.png,.jpg,.jpeg">
                            <div class="form-text">Recommended size: 35x35px or square aspect ratio. Used in collapsed sidebar.</div>
                          </div>
                        </div>
                        <div class="col-md-6">
                          <div class="mb-3">
                            <label class="form-label">Current Logos</label>
                            <div class="border rounded p-3 text-center">
                              <div class="mb-3">
                                <strong>Main Logo:</strong><br>
                                <img id="current_main_logo" src="../<?php echo getSetting('main_logo', 'assets/images/logo.svg'); ?>" alt="Main Logo" style="max-height: 60px; max-width: 200px;" class="img-fluid">
                              </div>
                              <div>
                                <strong>Mini Logo:</strong><br>
                                <img id="current_mini_logo" src="../<?php echo getSetting('mini_logo', 'assets/images/logo-text-white.svg'); ?>" alt="Mini Logo" style="max-height: 40px; max-width: 40px;" class="img-fluid">
                              </div>
                            </div>
                          </div>
                        </div>
                      </div>
                      <div class="mb-3">
                        <div class="form-check">
                          <input class="form-check-input" type="checkbox" id="reset_logos" name="reset_logos" value="1">
                          <label class="form-check-label" for="reset_logos">
                            Reset to default logos
                          </label>
                        </div>
                      </div>
                      <button type="submit" class="btn btn-primary">Update Logo Settings</button>
                    </form>
                  </div>

                  <!-- Favicon Settings Tab -->
                  <div class="tab-pane fade" id="favicon" role="tabpanel" aria-labelledby="favicon-tab">
                    <form method="POST" enctype="multipart/form-data">
                      <input type="hidden" name="action" value="update_favicon">
                      <div class="row">
                        <div class="col-md-6">
                          <div class="mb-3">
                            <label for="favicon" class="form-label">Favicon (ICO/PNG/SVG)</label>
                            <input type="file" class="form-control" id="favicon" name="favicon" accept=".ico,.png,.jpg,.jpeg,.svg">
                            <div class="form-text">Recommended formats: ICO (16x16, 32x32), PNG (32x32), or SVG. ICO is preferred for best browser compatibility.</div>
                          </div>
                          <div class="mb-3">
                            <div class="form-check">
                              <input class="form-check-input" type="checkbox" id="reset_favicon" name="reset_favicon" value="1">
                              <label class="form-check-label" for="reset_favicon">
                                Reset to default favicon
                              </label>
                            </div>
                          </div>
                        </div>
                        <div class="col-md-6">
                          <div class="mb-3">
                            <label class="form-label">Current Favicon</label>
                            <div class="border rounded p-3 text-center">
                              <div class="mb-3">
                                <strong>Favicon Preview:</strong><br>
                                <img id="current_favicon" src="../<?php echo getSetting('favicon', 'assets/images/favicon.png'); ?>" alt="Favicon" style="width: 32px; height: 32px;" class="img-fluid">
                              </div>
                              <div class="small text-muted">
                                <strong>Current Path:</strong><br>
                                <code><?php echo getSetting('favicon', 'assets/images/favicon.png'); ?></code>
                              </div>
                            </div>
                          </div>
                        </div>
                      </div>
                      <div class="alert alert-info">
                        <i class="fi fi-rr-info me-2"></i>
                        <strong>Note:</strong> The favicon will be updated across all pages (frontend, user dashboard, and admin dashboard) after saving.
                      </div>
                      <button type="submit" class="btn btn-primary">Update Favicon</button>
                    </form>
                  </div>

                  <!-- System Settings Tab -->
                  <div class="tab-pane fade" id="system" role="tabpanel" aria-labelledby="system-tab">
                    <form method="POST">
                      <input type="hidden" name="action" value="update_settings">
                      <input type="hidden" name="category" value="system">
                      <div class="row">
                        <div class="col-md-6">
                          <div class="mb-3">
                            <div class="form-check form-switch">
                              <input class="form-check-input" type="checkbox" id="maintenance_mode" name="settings[maintenance_mode]" value="1" <?php echo $systemSettings['maintenance_mode'] ? 'checked' : ''; ?>>
                              <label class="form-check-label" for="maintenance_mode">Maintenance Mode</label>
                            </div>
                          </div>
                        </div>
                        <div class="col-md-6">
                          <div class="mb-3">
                            <div class="form-check form-switch">
                              <input class="form-check-input" type="checkbox" id="registration_enabled" name="settings[registration_enabled]" value="1" <?php echo $systemSettings['registration_enabled'] ? 'checked' : ''; ?>>
                              <label class="form-check-label" for="registration_enabled">Allow Registration</label>
                            </div>
                          </div>
                        </div>
                      </div>
                      <div class="row">
                        <div class="col-md-6">
                          <div class="mb-3">
                            <div class="form-check form-switch">
                              <input class="form-check-input" type="checkbox" id="email_verification" name="settings[email_verification]" value="1" <?php echo $systemSettings['email_verification'] ? 'checked' : ''; ?>>
                              <label class="form-check-label" for="email_verification">Email Verification Required</label>
                            </div>
                          </div>
                        </div>
                        <div class="col-md-6">
                          <div class="mb-3">
                            <label for="max_downloads" class="form-label">Max Downloads per Product</label>
                            <input type="number" class="form-control" id="max_downloads" name="settings[max_downloads]" value="<?php echo htmlspecialchars($systemSettings['max_downloads']); ?>" min="1">
                          </div>
                        </div>
                      </div>
                      <div class="mb-3">
                        <label for="download_expiry_days" class="form-label">Download Expiry (Days)</label>
                        <input type="number" class="form-control" id="download_expiry_days" name="settings[download_expiry_days]" value="<?php echo htmlspecialchars($systemSettings['download_expiry_days']); ?>" min="1">
                      </div>
                      <button type="submit" class="btn btn-primary">Update System Settings</button>
                    </form>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>

      </div>

    </main>

    <!-- begin::Edura Footer -->
    <footer class="footer-wrapper bg-body text-center">
      <div class="container-fluid">
        <p class="mb-0">© <span class="currentYear">2025</span> Edura. Proudly powered by <a href="javascript:void(0);">LayoutDrop</a>.</p>
      </div>
    </footer>
    <!-- end::Edura Footer -->

  </div>

  <!-- begin::Edura Page Scripts -->
  <script src="assets/libs/global/global.min.js"></script>
  <script src="assets/js/appSettings.js"></script>
  <script src="assets/js/main.js"></script>
  <!-- end::Edura Page Scripts -->
  
  <script>
    function testPaystackConnection() {
      const publicKey = document.getElementById('paystack_public_key').value;
      const secretKey = document.getElementById('paystack_secret_key').value;
      const testResult = document.getElementById('paystack-test-result');
      const testMessage = document.getElementById('test-message');
      
      if (!publicKey || !secretKey) {
        showTestResult('Please enter both Public Key and Secret Key before testing.', 'warning');
        return;
      }
      
      // Show loading state
      showTestResult('Testing Paystack connection...', 'info');
      
      // Test the connection
      fetch('test_paystack_connection.php', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          public_key: publicKey,
          secret_key: secretKey
        })
      })
      .then(response => response.json())
      .then(data => {
        if (data.success) {
          showTestResult('✅ Paystack connection successful! API keys are valid.', 'success');
        } else {
          showTestResult('❌ Paystack connection failed: ' + data.message, 'danger');
        }
      })
      .catch(error => {
        showTestResult('❌ Error testing connection: ' + error.message, 'danger');
      });
    }
    
    function showTestResult(message, type) {
      const testResult = document.getElementById('paystack-test-result');
      const testMessage = document.getElementById('test-message');
      const alert = testResult.querySelector('.alert');
      
      testMessage.textContent = message;
      alert.className = `alert alert-${type}`;
      testResult.style.display = 'block';
      
      // Scroll to result
      testResult.scrollIntoView({ behavior: 'smooth' });
    }
  </script>

  </body>

</html>
