<?php
session_start();

// Check if user is admin
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: authentication/login.html');
    exit;
}

// Include database connection
require_once '../users/config/database.php';
require_once 'config/settings.php';
require_once 'config/branding.php';

// Get all lessons with their courses
try {
    $stmt = $pdo->prepare("
        SELECT cl.id, cl.lesson_title, cl.video_url, cl.video_file, cl.video_type, cl.duration,
               c.title as course_title, c.id as course_id
        FROM course_lessons cl 
        JOIN courses c ON cl.course_id = c.id 
        ORDER BY c.title, cl.lesson_number
    ");
    $stmt->execute();
    $lessons = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $lessons = [];
    $error = "Database error: " . $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Upload Lesson Videos - Admin Dashboard</title>
    <!-- Favicon -->
    <?php echo getFaviconHtml(); ?>
    <link href="assets/css/bootstrap.min.css" rel="stylesheet">
    <link href="assets/css/style.css" rel="stylesheet">
    <link href="assets/css/flaticon.css" rel="stylesheet">
    <style>
        .video-upload-area {
            border: 2px dashed #dee2e6;
            border-radius: 8px;
            padding: 40px;
            text-align: center;
            transition: all 0.3s ease;
            cursor: pointer;
        }
        .video-upload-area:hover {
            border-color: #007bff;
            background-color: #f8f9fa;
        }
        .video-upload-area.dragover {
            border-color: #007bff;
            background-color: #e3f2fd;
        }
        .upload-progress {
            display: none;
            margin-top: 15px;
        }
        .lesson-card {
            transition: all 0.3s ease;
        }
        .lesson-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
        }
        .video-status {
            display: inline-block;
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 12px;
            font-weight: 500;
        }
        .status-youtube {
            background-color: #ff4444;
            color: white;
        }
        .status-uploaded {
            background-color: #28a745;
            color: white;
        }
    </style>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="main-content">
        <div class="page-content">
            <div class="container-fluid">
                <div class="row">
                    <div class="col-12">
                        <div class="page-title-box d-sm-flex align-items-center justify-content-between">
                            <h4 class="mb-sm-0">Upload Lesson Videos</h4>
                            <div class="page-title-right">
                                <ol class="breadcrumb m-0">
                                    <li class="breadcrumb-item"><a href="index.php">Dashboard</a></li>
                                    <li class="breadcrumb-item active">Upload Videos</li>
                                </ol>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="row">
                    <div class="col-12">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="card-title mb-0">
                                    <i class="fi fi-rr-video me-2"></i>
                                    Lesson Videos Management
                                </h5>
                            </div>
                            <div class="card-body">
                                <?php if (isset($error)): ?>
                                    <div class="alert alert-danger">
                                        <i class="fi fi-rr-exclamation-triangle me-2"></i>
                                        <?php echo htmlspecialchars($error); ?>
                                    </div>
                                <?php endif; ?>

                                <div class="row">
                                    <?php foreach ($lessons as $lesson): ?>
                                        <div class="col-md-6 col-lg-4 mb-4">
                                            <div class="card lesson-card h-100">
                                                <div class="card-body">
                                                    <div class="d-flex justify-content-between align-items-start mb-3">
                                                        <h6 class="card-title mb-0"><?php echo htmlspecialchars($lesson['lesson_title']); ?></h6>
                                                        <span class="video-status status-<?php echo $lesson['video_type']; ?>">
                                                            <?php echo ucfirst($lesson['video_type']); ?>
                                                        </span>
                                                    </div>
                                                    <p class="text-muted small mb-3">
                                                        <strong>Course:</strong> <?php echo htmlspecialchars($lesson['course_title']); ?><br>
                                                        <strong>Duration:</strong> <?php echo htmlspecialchars($lesson['duration']); ?>
                                                    </p>
                                                    
                                                    <?php if ($lesson['video_type'] === 'upload' && $lesson['video_file']): ?>
                                                        <div class="alert alert-success small mb-3">
                                                            <i class="fi fi-rr-check me-1"></i>
                                                            Video uploaded: <?php echo htmlspecialchars($lesson['video_file']); ?>
                                                        </div>
                                                    <?php else: ?>
                                                        <div class="alert alert-warning small mb-3">
                                                            <i class="fi fi-rr-exclamation me-1"></i>
                                                            Using YouTube video
                                                        </div>
                                                    <?php endif; ?>

                                                    <div class="video-upload-area" data-lesson-id="<?php echo $lesson['id']; ?>">
                                                        <i class="fi fi-rr-cloud-upload" style="font-size: 2rem; color: #6c757d;"></i>
                                                        <h6 class="mt-2 mb-1">Upload Video</h6>
                                                        <p class="text-muted small mb-0">
                                                            Click or drag & drop video file<br>
                                                            <small>MP4, WebM, OGG, AVI, MOV (Max 500MB)</small>
                                                        </p>
                                                        <input type="file" class="d-none" accept="video/*" data-lesson-id="<?php echo $lesson['id']; ?>">
                                                    </div>
                                                    
                                                    <div class="upload-progress" data-lesson-id="<?php echo $lesson['id']; ?>">
                                                        <div class="progress">
                                                            <div class="progress-bar" role="progressbar" style="width: 0%"></div>
                                                        </div>
                                                        <small class="text-muted mt-1">Uploading...</small>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="assets/libs/jquery/jquery.min.js"></script>
    <script src="assets/libs/bootstrap/js/bootstrap.bundle.min.js"></script>
    
    <script>
        $(document).ready(function() {
            // Handle file input change
            $('.video-upload-area').on('click', function() {
                $(this).find('input[type="file"]').click();
            });
            
            // Handle drag and drop
            $('.video-upload-area').on('dragover', function(e) {
                e.preventDefault();
                $(this).addClass('dragover');
            });
            
            $('.video-upload-area').on('dragleave', function(e) {
                e.preventDefault();
                $(this).removeClass('dragover');
            });
            
            $('.video-upload-area').on('drop', function(e) {
                e.preventDefault();
                $(this).removeClass('dragover');
                
                const files = e.originalEvent.dataTransfer.files;
                if (files.length > 0) {
                    const lessonId = $(this).data('lesson-id');
                    uploadVideo(lessonId, files[0]);
                }
            });
            
            // Handle file selection
            $('input[type="file"]').on('change', function() {
                const lessonId = $(this).data('lesson-id');
                const file = this.files[0];
                if (file) {
                    uploadVideo(lessonId, file);
                }
            });
            
            function uploadVideo(lessonId, file) {
                const formData = new FormData();
                formData.append('video', file);
                formData.append('lesson_id', lessonId);
                
                const uploadArea = $(`.video-upload-area[data-lesson-id="${lessonId}"]`);
                const progressContainer = $(`.upload-progress[data-lesson-id="${lessonId}"]`);
                const progressBar = progressContainer.find('.progress-bar');
                
                // Show progress
                uploadArea.hide();
                progressContainer.show();
                
                $.ajax({
                    url: '../users/upload_video.php',
                    type: 'POST',
                    data: formData,
                    processData: false,
                    contentType: false,
                    xhr: function() {
                        const xhr = new window.XMLHttpRequest();
                        xhr.upload.addEventListener('progress', function(evt) {
                            if (evt.lengthComputable) {
                                const percentComplete = (evt.loaded / evt.total) * 100;
                                progressBar.css('width', percentComplete + '%');
                                progressBar.attr('aria-valuenow', percentComplete);
                            }
                        }, false);
                        return xhr;
                    },
                    success: function(response) {
                        try {
                            const data = JSON.parse(response);
                            if (data.success) {
                                // Reload page to show updated status
                                location.reload();
                            } else {
                                alert('Upload failed: ' + data.message);
                                progressContainer.hide();
                                uploadArea.show();
                            }
                        } catch (e) {
                            alert('Upload failed: Invalid response');
                            progressContainer.hide();
                            uploadArea.show();
                        }
                    },
                    error: function() {
                        alert('Upload failed: Network error');
                        progressContainer.hide();
                        uploadArea.show();
                    }
                });
            }
        });
    </script>
</body>
</html>
