<?php
require_once 'config/database.php';
require_once 'config/settings.php';
require_once 'config/branding.php';

$error = '';
$success = '';

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: authentication/login.php');
    exit();
}

// Handle user actions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $action = $_POST['action'] ?? '';
    $user_id = $_POST['user_id'] ?? '';
    
    try {
        $pdo = getConnection();
        
        switch ($action) {
            case 'suspend':
                $stmt = $pdo->prepare("UPDATE users SET status = 'suspended', updated_at = NOW() WHERE id = ?");
                $stmt->execute([$user_id]);
                $success = "User suspended successfully";
                break;
                
            case 'activate':
                $stmt = $pdo->prepare("UPDATE users SET status = 'active', updated_at = NOW() WHERE id = ?");
                $stmt->execute([$user_id]);
                $success = "User activated successfully";
                break;
                
            case 'delete':
                $stmt = $pdo->prepare("DELETE FROM users WHERE id = ?");
                $stmt->execute([$user_id]);
                $success = "User deleted successfully";
                break;
        }
    } catch (Exception $e) {
        $error = "Action failed: " . $e->getMessage();
    }
}

// Get users with pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 10;
$offset = ($page - 1) * $limit;

try {
    $pdo = getConnection();
    
    // Get total count
    $countStmt = $pdo->query("SELECT COUNT(*) FROM users");
    $totalUsers = $countStmt->fetchColumn();
    $totalPages = ceil($totalUsers / $limit);
    
    // Get users
    $stmt = $pdo->prepare("
        SELECT id, username, email, full_name, phone, status, created_at, updated_at, last_login
        FROM users 
        ORDER BY created_at DESC 
        LIMIT :limit OFFSET :offset
    ");
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $users = $stmt->fetchAll();
    
    // Get user statistics
    $activeUsers = $pdo->query("SELECT COUNT(*) FROM users WHERE status = 'active'")->fetchColumn();
    $suspendedUsers = $pdo->query("SELECT COUNT(*) FROM users WHERE status = 'suspended'")->fetchColumn();
    $inactiveUsers = $pdo->query("SELECT COUNT(*) FROM users WHERE status = 'inactive'")->fetchColumn();
    
} catch (Exception $e) {
    $error = "Database error: " . $e->getMessage();
    $users = [];
    $totalPages = 0;
    $totalUsers = 0;
    $activeUsers = 0;
    $suspendedUsers = 0;
    $inactiveUsers = 0;
}
?>
<!DOCTYPE html>
<html lang="en">

<head>

  <base href="">

  <!-- begin::Edura Meta Basic -->
  <meta charset="utf-8">
  <meta name="theme-color" content="#0D9488">
  <meta name="robots" content="index, follow">
  <meta name="author" content="Omateq Digitals">
  <meta name="format-detection" content="telephone=no">
  <meta name="keywords" content="Admin Dashboard, Omateq Digitals, User Management, Admin Panel, Dashboard Control">
  <meta name="description" content="Omateq Digitals Admin Dashboard - Manage users, suspend accounts, and control user access.">
  <!-- end::Edura Meta Basic -->

  <!-- begin::Edura Meta Social -->
  <meta property="og:url" content="">
  <meta property="og:site_name" content="Manage Users | Omateq Digitals">
  <meta property="og:type" content="website">
  <meta property="og:locale" content="en_us">
  <meta property="og:title" content="Manage Users | Omateq Digitals">
  <meta property="og:description" content="Omateq Digitals Admin Dashboard - Manage users, suspend accounts, and control user access.">
  <!-- end::Edura Meta Social -->

  <!-- begin::Edura Meta Twitter -->
  <meta name="twitter:card" content="summary">
  <meta name="twitter:url" content="">
  <meta name="twitter:creator" content="@omateqdigitals">
  <meta name="twitter:title" content="Manage Users | Omateq Digitals">
  <meta name="twitter:description" content="Omateq Digitals Admin Dashboard - Manage users, suspend accounts, and control user access.">
  <!-- end::Edura Meta Twitter -->

  <!-- begin::Edura Website Page Title -->
  <title>Manage Users | Omateq Digitals</title>
  <!-- end::Edura Website Page Title -->

  <!-- begin::Edura Mobile Specific -->
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <!-- end::Edura Mobile Specific -->

  <!-- begin::Edura Favicon Tags -->
  <?php echo getFaviconHtml(); ?>
  <!-- end::Edura Favicon Tags -->

  <!-- begin::Edura Google Fonts -->
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Plus+Jakarta+Sans:ital,wght@0,200..800;1,200..800&display=swap" rel="stylesheet">
  <!-- end::Edura Google Fonts -->

  <!-- begin::Edura Required Stylesheet -->
  <link rel="stylesheet" href="assets/libs/flaticon/css/all/all.css">
  <link rel="stylesheet" href="assets/libs/lucide/lucide.css">
  <link rel="stylesheet" href="assets/libs/fontawesome/css/all.min.css">
  <link rel="stylesheet" href="assets/libs/simplebar/simplebar.css">
  <link rel="stylesheet" href="assets/libs/node-waves/waves.css">
  <link rel="stylesheet" href="assets/libs/bootstrap-select/css/bootstrap-select.min.css">
  <!-- end::Edura Required Stylesheet -->

  <!-- begin::Edura CSS Stylesheet -->
  <link rel="stylesheet" href="assets/css/styles.css">
  <!-- end::Edura CSS Stylesheet -->

  <!-- begin::Edura Googletagmanager -->
  <script async src="https://www.googletagmanager.com/gtag/js?id=G-XWVQM68HHQ"></script>
  <script>
		window.dataLayer = window.dataLayer || [];
		function gtag(){dataLayer.push(arguments);}
		gtag('js', new Date());

		gtag('config', 'G-XWVQM68HHQ', {
			'cookie_flags': 'SameSite=None;Secure',
			'send_page_view': true
		});
	</script>
  <!-- end::Edura Googletagmanager -->

</head>

<body>
  <div class="page-layout">

    <?php include 'includes/header.php'; ?>

    <!-- begin::Edura Sidebar Menu -->
    <aside class="app-menubar" id="menubar">
      <div class="app-navbar-brand">
        <a class="navbar-brand-logo" href="index.php">
          <img src="../<?php echo getSetting('main_logo', 'assets/images/logo.svg'); ?>" alt="logo">
        </a>
        <a class="navbar-brand-mini" href="index.php">
          <img src="../<?php echo getSetting('mini_logo', 'assets/images/logo-text-white.svg'); ?>" alt="logo">
        </a>
      </div>
      <nav class="app-navbar" data-simplebar>
        <ul class="menubar">
          <li class="menu-item">
            <a class="menu-link" href="index.php">
              <i class="fi fi-rr-apps"></i>
                  <span class="menu-label">Dashboard</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/users.php">
              <i class="fi fi-rr-users"></i>
              <span class="menu-label">Manage Users</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/portfolio.php">
              <i class="fi fi-rr-portfolio"></i>
              <span class="menu-label">Manage Portfolio</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/products.php">
              <i class="fi fi-rr-shopping-cart"></i>
              <span class="menu-label">Manage Products</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/orders.php">
              <i class="fi fi-rr-receipt"></i>
              <span class="menu-label">Manage Orders</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/courses.php">
              <i class="fi fi-rr-book-alt"></i>
              <span class="menu-label">Manage Courses</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/script_requests.php">
              <i class="fi fi-rr-file-code"></i>
              <span class="menu-label">Manage Script Request</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="../admin-dashboard/settings.php">
              <i class="fi fi-rr-settings"></i>
              <span class="menu-label">Settings</span>
            </a>
          </li>
        </ul>
      </nav>
      <div class="app-footer">
        <a href="pages/faq.html" class="btn btn-outline-white waves-effect btn-app-nav w-100">
          <i class="fi fi-rs-interrogation"></i>
          <span class="nav-text">Help and Support</span>
        </a>
      </div>
    </aside>
    <!-- end::Edura Sidebar Menu -->

    <main class="app-wrapper">

      <div class="container-fluid">

        <div class="app-page-head d-flex flex-wrap gap-3 align-items-center justify-content-between">
          <div class="clearfix">
            <h1 class="app-page-title">Manage Users</h1>
            <nav aria-label="breadcrumb">
              <ol class="breadcrumb mb-0">
                <li class="breadcrumb-item">
                  <a href="index.php">Dashboard</a>
                </li>
                <li class="breadcrumb-item active" aria-current="page">Manage Users</li>
              </ol>
            </nav>
          </div>
        </div>

        <?php if ($success): ?>
          <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fi fi-rr-check-circle me-2"></i><?php echo htmlspecialchars($success); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
          </div>
        <?php endif; ?>
        
        <?php if ($error): ?>
          <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fi fi-rr-exclamation-circle me-2"></i><?php echo htmlspecialchars($error); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
          </div>
        <?php endif; ?>

        <div class="row">

          <div class="col-lg-12">
            <div class="card overflow-hidden">
              <div class="card-header d-flex align-items-center justify-content-between border-0">
                <h6 class="card-title mb-0">Users List (<?php echo $totalUsers; ?> total users)</h6>
                <div class="mail-header">
                  <form class="d-flex align-items-center h-100 w-200px w-md-250px position-relative me-auto" action="users.php" method="GET">
                    <button type="button" class="btn btn-sm border-0 position-absolute start-0 ms-3 p-0">
                      <i class="fi fi-rr-search"></i>
                    </button>
                    <input type="text" class="form-control ps-5" placeholder="Search users..." name="search">
                  </form>
                </div>
              </div>
              <div class="card-body p-0">
                <div class="table table-responsive mb-0">
                  <table class="table mb-0">
                    <thead class="table-light">
                      <tr>
                        <th class="minw-100px">ID</th>
                        <th class="minw-150px">User</th>
                        <th class="minw-200px">Email</th>
                        <th class="minw-120px">Phone</th>
                        <th class="minw-100px">Status</th>
                        <th class="minw-150px">Last Login</th>
                        <th class="minw-150px">Created</th>
                        <th class="minw-200px">Actions</th>
                      </tr>
                    </thead>
                    <tbody>
                      <?php if (empty($users)): ?>
                        <tr>
                          <td colspan="8" class="text-center py-4">
                            <div class="d-flex flex-column align-items-center">
                              <i class="fi fi-rr-users text-muted mb-2" style="font-size: 2rem;"></i>
                              <p class="text-muted mb-0">No users found</p>
                          </div>
                        </td>
                      </tr>
                      <?php else: ?>
                        <?php foreach ($users as $user): ?>
                      <tr>
                        <td>
                              <span class="text-primary">#<?php echo $user['id']; ?></span>
                        </td>
                        <td>
                          <div class="d-flex align-items-center mw-175px">
                                <div class="avatar avatar-xxs rounded-circle bg-primary text-white d-flex align-items-center justify-content-center">
                                  <?php echo strtoupper(substr($user['full_name'] ?: $user['username'], 0, 1)); ?>
                            </div>
                                <div class="ms-2 me-auto">
                                  <div class="fw-semibold"><?php echo htmlspecialchars($user['full_name'] ?: $user['username']); ?></div>
                                  <small class="text-muted">@<?php echo htmlspecialchars($user['username']); ?></small>
                            </div>
                          </div>
                        </td>
                            <td><?php echo htmlspecialchars($user['email']); ?></td>
                            <td><?php echo htmlspecialchars($user['phone'] ?: 'N/A'); ?></td>
                            <td>
                              <?php
                              $statusClass = '';
                              switch ($user['status']) {
                                case 'active':
                                  $statusClass = 'bg-success-subtle text-success';
                                  break;
                                case 'suspended':
                                  $statusClass = 'bg-warning-subtle text-warning';
                                  break;
                                case 'inactive':
                                  $statusClass = 'bg-danger-subtle text-danger';
                                  break;
                                default:
                                  $statusClass = 'bg-secondary-subtle text-secondary';
                              }
                              ?>
                              <span class="badge <?php echo $statusClass; ?>"><?php echo ucfirst($user['status']); ?></span>
                        </td>
                        <td>
                              <?php 
                              if ($user['last_login']) {
                                echo date('M d, Y H:i', strtotime($user['last_login']));
                              } else {
                                echo 'Never';
                              }
                              ?>
                        </td>
                            <td><?php echo date('M d, Y', strtotime($user['created_at'])); ?></td>
                            <td>
                              <div class="d-flex gap-1 align-items-center">
                                <?php if ($user['status'] === 'active'): ?>
                                  <form method="POST" style="display: inline;">
                                    <input type="hidden" name="action" value="suspend">
                                    <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                    <button type="submit" class="btn btn-sm btn-warning" onclick="return confirm('Are you sure you want to suspend this user?')">
                                      <i class="fi fi-rr-pause"></i> Suspend
                            </button>
                                  </form>
                                <?php else: ?>
                                  <form method="POST" style="display: inline;">
                                    <input type="hidden" name="action" value="activate">
                                    <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                    <button type="submit" class="btn btn-sm btn-success" onclick="return confirm('Are you sure you want to activate this user?')">
                                      <i class="fi fi-rr-play"></i> Activate
                              </button>
                                  </form>
                                <?php endif; ?>
                                
                                <form method="POST" style="display: inline;">
                                  <input type="hidden" name="action" value="delete">
                                  <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                  <button type="submit" class="btn btn-sm btn-danger" onclick="return confirm('Are you sure you want to delete this user? This action cannot be undone.')">
                                    <i class="fi fi-rr-trash"></i> Delete
                            </button>
                                </form>
                          </div>
                        </td>
                      </tr>
                        <?php endforeach; ?>
                      <?php endif; ?>
                    </tbody>
                  </table>
                </div>
              </div>
              <div class="card-footer border-0 p-3">
                <nav aria-label="Page navigation example">
                  <ul class="pagination pagination-sm mb-0">
                    <?php if ($page > 1): ?>
                    <li class="page-item">
                        <a class="page-link" href="?page=<?php echo $page - 1; ?>" aria-label="Previous">
                        <i class="fi fi-rr-angle-double-left"></i>
                      </a>
                    </li>
                    <?php endif; ?>
                    
                    <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                      <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                        <a class="page-link" href="?page=<?php echo $i; ?>"><?php echo $i; ?></a>
                    </li>
                    <?php endfor; ?>
                    
                    <?php if ($page < $totalPages): ?>
                    <li class="page-item">
                        <a class="page-link" href="?page=<?php echo $page + 1; ?>" aria-label="Next">
                        <i class="fi fi-rr-angle-double-right"></i>
                      </a>
                    </li>
                    <?php endif; ?>
                  </ul>
                </nav>
              </div>
            </div>
          </div>

        </div>

      </div>

    </main>

    <!-- begin::Edura Footer -->
    <footer class="footer-wrapper bg-body text-center">
      <div class="container-fluid">
        <p class="mb-0">© <span class="currentYear">2025</span> Edura. Proudly powered by <a href="javascript:void(0);">LayoutDrop</a>.</p>
      </div>
    </footer>
    <!-- end::Edura Footer -->

  </div>

  <!-- begin::Edura Page Scripts -->
  <script src="assets/libs/global/global.min.js"></script>
  <script src="assets/js/appSettings.js"></script>
  <script src="assets/js/main.js"></script>
  <!-- end::Edura Page Scripts -->
</body>

</html>