#!/bin/bash

# Omateq Digitals Deployment Script
# This script helps prepare the project for production deployment

echo "🚀 Omateq Digitals Deployment Preparation"
echo "========================================"

# Check if we're in the right directory
if [ ! -f "index.php" ]; then
    echo "❌ Error: Please run this script from the project root directory"
    exit 1
fi

echo "✅ Project root directory confirmed"

# Create backup of current files
echo "📦 Creating backup of current files..."
if [ ! -d "backup" ]; then
    mkdir backup
fi

tar -czf "backup/omateqdigitals_backup_$(date +%Y%m%d_%H%M%S).tar.gz" \
    --exclude="backup" \
    --exclude=".git" \
    --exclude="node_modules" \
    --exclude="*.log" \
    .

echo "✅ Backup created in backup/ directory"

# Check for sensitive files
echo "🔍 Checking for sensitive files..."

SENSITIVE_FILES=(
    "users/config/database.php"
    "admin-dashboard/config/database.php"
    ".env"
    "config.php"
)

for file in "${SENSITIVE_FILES[@]}"; do
    if [ -f "$file" ]; then
        echo "⚠️  Found sensitive file: $file"
        echo "   Make sure to update this with production credentials"
    fi
done

# Check file permissions
echo "🔐 Checking file permissions..."

# Set proper permissions for directories
find . -type d -exec chmod 755 {} \;

# Set proper permissions for files
find . -type f -exec chmod 644 {} \;

# Set special permissions for upload directories
if [ -d "uploads" ]; then
    chmod 755 uploads/
    echo "✅ Set uploads directory permissions"
fi

if [ -d "uploads/course_videos" ]; then
    chmod 755 uploads/course_videos/
    echo "✅ Set course_videos directory permissions"
fi

if [ -d "uploads/logos" ]; then
    chmod 755 uploads/logos/
    echo "✅ Set logos directory permissions"
fi

# Check for required PHP extensions
echo "🔧 Checking PHP requirements..."

REQUIRED_EXTENSIONS=("pdo" "curl" "gd" "openssl" "mbstring" "json")

for ext in "${REQUIRED_EXTENSIONS[@]}"; do
    if php -m | grep -q "$ext"; then
        echo "✅ PHP extension $ext is available"
    else
        echo "❌ PHP extension $ext is missing"
    fi
done

# Check PHP version
PHP_VERSION=$(php -r "echo PHP_VERSION;")
echo "📋 PHP Version: $PHP_VERSION"

if php -r "exit(version_compare(PHP_VERSION, '7.4.0', '>=') ? 0 : 1);"; then
    echo "✅ PHP version is compatible (7.4+)"
else
    echo "❌ PHP version is too old. Please upgrade to PHP 7.4 or higher"
fi

# Create production configuration files
echo "📝 Creating production configuration files..."

if [ ! -f "users/config/database_production.php" ]; then
    echo "⚠️  Production database config not found. Please create it manually."
fi

if [ ! -f "admin-dashboard/config/database_production.php" ]; then
    echo "⚠️  Production admin database config not found. Please create it manually."
fi

if [ ! -f ".htaccess_production" ]; then
    echo "⚠️  Production .htaccess not found. Please create it manually."
fi

# Check for database schema
echo "🗄️  Checking database schema..."
if [ -f "admin-dashboard/database/omateqdigitals_admin.sql" ]; then
    echo "✅ Database schema file found"
else
    echo "❌ Database schema file not found"
fi

# Display deployment checklist
echo ""
echo "📋 DEPLOYMENT CHECKLIST"
echo "======================="
echo ""
echo "Before deploying to production, please ensure:"
echo ""
echo "1. ✅ Update database credentials in:"
echo "   - users/config/database_production.php"
echo "   - admin-dashboard/config/database_production.php"
echo ""
echo "2. ✅ Update Paystack API keys in admin dashboard:"
echo "   - Use LIVE keys (pk_live_... and sk_live_...)"
echo "   - Update callback URLs to production domain"
echo ""
echo "3. ✅ Configure web server:"
echo "   - Copy .htaccess_production to .htaccess"
echo "   - Enable mod_rewrite and mod_ssl"
echo "   - Set up SSL certificate"
echo ""
echo "4. ✅ Set up database:"
echo "   - Create production database"
echo "   - Import omateqdigitals_admin.sql"
echo "   - Create database user with proper permissions"
echo ""
echo "5. ✅ Test functionality:"
echo "   - User registration and login"
echo "   - Course enrollment and payment"
echo "   - File uploads and downloads"
echo "   - Admin dashboard access"
echo ""
echo "6. ✅ Security measures:"
echo "   - Change default passwords"
echo "   - Enable HTTPS redirect"
echo "   - Set up regular backups"
echo "   - Monitor error logs"
echo ""
echo "📚 For detailed instructions, see DEPLOYMENT_GUIDE.md"
echo ""
echo "🎉 Deployment preparation complete!"
echo ""
echo "Next steps:"
echo "1. Upload files to production server"
echo "2. Update configuration files"
echo "3. Set up database"
echo "4. Test all functionality"
echo "5. Go live! 🚀"
