<?php
// Google Reviews Integration
// Note: This is a simplified implementation. For production use, you'll need to:
// 1. Set up Google My Business API credentials
// 2. Implement proper authentication
// 3. Handle rate limiting and caching

header('Content-Type: application/json');

// Configuration
$google_place_id = 'YOUR_GOOGLE_PLACE_ID'; // Replace with your actual Google Place ID
$api_key = 'YOUR_GOOGLE_API_KEY'; // Replace with your Google API key
$cache_duration = 3600; // Cache for 1 hour
$cache_file = 'google_reviews_cache.json';

// Alternative: Use Google Places API with business name search
$business_name = 'Omateq Digitals';
$business_location = 'Abuja, Nigeria';

// Function to get cached reviews
function getCachedReviews($cache_file, $cache_duration) {
    if (file_exists($cache_file)) {
        $cache_data = json_decode(file_get_contents($cache_file), true);
        if ($cache_data && isset($cache_data['timestamp'])) {
            if ((time() - $cache_data['timestamp']) < $cache_duration) {
                return $cache_data['reviews'];
            }
        }
    }
    return null;
}

// Function to cache reviews
function cacheReviews($reviews, $cache_file) {
    $cache_data = [
        'timestamp' => time(),
        'reviews' => $reviews
    ];
    file_put_contents($cache_file, json_encode($cache_data));
}

// Function to search for business using Google Places API
function searchBusiness($business_name, $location, $api_key) {
    $search_query = urlencode($business_name . ' ' . $location);
    $url = "https://maps.googleapis.com/maps/api/place/textsearch/json?query={$search_query}&key={$api_key}";
    
    $response = @file_get_contents($url);
    if ($response === false) {
        return null;
    }
    
    $data = json_decode($response, true);
    if ($data['status'] === 'OK' && !empty($data['results'])) {
        return $data['results'][0]['place_id']; // Return first result's place_id
    }
    
    return null;
}

// Function to fetch Google Reviews using Google Places API
function fetchGoogleReviews($place_id, $api_key) {
    // If no place_id provided, try to search for the business
    if ($place_id === 'YOUR_GOOGLE_PLACE_ID') {
        global $business_name, $business_location;
        $place_id = searchBusiness($business_name, $business_location, $api_key);
        
        if (!$place_id) {
            // Fall back to sample reviews if business not found
            return getSampleReviews();
        }
    }
    
    // Fetch reviews using Google Places API
    $url = "https://maps.googleapis.com/maps/api/place/details/json?place_id={$place_id}&fields=reviews,rating,user_ratings_total&key={$api_key}";
    
    $response = @file_get_contents($url);
    if ($response === false) {
        return getSampleReviews();
    }
    
    $data = json_decode($response, true);
    if ($data['status'] === 'OK' && isset($data['result']['reviews'])) {
        return $data['result']['reviews'];
    }
    
    // Fall back to sample reviews if API fails
    return getSampleReviews();
}

// Function to get sample reviews (fallback)
function getSampleReviews() {
    // This is a simplified implementation
    // In production, you would use the Google My Business API or Google Places API
    
    // For now, we'll return sample reviews that match your current testimonials
    // In production, replace this with actual API calls
    
    $sample_reviews = [
        [
            'author_name' => 'Dr. Sarah Johnson',
            'author_url' => '',
            'profile_photo_url' => 'assets/img/client/testimonial-1.jpg',
            'rating' => 5,
            'relative_time_description' => '2 months ago',
            'text' => 'Omateq Digitals transformed our school management system completely. The portal they developed has streamlined our administrative processes and improved communication between teachers, students, and parents significantly.',
            'time' => time() - (60 * 24 * 60 * 60) // 2 months ago
        ],
        [
            'author_name' => 'Michael Chen',
            'author_url' => '',
            'profile_photo_url' => 'assets/img/client/testimonial-2.jpg',
            'rating' => 5,
            'relative_time_description' => '1 month ago',
            'text' => 'The inventory management software developed by Omateq Digitals has revolutionized our business operations. Real-time tracking and automated reports have saved us countless hours and reduced errors dramatically.',
            'time' => time() - (30 * 24 * 60 * 60) // 1 month ago
        ],
        [
            'author_name' => 'Alexandra Rodriguez',
            'author_url' => '',
            'profile_photo_url' => 'assets/img/client/testimonial-3.jpg',
            'rating' => 5,
            'relative_time_description' => '3 weeks ago',
            'text' => 'Their digital skills training program was exceptional. I learned web development from scratch and now I\'m building my own applications. The instructors are knowledgeable and the curriculum is well-structured.',
            'time' => time() - (21 * 24 * 60 * 60) // 3 weeks ago
        ],
        [
            'author_name' => 'David Thompson',
            'author_url' => '',
            'profile_photo_url' => 'assets/img/client/default-avatar.svg',
            'rating' => 5,
            'relative_time_description' => '1 week ago',
            'text' => 'Excellent service! Omateq Digitals delivered our custom web application on time and within budget. The team was professional and responsive throughout the entire project.',
            'time' => time() - (7 * 24 * 60 * 60) // 1 week ago
        ],
        [
            'author_name' => 'Maria Garcia',
            'author_url' => '',
            'profile_photo_url' => 'assets/img/client/default-avatar.svg',
            'rating' => 5,
            'relative_time_description' => '5 days ago',
            'text' => 'The AI automation training course was outstanding. I now understand how to implement AI solutions in my business. Highly recommended for anyone looking to learn AI.',
            'time' => time() - (5 * 24 * 60 * 60) // 5 days ago
        ]
    ];
    
    return $sample_reviews;
}

// Function to generate star rating HTML
function generateStarRating($rating) {
    $stars = '';
    for ($i = 1; $i <= 5; $i++) {
        if ($i <= $rating) {
            $stars .= '<li><i class="fa fa-star"></i></li>';
        } else {
            $stars .= '<li><i class="fa fa-star-o"></i></li>';
        }
    }
    return $stars;
}

// Function to format time ago
function timeAgo($timestamp) {
    $time = time() - $timestamp;
    if ($time < 60) return 'just now';
    if ($time < 3600) return floor($time/60) . ' minutes ago';
    if ($time < 86400) return floor($time/3600) . ' hours ago';
    if ($time < 2592000) return floor($time/86400) . ' days ago';
    if ($time < 31536000) return floor($time/2592000) . ' months ago';
    return floor($time/31536000) . ' years ago';
}

try {
    // Check if we have cached reviews
    $reviews = getCachedReviews($cache_file, $cache_duration);
    
    if (!$reviews) {
        // Fetch fresh reviews
        $reviews = fetchGoogleReviews($google_place_id, $api_key);
        
        // Cache the reviews
        cacheReviews($reviews, $cache_file);
    }
    
    // Format reviews for display
    $formatted_reviews = [];
    foreach ($reviews as $review) {
        $formatted_reviews[] = [
            'author_name' => $review['author_name'],
            'profile_photo_url' => $review['profile_photo_url'],
            'rating' => $review['rating'],
            'star_rating_html' => generateStarRating($review['rating']),
            'text' => $review['text'],
            'relative_time' => $review['relative_time_description'],
            'time_ago' => timeAgo($review['time'])
        ];
    }
    
    echo json_encode([
        'success' => true,
        'reviews' => $formatted_reviews,
        'total_reviews' => count($formatted_reviews),
        'cached' => $reviews !== null
    ]);
    
} catch (Exception $e) {
    error_log("Google Reviews Error: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => 'Failed to load reviews',
        'error' => $e->getMessage()
    ]);
}
?>
