// Custom Video Player Component
class CustomVideoPlayer {
    constructor(container, options = {}) {
        this.container = container;
        this.options = {
            autoplay: false,
            controls: true,
            progressUpdateInterval: 1000,
            ...options
        };
        
        this.video = null;
        this.isPlaying = false;
        this.currentTime = 0;
        this.duration = 0;
        this.progressInterval = null;
        this.lastProgressUpdate = 0;
        this.autoCompleted = false;
        
        this.init();
    }
    
    init() {
        this.createPlayer();
        this.bindEvents();
        this.startProgressTracking();
    }
    
    createPlayer() {
        const playerHTML = `
            <div class="custom-video-player">
                <div class="video-container">
                    <video class="video-element" preload="metadata" controlsList="nodownload nofullscreen noremoteplayback" disablePictureInPicture>
                        <source src="" type="video/mp4">
                        Your browser does not support the video tag.
                    </video>
                    <div class="video-overlay">
                        <div class="play-button" id="playButton">
                            <i class="fi fi-rr-play"></i>
                        </div>
                    </div>
                </div>
                <div class="video-controls">
                    <div class="control-left">
                        <button class="control-btn play-pause-btn" id="playPauseBtn">
                            <i class="fi fi-rr-play"></i>
                        </button>
                        <div class="time-display">
                            <span class="current-time">0:00</span>
                            <span class="time-separator">/</span>
                            <span class="total-time">0:00</span>
                        </div>
                    </div>
                    <div class="control-center">
                        <div class="progress-container">
                            <div class="progress-bar">
                                <div class="progress-filled" id="progressFilled"></div>
                                <div class="progress-thumb" id="progressThumb"></div>
                            </div>
                        </div>
                    </div>
                    <div class="control-right">
                        <button class="control-btn volume-btn" id="volumeBtn">
                            <i class="fi fi-rr-volume"></i>
                        </button>
                        <button class="control-btn fullscreen-btn" id="fullscreenBtn">
                            <i class="fi fi-rr-expand"></i>
                        </button>
                    </div>
                </div>
            </div>
        `;
        
        this.container.innerHTML = playerHTML;
        this.video = this.container.querySelector('.video-element');
        this.setupVideoElement();
    }
    
    setupVideoElement() {
        // Disable right-click context menu
        this.video.addEventListener('contextmenu', (e) => {
            e.preventDefault();
            return false;
        });
        
        // Disable keyboard shortcuts that could save video
        this.video.addEventListener('keydown', (e) => {
            // Disable Ctrl+S, Ctrl+A, F12, Ctrl+Shift+I, etc.
            if (e.ctrlKey && (e.key === 's' || e.key === 'S' || e.key === 'a' || e.key === 'A')) {
                e.preventDefault();
                return false;
            }
            if (e.key === 'F12' || (e.ctrlKey && e.shiftKey && e.key === 'I')) {
                e.preventDefault();
                return false;
            }
        });
        
        // Disable drag and drop
        this.video.addEventListener('dragstart', (e) => {
            e.preventDefault();
            return false;
        });
        
        this.video.addEventListener('loadedmetadata', () => {
            this.duration = this.video.duration;
            this.updateTotalTime();
        });
        
        this.video.addEventListener('timeupdate', () => {
            this.currentTime = this.video.currentTime;
            this.updateProgress();
            this.updateCurrentTime();
        });
        
        this.video.addEventListener('play', () => {
            this.isPlaying = true;
            this.updatePlayButton();
            this.hideOverlay();
        });
        
        this.video.addEventListener('pause', () => {
            this.isPlaying = false;
            this.updatePlayButton();
            // Save progress when video is paused
            this.saveProgress();
        });
        
        this.video.addEventListener('ended', () => {
            this.isPlaying = false;
            this.updatePlayButton();
            this.showOverlay();
            // Mark lesson as completed when video ends
            this.markLessonCompleted();
        });
        
        // Save progress when seeking
        this.video.addEventListener('seeked', () => {
            this.saveProgress();
        });
    }
    
    bindEvents() {
        // Play/Pause button
        const playPauseBtn = this.container.querySelector('#playPauseBtn');
        const playButton = this.container.querySelector('#playButton');
        
        playPauseBtn.addEventListener('click', () => this.togglePlayPause());
        playButton.addEventListener('click', () => this.togglePlayPause());
        
        // Progress bar
        const progressBar = this.container.querySelector('.progress-bar');
        progressBar.addEventListener('click', (e) => this.seekTo(e));
        
        // Volume button
        const volumeBtn = this.container.querySelector('#volumeBtn');
        volumeBtn.addEventListener('click', () => this.toggleMute());
        
        // Fullscreen button
        const fullscreenBtn = this.container.querySelector('#fullscreenBtn');
        fullscreenBtn.addEventListener('click', () => this.toggleFullscreen());
        
        // Keyboard controls
        document.addEventListener('keydown', (e) => {
            if (this.container.contains(document.activeElement) || this.isPlaying) {
                switch(e.code) {
                    case 'Space':
                        e.preventDefault();
                        this.togglePlayPause();
                        break;
                    case 'ArrowLeft':
                        e.preventDefault();
                        this.seek(-10);
                        break;
                    case 'ArrowRight':
                        e.preventDefault();
                        this.seek(10);
                        break;
                    case 'KeyF':
                        e.preventDefault();
                        this.toggleFullscreen();
                        break;
                    case 'KeyM':
                        e.preventDefault();
                        this.toggleMute();
                        break;
                }
            }
        });
    }
    
    loadVideo(videoSrc, resumeTime = 0) {
        console.log('CustomVideoPlayer: Loading video:', videoSrc, 'Resume from:', resumeTime);
        this.video.src = videoSrc;
        this.video.load();
        
        // Add error handling
        this.video.addEventListener('error', (e) => {
            console.error('Video load error:', e);
            console.error('Video src:', videoSrc);
            console.error('Error details:', this.video.error);
        });
        
        this.video.addEventListener('loadeddata', () => {
            console.log('Video loaded successfully:', videoSrc);
            // Resume from saved position if available
            if (resumeTime > 0 && resumeTime < this.video.duration) {
                this.video.currentTime = resumeTime;
                this.currentTime = resumeTime;
                console.log('Resuming video from:', resumeTime);
            }
            this.updateProgress();
            this.updateCurrentTime();
        });
        
        this.video.addEventListener('loadedmetadata', () => {
            this.duration = this.video.duration;
            this.updateTotalTime();
        });
    }
    
    togglePlayPause() {
        if (this.video.paused) {
            this.video.play();
        } else {
            this.video.pause();
        }
    }
    
    seek(seconds) {
        this.video.currentTime = Math.max(0, Math.min(this.duration, this.video.currentTime + seconds));
    }
    
    seekTo(event) {
        const progressBar = this.container.querySelector('.progress-bar');
        const rect = progressBar.getBoundingClientRect();
        const clickX = event.clientX - rect.left;
        const percentage = clickX / rect.width;
        this.video.currentTime = percentage * this.duration;
    }
    
    toggleMute() {
        this.video.muted = !this.video.muted;
        const volumeBtn = this.container.querySelector('#volumeBtn i');
        volumeBtn.className = this.video.muted ? 'fi fi-rr-volume-cross' : 'fi fi-rr-volume';
    }
    
    toggleFullscreen() {
        if (!document.fullscreenElement) {
            this.container.requestFullscreen();
        } else {
            document.exitFullscreen();
        }
    }
    
    updatePlayButton() {
        const playPauseBtn = this.container.querySelector('#playPauseBtn i');
        const playButton = this.container.querySelector('#playButton');
        
        playPauseBtn.className = this.isPlaying ? 'fi fi-rr-pause' : 'fi fi-rr-play';
        
        if (this.isPlaying) {
            playButton.style.display = 'none';
        } else {
            playButton.style.display = 'flex';
        }
    }
    
    updateProgress() {
        // Validate duration to prevent NaN
        if (!this.duration || this.duration <= 0) {
            return;
        }
        
        const progress = (this.currentTime / this.duration) * 100;
        const progressFilled = this.container.querySelector('#progressFilled');
        const progressThumb = this.container.querySelector('#progressThumb');
        
        if (progressFilled) {
            progressFilled.style.width = `${progress}%`;
        }
        if (progressThumb) {
            progressThumb.style.left = `${progress}%`;
        }
        
        // Auto-complete lesson when reaching 99% progress
        if (progress >= 99 && !this.autoCompleted) {
            this.autoCompleted = true;
            console.log(`Auto-completing lesson at ${Math.round(progress)}% progress`);
            
            // Show auto-completion notification
            if (typeof window.showNotification === 'function') {
                window.showNotification(`Lesson auto-completed at ${Math.round(progress)}% progress! Moving to next lesson...`, 'success');
            }
            
            this.markLessonCompleted();
        }
        
        // Update modal progress with throttling to prevent blinking (max once per second)
        if (typeof window.updateProgressDisplay === 'function') {
            const now = Date.now();
            if (now - this.lastProgressUpdate >= 1000) { // Update max once per second
                // Only update if values are valid
                if (!isNaN(progress) && !isNaN(this.currentTime) && !isNaN(this.duration)) {
                    window.updateProgressDisplay(progress, this.currentTime, this.duration);
                    this.lastProgressUpdate = now;
                }
            }
        }
    }
    
    updateCurrentTime() {
        const currentTimeElement = this.container.querySelector('.current-time');
        currentTimeElement.textContent = this.formatTime(this.currentTime);
    }
    
    updateTotalTime() {
        const totalTimeElement = this.container.querySelector('.total-time');
        totalTimeElement.textContent = this.formatTime(this.duration);
    }
    
    saveProgress() {
        // Save progress to server if lesson ID is available
        if (window.currentModalLessonId && this.video && this.duration > 0) {
            const progressData = {
                lesson_id: window.currentModalLessonId,
                current_time: this.currentTime,
                duration: this.duration,
                is_completed: false
            };
            
            // Send progress to server
            fetch('track_video_progress.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams(progressData)
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    console.log('Progress saved:', this.currentTime, '/', this.duration);
                } else {
                    console.error('Failed to save progress:', data.message);
                }
            })
            .catch(error => {
                console.error('Error saving progress:', error);
            });
        }
    }
    
    markLessonCompleted() {
        // Mark lesson as completed when video ends
        if (window.currentModalLessonId && this.video && this.duration > 0) {
            const progressData = {
                lesson_id: window.currentModalLessonId,
                current_time: this.duration, // Set to full duration
                duration: this.duration,
                is_completed: true
            };
            
            // Send completion to server
            fetch('track_video_progress.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams(progressData)
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    console.log('Lesson marked as completed automatically');
                    // Call the completion handler from the main page
                    if (typeof window.handleLessonCompleted === 'function') {
                        window.handleLessonCompleted(window.currentModalLessonId);
                    }
                } else {
                    console.error('Failed to mark lesson as completed:', data.message);
                }
            })
            .catch(error => {
                console.error('Error marking lesson as completed:', error);
            });
        }
    }
    
    formatTime(seconds) {
        const mins = Math.floor(seconds / 60);
        const secs = Math.floor(seconds % 60);
        return `${mins}:${secs.toString().padStart(2, '0')}`;
    }
    
    showOverlay() {
        const overlay = this.container.querySelector('.video-overlay');
        overlay.style.display = 'flex';
    }
    
    hideOverlay() {
        const overlay = this.container.querySelector('.video-overlay');
        overlay.style.display = 'none';
    }
    
    startProgressTracking() {
        this.progressInterval = setInterval(() => {
            if (this.isPlaying && typeof window.trackVideoProgress === 'function') {
                window.trackVideoProgress(this.currentTime, this.duration, true);
            }
        }, this.options.progressUpdateInterval);
    }
    
    stopProgressTracking() {
        if (this.progressInterval) {
            clearInterval(this.progressInterval);
        }
    }
    
    destroy() {
        this.stopProgressTracking();
        if (this.video) {
            this.video.pause();
            this.video.src = '';
        }
    }
}

// Initialize custom video player when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    // This will be called when the modal is opened
    window.initCustomVideoPlayer = function(containerId, videoSrc, resumeTime = 0) {
        const container = document.getElementById(containerId);
        if (container) {
            // Clear any existing content
            container.innerHTML = '';
            
            // Create new video player
            const player = new CustomVideoPlayer(container);
            player.loadVideo(videoSrc, resumeTime);
            return player;
        }
        return null;
    };
});
