<?php
// Email Verification Handler

class EmailVerificationHandler {
    private $pdo;
    private $token_expiry_hours = 24; // Token valid for 24 hours
    
    public function __construct(PDO $pdo) {
        $this->pdo = $pdo;
    }
    
    /**
     * Generate a secure verification token
     */
    private function generateToken() {
        return bin2hex(random_bytes(32)); // 64-character hex string
    }
    
    /**
     * Generate a 6-digit verification code
     */
    private function generateVerificationCode() {
        return str_pad(rand(0, 999999), 6, '0', STR_PAD_LEFT);
    }
    
    /**
     * Create and send email verification code
     */
    public function createVerificationCode($userId, $email) {
        try {
            // Generate 6-digit verification code
            $verification_code = $this->generateVerificationCode();
            
            // Update user with verification code
            $stmt = $this->pdo->prepare("UPDATE users SET verification_code = :code, verification_sent_at = NOW() WHERE id = :user_id");
            if ($stmt->execute(['code' => $verification_code, 'user_id' => $userId])) {
                return ['success' => true, 'code' => $verification_code];
            }
            return ['success' => false, 'message' => 'Failed to create verification code.'];
            
        } catch (Exception $e) {
            error_log("Email verification code creation failed: " . $e->getMessage());
            return ['success' => false, 'message' => 'Failed to create verification code.'];
        }
    }
    
    /**
     * Create and send email verification token
     */
    public function createVerificationToken($userId, $email) {
        try {
            // Invalidate any existing tokens for this user
            $stmt = $this->pdo->prepare("UPDATE email_verification_tokens SET used = TRUE WHERE user_id = :user_id AND expires_at > NOW() AND used = FALSE");
            $stmt->execute(['user_id' => $userId]);
            
            $token = $this->generateToken();
            $expires_at = date('Y-m-d H:i:s', strtotime("+" . $this->token_expiry_hours . " hours"));
            
            $stmt = $this->pdo->prepare("INSERT INTO email_verification_tokens (user_id, token, expires_at) VALUES (:user_id, :token, :expires_at)");
            if ($stmt->execute(['user_id' => $userId, 'token' => $token, 'expires_at' => $expires_at])) {
                return ['success' => true, 'token' => $token];
            }
            return ['success' => false, 'message' => 'Failed to create verification token.'];
            
        } catch (Exception $e) {
            error_log("Email verification token creation failed: " . $e->getMessage());
            return ['success' => false, 'message' => 'Failed to create verification token.'];
        }
    }
    
    /**
     * Send email verification code email
     */
    public function sendVerificationCodeEmail($email, $code, $userName = '', $isLoginOTP = false) {
        try {
            require_once __DIR__ . '/smtp_config.php';
            require_once __DIR__ . '/smtp_mailer.php';
            
            $config = SMTPConfig::getConfig();
            $mailer = new SMTPMailer($config);
            
            $subject = $isLoginOTP ? "Your Verification Code - Omateq Digitals" : "Verify Your Email - Omateq Digitals";
            $message = $this->buildVerificationCodeEmailMessage($code, $userName, $isLoginOTP);
            
            $result = $mailer->send($email, $subject, $message, true);
            
            if ($result['success']) {
                // Update verification_sent_at timestamp
                $stmt = $this->pdo->prepare("UPDATE users SET verification_sent_at = NOW() WHERE email = :email");
                $stmt->execute(['email' => $email]);
            }
            
            return $result;
            
        } catch (Exception $e) {
            $logMessage = $isLoginOTP ? "Login OTP send failed" : "Email verification code send failed";
            error_log($logMessage . ": " . $e->getMessage());
            $errorMessage = $isLoginOTP ? 'Failed to send login OTP email.' : 'Failed to send verification code email.';
            return ['success' => false, 'message' => $errorMessage];
        }
    }
    
    /**
     * Send email verification email
     */
    public function sendVerificationEmail($email, $token, $userName = '') {
        try {
            require_once __DIR__ . '/smtp_config.php';
            require_once __DIR__ . '/smtp_mailer.php';
            
            $config = SMTPConfig::getConfig();
            $mailer = new SMTPMailer($config);
            
            $verificationLink = "http://" . (isset($_SERVER['HTTP_HOST']) ? $_SERVER['HTTP_HOST'] : 'localhost') . "/users/authentication/verify-email.php?token=" . $token;
            
            $subject = "Verify Your Email - Omateq Digitals";
            $message = $this->buildVerificationEmailMessage($verificationLink, $userName);
            
            $result = $mailer->send($email, $subject, $message, true);
            
            if ($result['success']) {
                // Update verification_sent_at timestamp
                $stmt = $this->pdo->prepare("UPDATE users SET verification_sent_at = NOW() WHERE email = :email");
                $stmt->execute(['email' => $email]);
            }
            
            return $result;
            
        } catch (Exception $e) {
            error_log("Email verification send failed: " . $e->getMessage());
            return ['success' => false, 'message' => 'Failed to send verification email.'];
        }
    }
    
    /**
     * Validate email verification code
     */
    public function validateCode($email, $code, $isLoginOTP = false) {
        try {
            // For login OTP, use 15 minutes expiry. For email verification, use 24 hours
            $expiryInterval = $isLoginOTP ? '15 MINUTE' : '24 HOUR';
            
            $stmt = $this->pdo->prepare("
                SELECT id, email, full_name, email_verified, verification_code, verification_sent_at 
                FROM users 
                WHERE email = :email AND verification_code = :code AND verification_sent_at > DATE_SUB(NOW(), INTERVAL {$expiryInterval})
            ");
            $stmt->execute(['email' => $email, 'code' => $code]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$user) {
                $expiryMessage = $isLoginOTP ? 'OTP code has expired (15 minutes). Please request a new one.' : 'Verification code has expired (24 hours). Please request a new one.';
                return ['success' => false, 'message' => $expiryMessage];
            }
            
            if ($user['email_verified']) {
                return ['success' => false, 'message' => 'Email is already verified.'];
            }
            
            return ['success' => true, 'user_id' => $user['id'], 'email' => $user['email']];
            
        } catch (Exception $e) {
            error_log("Email verification code validation failed: " . $e->getMessage());
            return ['success' => false, 'message' => 'Failed to validate verification code.'];
        }
    }
    
    /**
     * Mark email as verified using code
     */
    public function markEmailVerifiedByCode($userId, $email) {
        try {
            $this->pdo->beginTransaction();
            
            // Mark user email as verified and clear verification code
            $stmt = $this->pdo->prepare("UPDATE users SET email_verified = TRUE, verification_code = NULL WHERE id = :id");
            $stmt->execute(['id' => $userId]);
            
            $this->pdo->commit();
            return ['success' => true, 'message' => 'Email verified successfully.'];
            
        } catch (Exception $e) {
            $this->pdo->rollBack();
            error_log("Email verification mark failed: " . $e->getMessage());
            return ['success' => false, 'message' => 'Failed to verify email.'];
        }
    }
    
    /**
     * Validate email verification token
     */
    public function validateToken($token) {
        try {
            $stmt = $this->pdo->prepare("
                SELECT evt.*, u.email, u.full_name, u.email_verified 
                FROM email_verification_tokens evt 
                JOIN users u ON evt.user_id = u.id 
                WHERE evt.token = :token AND evt.expires_at > NOW() AND evt.used = FALSE
            ");
            $stmt->execute(['token' => $token]);
            $tokenData = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$tokenData) {
                return ['success' => false, 'message' => 'Invalid or expired verification token.'];
            }
            
            if ($tokenData['email_verified']) {
                return ['success' => false, 'message' => 'Email is already verified.'];
            }
            
            return ['success' => true, 'user_id' => $tokenData['user_id'], 'email' => $tokenData['email'], 'token_id' => $tokenData['id']];
            
        } catch (Exception $e) {
            error_log("Email verification validation failed: " . $e->getMessage());
            return ['success' => false, 'message' => 'Failed to validate verification token.'];
        }
    }
    
    /**
     * Mark email as verified
     */
    public function markEmailVerified($tokenId, $userId) {
        try {
            $this->pdo->beginTransaction();
            
            // Mark token as used
            $stmt = $this->pdo->prepare("UPDATE email_verification_tokens SET used = TRUE WHERE id = :id");
            $stmt->execute(['id' => $tokenId]);
            
            // Mark user email as verified
            $stmt = $this->pdo->prepare("UPDATE users SET email_verified = TRUE WHERE id = :id");
            $stmt->execute(['id' => $userId]);
            
            $this->pdo->commit();
            return ['success' => true, 'message' => 'Email verified successfully.'];
            
        } catch (Exception $e) {
            $this->pdo->rollBack();
            error_log("Email verification mark failed: " . $e->getMessage());
            return ['success' => false, 'message' => 'Failed to verify email.'];
        }
    }
    
    /**
     * Check if user needs email verification
     */
    public function needsVerification($email) {
        try {
            $stmt = $this->pdo->prepare("SELECT email_verified FROM users WHERE email = :email");
            $stmt->execute(['email' => $email]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$user) {
                return false; // User doesn't exist
            }
            
            return !$user['email_verified']; // Return true if not verified
            
        } catch (Exception $e) {
            error_log("Email verification check failed: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Build HTML email message for verification code
     */
    private function buildVerificationCodeEmailMessage($code, $userName = '', $isLoginOTP = false) {
        $greeting = $userName ? "Hello " . htmlspecialchars($userName) . "," : "Hello,";
        
        if ($isLoginOTP) {
            $title = "Verification Code";
            $heading = "Your Verification Code";
            $description = "You have requested to access your Omateq Digitals account. To complete your login, please use the verification code below:";
            $instruction = "Enter this code on the login page to complete your access.";
            $expiry = "This verification code will expire in 15 minutes for security reasons.";
            $footer = "If you didn't request this access, please ignore this email and consider changing your password.";
        } else {
            $title = "Verify Your Email";
            $heading = "Verify Your Email Address";
            $description = "Thank you for registering with Omateq Digitals! To complete your registration and access all features, please verify your email address using the code below:";
            $instruction = "Enter this code on the login page to verify your email.";
            $expiry = "This verification code will expire in 24 hours for security reasons.";
            $footer = "If you didn't create an account with us, please ignore this email.";
        }
        
        return "
        <html>
        <head>
            <title>{$title}</title>
        </head>
        <body>
            <div style='font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px;'>
                <div style='text-align: center; margin-bottom: 30px;'>
                    <h1 style='color: #00022E; margin: 0;'>Omateq Digitals</h1>
                </div>
                
                <h2 style='color: #00022E;'>{$heading}</h2>
                <p>{$greeting}</p>
                <p>{$description}</p>
                
                <div style='text-align: center; margin: 30px 0; background: #f8f9fa; padding: 30px; border-radius: 10px; border: 2px dashed #00022E;'>
                    <h3 style='color: #00022E; margin: 0 0 15px 0;'>Your Verification Code</h3>
                    <div style='background: #00022E; color: white; padding: 20px; font-size: 32px; font-weight: bold; letter-spacing: 5px; border-radius: 8px; display: inline-block; font-family: monospace;'>
                        {$code}
                    </div>
                    <p style='color: #666; font-size: 14px; margin-top: 15px;'>{$instruction}</p>
                </div>
                
                <p style='color: #666; font-size: 14px;'>{$expiry}</p>
                <p style='color: #666; font-size: 14px;'>{$footer}</p>
                
                <hr style='margin: 30px 0; border: none; border-top: 1px solid #eee;'>
                
                <div style='text-align: center; color: #666; font-size: 12px;'>
                    <p>This is an automated message from Omateq Digitals.</p>
                    <p>Please do not reply to this email.</p>
                </div>
            </div>
        </body>
        </html>
        ";
    }
    
    /**
     * Build HTML email message for verification
     */
    private function buildVerificationEmailMessage($verificationLink, $userName = '') {
        $greeting = $userName ? "Hello " . htmlspecialchars($userName) . "," : "Hello,";
        
        return "
        <html>
        <head>
            <title>Verify Your Email</title>
        </head>
        <body>
            <div style='font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px;'>
                <div style='text-align: center; margin-bottom: 30px;'>
                    <h1 style='color: #00022E; margin: 0;'>Omateq Digitals</h1>
                </div>
                
                <h2 style='color: #00022E;'>Verify Your Email Address</h2>
                <p>{$greeting}</p>
                <p>Thank you for registering with Omateq Digitals! To complete your registration and access all features, please verify your email address by clicking the button below:</p>
                
                <div style='text-align: center; margin: 30px 0;'>
                    <a href='{$verificationLink}' style='background-color: #00022E; color: white; padding: 15px 30px; text-decoration: none; border-radius: 5px; display: inline-block; font-weight: bold;'>Verify Email Address</a>
                </div>
                
                <p style='color: #666; font-size: 14px;'>This verification link will expire in 24 hours for security reasons.</p>
                <p style='color: #666; font-size: 14px;'>If you didn't create an account with us, please ignore this email.</p>
                
                <hr style='margin: 30px 0; border: none; border-top: 1px solid #eee;'>
                
                <div style='text-align: center; color: #666; font-size: 12px;'>
                    <p>This is an automated message from Omateq Digitals.</p>
                    <p>Please do not reply to this email.</p>
                </div>
            </div>
        </body>
        </html>
        ";
    }
}
?>
