<?php
// Exchange Rate Configuration
// Fetches live USD to NGN exchange rate from multiple reliable sources

function getUSDToNGNRate() {
    // Cache the rate for 30 minutes to get more frequent updates
    $cache_file = __DIR__ . '/exchange_rate_cache.json';
    $cache_duration = 1800; // 30 minutes in seconds
    
    // Check if cache exists and is still valid
    if (file_exists($cache_file)) {
        $cache_data = json_decode(file_get_contents($cache_file), true);
        if ($cache_data && isset($cache_data['timestamp']) && (time() - $cache_data['timestamp']) < $cache_duration) {
            return $cache_data['rate'];
        }
    }
    
    // Try multiple sources to get the most accurate rate
    $rate = getBestExchangeRateFromMultipleSources();
    
    if ($rate && $rate > 1000) {
        // Cache the rate
        $cache_data = [
            'rate' => $rate,
            'timestamp' => time(),
            'source' => 'multiple_apis'
        ];
        file_put_contents($cache_file, json_encode($cache_data));
        return $rate;
    }
    
    // Fallback rate if all APIs fail
    return 1600; // Updated fallback rate
}

// New function to get rates from multiple sources
function getBestExchangeRateFromMultipleSources() {
    $rates = [];
    
    // Try Fixer.io API (more reliable)
    $fixer_rate = getRateFromFixerIO();
    if ($fixer_rate) $rates[] = $fixer_rate;
    
    // Try ExchangeRate-API
    $exchange_rate = getRateFromExchangeRateAPI();
    if ($exchange_rate) $rates[] = $exchange_rate;
    
    // Try CurrencyLayer API
    $currency_layer_rate = getRateFromCurrencyLayer();
    if ($currency_layer_rate) $rates[] = $currency_layer_rate;
    
    // Try AbokiFX as backup
    $aboki_rate = fetchLiveExchangeRate();
    if ($aboki_rate) $rates[] = $aboki_rate;
    
    // Return the average of all successful rates, or the highest if we want to be conservative
    if (!empty($rates)) {
        // Filter out unrealistic rates (too low or too high)
        $valid_rates = array_filter($rates, function($rate) {
            return $rate >= 1000 && $rate <= 2000;
        });
        
        if (!empty($valid_rates)) {
            // Return the average for more accuracy
            return array_sum($valid_rates) / count($valid_rates);
        }
    }
    
    return false;
}

function getRateFromFixerIO() {
    try {
        // Using a free tier API key (you should get your own from fixer.io)
        $api_key = 'YOUR_FIXER_IO_API_KEY'; // Replace with actual API key
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, "http://data.fixer.io/api/latest?access_key={$api_key}&symbols=NGN&base=USD");
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code === 200 && $response) {
            $data = json_decode($response, true);
            if (isset($data['rates']['NGN'])) {
                return floatval($data['rates']['NGN']);
            }
        }
    } catch (Exception $e) {
        error_log('Fixer.io API error: ' . $e->getMessage());
    }
    
    return false;
}

function getRateFromCurrencyLayer() {
    try {
        // Using a free tier API key (you should get your own from currencylayer.com)
        $api_key = 'YOUR_CURRENCY_LAYER_API_KEY'; // Replace with actual API key
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, "http://api.currencylayer.com/live?access_key={$api_key}&currencies=NGN&source=USD");
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code === 200 && $response) {
            $data = json_decode($response, true);
            if (isset($data['quotes']['USDNGN'])) {
                return floatval($data['quotes']['USDNGN']);
            }
        }
    } catch (Exception $e) {
        error_log('CurrencyLayer API error: ' . $e->getMessage());
    }
    
    return false;
}

function fetchLiveExchangeRate() {
    try {
        // Use cURL to fetch the exchange rate from AbokiFX
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, 'https://www.abokifx.app/black-market-exchange-rate/');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        
        $html = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code === 200 && $html) {
            // Parse the HTML to extract USD rate
            // Look for USD rate in the HTML content
            if (preg_match('/USD.*?(\d{3,4}(?:\.\d{2})?)/i', $html, $matches)) {
                return floatval($matches[1]);
            }
        }
    } catch (Exception $e) {
        error_log('AbokiFX fetch error: ' . $e->getMessage());
    }
    
    return false;
}

// Alternative method using a more reliable API
function getUSDToNGNFromAPI() {
    try {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, 'https://api.exchangerate-api.com/v4/latest/USD');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code === 200 && $response) {
            $data = json_decode($response, true);
            if (isset($data['rates']['NGN'])) {
                return floatval($data['rates']['NGN']);
            }
        }
    } catch (Exception $e) {
        error_log('Exchange rate API error: ' . $e->getMessage());
    }
    
    return false;
}

// Updated function name to match the new structure
function getRateFromExchangeRateAPI() {
    return getUSDToNGNFromAPI();
}

// Get the best available rate
function getBestExchangeRate() {
    // Use the new multi-source approach
    $rate = getUSDToNGNRate();
    
    // Final fallback with more realistic rate
    if (!$rate || $rate < 1000) {
        $rate = 1600; // Updated fallback rate based on current market
    }
    
    return $rate;
}

// Add a function to get rate with source information
function getExchangeRateWithInfo() {
    $cache_file = __DIR__ . '/exchange_rate_cache.json';
    
    if (file_exists($cache_file)) {
        $cache_data = json_decode(file_get_contents($cache_file), true);
        if ($cache_data && isset($cache_data['timestamp'])) {
            return [
                'rate' => $cache_data['rate'],
                'timestamp' => $cache_data['timestamp'],
                'source' => $cache_data['source'] ?? 'unknown',
                'age_minutes' => round((time() - $cache_data['timestamp']) / 60, 1)
            ];
        }
    }
    
    // If no cache, get fresh rate
    $rate = getBestExchangeRate();
    return [
        'rate' => $rate,
        'timestamp' => time(),
        'source' => 'live',
        'age_minutes' => 0
    ];
}
?>
