<?php
// Password Reset Handler Class

class PasswordResetHandler {
    private $db;
    
    public function __construct($database) {
        $this->db = $database;
    }
    
    /**
     * Generate a secure random token
     */
    public function generateToken() {
        return bin2hex(random_bytes(32));
    }
    
    /**
     * Create a password reset token for a user
     */
    public function createResetToken($email) {
        try {
            // First, check if user exists
            $stmt = $this->db->prepare("SELECT id FROM users WHERE email = ?");
            $stmt->execute([$email]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$user) {
                return ['success' => false, 'message' => 'Email not found in our system.'];
            }
            
            // Delete any existing tokens for this user
            $stmt = $this->db->prepare("DELETE FROM password_reset_tokens WHERE user_id = ?");
            $stmt->execute([$user['id']]);
            
            // Create new token
            $token = $this->generateToken();
            $expiresAt = date('Y-m-d H:i:s', strtotime('+1 hour')); // Token expires in 1 hour
            
            $stmt = $this->db->prepare("INSERT INTO password_reset_tokens (user_id, token, expires_at) VALUES (?, ?, ?)");
            $stmt->execute([$user['id'], $token, $expiresAt]);
            
            return [
                'success' => true, 
                'token' => $token, 
                'user_id' => $user['id'],
                'expires_at' => $expiresAt
            ];
            
        } catch (Exception $e) {
            return ['success' => false, 'message' => 'Failed to create reset token.'];
        }
    }
    
    /**
     * Validate a password reset token
     */
    public function validateToken($token) {
        try {
            $stmt = $this->db->prepare("
                SELECT prt.*, u.email, u.full_name 
                FROM password_reset_tokens prt 
                JOIN users u ON prt.user_id = u.id 
                WHERE prt.token = ? AND prt.expires_at > NOW() AND prt.used_at IS NULL
            ");
            $stmt->execute([$token]);
            $tokenData = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$tokenData) {
                return ['success' => false, 'message' => 'Invalid or expired token.'];
            }
            
            return ['success' => true, 'data' => $tokenData];
            
        } catch (Exception $e) {
            return ['success' => false, 'message' => 'Failed to validate token.'];
        }
    }
    
    /**
     * Reset user password
     */
    public function resetPassword($token, $newPassword) {
        try {
            // Validate token first
            $tokenValidation = $this->validateToken($token);
            if (!$tokenValidation['success']) {
                return $tokenValidation;
            }
            
            $tokenData = $tokenValidation['data'];
            
            // Hash the new password
            $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
            
            // Update user password
            $stmt = $this->db->prepare("UPDATE users SET password = ? WHERE id = ?");
            $stmt->execute([$hashedPassword, $tokenData['user_id']]);
            
            // Mark token as used
            $stmt = $this->db->prepare("UPDATE password_reset_tokens SET used_at = NOW() WHERE token = ?");
            $stmt->execute([$token]);
            
            return ['success' => true, 'message' => 'Password reset successfully.'];
            
        } catch (Exception $e) {
            return ['success' => false, 'message' => 'Failed to reset password.'];
        }
    }
    
    /**
     * Send password reset email
     */
    public function sendResetEmail($email, $token) {
        try {
            // Try SMTP first, fallback to basic mail() if SMTP fails
            $smtpResult = $this->sendResetEmailSMTP($email, $token);
            if ($smtpResult['success']) {
                return $smtpResult;
            }
            
            // Fallback to basic mail() function
            return $this->sendResetEmailBasic($email, $token);
            
        } catch (Exception $e) {
            return ['success' => false, 'message' => 'Failed to send reset email: ' . $e->getMessage()];
        }
    }
    
    /**
     * Send password reset email using SMTP
     */
    private function sendResetEmailSMTP($email, $token) {
        try {
            require_once __DIR__ . '/smtp_config.php';
            require_once __DIR__ . '/smtp_mailer.php';
            
            $config = SMTPConfig::getConfig();
            
            // Debug: Log the configuration being used
            error_log("SMTP Config: " . json_encode([
                'host' => $config['host'],
                'port' => $config['port'],
                'encryption' => $config['encryption'],
                'username' => $config['username']
            ]));
            
            $mailer = new SMTPMailer($config);
            
            $resetLink = "http://" . (isset($_SERVER['HTTP_HOST']) ? $_SERVER['HTTP_HOST'] : 'localhost') . dirname($_SERVER['PHP_SELF']) . "/reset-password.php?token=" . $token;
            
            $subject = "Password Reset Request - Omateq Digitals";
            $message = $this->buildEmailMessage($resetLink);
            
            $result = $mailer->send($email, $subject, $message, true);
            
            // Debug: Log the result
            error_log("SMTP Result: " . json_encode($result));
            
            return $result;
            
        } catch (Exception $e) {
            error_log("SMTP Error: " . $e->getMessage());
            return ['success' => false, 'message' => 'SMTP failed: ' . $e->getMessage()];
        }
    }
    
    /**
     * Send password reset email using basic mail() function
     */
    private function sendResetEmailBasic($email, $token) {
        try {
            $resetLink = "http://" . (isset($_SERVER['HTTP_HOST']) ? $_SERVER['HTTP_HOST'] : 'localhost') . dirname($_SERVER['PHP_SELF']) . "/reset-password.php?token=" . $token;
            
            $subject = "Password Reset Request - Omateq Digitals";
            $message = $this->buildEmailMessage($resetLink);
            
            $headers = [
                'MIME-Version: 1.0',
                'Content-type: text/html; charset=UTF-8',
                'From: Omateq Digitals <noreply@omateqdigitals.com>',
                'Reply-To: support@omateqdigitals.com',
                'X-Mailer: PHP/' . phpversion()
            ];
            
            $success = mail($email, $subject, $message, implode("\r\n", $headers));
            
            if ($success) {
                return ['success' => true, 'message' => 'Password reset email sent successfully.'];
            } else {
                return ['success' => false, 'message' => 'Failed to send email. Please check your email configuration.'];
            }
            
        } catch (Exception $e) {
            return ['success' => false, 'message' => 'Failed to send reset email.'];
        }
    }
    
    /**
     * Build HTML email message
     */
    private function buildEmailMessage($resetLink) {
        return "
        <html>
        <head>
            <title>Password Reset Request</title>
        </head>
        <body>
            <div style='font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px;'>
                <div style='text-align: center; margin-bottom: 30px;'>
                    <h1 style='color: #00022E; margin: 0;'>Omateq Digitals</h1>
                </div>
                
                <h2 style='color: #00022E;'>Password Reset Request</h2>
                <p>Hello,</p>
                <p>You have requested to reset your password for your Omateq Digitals account.</p>
                <p>Click the button below to reset your password:</p>
                
                <div style='text-align: center; margin: 30px 0;'>
                    <a href='{$resetLink}' style='background-color: #00022E; color: white; padding: 15px 30px; text-decoration: none; border-radius: 5px; display: inline-block; font-weight: bold;'>Reset Password</a>
                </div>
                
                <p style='color: #666; font-size: 14px;'>This link will expire in 1 hour for security reasons.</p>
                <p style='color: #666; font-size: 14px;'>If you didn't request this password reset, please ignore this email.</p>
                
                <hr style='margin: 30px 0; border: none; border-top: 1px solid #eee;'>
                
                <div style='text-align: center; color: #666; font-size: 12px;'>
                    <p>This is an automated message from Omateq Digitals.</p>
                    <p>Please do not reply to this email.</p>
                </div>
            </div>
        </body>
        </html>
        ";
    }
}
?>
