<?php
// Simple SMTP Mailer Class (without external dependencies)

class SMTPMailer {
    private $config;
    private $socket;
    
    public function __construct($config) {
        $this->config = $config;
    }
    
    /**
     * Send email using SMTP
     */
    public function send($to, $subject, $message, $isHtml = true) {
        try {
            // Connect to SMTP server
            $this->connect();
            
        // Send SMTP commands
        $this->sendCommand("EHLO " . (isset($_SERVER['HTTP_HOST']) ? $_SERVER['HTTP_HOST'] : 'localhost'));
            $this->sendCommand("AUTH LOGIN");
            $this->sendCommand(base64_encode($this->config['username']));
            $this->sendCommand(base64_encode($this->config['password']));
            $this->sendCommand("MAIL FROM: <" . $this->config['from_email'] . ">");
            $this->sendCommand("RCPT TO: <" . $to . ">");
            $this->sendCommand("DATA");
            
            // Send email headers and body
            $headers = $this->buildHeaders($to, $subject, $isHtml);
            $this->sendCommand($headers . "\r\n" . $message . "\r\n.");
            
            $this->sendCommand("QUIT");
            $this->disconnect();
            
            return ['success' => true, 'message' => 'Email sent successfully'];
            
        } catch (Exception $e) {
            $this->disconnect();
            return ['success' => false, 'message' => 'Failed to send email: ' . $e->getMessage()];
        }
    }
    
    private function connect() {
        $context = stream_context_create();
        
        // Build connection string based on encryption type
        $connectionString = $this->config['host'] . ':' . $this->config['port'];
        
        if ($this->config['encryption'] === 'ssl') {
            $connectionString = 'ssl://' . $connectionString;
        }
        
        $this->socket = stream_socket_client(
            $connectionString,
            $errno, $errstr, 30, STREAM_CLIENT_CONNECT, $context
        );
        
        if (!$this->socket) {
            throw new Exception("Failed to connect to SMTP server: $errstr ($errno)");
        }
        
        // Enable TLS if required (for non-SSL connections)
        if ($this->config['encryption'] === 'tls') {
            if (!stream_socket_enable_crypto($this->socket, true, STREAM_CRYPTO_METHOD_TLS_CLIENT)) {
                throw new Exception("Failed to enable TLS encryption");
            }
        }
        
        // Read server response
        $response = fgets($this->socket, 512);
        if (substr($response, 0, 3) !== '220') {
            throw new Exception("SMTP server error: $response");
        }
    }
    
    private function sendCommand($command) {
        fwrite($this->socket, $command . "\r\n");
        $response = fgets($this->socket, 512);
        
        if (substr($response, 0, 3) >= '400') {
            throw new Exception("SMTP command failed: $command - Response: $response");
        }
        
        return $response;
    }
    
    private function buildHeaders($to, $subject, $isHtml) {
        $headers = "From: " . $this->config['from_name'] . " <" . $this->config['from_email'] . ">\r\n";
        $headers .= "Reply-To: " . $this->config['reply_to'] . "\r\n";
        $headers .= "To: $to\r\n";
        $headers .= "Subject: $subject\r\n";
        $headers .= "Date: " . date('r') . "\r\n";
        $headers .= "Message-ID: <" . time() . "." . uniqid() . "@" . (isset($_SERVER['HTTP_HOST']) ? $_SERVER['HTTP_HOST'] : 'localhost') . ">\r\n";
        
        // Add authentication and delivery improvement headers
        $headers .= "X-Mailer: Omateq Digitals PHP Mailer\r\n";
        $headers .= "X-Priority: 3\r\n";
        $headers .= "X-MSMail-Priority: Normal\r\n";
        $headers .= "Importance: Normal\r\n";
        
        // Add list-unsubscribe header for better deliverability
        $headers .= "List-Unsubscribe: <mailto:unsubscribe@" . (isset($_SERVER['HTTP_HOST']) ? $_SERVER['HTTP_HOST'] : 'localhost') . ">\r\n";
        
        if ($isHtml) {
            $headers .= "MIME-Version: 1.0\r\n";
            $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
            $headers .= "Content-Transfer-Encoding: 8bit\r\n";
        } else {
            $headers .= "Content-Type: text/plain; charset=UTF-8\r\n";
            $headers .= "Content-Transfer-Encoding: 8bit\r\n";
        }
        
        return $headers;
    }
    
    private function disconnect() {
        if ($this->socket) {
            fclose($this->socket);
            $this->socket = null;
        }
    }
}
?>
