<?php
require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/../admin-dashboard/config/settings.php';
require_once __DIR__ . '/../admin-dashboard/config/branding.php';
require_once __DIR__ . '/config/session_manager.php';

// Initialize session and check login with timeout
UserSessionManager::initSession();
UserSessionManager::requireLogin('authentication/login.php');

// Get user data
try {
    $pdo = getConnection();
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $user = $stmt->fetch();
    
    if (!$user) {
        session_destroy();
        header('Location: authentication/login.php');
        exit();
    }
} catch (Exception $e) {
    session_destroy();
    header('Location: authentication/login.php');
    exit();
}

$error = '';
$success = '';

// Handle course enrollment - show payment modal instead of redirecting
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'enroll') {
        $course_id = $_POST['course_id'];
        
        try {
            // Check if user is already enrolled
            $stmt = $pdo->prepare("SELECT id FROM course_enrollments WHERE user_id = ? AND course_id = ?");
            $stmt->execute([$_SESSION['user_id'], $course_id]);
            
            if ($stmt->fetch()) {
                $error = "You are already enrolled in this course.";
            } else {
                // Get course details for payment
                $stmt = $pdo->prepare("SELECT * FROM courses WHERE id = ? AND status = 'active'");
                $stmt->execute([$course_id]);
                $course = $stmt->fetch();
                
                if ($course) {
                    // Store course details in session for payment processing
                    $_SESSION['pending_course_enrollment'] = [
                        'course_id' => $course_id,
                        'course_title' => $course['title'],
                        'course_price' => $course['price'],
                        'course_description' => $course['description']
                    ];
                    
                    // Return success to trigger payment modal
                    echo json_encode(['success' => true, 'message' => 'Course details loaded for payment']);
                    exit();
                } else {
                    $error = "Course not found or not available.";
                }
            }
        } catch (Exception $e) {
            $error = "Enrollment failed: " . $e->getMessage();
        }
    }
}

// Enrolled courses removed - users should go to my_courses.php to view their enrolled courses

// Get all available courses (excluding those user is already enrolled in)
$available_courses = [];
try {
    $stmt = $pdo->prepare("
        SELECT c.*, 
               COUNT(ce.id) as enrollment_count
        FROM courses c
        LEFT JOIN course_enrollments ce ON c.id = ce.course_id AND ce.user_id = ?
        WHERE c.status = 'active' AND ce.id IS NULL
        GROUP BY c.id
        ORDER BY c.created_at DESC
    ");
    $stmt->execute([$_SESSION['user_id']]);
    $available_courses = $stmt->fetchAll();
} catch (Exception $e) {
    $error = "Failed to load courses: " . $e->getMessage();
}

// Get script request replies
$script_replies = [];
$script_replies_count = 0;
try {
    $stmt = $pdo->prepare("SELECT * FROM script_requests WHERE user_id = ? AND admin_reply IS NOT NULL ORDER BY created_at DESC LIMIT 5");
    $stmt->execute([$_SESSION['user_id']]);
    $script_replies = $stmt->fetchAll();
    $script_replies_count = count($script_replies);
} catch (Exception $e) {
    // Handle error silently
}

// Get transactions (purchases and course enrollments)
$transactions = [];
$transactions_count = 0;
try {
    $stmt = $pdo->prepare("SELECT 'purchase' as type, product_name as item_name, total_amount as amount, status, created_at FROM orders WHERE user_id = ? 
                          UNION ALL 
                          SELECT 'course' as type, title as item_name, price as amount, 'completed' as status, created_at FROM course_enrollments ce 
                          JOIN courses c ON ce.course_id = c.id WHERE ce.user_id = ? 
                          ORDER BY created_at DESC LIMIT 5");
    $stmt->execute([$_SESSION['user_id'], $_SESSION['user_id']]);
    $transactions = $stmt->fetchAll();
    $transactions_count = count($transactions);
} catch (Exception $e) {
    // Handle error silently
}

// Function to format file size
function formatFileSize($bytes) {
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    } else {
        return $bytes . ' bytes';
    }
}

// Function to get time ago
function time_ago($datetime) {
    $time = time() - strtotime($datetime);
    
    if ($time < 60) return 'just now';
    if ($time < 3600) return floor($time/60) . ' min ago';
    if ($time < 86400) return floor($time/3600) . ' hr ago';
    if ($time < 2592000) return floor($time/86400) . ' days ago';
    return date('M j', strtotime($datetime));
}
?>
<!DOCTYPE html>
<html lang="en">

<head>
  <!-- begin::Edura Meta Basic -->
  <meta charset="utf-8">
  <meta name="theme-color" content="#0D9488">
  <meta name="robots" content="index, follow">
  <meta name="author" content="LayoutDrop">
  <meta name="format-detection" content="telephone=no">
  <meta name="keywords" content="School Management, Education Admin Dashboard, Student Information System, University Dashboard, College Admin Template, Learning Management System, Education CRM, Teacher Management, Student Attendance Dashboard, Online Education Admin, Responsive Admin Template, Bootstrap 5 Education Admin, eLearning Dashboard, School ERP, Class Management Dashboard, Academic Dashboard, Education Web Application, Modern Admin Template, Admin UI Kit, ThemeForest Admin Template">
  <meta name="description" content="Edura is a professional and modern School Management & Education Admin Dashboard Template built with Bootstrap. It includes light and dark modes, and is perfect for managing students, teachers, courses, attendance, fees, and academic activities — ideal for schools, colleges, universities, and eLearning platforms.">
  <!-- end::Edura Meta Basic -->

  <!-- begin::Edura Meta Social -->
  <meta property="og:url" content="https://demos.layoutdrop.com/edura/demo/">
  <meta property="og:site_name" content="Dashboard | Edura School Management & Education Admin Dashboard Template">
  <meta property="og:type" content="website">
  <meta property="og:locale" content="en_us">
  <meta property="og:title" content="Dashboard | Edura School Management & Education Admin Dashboard Template">
  <meta property="og:description" content="Edura is a professional and modern School Management & Education Admin Dashboard Template built with Bootstrap. It includes light and dark modes, and is perfect for managing students, teachers, courses, attendance, fees, and academic activities — ideal for schools, colleges, universities, and eLearning platforms.">
  <meta property="og:image" content="https://demos.layoutdrop.com/edura/demo/preview.png">
  <!-- end::Edura Meta Social -->

  <!-- begin::Edura Meta Twitter -->
  <meta name="twitter:card" content="summary_large_image">
  <meta name="twitter:site" content="@LayoutDrop">
  <meta name="twitter:creator" content="@LayoutDrop">
  <meta name="twitter:title" content="Dashboard | Edura School Management & Education Admin Dashboard Template">
  <meta name="twitter:description" content="Edura is a professional and modern School Management & Education Admin Dashboard Template built with Bootstrap. It includes light and dark modes, and is perfect for managing students, teachers, courses, attendance, fees, and academic activities — ideal for schools, colleges, universities, and eLearning platforms.">
  <meta name="twitter:image" content="https://demos.layoutdrop.com/edura/demo/preview.png">
  <!-- end::Edura Meta Twitter -->

  <!-- begin::Edura Website Page Title -->
  <title>Courses | Omateq Digitals</title>
  <!-- end::Edura Website Page Title -->

  <!-- begin::Edura Mobile Specific -->
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <!-- end::Edura Mobile Specific -->

  <!-- begin::Edura Favicon Tags -->
  <?php echo getFaviconHtml(); ?>
  <!-- end::Edura Favicon Tags -->

  <!-- begin::Edura Google Fonts -->
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Plus+Jakarta+Sans:ital,wght@0,200..800;1,200..800&display=swap" rel="stylesheet">
  <!-- end::Edura Google Fonts -->

  <!-- begin::Edura Required Stylesheet -->
  <link rel="stylesheet" href="assets/libs/flaticon/css/all/all.css">
  <link rel="stylesheet" href="assets/libs/lucide/lucide.css">
  <link rel="stylesheet" href="assets/libs/fontawesome/css/all.min.css">
  <link rel="stylesheet" href="assets/libs/simplebar/simplebar.css">
  <link rel="stylesheet" href="assets/libs/node-waves/waves.css">
  <link rel="stylesheet" href="assets/libs/bootstrap-select/css/bootstrap-select.min.css">
  <!-- end::Edura Required Stylesheet -->

  <!-- begin::Edura CSS Stylesheet -->
  <link rel="stylesheet" href="assets/css/styles.css">
  <!-- end::Edura CSS Stylesheet -->
  <style>
    /* Logo sizing for better fit */
    .app-navbar-brand .navbar-brand-logo img {
      max-height: 50px;
      max-width: 180px;
      width: auto;
      height: auto;
      object-fit: contain;
    }
    
    .app-navbar-brand .navbar-brand-mini img {
      max-height: 35px;
      max-width: 35px;
      width: auto;
      height: auto;
      object-fit: contain;
    }
  </style>
  
  <!-- Custom CSS for course layout -->
  <style>
    .course-card {
      transition: all 0.3s ease;
      border: none;
      box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    }
    .course-card:hover {
      transform: translateY(-5px);
      box-shadow: 0 5px 20px rgba(0,0,0,0.15);
    }
    .course-thumbnail {
      height: 200px;
      object-fit: cover;
      border-radius: 10px 10px 0 0;
    }
    .enrolled-badge {
      position: absolute;
      top: 10px;
      right: 10px;
      z-index: 2;
    }
    .course-level {
      font-size: 0.8rem;
      padding: 4px 8px;
      border-radius: 12px;
    }
    .level-beginner { background-color: #d4edda; color: #155724; }
    .level-intermediate { background-color: #fff3cd; color: #856404; }
    .level-advanced { background-color: #f8d7da; color: #721c24; }
    .stats-card {
      background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
      color: white;
      border-radius: 15px;
    }
    .stats-card .card-body {
      padding: 1.5rem;
    }
    .stats-number {
      font-size: 2rem;
      font-weight: bold;
      margin-bottom: 0.5rem;
    }
    .stats-label {
      font-size: 0.9rem;
      opacity: 0.9;
    }
  </style>
</head>

<body>
  <div class="page-layout">

    <!-- begin::Edura Page Header -->
    <header class="app-header">
      <div class="app-header-inner">
        <button class="app-toggler" type="button">
          <span></span>
          <span></span>
          <span></span>
        </button>
        <div class="app-header-start">
          <form class="d-none d-md-flex align-items-center h-100 w-lg-250px w-xxl-300px position-relative" action="index.html#">
            <button type="button" class="btn btn-sm border-0 position-absolute start-0 ms-3 p-0">
              <i class="fi fi-rr-search"></i>
            </button>
            <input type="text" class="form-control ps-5" placeholder="Search courses...">
          </form>
        </div>
        <div class="app-header-end">
          <div class="px-lg-3 px-2 ps-0 d-flex align-items-center">
            <a href="javascript:void(0);" class="theme-btn">
              <i class="fi fi-rr-brightness icon-light"></i>
              <div class="theme-toggle"></div>
              <i class="fi fi-rr-moon icon-dark"></i>
            </a>
          </div>
          <div class="vr my-3"></div>
          <div class="d-flex align-items-center gap-sm-2 gap-0 px-lg-4 px-sm-2 px-1">
            <div class="dropdown text-end">
              <a href="javascript:void(0);" class="btn btn-icon btn-action-gray rounded-circle waves-effect waves-light" data-bs-toggle="dropdown" data-bs-auto-close="outside" aria-expanded="true">
                <i class="fi fi-rr-envelope"></i>
              </a>
              <div class="dropdown-menu dropdown-menu-lg-end p-0 w-300px mt-2">
                <div class="px-3 py-3 border-bottom d-flex justify-content-between align-items-center">
                  <h6 class="mb-0">Messages <span class="badge badge-sm rounded-pill bg-primary ms-2"><?php echo $script_replies_count ?? 0; ?></span></h6>
                  <i class="bi bi-x-lg cursor-pointer"></i>
                </div>
                <div class="p-2" style="height: 300px;" data-simplebar>
                  <ul class="list-group list-group-hover list-group-smooth list-group-unlined">
                    <?php if (!empty($script_replies)): ?>
                      <?php foreach ($script_replies as $reply): ?>
                        <li class="list-group-item d-flex justify-content-between align-items-center">
                          <div class="avatar avatar-xs bg-info rounded-circle text-white">
                            <i class="fi fi-rr-user"></i>
                          </div>
                          <div class="ms-2 me-auto">
                            <h6 class="mb-0">Admin Reply</h6>
                            <small class="text-body d-block"><?php echo htmlspecialchars(substr($reply['admin_reply'], 0, 50)) . '...'; ?></small>
                            <small class="text-muted position-absolute end-0 top-0 mt-2 me-3"><?php echo time_ago($reply['created_at']); ?></small>
                          </div>
                        </li>
                      <?php endforeach; ?>
                    <?php else: ?>
                      <li class="list-group-item text-center py-4">
                        <i class="fi fi-rr-envelope text-muted" style="font-size: 2rem;"></i>
                        <p class="text-muted mt-2 mb-0">No script request replies yet</p>
                      </li>
                    <?php endif; ?>
                  </ul>
                </div>
                <div class="p-2">
                  <a href="script-request.php" class="btn w-100 btn-primary waves-effect waves-light">View all replies</a>
                </div>
              </div>
            </div>
            <div class="dropdown text-end">
              <a href="javascript:void(0);" class="btn btn-icon btn-action-gray rounded-circle waves-effect waves-light" data-bs-toggle="dropdown" data-bs-auto-close="outside" aria-expanded="true">
                <i class="fi fi-rr-bell"></i>
              </a>
              <div class="dropdown-menu dropdown-menu-lg-end p-0 w-300px mt-2">
                <div class="px-3 py-3 border-bottom d-flex justify-content-between align-items-center">
                  <h6 class="mb-0">Transactions <span class="badge badge-sm rounded-pill bg-primary ms-2"><?php echo $transactions_count ?? 0; ?></span></h6>
                  <i class="bi bi-x-lg cursor-pointer"></i>
                </div>
                <div class="p-2" style="height: 300px;" data-simplebar>
                  <ul class="list-group list-group-hover list-group-smooth list-group-unlined">
                    <?php if (!empty($transactions)): ?>
                      <?php foreach ($transactions as $transaction): ?>
                        <li class="list-group-item d-flex justify-content-between align-items-center">
                          <div class="avatar avatar-xs bg-<?php echo $transaction['type'] === 'purchase' ? 'success' : 'info'; ?> rounded-circle text-white">
                            <i class="fi fi-rr-<?php echo $transaction['type'] === 'purchase' ? 'shopping-cart' : 'graduation-cap'; ?>"></i>
                          </div>
                          <div class="ms-2 me-auto">
                            <h6 class="mb-0"><?php echo ucfirst($transaction['type']); ?> - <?php echo htmlspecialchars($transaction['item_name']); ?></h6>
                            <small class="text-body d-block"><?php echo $transaction['status'] === 'completed' ? 'Payment successful' : 'Payment pending'; ?> - $<?php echo number_format($transaction['amount'], 2); ?></small>
                            <small class="text-muted position-absolute end-0 top-0 mt-2 me-3"><?php echo time_ago($transaction['created_at']); ?></small>
                          </div>
                        </li>
                      <?php endforeach; ?>
                    <?php else: ?>
                      <li class="list-group-item text-center py-4">
                        <i class="fi fi-rr-shopping-cart text-muted" style="font-size: 2rem;"></i>
                        <p class="text-muted mt-2 mb-0">No transactions yet</p>
                      </li>
                    <?php endif; ?>
                  </ul>
                </div>
                <div class="p-2">
                  <a href="products.php" class="btn w-100 btn-primary waves-effect waves-light">View all transactions</a>
                </div>
              </div>
            </div>
          </div>
          <div class="vr my-3"></div>
          <div class="dropdown text-end ms-2">
            <a href="javascript:void(0);" class="d-flex align-items-center py-2" data-bs-toggle="dropdown" data-bs-auto-close="outside" aria-expanded="true">
              <div class="avatar avatar-sm rounded-circle avatar-status-success">
                <?php if ($user['avatar'] && file_exists($user['avatar'])): ?>
                  <img src="<?php echo htmlspecialchars($user['avatar']); ?>" alt="<?php echo htmlspecialchars($user['full_name']); ?>">
                <?php else: ?>
                  <img src="assets/images/avatar/avatar1.webp" alt="<?php echo htmlspecialchars($user['full_name']); ?>">
                <?php endif; ?>
              </div>
            </a>
            <ul class="dropdown-menu dropdown-menu-end w-225px mt-1">
              <li class="d-flex align-items-center p-2">
                <div class="avatar avatar-sm rounded-circle">
                  <?php if ($user['avatar'] && file_exists($user['avatar'])): ?>
                    <img src="<?php echo htmlspecialchars($user['avatar']); ?>" alt="<?php echo htmlspecialchars($user['full_name']); ?>">
                  <?php else: ?>
                    <img src="assets/images/avatar/avatar1.webp" alt="<?php echo htmlspecialchars($user['full_name']); ?>">
                  <?php endif; ?>
                </div>
                <div class="ms-2">
                  <div class="fw-bold text-dark"><?php echo htmlspecialchars($user['full_name']); ?></div>
                  <small class="text-body d-block lh-sm"><?php echo htmlspecialchars($user['email']); ?></small>
                </div>
              </li>
              <li>
                <div class="dropdown-divider my-1"></div>
              </li>
              <li>
                <a class="dropdown-item d-flex align-items-center gap-2" href="profile.php">
                  <i class="fi fi-rr-user scale-1x"></i> Profile
                </a>
              </li>
              <li>
                <a class="dropdown-item d-flex align-items-center gap-2" href="help-center.php">
                  <i class="fi fi-rs-interrogation scale-1x"></i> Help Center
                </a>
              </li>
              <li>
                <div class="dropdown-divider my-1"></div>
              </li>
              <li>
                <a class="dropdown-item d-flex align-items-center gap-2 text-danger" href="authentication/logout.php">
                  <i class="fi fi-sr-exit scale-1x"></i> Logout
                </a>
              </li>
            </ul>
          </div>
        </div>
      </div>
    </header>
    <!-- end::Edura Page Header -->

    <!-- begin::Edura Sidebar Menu -->
    <aside class="app-menubar" id="menubar">
      <div class="app-navbar-brand">
        <a class="navbar-brand-logo" href="index.php">
          <?php echo '<img src="' . '../' . getSetting('main_logo', 'assets/images/logo.svg') . '" alt="logo">'; ?>
        </a>
        <a class="navbar-brand-mini" href="index.php">
          <?php echo '<img src="' . '../' . getSetting('mini_logo', 'assets/images/logo-text-white.svg') . '" alt="logo">'; ?>
        </a>
      </div>
      <nav class="app-navbar" data-simplebar>
        <ul class="menubar">
              <li class="menu-item">
                <a class="menu-link" href="index.php">
              <i class="fi fi-rr-apps"></i>
                  <span class="menu-label">Dashboard</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="products.php">
              <i class="fi fi-rr-shopping-cart"></i>
              <span class="menu-label">Products</span>
                </a>
              </li>
              <li class="menu-item active">
            <a class="menu-link" href="course.php">
              <i class="fi fi-rr-graduation-cap"></i>
              <span class="menu-label">Course</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="my_courses.php">
              <i class="fi fi-rr-book"></i>
              <span class="menu-label">My Courses</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="profile.php">
              <i class="fi fi-rr-user"></i>
              <span class="menu-label">Profile</span>
                </a>
              </li>
              
        </ul>
      </nav>
      <div class="app-footer">
        <div class="text-center py-2">
          <small class="text-white-50">
            <i class="fi fi-rr-copyright me-1"></i>
            2025 Omateq Digitals
          </small>
        </div>
        <a href="mailto:info@omateqdigitals.com" class="btn btn-outline-white waves-effect btn-app-nav w-100">
          <i class="fi fi-rs-interrogation"></i>
          <span class="nav-text">Help and Support</span>
        </a>
      </div>
    </aside>
    <!-- end::Edura Sidebar Menu -->

    <main class="app-wrapper">
      <div class="container-fluid">
        <div class="app-page-head d-flex flex-wrap gap-3 align-items-center justify-content-between">
          <div class="clearfix">
            <h1 class="app-page-title">Courses</h1>
            <span>Browse and enroll in our courses</span>
          </div>
        </div>

        <!-- Success/Error Messages -->
        <?php if ($error): ?>
          <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fi fi-rr-exclamation-triangle me-2"></i>
            <?php echo htmlspecialchars($error); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
          </div>
        <?php endif; ?>
        <?php if ($success): ?>
          <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fi fi-rr-check-circle me-2"></i>
            <?php echo htmlspecialchars($success); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
          </div>
        <?php endif; ?>

        <!-- Course Statistics -->
        <div class="row mb-4">
          <div class="col-md-4 mb-3">
            <div class="card stats-card p-3" style="background: linear-gradient(135deg, #a18cd1 0%, #fbc2eb 100%);">
              <div class="d-flex align-items-center">
                <div class="icon-square bg-white-opacity-20 rounded-3 me-3">
                  <i class="fi fi-rr-book text-white fs-4"></i>
                </div>
                <div>
                  <h5 class="mb-0 text-white"><?php echo count($available_courses); ?></h5>
                  <p class="text-white-50 mb-0">Available Courses</p>
                </div>
              </div>
            </div>
          </div>
          <div class="col-md-4 mb-3">
            <div class="card stats-card p-3" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);">
              <div class="d-flex align-items-center">
                <div class="icon-square bg-white-opacity-20 rounded-3 me-3">
                  <i class="fi fi-rr-graduation-cap text-white fs-4"></i>
                </div>
                <div>
                  <a href="my_courses.php" class="text-white text-decoration-none">
                    <h5 class="mb-0 text-white">View My Courses</h5>
                    <p class="text-white-50 mb-0">Check your enrolled courses</p>
                  </a>
                </div>
              </div>
            </div>
          </div>
          <div class="col-md-4 mb-3">
            <div class="card stats-card p-3" style="background: linear-gradient(135deg, #f6d365 0%, #fda085 100%);">
              <div class="d-flex align-items-center">
                <div class="icon-square bg-white-opacity-20 rounded-3 me-3">
                  <i class="fi fi-rr-plus text-white fs-4"></i>
                </div>
                <div>
                  <h5 class="mb-0 text-white">Start Learning</h5>
                  <p class="text-white-50 mb-0">Browse and enroll in courses</p>
                </div>
              </div>
            </div>
          </div>
        </div>

        <!-- Enrolled courses section removed - users should go to my_courses.php to view their enrolled courses -->

        <!-- Available Courses -->
        <div class="row">
          <div class="col-12">
            <div class="card">
              <div class="card-header">
                <h5 class="card-title mb-0">
                  <i class="fi fi-rr-book me-2"></i>
                  Available Courses
                </h5>
              </div>
              <div class="card-body">
                <?php if (!empty($available_courses)): ?>
                  <div class="row">
                    <?php foreach ($available_courses as $course): ?>
                      <div class="col-lg-4 col-md-6 mb-4">
                        <div class="card course-card h-100">
                          <div class="position-relative">
                            <?php if (isset($course['thumbnail']) && !empty($course['thumbnail'])): ?>
                              <img src="../<?php echo htmlspecialchars($course['thumbnail']); ?>" class="card-img-top course-thumbnail" alt="<?php echo htmlspecialchars($course['title'] ?? 'Course'); ?>">
                            <?php else: ?>
                              <div class="card-img-top course-thumbnail bg-light d-flex align-items-center justify-content-center">
                                <i class="fi fi-rr-graduation-cap text-muted" style="font-size: 3rem;"></i>
                              </div>
                            <?php endif; ?>
                            <div class="enrolled-badge">
                              <span class="badge bg-info">
                                <i class="fi fi-rr-users me-1"></i>
                                <?php echo $course['enrollment_count'] ?? 0; ?> enrolled
                              </span>
                            </div>
                          </div>
                          <div class="card-body d-flex flex-column">
                            <h5 class="card-title"><?php echo htmlspecialchars($course['title'] ?? 'Course'); ?></h5>
                            <p class="card-text text-muted"><?php echo htmlspecialchars(substr($course['description'] ?? '', 0, 100)) . '...'; ?></p>
                            <div class="mt-auto">
                              <div class="d-flex justify-content-between align-items-center mb-2">
                                <span class="badge course-level level-<?php echo strtolower($course['level'] ?? 'beginner'); ?>"><?php echo ucfirst($course['level'] ?? 'Beginner'); ?></span>
                                <span class="text-primary fw-bold">$<?php echo number_format($course['price'] ?? 0, 2); ?></span>
                              </div>
                              <div class="d-flex gap-2">
                                <button type="button" class="btn btn-success btn-sm flex-fill" onclick="enrollInCourse(<?php echo $course['id']; ?>)">
                                  <i class="fi fi-rr-plus me-1"></i>
                                  Enroll Now
                                </button>
                                <button class="btn btn-outline-primary btn-sm" data-bs-toggle="modal" data-bs-target="#courseModal<?php echo $course['id']; ?>">
                                  <i class="fi fi-rr-eye"></i>
                                </button>
                              </div>
                            </div>
                          </div>
                        </div>
                      </div>
                    <?php endforeach; ?>
                  </div>
                <?php else: ?>
                  <div class="text-center py-5">
                    <i class="fi fi-rr-graduation-cap text-muted" style="font-size: 4rem;"></i>
                    <h4 class="text-muted mt-3">No courses available</h4>
                    <p class="text-muted">Check back later for new courses!</p>
                  </div>
                <?php endif; ?>
              </div>
            </div>
          </div>
        </div>

      </div>
    </main>

  </div>

  <!-- Course Details Modal -->
  <div class="modal fade" id="courseModal" tabindex="-1" aria-labelledby="courseModalLabel" aria-hidden="true" data-bs-backdrop="true" data-bs-keyboard="true">
    <div class="modal-dialog modal-lg">
      <div class="modal-content">
        <div class="modal-header">
          <h5 class="modal-title" id="courseModalLabel">
            <i class="fi fi-rr-graduation-cap me-2"></i>
            Course Details
          </h5>
          <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
        </div>
        <div class="modal-body" id="courseModalBody">
          <!-- Course details will be loaded here -->
        </div>
        <div class="modal-footer">
          <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
          <button type="button" class="btn btn-success" id="purchaseCourseBtn" style="display: none;">
            <i class="fi fi-rr-shopping-cart me-2"></i>
            Purchase Course
          </button>
        </div>
      </div>
    </div>
  </div>

  <!-- Payment Modal -->
  <div class="modal fade" id="paymentModal" tabindex="-1" aria-labelledby="paymentModalLabel" aria-hidden="true" data-bs-backdrop="true" data-bs-keyboard="true">
    <div class="modal-dialog modal-lg">
      <div class="modal-content">
        <div class="modal-header">
          <h5 class="modal-title" id="paymentModalLabel">
            <i class="fi fi-rr-credit-card me-2"></i>
            Complete Payment
          </h5>
          <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
        </div>
        <div class="modal-body">
          <div class="row">
            <div class="col-md-6">
              <div class="course-summary">
                <h6>Course Summary</h6>
                <div class="course-info" id="paymentCourseInfo">
                  <!-- Course info will be loaded here -->
                </div>
              </div>
            </div>
            <div class="col-md-6">
              <div class="payment-options">
                <h6>Payment Method</h6>
                <div class="payment-details mb-3">
                  <div class="d-flex justify-content-between mb-2">
                    <span>Amount (USD):</span>
                    <span class="fw-bold" id="paymentAmountUSD">$0.00</span>
                  </div>
                  <div class="d-flex justify-content-between mb-2">
                    <span>Exchange Rate:</span>
                    <span class="fw-bold" id="paymentExchangeRate">1 USD = ₦<?php 
                      require_once __DIR__ . '/config/exchange_rate.php';
                      try {
                        $exchange_rate_info = getExchangeRateWithInfo();
                        $exchange_rate = $exchange_rate_info['rate'] ?? 1500;
                        echo number_format($exchange_rate, 2);
                      } catch (Exception $e) {
                        echo number_format(1500, 2);
                      }
                    ?></span>
                  </div>
                  <div class="d-flex justify-content-between mb-2">
                    <span>Amount (NGN):</span>
                    <span class="fw-bold text-primary" id="paymentAmountNGN">₦0.00</span>
                  </div>
                </div>
                <div class="d-grid gap-2">
                  <button class="btn btn-primary btn-lg" id="paystackBtn">
                    <i class="fi fi-rr-credit-card me-2"></i>
                    Pay with Paystack
                  </button>
                </div>
                <div class="mt-3">
                  <small class="text-muted">
                    <i class="fi fi-rr-shield-check me-1"></i>
                    Secure payment processing powered by Paystack
                  </small>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  </div>

  <!-- begin::Edura Scripts -->
  <script src="assets/libs/global/global.min.js"></script>
  <script src="assets/js/appSettings.js"></script>
  <script src="assets/js/main.js"></script>
  <!-- end::Edura Scripts -->

  <!-- Disable Bootstrap automatic modal initialization -->
  <script>
    // Override Bootstrap modal initialization before any other scripts run
    (function() {
      if (typeof bootstrap !== 'undefined' && bootstrap.Modal) {
        const originalModal = bootstrap.Modal;
        bootstrap.Modal = function(element, config) {
          const defaultConfig = {
            backdrop: true,
            keyboard: true,
            focus: true,
            ...config
          };
          try {
            return new originalModal(element, defaultConfig);
          } catch (error) {
            console.warn('Bootstrap modal error prevented:', error);
            return {
              show: function() {
                const modal = document.getElementById(element.id || element);
                if (modal) {
                  modal.style.display = 'block';
                  modal.classList.add('show');
                  document.body.classList.add('modal-open');
                }
              },
              hide: function() {
                const modal = document.getElementById(element.id || element);
                if (modal) {
                  modal.style.display = 'none';
                  modal.classList.remove('show');
                  document.body.classList.remove('modal-open');
                }
              },
              dispose: function() {
                this.hide();
              }
            };
          }
        };
        
        // Override getOrCreateInstance
        bootstrap.Modal.getOrCreateInstance = function(element, config) {
          const defaultConfig = {
            backdrop: true,
            keyboard: true,
            focus: true,
            ...config
          };
          try {
            return originalModal.getOrCreateInstance(element, defaultConfig);
          } catch (error) {
            console.warn('Bootstrap modal getOrCreateInstance error prevented:', error);
            return {
              show: function() {
                const modal = document.getElementById(element.id || element);
                if (modal) {
                  modal.style.display = 'block';
                  modal.classList.add('show');
                  document.body.classList.add('modal-open');
                }
              },
              hide: function() {
                const modal = document.getElementById(element.id || element);
                if (modal) {
                  modal.style.display = 'none';
                  modal.classList.remove('show');
                  document.body.classList.remove('modal-open');
                }
              },
              dispose: function() {
                this.hide();
              }
            };
          }
        };
      }
    })();
  </script>

  <script>
    let currentCourseId = null;
    let currentCoursePrice = 0;

    // Disable automatic Bootstrap modal initialization
    if (typeof bootstrap !== 'undefined') {
      // Override Bootstrap modal initialization to prevent errors
      const originalModal = bootstrap.Modal;
      bootstrap.Modal = function(element, config) {
        // Provide default config to prevent backdrop errors
        const defaultConfig = {
          backdrop: true,
          keyboard: true,
          focus: true,
          ...config
        };
        try {
          return new originalModal(element, defaultConfig);
        } catch (error) {
          console.warn('Bootstrap modal initialization failed, using fallback:', error);
          return {
            show: () => showModal(element.id),
            hide: () => hideModal(element.id),
            dispose: () => hideModal(element.id)
          };
        }
      };
      
      // Override getOrCreateInstance to prevent automatic initialization
      bootstrap.Modal.getOrCreateInstance = function(element, config) {
        const defaultConfig = {
          backdrop: true,
          keyboard: true,
          focus: true,
          ...config
        };
        try {
          return originalModal.getOrCreateInstance(element, defaultConfig);
        } catch (error) {
          console.warn('Bootstrap modal getOrCreateInstance failed, using fallback:', error);
          return {
            show: () => showModal(element.id),
            hide: () => hideModal(element.id),
            dispose: () => hideModal(element.id)
          };
        }
      };
    }

    // Simple modal fallback functions
    function showModal(modalId) {
      const modal = document.getElementById(modalId);
      if (modal) {
        modal.style.display = 'block';
        modal.classList.add('show');
        document.body.classList.add('modal-open');
        
        // Add backdrop
        const backdrop = document.createElement('div');
        backdrop.className = 'modal-backdrop fade show';
        backdrop.id = 'modal-backdrop';
        document.body.appendChild(backdrop);
      }
    }

    function hideModal(modalId) {
      const modal = document.getElementById(modalId);
      if (modal) {
        modal.style.display = 'none';
        modal.classList.remove('show');
        document.body.classList.remove('modal-open');
        
        // Remove backdrop
        const backdrop = document.getElementById('modal-backdrop');
        if (backdrop) {
          backdrop.remove();
        }
      }
    }

    // Wait for DOM to be fully loaded
    document.addEventListener('DOMContentLoaded', function() {
      // Disable automatic modal initialization by removing data-bs-toggle attributes
      const modalTriggers = document.querySelectorAll('[data-bs-toggle="modal"]');
      modalTriggers.forEach(trigger => {
        trigger.removeAttribute('data-bs-toggle');
        trigger.setAttribute('data-custom-modal', 'true');
      });
      
      // Initialize Bootstrap components
      if (typeof bootstrap === 'undefined') {
        console.warn('Bootstrap is not loaded. Using fallback modal functionality.');
      }

      // Handle course view button clicks
      document.addEventListener('click', function(e) {
        console.log('Button clicked:', e.target);
        
        // Handle both data-bs-target and data-custom-modal attributes
        const button = e.target.closest('[data-bs-target^="#courseModal"], [data-custom-modal="true"]');
        if (button) {
          console.log('Course modal button found:', button);
          let courseId = button.getAttribute('data-bs-target');
          if (courseId) {
            courseId = courseId.replace('#courseModal', '');
          } else {
            // If no data-bs-target, try to get course ID from other attributes
            courseId = button.getAttribute('data-course-id') || '1';
          }
          console.log('Loading course details for ID:', courseId);
          loadCourseDetails(courseId);
        }
      });

      // Handle modal close buttons (fallback)
      document.addEventListener('click', function(e) {
        if (e.target.classList.contains('btn-close') || e.target.classList.contains('btn-secondary')) {
          const modal = e.target.closest('.modal');
          if (modal) {
            if (typeof bootstrap !== 'undefined') {
              const modalInstance = bootstrap.Modal.getInstance(modal);
              if (modalInstance) {
                modalInstance.hide();
              }
            } else {
              hideModal(modal.id);
            }
          }
        }
      });

      // Handle backdrop clicks (fallback)
      document.addEventListener('click', function(e) {
        if (e.target.classList.contains('modal-backdrop')) {
          const openModals = document.querySelectorAll('.modal.show');
          openModals.forEach(modal => {
            if (typeof bootstrap !== 'undefined') {
              const modalInstance = bootstrap.Modal.getInstance(modal);
              if (modalInstance) {
                modalInstance.hide();
              }
            } else {
              hideModal(modal.id);
            }
          });
        }
      });
    });

    // Load course details
    function loadCourseDetails(courseId) {
      console.log('loadCourseDetails called with courseId:', courseId);
      currentCourseId = courseId;
      
      // Show loading state
      const modalBody = document.getElementById('courseModalBody');
      if (!modalBody) {
        console.error('courseModalBody not found');
        return;
      }
      
      modalBody.innerHTML = '<div class="text-center py-4"><div class="spinner-border" role="status"><span class="visually-hidden">Loading...</span></div><p class="mt-2">Loading course details...</p></div>';
      
      // Show the modal
      console.log('Showing course modal');
      if (typeof bootstrap !== 'undefined') {
        try {
          const courseModal = new bootstrap.Modal(document.getElementById('courseModal'));
          courseModal.show();
        } catch (error) {
          console.error('Error showing Bootstrap modal:', error);
          showModal('courseModal');
        }
      } else {
        showModal('courseModal');
      }
      
      // Find course data from the server
      findCourseData(courseId).then(courseData => {
        console.log('Course data received:', courseData);
        if (!courseData) {
          modalBody.innerHTML = '<div class="alert alert-danger">Course not found</div>';
          return;
        }

        currentCoursePrice = parseFloat(courseData.price) || 0;

        // Populate modal content
        modalBody.innerHTML = `
        <div class="row">
          <div class="col-md-4">
            <div class="course-image">
              ${courseData.thumbnail ? 
                `<img src="../${courseData.thumbnail}" class="img-fluid rounded" alt="${courseData.title}">` :
                `<div class="bg-light d-flex align-items-center justify-content-center rounded" style="height: 200px;">
                  <i class="fi fi-rr-graduation-cap text-muted" style="font-size: 3rem;"></i>
                </div>`
              }
            </div>
          </div>
          <div class="col-md-8">
            <h4>${courseData.title}</h4>
            <p class="text-muted">${courseData.description || 'No description available'}</p>
            
            <div class="row mb-3">
              <div class="col-sm-6">
                <strong>Level:</strong> ${courseData.level ? courseData.level.charAt(0).toUpperCase() + courseData.level.slice(1) : 'Beginner'}
              </div>
              <div class="col-sm-6">
                <strong>Duration:</strong> ${courseData.duration || 'Not specified'}
              </div>
            </div>
            
            <div class="row mb-3">
              <div class="col-sm-6">
                <strong>Instructor:</strong> ${courseData.instructor || 'TBA'}
              </div>
              <div class="col-sm-6">
                <strong>Price:</strong> <span class="text-primary fw-bold">$${parseFloat(courseData.price || 0).toFixed(2)}</span>
              </div>
            </div>

            ${courseData.curriculum ? `
              <div class="curriculum">
                <h6>What You'll Learn:</h6>
                <div class="curriculum-content">${courseData.curriculum}</div>
              </div>
            ` : ''}
          </div>
        </div>
      `;

        // Show/hide purchase button based on enrollment status
        const purchaseBtn = document.getElementById('purchaseCourseBtn');
        if (courseData.isEnrolled) {
          purchaseBtn.style.display = 'none';
        } else {
          purchaseBtn.style.display = 'inline-block';
          purchaseBtn.onclick = () => showPaymentModal(courseData);
        }
        
        // If this is an enrollment request (not just viewing), show payment modal directly
        if (window.location.hash === '#enroll') {
          setTimeout(() => {
            showPaymentModal(courseData);
            // Remove the hash to prevent showing payment modal on page refresh
            window.history.replaceState(null, null, window.location.pathname);
          }, 500);
        }
      });
    }

    // Find course data from the page
    function findCourseData(courseId) {
      // Fetch course details from server
      return fetch(`get_course_details.php?id=${courseId}`)
        .then(response => response.json())
        .then(data => {
          if (data.success) {
            return data.course;
          } else {
            throw new Error(data.message || 'Failed to load course details');
          }
        })
        .catch(error => {
          console.error('Error loading course details:', error);
          alert('Failed to load course details: ' + error.message);
          return null;
        });
    }

    // Show payment modal
    function showPaymentModal(courseData) {
      const paymentModalElement = document.getElementById('paymentModal');
      const courseInfo = document.getElementById('paymentCourseInfo');
      
      if (!paymentModalElement || !courseInfo) {
        console.error('Payment modal elements not found');
        alert('Payment modal not available. Please refresh the page and try again.');
        return;
      }
      
      courseInfo.innerHTML = `
        <div class="course-item">
          <h6>${courseData.title}</h6>
          <p class="text-muted mb-2">${courseData.description || 'No description available'}</p>
          <div class="d-flex justify-content-between">
            <span>Level: ${courseData.level ? courseData.level.charAt(0).toUpperCase() + courseData.level.slice(1) : 'Beginner'}</span>
            <span class="fw-bold text-primary">$${parseFloat(courseData.price || 0).toFixed(2)}</span>
          </div>
        </div>
      `;

      // Update payment amounts in the modal
      const exchangeRate = <?php 
        require_once __DIR__ . '/config/exchange_rate.php';
        try {
          $exchange_rate_info = getExchangeRateWithInfo();
          $exchange_rate = $exchange_rate_info['rate'] ?? 1500;
          echo json_encode($exchange_rate);
        } catch (Exception $e) {
          echo json_encode(1500);
        }
      ?>;
      
      const coursePrice = parseFloat(courseData.price || 0);
      const amountNGN = coursePrice * exchangeRate;
      
      // Update payment display
      document.getElementById('paymentAmountUSD').textContent = `$${coursePrice.toFixed(2)}`;
      document.getElementById('paymentAmountNGN').textContent = `₦${amountNGN.toLocaleString()}`;

      // Initialize modal with proper configuration
      try {
        if (typeof bootstrap !== 'undefined') {
          const paymentModal = new bootstrap.Modal(paymentModalElement, {
            backdrop: true,
            keyboard: true,
            focus: true
          });
          
          paymentModal.show();
        } else {
          // Use fallback modal
          showModal('paymentModal');
        }
      } catch (error) {
        console.error('Error initializing payment modal:', error);
        // Fallback: show modal using jQuery if available
        if (typeof $ !== 'undefined') {
          $(paymentModalElement).modal('show');
        } else {
          // Use custom fallback
          showModal('paymentModal');
        }
      }
    }

    // Handle payment method selection
    document.getElementById('paystackBtn').addEventListener('click', function() {
      if (currentCourseId) {
        console.log('Paystack payment initiated for course:', currentCourseId);
        initializePaystackPayment();
      }
    });

    // Process payment (only Paystack now)
    function processPayment(method) {
      if (method === 'paystack') {
        // Initialize Paystack payment
        initializePaystackPayment();
      }
    }

    // Initialize Paystack payment
    function initializePaystackPayment() {
      console.log('Initializing Paystack payment...');
      console.log('PaystackPop available:', typeof PaystackPop !== 'undefined');
      
      if (typeof PaystackPop === 'undefined') {
        alert('Paystack is not loaded. Please refresh the page and try again.');
        return;
      }
      
      // Get exchange rate from server to ensure consistency with payment_handler.php
      // Use the same exchange rate that was displayed in the modal
      const exchangeRate = <?php 
        require_once __DIR__ . '/config/exchange_rate.php';
        try {
          $exchange_rate_info = getExchangeRateWithInfo();
          $exchange_rate = $exchange_rate_info['rate'] ?? 1500;
          echo json_encode($exchange_rate);
        } catch (Exception $e) {
          echo json_encode(1500);
        }
      ?>;
      const amountInNGN = currentCoursePrice * exchangeRate;
      const amountInKobo = Math.round(amountInNGN * 100); // Convert to kobo
      
      const email = '<?php echo $user['email']; ?>';
      const reference = 'COURSE_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
      
      console.log('Paystack payment details:', {
        coursePrice: currentCoursePrice,
        exchangeRate: exchangeRate,
        amountInNGN: amountInNGN,
        amountInKobo: amountInKobo,
        email: email,
        reference: reference
      });
      
      try {
        const handler = PaystackPop.setup({
          key: '<?php 
            require_once __DIR__ . '/config/payment_config.php';
            $payment_config = include __DIR__ . '/config/payment_config.php';
            echo $payment_config['paystack']['public_key'];
          ?>',
          email: email,
          amount: amountInKobo,
          currency: 'NGN',
          ref: reference,
          callback: function(response) {
            console.log('Paystack payment successful:', response);
            console.log('Payment reference from Paystack:', response.reference);
            handlePaymentSuccess(response.reference);
          },
          onClose: function() {
            console.log('Payment cancelled by user');
          }
        });
        
        handler.openIframe();
      } catch (error) {
        console.error('Error initializing Paystack payment:', error);
        alert('Payment initialization failed. Please try again.');
      }
    }

    // Handle course enrollment button click
    function enrollInCourse(courseId) {
      console.log('Enrolling in course:', courseId);
      
      // Show loading state
      const button = event.target;
      const originalText = button.innerHTML;
      button.innerHTML = '<i class="fi fi-rr-spinner me-1"></i>Loading...';
      button.disabled = true;
      
      // Send enrollment request
      fetch('course.php', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: `action=enroll&course_id=${courseId}`
      })
      .then(response => response.json())
      .then(data => {
        if (data.success) {
          // Set hash to indicate this is an enrollment request
          window.location.hash = '#enroll';
          // Load course details and show payment modal
          loadCourseDetails(courseId);
        } else {
          alert('Error: ' + data.message);
          button.innerHTML = originalText;
          button.disabled = false;
        }
      })
      .catch(error => {
        console.error('Error enrolling in course:', error);
        alert('Error enrolling in course. Please try again.');
        button.innerHTML = originalText;
        button.disabled = false;
      });
    }

    // Handle successful payment
    function handlePaymentSuccess(reference) {
      console.log('Processing payment success for reference:', reference);
      console.log('Reference type:', typeof reference);
      console.log('Reference value:', reference);
      
      // Send payment verification to server
      fetch('process_course_payment.php', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          course_id: currentCourseId,
          reference: reference,
          amount: currentCoursePrice
        })
      })
      .then(response => response.json())
      .then(data => {
        console.log('Payment processing response:', data);
        
        if (data.success) {
          // Close modals properly
          if (typeof bootstrap !== 'undefined') {
            const paymentModalElement = document.getElementById('paymentModal');
            const courseModalElement = document.getElementById('courseModal');
            
            if (paymentModalElement) {
              const paymentModal = bootstrap.Modal.getInstance(paymentModalElement);
              if (paymentModal) {
                paymentModal.hide();
              }
            }
            
            if (courseModalElement) {
              const courseModal = bootstrap.Modal.getInstance(courseModalElement);
              if (courseModal) {
                courseModal.hide();
              }
            }
          } else {
            // Use fallback modal closing
            hideModal('paymentModal');
            hideModal('courseModal');
          }
          
          // Show success message
          alert('Payment successful! You are now enrolled in the course.');
          
          // Redirect to my courses page
          window.location.href = 'my_courses.php';
        } else {
          alert('Payment verification failed: ' + data.message);
        }
      })
      .catch(error => {
        console.error('Error processing payment:', error);
        alert('Payment processing error. Please contact support.');
      });
    }
  </script>

  <!-- WhatsApp Chat Button -->
  <div class="whatsapp-chat-button">
    <a href="https://wa.me/2347069378861?text=Hello%20Omateq%20Digitals!%20I%20would%20like%20to%20know%20more%20about%20your%20services." 
       target="_blank" 
       rel="noopener noreferrer"
       class="whatsapp-link"
       title="Chat with us on WhatsApp">
      <i class="fab fa-whatsapp"></i>
      <span class="whatsapp-text">Chat with us</span>
    </a>
  </div>
  
  <style>
  /* WhatsApp Chat Button Styles */
  .whatsapp-chat-button {
    position: fixed;
    bottom: 20px;
    right: 20px;
    z-index: 1000;
    animation: pulse 2s infinite;
  }
  
  .whatsapp-link {
    display: flex;
    align-items: center;
    background: #25D366;
    color: white;
    text-decoration: none;
    padding: 12px 16px;
    border-radius: 50px;
    box-shadow: 0 4px 12px rgba(37, 211, 102, 0.4);
    transition: all 0.3s ease;
    font-family: 'Poppins', sans-serif;
    font-weight: 500;
    font-size: 14px;
  }
  
  .whatsapp-link:hover {
    background: #128C7E;
    color: white;
    text-decoration: none;
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(37, 211, 102, 0.6);
  }
  
  .whatsapp-link i {
    font-size: 24px;
    margin-right: 8px;
  }
  
  .whatsapp-text {
    white-space: nowrap;
  }
  
  @keyframes pulse {
    0% {
      box-shadow: 0 0 0 0 rgba(37, 211, 102, 0.7);
    }
    70% {
      box-shadow: 0 0 0 10px rgba(37, 211, 102, 0);
    }
    100% {
      box-shadow: 0 0 0 0 rgba(37, 211, 102, 0);
    }
  }
  
  /* Mobile responsiveness */
  @media (max-width: 768px) {
    .whatsapp-chat-button {
      bottom: 15px;
      right: 15px;
    }
    
    .whatsapp-link {
      padding: 10px 14px;
      font-size: 13px;
    }
    
    .whatsapp-link i {
      font-size: 22px;
      margin-right: 6px;
    }
    
    .whatsapp-text {
      display: none;
    }
  }
  </style>

  <!-- Paystack Script -->
  <script src="https://js.paystack.co/v1/inline.js"></script>

</body>
</html>
<?php
// End of PHP processing
?>