<?php
session_start();
require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/../admin-dashboard/config/settings.php';
require_once __DIR__ . '/../admin-dashboard/config/branding.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: authentication/login.php');
    exit();
}

// Get course and lesson IDs from URL
$course_id = (int)($_GET['course_id'] ?? 0);
$lesson_id = (int)($_GET['lesson_id'] ?? 0);

// Initialize variables
$course = null;
$user = null;
$lessons = [];
$current_lesson = null;
$lesson_progress = [];
$error = null;

try {
    $pdo = getConnection();
    
    // Get course details
    if ($course_id) {
        $stmt = $pdo->prepare("SELECT * FROM courses WHERE id = ? AND status = 'active'");
        $stmt->execute([$course_id]);
        $course = $stmt->fetch();
        
        // Fix instructor field if it contains course title instead of instructor name
        if ($course && $course['instructor'] === $course['title']) {
            $course['instructor'] = 'Omateq Digitals Team';
        }
        
        if (!$course) {
            $error = "Course not found or not available.";
        }
    } else {
        $error = "Course ID is required.";
    }
    
    // Get user details
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $user = $stmt->fetch();
    
    if (!$user) {
        $error = "User not found. Please log in again.";
    }
    
    // Check if user is enrolled in the course
    if ($course) {
        $stmt = $pdo->prepare("SELECT * FROM course_enrollments WHERE user_id = ? AND course_id = ?");
        $stmt->execute([$_SESSION['user_id'], $course_id]);
        $enrollment = $stmt->fetch();
        
        if (!$enrollment) {
            $error = "You are not enrolled in this course.";
        }
    }

} catch (Exception $e) {
    error_log("Database connection error: " . $e->getMessage());
    error_log("Stack trace: " . $e->getTraceAsString());
    $error = "Database connection error: " . $e->getMessage();
}

// Get course lessons
if ($course && !$error) {
    try {
        // Get all lessons for this course
        $stmt = $pdo->prepare("SELECT * FROM course_lessons WHERE course_id = ? AND is_active = 'yes' ORDER BY lesson_number");
        $stmt->execute([$course_id]);
        $lessons = $stmt->fetchAll();
        
        // Get user's progress for each lesson
        if (!empty($lessons)) {
            $lesson_ids = array_column($lessons, 'id');
            $placeholders = str_repeat('?,', count($lesson_ids) - 1) . '?';
            $stmt = $pdo->prepare("SELECT lesson_id, is_completed, video_current_time as watched_duration, last_updated as last_watched_at FROM video_progress WHERE user_id = ? AND lesson_id IN ($placeholders)");
            $stmt->execute(array_merge([$_SESSION['user_id']], $lesson_ids));
            $progress_data = $stmt->fetchAll();
            
            foreach ($progress_data as $progress) {
                $lesson_progress[$progress['lesson_id']] = $progress;
            }
        }
        
        // Determine current lesson
        if ($lesson_id) {
            // Use specific lesson from URL
            $stmt = $pdo->prepare("SELECT * FROM course_lessons WHERE id = ? AND course_id = ? AND is_active = 'yes'");
            $stmt->execute([$lesson_id, $course_id]);
            $current_lesson = $stmt->fetch();
            
            if (!$current_lesson) {
                $error = "Lesson not found or not available.";
            }
        } else {
            // Auto-select lesson: first incomplete lesson
            if (!empty($lessons)) {
                foreach ($lessons as $lesson) {
                    if (!isset($lesson_progress[$lesson['id']]) || !$lesson_progress[$lesson['id']]['is_completed']) {
                        $current_lesson = $lesson;
                        break;
                    }
                }
                
                // If all lessons completed, show last lesson
                if (!$current_lesson) {
                    $current_lesson = end($lessons);
                }
            }
        }
        
        // Check if course is completed
        $course_completed = false;
        $completed_lessons = 0;
        if (!empty($lessons)) {
            foreach ($lessons as $lesson) {
                if (isset($lesson_progress[$lesson['id']]) && $lesson_progress[$lesson['id']]['is_completed']) {
                    $completed_lessons++;
                }
            }
            $course_completed = ($completed_lessons === count($lessons));
        }
        
    } catch (Exception $e) {
        error_log("Lessons query error: " . $e->getMessage());
        error_log("Stack trace: " . $e->getTraceAsString());
        $error = "Error loading course lessons: " . $e->getMessage();
    }
}
?>
<!DOCTYPE html>
<html lang="en">

<head>

  <!-- begin::Edura Meta Basic -->
  <meta charset="utf-8">
  <meta name="theme-color" content="#0D9488">
  <meta name="robots" content="index, follow">
  <meta name="author" content="LayoutDrop">
  <meta name="format-detection" content="telephone=no">
  <meta name="keywords" content="Course Learning, Video Lessons, Online Education, eLearning Platform">
  <meta name="description" content="Learn with our comprehensive video courses and track your progress in real-time.">
  <!-- end::Edura Meta Basic -->

  <!-- begin::Edura Website Page Title -->
  <title><?php echo $course ? htmlspecialchars($course['title']) : 'Course Not Found'; ?> - Omateq Digitals</title>
  <!-- end::Edura Website Page Title -->

  <!-- begin::Edura Mobile Specific -->
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <!-- end::Edura Mobile Specific -->

  <!-- begin::Edura Favicon Tags -->
  <?php echo getFaviconHtml(); ?>
  <!-- end::Edura Favicon Tags -->

  <!-- begin::Edura Google Fonts -->
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Plus+Jakarta+Sans:ital,wght@0,200..800;1,200..800&display=swap" rel="stylesheet">
  <!-- end::Edura Google Fonts -->

  <!-- begin::Edura Required Stylesheet -->
  <link rel="stylesheet" href="assets/libs/flaticon/css/all/all.css">
  <link rel="stylesheet" href="assets/libs/lucide/lucide.css">
  <link rel="stylesheet" href="assets/libs/fontawesome/css/all.min.css">
  <link rel="stylesheet" href="assets/libs/simplebar/simplebar.css">
  <link rel="stylesheet" href="assets/libs/node-waves/waves.css">
  <link rel="stylesheet" href="assets/libs/bootstrap-select/css/bootstrap-select.min.css">
  <!-- end::Edura Required Stylesheet -->

  <!-- begin::Edura CSS Stylesheet -->
  <link rel="stylesheet" href="assets/css/styles.css">
  <link rel="stylesheet" href="assets/css/custom-video-player.css">
  <!-- end::Edura CSS Stylesheet -->
  
  <style>
    /* Logo sizing for better fit */
    .app-navbar-brand .navbar-brand-logo img {
      max-height: 50px;
      max-width: 180px;
      width: auto;
      height: auto;
      object-fit: contain;
    }
    
    .app-navbar-brand .navbar-brand-mini img {
      max-height: 35px;
      max-width: 35px;
      width: auto;
      height: auto;
      object-fit: contain;
    }
  </style>
    
    <style>
        .video-thumbnail-container {
            transition: transform 0.3s ease;
        }
        .video-thumbnail-container:hover {
            transform: scale(1.02);
        }
        .lesson-preview {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 12px;
            overflow: hidden;
        }
        .lesson-list-item {
            transition: all 0.3s ease;
            cursor: pointer;
        }
        .lesson-list-item:hover {
            transform: translateX(5px);
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        .avatar {
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
        }
        .avatar-sm {
            width: 2rem;
            height: 2rem;
            font-size: 0.875rem;
        }
        .notification {
            position: fixed;
            top: 20px;
            right: 20px;
            z-index: 1050;
            min-width: 300px;
        }
        .video-progress {
            background: rgba(0, 0, 0, 0.8);
            border-radius: 8px;
            padding: 1rem;
        }
        /* Modal progress styling */
        .modal .video-progress {
            background: rgba(108, 117, 125, 0.9);
            border-radius: 0;
            padding: 1rem;
            margin: 0;
        }
        .modal .progress {
            background: rgba(255, 255, 255, 0.2);
        }
        .modal .progress-bar {
            background: linear-gradient(90deg, #007bff, #0056b3);
            transition: width 0.3s ease;
        }
    </style>
</head>

<body>
  <div class="page-layout">

    <!-- begin::Edura Page Header -->
    <header class="app-header">
      <div class="app-header-inner">
        <button class="app-toggler" type="button">
          <span></span>
          <span></span>
          <span></span>
        </button>
        <div class="app-header-start">
          <form class="d-none d-md-flex align-items-center h-100 w-lg-250px w-xxl-300px position-relative" action="index.html#">
            <button type="button" class="btn btn-sm border-0 position-absolute start-0 ms-3 p-0">
              <i class="fi fi-rr-search"></i>
            </button>
            <input type="text" class="form-control ps-5" placeholder="Search anything's">
          </form>
        </div>
        <div class="app-header-end">
          <div class="px-lg-3 px-2 ps-0 d-flex align-items-center">
            <a href="javascript:void(0);" class="theme-btn">
              <i class="fi fi-rr-brightness icon-light"></i>
              <div class="theme-toggle"></div>
              <i class="fi fi-rr-moon icon-dark"></i>
            </a>
          </div>
          <div class="vr my-3"></div>
          <div class="d-flex align-items-center gap-sm-2 gap-0 px-lg-4 px-sm-2 px-1">
            <div class="dropdown text-end">
              <a href="javascript:void(0);" class="btn btn-icon btn-action-gray rounded-circle waves-effect waves-light position-relative" data-bs-toggle="dropdown" data-bs-auto-close="outside" aria-expanded="true">
                <i class="fi fi-rr-envelope"></i>
              </a>
              <div class="dropdown-menu dropdown-menu-lg-end p-0 w-300px mt-2">
                <div class="px-3 py-3 border-bottom d-flex justify-content-between align-items-center">
                  <h6 class="mb-0">Notifications</h6>
                  <i class="bi bi-x-lg cursor-pointer"></i>
                </div>
                <div class="p-2" style="height: 300px;" data-simplebar>
                  <div class="text-center py-4">
                    <i class="fi fi-rr-envelope text-muted" style="font-size: 2rem;"></i>
                    <p class="text-muted mt-2 mb-0">No notifications yet</p>
                  </div>
                </div>
              </div>
            </div>
            <div class="dropdown text-end">
              <a href="javascript:void(0);" class="btn btn-icon btn-action-gray rounded-circle waves-effect waves-light" data-bs-toggle="dropdown" data-bs-auto-close="outside" aria-expanded="true">
                <i class="fi fi-rr-bell"></i>
              </a>
              <div class="dropdown-menu dropdown-menu-lg-end p-0 w-300px mt-2">
                <div class="px-3 py-3 border-bottom d-flex justify-content-between align-items-center">
                  <h6 class="mb-0">Updates</h6>
                  <i class="bi bi-x-lg cursor-pointer"></i>
                </div>
                <div class="p-2" style="height: 300px;" data-simplebar>
                  <div class="text-center py-4">
                    <i class="fi fi-rr-bell text-muted" style="font-size: 2rem;"></i>
                    <p class="text-muted mt-2 mb-0">No updates yet</p>
                  </div>
                </div>
              </div>
            </div>
          </div>
          <div class="vr my-3"></div>
          <div class="dropdown text-end ms-2">
            <a href="javascript:void(0);" class="d-flex align-items-center py-2" data-bs-toggle="dropdown" data-bs-auto-close="outside" aria-expanded="true">
              <div class="avatar avatar-sm rounded-circle avatar-status-success">
                <?php if ($user && $user['avatar'] && file_exists($user['avatar'])): ?>
                  <img src="<?php echo htmlspecialchars($user['avatar']); ?>" alt="<?php echo htmlspecialchars($user['full_name']); ?>">
                <?php else: ?>
                  <img src="assets/images/avatar/avatar1.webp" alt="<?php echo htmlspecialchars($user ? $user['full_name'] : 'User'); ?>">
                <?php endif; ?>
              </div>
            </a>
            <ul class="dropdown-menu dropdown-menu-end w-225px mt-1">
              <li class="d-flex align-items-center p-2">
                <div class="avatar avatar-sm rounded-circle">
                  <?php if ($user && $user['avatar'] && file_exists($user['avatar'])): ?>
                    <img src="<?php echo htmlspecialchars($user['avatar']); ?>" alt="<?php echo htmlspecialchars($user['full_name']); ?>">
                  <?php else: ?>
                    <img src="assets/images/avatar/avatar1.webp" alt="<?php echo htmlspecialchars($user ? $user['full_name'] : 'User'); ?>">
                  <?php endif; ?>
                </div>
                <div class="ms-2">
                  <div class="fw-bold text-dark"><?php echo htmlspecialchars($user ? $user['full_name'] : 'User'); ?></div>
                  <small class="text-body d-block lh-sm"><?php echo htmlspecialchars($user ? $user['email'] : 'user@example.com'); ?></small>
                </div>
              </li>
              <li>
                <div class="dropdown-divider my-1"></div>
              </li>
              <li>
                <a class="dropdown-item d-flex align-items-center gap-2" href="profile.php">
                  <i class="fi fi-rr-user scale-1x"></i> Profile
                </a>
              </li>
              <li>
                <a class="dropdown-item d-flex align-items-center gap-2" href="my_courses.php">
                  <i class="fi fi-rr-book scale-1x"></i> My Courses
                </a>
              </li>
              <li>
                <div class="dropdown-divider my-1"></div>
              </li>
              <li>
                <a class="dropdown-item d-flex align-items-center gap-2 text-danger" href="authentication/logout.php">
                  <i class="fi fi-sr-exit scale-1x"></i> Logout
                </a>
              </li>
            </ul>
          </div>
        </div>
      </div>
    </header>
    <!-- end::Edura Page Header -->

    <!-- begin::Edura Sidebar Menu -->
    <aside class="app-menubar" id="menubar">
      <div class="app-navbar-brand">
        <a class="navbar-brand-logo" href="index.php">
          <?php echo '<img src="' . '../' . getSetting('main_logo', 'assets/images/logo.svg') . '" alt="logo">'; ?>
        </a>
        <a class="navbar-brand-mini" href="index.php">
          <?php echo '<img src="' . '../' . getSetting('mini_logo', 'assets/images/logo-text-white.svg') . '" alt="logo">'; ?>
        </a>
      </div>
      <nav class="app-navbar" data-simplebar>
        <ul class="menubar">
              <li class="menu-item">
                <a class="menu-link" href="index.php">
              <i class="fi fi-rr-apps"></i>
                  <span class="menu-label">Dashboard</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="products.php">
              <i class="fi fi-rr-users"></i>
              <span class="menu-label">Products</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="course.php">
              <i class="fi fi-rr-graduation-cap"></i>
              <span class="menu-label">Course</span>
                </a>
              </li>
              <li class="menu-item active">
            <a class="menu-link" href="my_courses.php">
              <i class="fi fi-rr-book"></i>
              <span class="menu-label">My Courses</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="profile.php">
              <i class="fi fi-rr-receipt"></i>
              <span class="menu-label">Profile</span>
                </a>
              </li>
              
        </ul>
      </nav>
      <div class="app-footer">
        <div class="text-center py-2">
          <small class="text-white-50">
            <i class="fi fi-rr-copyright me-1"></i>
            2025 Omateq Digitals
          </small>
        </div>
        <a href="mailto:info@omateqdigitals.com" class="btn btn-outline-white waves-effect btn-app-nav w-100">
          <i class="fi fi-rs-interrogation"></i>
          <span class="nav-text">Help and Support</span>
        </a>
      </div>
    </aside>
    <!-- end::Edura Sidebar Menu -->

    <main class="app-wrapper">

      <div class="container-fluid">

        <!-- Page Header -->
        <div class="app-page-head d-flex flex-wrap gap-3 align-items-center justify-content-between">
          <div class="clearfix">
            <h1 class="app-page-title">Course Learning</h1>
            <span>Start your learning journey</span>
          </div>
          <div class="d-flex gap-2">
            <a href="my_courses.php" class="btn btn-outline-primary waves-effect btn-shadow">
              <i class="fi fi-rr-arrow-left me-1"></i>
              Back to My Courses
            </a>
          </div>
        </div>

        <?php if ($error): ?>
            <div class="alert alert-danger">
                <i class="fi fi-rr-exclamation-triangle me-2"></i>
                <?php echo htmlspecialchars($error); ?>
            </div>
        <?php else: ?>
            
            <!-- Course Info Cards -->
            <div class="row mb-4">
                <div class="col-xxl-3 col-md-6">
                    <div class="card">
                        <div class="card-body">
                            <div class="d-flex flex-column mb-3">
                                <h6 class="text-body text-uppercase text-2xs mb-1">Total Lessons</h6>
                                <h2 class="mb-0"><?php echo count($lessons); ?></h2>
                            </div>
                            <div class="avatar bg-primary bg-opacity-05 shadow-info rounded-circle avatar-xl text-primary mb-0 mx-auto">
                                <i class="fi fi-rr-book"></i>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-xxl-3 col-md-6">
                    <div class="card">
                        <div class="card-body">
                            <div class="d-flex flex-column mb-3">
                                <h6 class="text-body text-uppercase text-2xs mb-1">Course Duration</h6>
                                <h2 class="mb-0"><?php echo $course['duration'] ?? 'Self-paced'; ?></h2>
                            </div>
                            <div class="avatar bg-success bg-opacity-05 shadow-success rounded-circle avatar-xl text-success mb-0 mx-auto">
                                <i class="fi fi-rr-clock"></i>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-xxl-3 col-md-6">
                    <div class="card">
                        <div class="card-body">
                            <div class="d-flex flex-column mb-3">
                                <h6 class="text-body text-uppercase text-2xs mb-1">Level</h6>
                                <h2 class="mb-0"><?php echo ucfirst($course['level'] ?? 'Beginner'); ?></h2>
                            </div>
                            <div class="avatar bg-warning bg-opacity-05 shadow-warning rounded-circle avatar-xl text-warning mb-0 mx-auto">
                                <i class="fi fi-rr-graduation-cap"></i>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-xxl-3 col-md-6">
                    <div class="card">
                        <div class="card-body">
                            <div class="d-flex flex-column mb-3">
                                <h6 class="text-body text-uppercase text-2xs mb-1">Instructor</h6>
                                <h2 class="mb-0"><?php echo htmlspecialchars($course['instructor'] ?? 'Omateq Digitals Team'); ?></h2>
                            </div>
                            <div class="avatar bg-info bg-opacity-05 shadow-info rounded-circle avatar-xl text-info mb-0 mx-auto">
                                <i class="fi fi-rr-user"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

                <div class="row">
                    <!-- Lessons Sidebar -->
                    <div class="col-lg-4 mb-4">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0">
                                    <i class="fi fi-rr-play me-2"></i>
                                    Course Lessons
                                </h5>
                            </div>
                            <div class="card-body">
                                <!-- List View Only -->
                                <?php foreach ($lessons as $index => $lesson): ?>
                                    <?php 
                                    $is_completed = isset($lesson_progress[$lesson['id']]) && $lesson_progress[$lesson['id']]['is_completed'];
                                    $is_current = $current_lesson && $current_lesson['id'] == $lesson['id'];
                                    ?>
                                    <div class="lesson-list-item mb-3 p-3 border rounded <?php echo $is_current ? 'border-primary bg-light' : ''; ?>" 
                                         data-lesson-id="<?php echo $lesson['id']; ?>"
                                         onclick="loadLesson(<?php echo $lesson['id']; ?>)">
                                        <div class="d-flex align-items-center">
                                            <div class="lesson-number me-3">
                                                <div class="avatar avatar-sm rounded-circle <?php echo $is_completed ? 'bg-success' : ($is_current ? 'bg-primary' : 'bg-light text-dark'); ?>">
                                                    <?php if ($is_completed): ?>
                                                        <i class="fi fi-rr-check"></i>
                                                    <?php else: ?>
                                                        <?php echo $lesson['lesson_number']; ?>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                            <div class="flex-grow-1">
                                                <h6 class="mb-1"><?php echo htmlspecialchars($lesson['lesson_title']); ?></h6>
                                                <p class="text-muted small mb-2">
                                                    <?php echo htmlspecialchars(substr($lesson['lesson_description'] ?? '', 0, 100)) . '...'; ?>
                                                </p>
                                                <?php if (isset($lesson_progress[$lesson['id']]) && $lesson_progress[$lesson['id']]['watched_duration'] > 0): ?>
                                                    <div class="progress mb-2" style="height: 4px;">
                                                        <?php 
                                                        $duration_seconds = 0;
                                                        if (preg_match('/(\d+):(\d+)/', $lesson['duration'], $matches)) {
                                                            $duration_seconds = ($matches[1] * 60) + $matches[2];
                                                        }
                                                        $watched_seconds = $lesson_progress[$lesson['id']]['watched_duration'];
                                                        $progress_percentage = $duration_seconds > 0 ? ($watched_seconds / $duration_seconds) * 100 : 0;
                                                        ?>
                                                        <div class="progress-bar bg-primary" style="width: <?php echo min(100, $progress_percentage); ?>%"></div>
                                                    </div>
                                                    <small class="text-muted">
                                                        <i class="fi fi-rr-play me-1"></i>
                                                        Resumed at <?php echo gmdate('i:s', $watched_seconds); ?>
                                                    </small>
                                                <?php endif; ?>
                                            </div>
                                            <div class="text-end">
                                                <small class="text-muted me-3">
                                                    <i class="fi fi-rr-clock me-1"></i>
                                                    <?php echo $lesson['duration']; ?>
                                                </small>
                                                <?php if ($is_completed): ?>
                                                    <span class="badge bg-success lesson-status">Completed</span>
                                                <?php elseif ($is_current): ?>
                                                    <span class="badge bg-primary lesson-status">Current</span>
                                                <?php elseif (isset($lesson_progress[$lesson['id']]) && $lesson_progress[$lesson['id']]['watched_duration'] > 0): ?>
                                                    <span class="badge bg-warning lesson-status">In Progress</span>
                                                <?php else: ?>
                                                    <span class="badge bg-secondary lesson-status">Not Started</span>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>

                    <!-- Main Content -->
                    <div class="col-lg-8">
                        <?php if ($course_completed): ?>
                            <!-- Certificate of Completion -->
                            <div class="card border-success">
                                <div class="card-header bg-success text-white">
                                    <h4 class="mb-0 text-center">
                                        <i class="fi fi-rr-trophy me-2"></i>
                                        Congratulations! Course Completed
                                    </h4>
                                </div>
                                <div class="card-body">
                                    <div class="certificate-container text-center p-4" style="background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%); border: 3px solid #28a745; border-radius: 15px;">
                                        <div class="certificate-header mb-4">
                                            <h2 class="text-success mb-2" style="font-family: \\'Times New Roman\\', serif; font-weight: bold;">
                                                CERTIFICATE OF COMPLETION
                                            </h2>
                                            <div class="decorative-line" style="height: 3px; background: linear-gradient(90deg, #28a745, #20c997, #28a745); margin: 0 auto 20px; width: 200px; border-radius: 2px;"></div>
                                            <p class="text-muted mb-0" style="font-style: italic;">This certifies that</p>
                                        </div>
                                        
                                        <div class="student-name mb-4">
                                            <h3 class="text-dark mb-0" style="font-family: \\'Georgia\\', serif; font-weight: bold; text-transform: uppercase; letter-spacing: 1px;">
                                                <?php echo htmlspecialchars($user ? $user['full_name'] : 'Student'); ?>
                                            </h3>
                                        </div>
                                        
                                        <div class="certificate-body mb-4">
                                            <p class="text-dark mb-3" style="font-size: 1.1rem; line-height: 1.6;">
                                                has successfully completed the course
                                            </p>
                                            <h4 class="text-primary mb-3" style="font-family: \\'Georgia\\', serif; font-weight: bold;">
                                                "<?php echo htmlspecialchars($course ? $course['title'] : 'Course Not Found'); ?>"
                                            </h4>
                                            <p class="text-muted mb-0" style="font-size: 0.95rem;">
                                                on <?php echo date('F j, Y'); ?>
                                            </p>
                                        </div>
                                        
                                        <div class="certificate-footer mt-4">
                                            <div class="row">
                                                <div class="col-md-6 text-center">
                                                    <div class="signature-section">
                                                        <div class="signature-line" style="border-top: 2px solid #333; width: 150px; margin: 0 auto 5px;"></div>
                                                        <p class="text-dark mb-0" style="font-weight: bold;"><?php echo htmlspecialchars($course ? ($course['instructor'] ?? 'Omateq Digitals Team') : 'Omateq Digitals Team'); ?></p>
                                                        <small class="text-muted">Instructor</small>
                                                    </div>
                                                </div>
                                                <div class="col-md-6 text-center">
                                                    <div class="organization-section">
                                                        <div class="signature-image mb-2">
                                                            <img src="assets/images/omateq-signature.png" alt="Omateq Digitals Signature" style="max-height: 60px; max-width: 150px;" onerror="this.src='assets/images/logo-full.svg'; this.style.maxHeight='50px'; this.style.maxWidth='120px';">
                                                            <div class="signature-line" style="border-top: 2px solid #333; width: 150px; margin: 0 auto 5px; display: none;"></div>
                                                        </div>
                                                        <p class="text-dark mb-0" style="font-weight: bold;">Omateq Digitals</p>
                                                        <small class="text-muted">Training Organization</small>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <div class="certificate-seal mt-4">
                                            <div class="seal" style="width: 80px; height: 80px; border: 3px solid #28a745; border-radius: 50%; margin: 0 auto; display: flex; align-items: center; justify-content: center; background: linear-gradient(135deg, #fff 0%, #f8f9fa 100%); box-shadow: 0 4px 8px rgba(0,0,0,0.1);">
                                                <i class="fi fi-rr-trophy text-success" style="font-size: 2rem;"></i>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="text-center mt-4">
                                        <button class="btn btn-success btn-lg me-3" onclick="downloadCertificate()">
                                            <i class="fi fi-rr-download me-2"></i>
                                            Download Certificate
                                        </button>
                                        <a href="my_courses.php" class="btn btn-outline-primary btn-lg">
                                            <i class="fi fi-rr-arrow-left me-2"></i>
                                            Back to My Courses
                                        </a>
                                    </div>
                                </div>
                            </div>
                        <?php elseif ($current_lesson): ?>
                            <div class="card">
                                <div class="card-header">
                                    <!-- Course Title -->
                                    <div class="course-title-section mb-4">
                                        <div class="course-title-box" style="background-color: #ffffff; padding: 1rem 1.5rem; border-radius: 12px; border-left: 4px solid #3498db; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
                                            <h6 class="course-title" style="color: #2c3e50; font-weight: 600; margin-bottom: 0; text-align: center; font-size: 0.95rem; line-height: 1.2;">
                                                <?php echo htmlspecialchars($course ? $course['title'] : 'Course Not Found'); ?>
                                            </h6>
                                        </div>
                                    </div>
                                    
                                    <!-- Lesson Title -->
                                    <h5 class="mb-0" style="color: #ffffff; border-top: 2px solid rgba(255, 255, 255, 0.3); padding-top: 1rem;">
                                        <i class="fi fi-rr-play me-2" style="color: #ffffff;"></i>
                                        Lesson <?php echo $current_lesson['lesson_number']; ?>: <?php echo htmlspecialchars($current_lesson['lesson_title']); ?>
                                    </h5>
                                </div>
                                <div class="card-body">
                                    <!-- Lesson Preview -->
                                    <div class="lesson-preview mb-4">
                                        <!-- Video Thumbnail (when not playing) -->
                                        <div id="videoThumbnail" class="video-thumbnail-container position-relative" style="cursor: pointer;" 
                                             data-lesson-id="<?php echo $current_lesson['id']; ?>"
                                             data-lesson-title="<?php echo htmlspecialchars($current_lesson['lesson_title']); ?>"
                                             data-video-type="<?php echo $current_lesson['video_type'] ?? 'upload'; ?>"
                                             data-video-file="<?php echo htmlspecialchars($current_lesson['video_file'] ?? ''); ?>">
                                            <div class="ratio ratio-16x9 bg-dark rounded">
                                                <div class="d-flex align-items-center justify-content-center">
                                                    <div class="text-center text-white">
                                                        <i class="fi fi-rr-play" style="font-size: 4rem;"></i>
                                                        <h5 class="mt-3">Click to Start Learning</h5>
                                                        <p class="text-muted">Duration: <?php echo $current_lesson['duration']; ?></p>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <!-- Video Player (when playing) -->
                                        <div id="inlineVideoPlayer" class="d-none">
                                            <div class="ratio ratio-16x9 bg-dark rounded">
                                                <div id="videoPlayerContainer"></div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="lesson-actions mt-4">
                                        <div class="d-flex justify-content-between align-items-center">
                                            <div>
                                                <button id="startLearningBtn" class="btn btn-primary me-2" 
                                                        data-lesson-id="<?php echo $current_lesson['id']; ?>"
                                                        data-lesson-title="<?php echo htmlspecialchars($current_lesson['lesson_title']); ?>"
                                                        data-video-type="<?php echo $current_lesson['video_type'] ?? 'upload'; ?>"
                                                        data-video-file="<?php echo htmlspecialchars($current_lesson['video_file'] ?? ''); ?>">
                                                    <i class="fi fi-rr-play me-1"></i> Start Learning
                                                </button>
                                                <button class="btn btn-success" onclick="markLessonComplete(<?php echo $current_lesson['id']; ?>)">
                                                    <i class="fi fi-rr-check me-1"></i> Mark as Complete
                                                </button>
                                            </div>
                                            <div>
                                                <?php if ($current_lesson['lesson_number'] > 1): ?>
                                                    <a href="?course_id=<?php echo $course_id; ?>&lesson_id=<?php echo $lessons[$current_lesson['lesson_number'] - 2]['id']; ?>" 
                                                       class="btn btn-outline-secondary me-2">
                                                        <i class="fi fi-rr-arrow-left me-1"></i> Previous
                                                    </a>
                                                <?php endif; ?>
                                                <?php if ($current_lesson['lesson_number'] < count($lessons)): ?>
                                                    <a href="?course_id=<?php echo $course_id; ?>&lesson_id=<?php echo $lessons[$current_lesson['lesson_number']]['id']; ?>" 
                                                       class="btn btn-primary">
                                                        Next <i class="fi fi-rr-arrow-right ms-1"></i>
                                                    </a>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="mt-4">
                                        <h6 style="color: #ffffff; font-weight: 500;">Lesson Description</h6>
                                        <div class="lesson-description" style="background-color: #f8f9fa; padding: 1rem; border-radius: 8px; border-left: 4px solid #3498db;">
                                            <p class="text-muted">Lesson content will be displayed here once available.</p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php else: ?>
                            <div class="card">
                                <div class="card-body text-center">
                                    <h5>No Lesson Selected</h5>
                                    <p class="text-muted">Please select a lesson from the sidebar to begin learning.</p>
                                    <?php if (!empty($lessons)): ?>
                                        <button id="startFirstLessonBtn" class="btn btn-primary" 
                                                data-lesson-id="<?php echo $lessons[0]['id']; ?>"
                                                data-lesson-title="<?php echo htmlspecialchars($lessons[0]['lesson_title']); ?>"
                                                data-video-type="<?php echo $lessons[0]['video_type'] ?? 'upload'; ?>"
                                                data-video-file="<?php echo htmlspecialchars($lessons[0]['video_file'] ?? ''); ?>">
                                            <i class="fi fi-rr-play me-1"></i> Start Learning
                                        </button>
                                    <?php endif; ?>
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endif; ?>
        </div>

      </div>

    </main>

    <!-- begin::Edura Footer -->
    <footer class="footer-wrapper bg-body text-center">
      <div class="container-fluid">
        <p class="mb-0">© <span class="currentYear">2025</span> Omateq Digitals. Proudly powered by Omateq Digitals Ltd.</p>
      </div>
    </footer>
    <!-- end::Edura Footer -->

    <!-- WhatsApp Chat Button -->
    <div class="whatsapp-chat-button">
      <a href="https://wa.me/2347069378861?text=Hello%20Omateq%20Digitals!%20I%20would%20like%20to%20know%20more%20about%20your%20services." 
         target="_blank" 
         rel="noopener noreferrer"
         class="whatsapp-link"
         title="Chat with us on WhatsApp">
        <i class="fab fa-whatsapp"></i>
        <span class="whatsapp-text">Chat with us</span>
      </a>
    </div>
    
    <style>
    /* WhatsApp Chat Button Styles */
    .whatsapp-chat-button {
      position: fixed;
      bottom: 20px;
      right: 20px;
      z-index: 1000;
      animation: pulse 2s infinite;
    }
    
    .whatsapp-link {
      display: flex;
      align-items: center;
      background: #25D366;
      color: white;
      text-decoration: none;
      padding: 12px 16px;
      border-radius: 50px;
      box-shadow: 0 4px 12px rgba(37, 211, 102, 0.4);
      transition: all 0.3s ease;
      font-family: 'Poppins', sans-serif;
      font-weight: 500;
      font-size: 14px;
    }
    
    .whatsapp-link:hover {
      background: #128C7E;
      color: white;
      text-decoration: none;
      transform: translateY(-2px);
      box-shadow: 0 6px 20px rgba(37, 211, 102, 0.6);
    }
    
    .whatsapp-link i {
      font-size: 24px;
      margin-right: 8px;
    }
    
    .whatsapp-text {
      white-space: nowrap;
    }
    
    @keyframes pulse {
      0% {
        box-shadow: 0 0 0 0 rgba(37, 211, 102, 0.7);
      }
      70% {
        box-shadow: 0 0 0 10px rgba(37, 211, 102, 0);
      }
      100% {
        box-shadow: 0 0 0 0 rgba(37, 211, 102, 0);
      }
    }
    
    /* Mobile responsiveness */
    @media (max-width: 768px) {
      .whatsapp-chat-button {
        bottom: 15px;
        right: 15px;
      }
      
      .whatsapp-link {
        padding: 10px 14px;
        font-size: 13px;
      }
      
      .whatsapp-link i {
        font-size: 22px;
        margin-right: 6px;
      }
      
      .whatsapp-text {
        display: none;
      }
    }
    </style>

  </div>

    <!-- Video Modal -->
    <div class="modal fade" id="videoModal" tabindex="-1" aria-labelledby="videoModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-xl modal-dialog-centered">
            <div class="modal-content bg-dark">
                <div class="modal-header border-secondary">
                    <h5 class="modal-title text-white" id="videoModalTitle">Lesson Video</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body p-0">
                    <!-- Progress Display Above Video -->
                    <div class="video-progress p-3 bg-secondary">
                        <div class="d-flex justify-content-between align-items-center mb-2">
                            <span class="text-white">Progress: <span id="modalProgressText">0%</span></span>
                            <span class="text-white">Time: <span id="modalTimeText">0:00 / 0:00</span></span>
                        </div>
                        <div class="progress" style="height: 8px;">
                            <div id="modalProgressBar" class="progress-bar bg-primary" role="progressbar" style="width: 0%"></div>
                        </div>
                    </div>
                    
                    <!-- Video Player -->
                    <div class="ratio ratio-16x9">
                        <div id="videoModalPlayer"></div>
                    </div>
                </div>
                <div class="modal-footer border-secondary">
                    <div class="d-flex justify-content-end w-100">
                        <button type="button" class="btn btn-success me-2" onclick="markLessonCompleteFromModal()">
                            <i class="fi fi-rr-check me-1"></i> Mark as Complete
                        </button>
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                            <i class="fi fi-rr-cross me-1"></i> Close
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>

  <!-- begin::Edura Page Scripts -->
  <script src="assets/libs/global/global.min.js"></script>
  <script src="assets/libs/sortable/Sortable.min.js"></script>
  <script src="assets/libs/flatpickr/flatpickr.min.js"></script>
  <script src="assets/js/appSettings.js"></script>
  <script src="assets/js/main.js"></script>
  <script src="assets/js/custom-video-player.js"></script>
  <!-- end::Edura Page Scripts -->
    
    <script>
        // Global variables
        const courseId = <?php echo isset($course_id) ? $course_id : 0; ?>;
        const currentLessonId = <?php echo isset($current_lesson) && $current_lesson ? $current_lesson['id'] : 'null'; ?>;
        let currentModalLessonId = null;
        let currentVideoPlayer = null;
        let progressInterval = null;
        
        // Debug logging
        console.log('PHP Debug - $current_lesson:', <?php echo json_encode($current_lesson); ?>);
        console.log('PHP Debug - $course_id:', <?php echo $course_id; ?>);
        console.log('PHP Debug - $lesson_id:', <?php echo $lesson_id; ?>);
        console.log('PHP Debug - Lessons count:', <?php echo count($lessons); ?>);
        
        document.addEventListener('DOMContentLoaded', function() {
            console.log('DOM loaded, currentLessonId:', currentLessonId);
            console.log('Course ID:', courseId);
            console.log('Current lesson title:', <?php echo json_encode($current_lesson ? $current_lesson['lesson_title'] : 'None'); ?>);
            
            // Add click event listener for video thumbnail
            const videoThumbnail = document.getElementById('videoThumbnail');
            if (videoThumbnail) {
                videoThumbnail.addEventListener('click', function() {
                    const lessonId = this.dataset.lessonId;
                    const lessonTitle = this.dataset.lessonTitle;
                    const videoType = this.dataset.videoType;
                    const videoFile = this.dataset.videoFile;
                    startVideoLesson(lessonId, lessonTitle, videoType, videoFile);
                });
            }
            
            // Add click event listener for start learning button
            const startLearningBtn = document.getElementById('startLearningBtn');
            if (startLearningBtn) {
                startLearningBtn.addEventListener('click', function() {
                    const lessonId = this.dataset.lessonId;
                    const lessonTitle = this.dataset.lessonTitle;
                    const videoType = this.dataset.videoType;
                    const videoFile = this.dataset.videoFile;
                    startVideoLesson(lessonId, lessonTitle, videoType, videoFile);
                });
            }
            
            // Add click event listener for start first lesson button
            const startFirstLessonBtn = document.getElementById('startFirstLessonBtn');
            if (startFirstLessonBtn) {
                startFirstLessonBtn.addEventListener('click', function() {
                    const lessonId = this.dataset.lessonId;
                    const lessonTitle = this.dataset.lessonTitle;
                    const videoType = this.dataset.videoType;
                    const videoFile = this.dataset.videoFile;
                    startVideoLesson(lessonId, lessonTitle, videoType, videoFile);
                });
            }
        });
        
        // View toggle functions removed - using list view only
        
        // Start video lesson inline
        function startVideoLesson(lessonId, lessonTitle, videoType = 'upload', videoFile = null) {
            console.log('startVideoLesson called with:', {lessonId, lessonTitle, videoType, videoFile});
            currentModalLessonId = lessonId;
            
            if (!videoFile || videoFile === '' || videoFile === 'null') {
                // Show a helpful message in the video player area
                document.getElementById('videoThumbnail').classList.add('d-none');
                document.getElementById('inlineVideoPlayer').classList.remove('d-none');
                
                const playerContainer = document.getElementById('videoPlayerContainer');
                playerContainer.innerHTML = `
                    <div class="d-flex align-items-center justify-content-center h-100 bg-dark rounded">
                        <div class="text-center text-white p-4">
                            <i class="fi fi-rr-exclamation-triangle" style="font-size: 3rem; color: #ffc107;"></i>
                            <h5 class="mt-3">No Video Available</h5>
                            <p class="text-muted">This lesson doesn't have a video file uploaded yet.</p>
                            <p class="small text-muted">Please contact your instructor or administrator.</p>
                            <button class="btn btn-secondary mt-2" onclick="resetInlineVideoPlayer()">
                                <i class="fi fi-rr-arrow-left me-1"></i> Back
                            </button>
                        </div>
                    </div>
                `;
                return;
            }
            
            // Hide thumbnail and show video player
            document.getElementById('videoThumbnail').classList.add('d-none');
            document.getElementById('inlineVideoPlayer').classList.remove('d-none');
            
            // Load lesson progress and initialize video player
            loadLessonProgressForInline(lessonId, videoFile);
        }
        
        // Load lesson progress for inline player
        function loadLessonProgressForInline(lessonId, videoFile) {
            fetch(`get_lesson_progress.php?lesson_id=${lessonId}`)
                .then(response => response.json())
                .then(data => {
                    const resumeTime = data.success && data.progress ? data.progress.watched_duration || 0 : 0;
                    console.log('Loaded progress for lesson:', lessonId, 'Resume time:', resumeTime);
                    
                    initializeInlineVideoPlayer(videoFile, resumeTime);
                })
                .catch(error => {
                    console.error('Error loading lesson progress:', error);
                    initializeInlineVideoPlayer(videoFile, 0);
                });
        }
        
        // Initialize inline video player
        function initializeInlineVideoPlayer(videoFile, resumeTime) {
            const playerContainer = document.getElementById('videoPlayerContainer');
            const videoSrc = '../uploads/course_videos/' + videoFile;
            
            console.log('Loading inline video:', videoSrc, 'Resume from:', resumeTime);
            
            if (typeof window.initCustomVideoPlayer === 'function') {
                currentVideoPlayer = window.initCustomVideoPlayer('videoPlayerContainer', videoSrc, resumeTime);
                
                if (currentVideoPlayer) {
                    console.log('Inline video player initialized successfully');
                    if (resumeTime > 0) {
                        setTimeout(() => {
                            showResumeNotification(resumeTime);
                        }, 1000);
                    }
                } else {
                    console.error('Failed to initialize inline video player');
                    playerContainer.innerHTML = `
                        <div class="alert alert-danger text-center">
                            <i class="fi fi-rr-exclamation-triangle me-2"></i>
                            Failed to load video player. Please try again.
                        </div>
                    `;
                }
            } else {
                console.error('initCustomVideoPlayer function not available');
                playerContainer.innerHTML = `
                    <div class="alert alert-danger text-center">
                        <i class="fi fi-rr-exclamation-triangle me-2"></i>
                        Video player not available. Please refresh the page.
                    </div>
                `;
            }
        }
        
        // Load lesson function
        function loadLesson(lessonId) {
            console.log('Loading lesson:', lessonId);
            if (!courseId || !lessonId) {
                console.error('Missing courseId or lessonId:', {courseId, lessonId});
                return;
            }
            // Reset inline video player before switching lessons
            resetInlineVideoPlayer();
            window.location.href = `?course_id=${courseId}&lesson_id=${lessonId}`;
        }
        
        // Check for next lesson
        function checkNextLesson() {
            console.log('Checking next lesson for course:', courseId, 'current lesson:', currentLessonId);
            fetch(`get_next_lesson.php?course_id=${courseId}&current_lesson_id=${currentLessonId}`)
                .then(response => response.json())
                .then(data => {
                    console.log('Next lesson response:', data);
                    if (data.success && data.next_lesson) {
                        if (data.auto_advance) {
                            console.log('Auto-advancing to next lesson:', data.next_lesson.lesson_title);
                            window.location.href = `?course_id=${courseId}&lesson_id=${data.next_lesson.id}`;
                        } else {
                            console.log('Next lesson available but not auto-advancing');
                        }
                    } else if (data.success && data.course_completed) {
                        showNotification('Congratulations! You have completed all lessons in this course!', 'success');
                    } else {
                        console.log('No next lesson found or course completed');
                    }
                })
                .catch(error => {
                    console.error('Error checking next lesson:', error);
                });
        }
        
        // Manual lesson completion
        function markLessonComplete(lessonId) {
            fetch('mark_lesson_complete.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'lesson_id=' + lessonId
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    console.log('Lesson marked as complete!', data);
                    showNotification('Lesson completed! Moving to next lesson...', 'success');
                    updateLessonStatusInUI(lessonId, true);
                    setTimeout(() => {
                        console.log('Resetting inline player and checking next lesson...');
                        resetInlineVideoPlayer();
                        checkNextLesson();
                    }, 2000);
                } else {
                    console.error('Failed to mark lesson as complete:', data);
                    showNotification('Error: ' + data.message, 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showNotification('An error occurred while marking the lesson as complete.', 'error');
            });
        }
        
        // Show notification
        function showNotification(message, type) {
            const alertClass = type === 'success' ? 'alert-success' : (type === 'warning' ? 'alert-warning' : 'alert-danger');
            const iconClass = type === 'success' ? 'fi fi-rr-check' : (type === 'warning' ? 'fi fi-rr-exclamation' : 'fi fi-rr-exclamation-triangle');
            
            const notification = document.createElement('div');
            notification.className = `alert ${alertClass} alert-dismissible fade show notification`;
            notification.innerHTML = `
                <i class="${iconClass} me-2"></i>
                ${message}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            `;
            
            document.body.appendChild(notification);
            
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 5000);
        }
        
        // Close video modal
        function closeVideoModal() {
            if (progressInterval) {
                clearInterval(progressInterval);
            }
            
            if (currentVideoPlayer) {
                currentVideoPlayer.destroy();
                currentVideoPlayer = null;
            }
            
            const playerContainer = document.getElementById('videoModalPlayer');
            if (playerContainer) {
                playerContainer.innerHTML = '';
            }
            
            const modal = bootstrap.Modal.getInstance(document.getElementById('videoModal'));
            if (modal) {
                modal.hide();
            }
        }
        
        // Reset inline video player (show thumbnail again)
        function resetInlineVideoPlayer() {
            if (currentVideoPlayer) {
                currentVideoPlayer.destroy();
                currentVideoPlayer = null;
            }
            
            const playerContainer = document.getElementById('videoPlayerContainer');
            if (playerContainer) {
                playerContainer.innerHTML = '';
            }
            
            // Show thumbnail and hide video player
            document.getElementById('videoThumbnail').classList.remove('d-none');
            document.getElementById('inlineVideoPlayer').classList.add('d-none');
        }
        
        function markLessonCompleteFromModal() {
            if (currentModalLessonId) {
                markLessonComplete(currentModalLessonId);
            }
        }
        
        // Handle lesson completion (called from video player)
        window.handleLessonCompleted = function(lessonId) {
            console.log('Handling lesson completion for:', lessonId);
            
            fetch('mark_lesson_complete.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'lesson_id=' + lessonId
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    console.log('Lesson marked as completed successfully');
                    showNotification('Lesson completed! Moving to next lesson...', 'success');
                    updateLessonStatusInUI(lessonId, true);
                    setTimeout(() => {
                        resetInlineVideoPlayer();
                        checkNextLesson();
                    }, 2000);
                } else {
                    console.error('Failed to mark lesson as completed:', data.message);
                }
            })
            .catch(error => {
                console.error('Error marking lesson as completed:', error);
            });
        };
        
        // Update lesson status in UI
        function updateLessonStatusInUI(lessonId, isCompleted) {
            const lessonCard = document.querySelector(`[data-lesson-id="${lessonId}"]`);
            if (lessonCard) {
                const statusBadge = lessonCard.querySelector('.lesson-status');
                if (statusBadge) {
                    if (isCompleted) {
                        statusBadge.className = 'badge bg-success lesson-status';
                        statusBadge.textContent = 'Completed';
                    } else {
                        statusBadge.className = 'badge bg-warning lesson-status';
                        statusBadge.textContent = 'Pending';
                    }
                }
            }
            
            const completeButton = document.querySelector('button[onclick="markLessonCompleteFromModal()"]');
            if (completeButton) {
                if (isCompleted) {
                    completeButton.className = 'btn btn-success me-2';
                    completeButton.innerHTML = '<i class="fi fi-rr-check me-1"></i> Completed';
                    completeButton.disabled = true;
                } else {
                    completeButton.className = 'btn btn-success me-2';
                    completeButton.innerHTML = '<i class="fi fi-rr-check me-1"></i> Mark as Complete';
                    completeButton.disabled = false;
                }
            }
        }
        
        // Handle modal events
        document.getElementById('videoModal').addEventListener('hidden.bs.modal', function () {
            closeVideoModal();
        });
        
        // Open video modal
        function openVideoModal(lessonId, videoUrl, lessonTitle, videoType = 'upload', videoFile = null) {
            currentModalLessonId = lessonId;
            
            document.getElementById('videoModalTitle').textContent = lessonTitle;
            
            document.getElementById('modalProgressText').textContent = '0%';
            document.getElementById('modalTimeText').textContent = '0:00 / 0:00';
            document.getElementById('modalProgressBar').style.width = '0%';
            
            const modal = new bootstrap.Modal(document.getElementById('videoModal'));
            modal.show();
            
            loadLessonProgressForModal(lessonId, videoFile);
        }
        
        // Load lesson progress for modal
        function loadLessonProgressForModal(lessonId, videoFile) {
            if (!videoFile) {
                const playerContainer = document.getElementById('videoModalPlayer');
                playerContainer.innerHTML = `
                    <div class="alert alert-warning text-center">
                        <i class="fi fi-rr-exclamation-triangle me-2"></i>
                        No video file found for this lesson.
                    </div>
                `;
                return;
            }
            
            fetch(`get_lesson_progress.php?lesson_id=${lessonId}`)
                .then(response => response.json())
                .then(data => {
                    const resumeTime = data.success && data.progress ? data.progress.watched_duration || 0 : 0;
                    console.log('Loaded progress for lesson:', lessonId, 'Resume time:', resumeTime);
                    
                    initializeVideoPlayerWithProgress(videoFile, resumeTime);
                })
                .catch(error => {
                    console.error('Error loading lesson progress:', error);
                    initializeVideoPlayerWithProgress(videoFile, 0);
                });
        }
        
        // Initialize video player with progress
        function initializeVideoPlayerWithProgress(videoFile, resumeTime) {
            const playerContainer = document.getElementById('videoModalPlayer');
            const videoSrc = '../uploads/course_videos/' + videoFile;
            
            console.log('Loading video:', videoSrc, 'Resume from:', resumeTime);
            
            if (typeof window.initCustomVideoPlayer === 'function') {
                currentVideoPlayer = window.initCustomVideoPlayer('videoModalPlayer', videoSrc, resumeTime);
                
                if (currentVideoPlayer) {
                    console.log('Video player initialized successfully');
                    if (resumeTime > 0) {
                        setTimeout(() => {
                            updateProgressDisplayFromResume(resumeTime);
                        }, 1000);
                    }
                } else {
                    console.error('Failed to initialize custom video player');
                    playerContainer.innerHTML = `
                        <div class="alert alert-danger text-center">
                            <i class="fi fi-rr-exclamation-triangle me-2"></i>
                            Failed to load video player. Please try again.
                        </div>
                    `;
                }
            } else {
                console.error('initCustomVideoPlayer function not available');
                playerContainer.innerHTML = `
                    <div class="alert alert-danger text-center">
                        <i class="fi fi-rr-exclamation-triangle me-2"></i>
                        Video player not available. Please refresh the page.
                    </div>
                `;
            }
        }
        
        // Update progress display from resume time
        function updateProgressDisplayFromResume(resumeTime) {
            if (currentVideoPlayer && currentVideoPlayer.video && currentVideoPlayer.duration > 0) {
                const progress = (resumeTime / currentVideoPlayer.duration) * 100;
                updateProgressDisplay(progress, resumeTime, currentVideoPlayer.duration);
                showResumeNotification(resumeTime);
            }
        }
        
        // Show resume notification
        function showResumeNotification(resumeTime) {
            const timeStr = formatTime(resumeTime);
            console.log(`Resuming video from ${timeStr}`);
            
            const notification = document.createElement('div');
            notification.className = 'alert alert-info position-fixed';
            notification.style.cssText = 'top: 20px; right: 20px; z-index: 1060; min-width: 300px;';
            notification.innerHTML = `
                <i class="fi fi-rr-play me-2"></i>
                Resuming from ${timeStr}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            `;
            
            document.body.appendChild(notification);
            
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 3000);
        }
        
        // Progress display function with fixed validation
        function updateProgressDisplay(progress, currentTime, totalTime) {
            // Validate inputs to prevent NaN and invalid values
            if (isNaN(progress) || isNaN(currentTime) || isNaN(totalTime) || 
                currentTime < 0 || totalTime <= 0 || currentTime > totalTime) {
                console.log('Invalid progress values:', { progress, currentTime, totalTime });
                return;
            }
            
            // Ensure values are valid numbers and currentTime doesn't exceed totalTime
            currentTime = Math.max(0, Math.min(currentTime, totalTime));
            progress = (currentTime / totalTime) * 100;
            progress = Math.max(0, Math.min(100, progress));
            
            const progressText = Math.round(progress) + '%';
            const timeText = formatTime(currentTime) + ' / ' + formatTime(totalTime);
            
            // Update modal progress with throttling to prevent blinking
            const modalProgressText = document.getElementById('modalProgressText');
            const modalTimeText = document.getElementById('modalTimeText');
            const modalProgressBar = document.getElementById('modalProgressBar');
            
            if (modalProgressText && modalProgressText.textContent !== progressText) {
                modalProgressText.textContent = progressText;
            }
            if (modalTimeText && modalTimeText.textContent !== timeText) {
                modalTimeText.textContent = timeText;
            }
            if (modalProgressBar && modalProgressBar.style.width !== progressText) {
                modalProgressBar.style.width = progressText;
            }
        }
        
        function formatTime(seconds) {
            const mins = Math.floor(seconds / 60);
            const secs = Math.floor(seconds % 60);
            return mins + ':' + (secs < 10 ? '0' : '') + secs;
        }
        
        // Certificate download function
        function downloadCertificate() {
            const courseId = <?php echo isset($course_id) ? $course_id : 0; ?>;
            const courseTitle = <?php echo json_encode(isset($course) && $course ? $course['title'] : 'Course Not Found'); ?>;
            const userName = <?php echo json_encode(isset($user) && $user ? $user['full_name'] : 'Student'); ?>;
            const completionDate = <?php echo json_encode(date('F j, Y')); ?>;
            const instructorName = <?php echo json_encode(isset($course) && $course ? ($course['instructor'] ?? 'Omateq Digitals Team') : 'Omateq Digitals Team'); ?>;
            
            // Create a new window for certificate generation
            const certificateWindow = window.open('', '_blank', 'width=800,height=600');
            
            const certificateHTML = `
                <!DOCTYPE html>
                <html>
                <head>
                    <title>Certificate of Completion - \${courseTitle}</title>
                    <style>
                        body {
                            font-family: \\'Times New Roman\\', serif;
                            margin: 0;
                            padding: 40px;
                            background: #f8f9fa;
                            display: flex;
                            justify-content: center;
                            align-items: center;
                            min-height: 100vh;
                        }
                        .certificate {
                            background: white;
                            border: 8px solid #28a745;
                            border-radius: 20px;
                            padding: 60px;
                            max-width: 800px;
                            text-align: center;
                            box-shadow: 0 10px 30px rgba(0,0,0,0.2);
                            position: relative;
                        }
                        .certificate::before {
                            content: '';
                            position: absolute;
                            top: 20px;
                            left: 20px;
                            right: 20px;
                            bottom: 20px;
                            border: 2px solid #28a745;
                            border-radius: 10px;
                            pointer-events: none;
                        }
                        .header {
                            margin-bottom: 40px;
                        }
                        .title {
                            font-size: 36px;
                            font-weight: bold;
                            color: #28a745;
                            margin-bottom: 20px;
                            text-transform: uppercase;
                            letter-spacing: 2px;
                        }
                        .subtitle {
                            font-size: 18px;
                            color: #6c757d;
                            font-style: italic;
                            margin-bottom: 40px;
                        }
                        .student-name {
                            font-size: 32px;
                            font-weight: bold;
                            color: #2c3e50;
                            margin-bottom: 40px;
                            text-transform: uppercase;
                            letter-spacing: 1px;
                        }
                        .body-text {
                            font-size: 20px;
                            color: #495057;
                            margin-bottom: 30px;
                            line-height: 1.6;
                        }
                        .course-title {
                            font-size: 24px;
                            font-weight: bold;
                            color: #007bff;
                            margin-bottom: 30px;
                            font-style: italic;
                        }
                        .date {
                            font-size: 18px;
                            color: #6c757d;
                            margin-bottom: 60px;
                        }
                        .signatures {
                            display: flex;
                            justify-content: space-between;
                            margin-top: 60px;
                        }
                        .signature {
                            text-align: center;
                            flex: 1;
                        }
                        .signature-line {
                            border-top: 2px solid #333;
                            width: 200px;
                            margin: 0 auto 10px;
                        }
                        .signature-name {
                            font-weight: bold;
                            font-size: 16px;
                            margin-bottom: 5px;
                        }
                        .signature-title {
                            font-size: 14px;
                            color: #6c757d;
                        }
                        .seal {
                            position: absolute;
                            top: 40px;
                            right: 40px;
                            width: 100px;
                            height: 100px;
                            border: 4px solid #28a745;
                            border-radius: 50%;
                            background: linear-gradient(135deg, #fff 0%, #f8f9fa 100%);
                            display: flex;
                            align-items: center;
                            justify-content: center;
                            font-size: 40px;
                            color: #28a745;
                        }
                        .organization {
                            margin-top: 40px;
                            font-size: 16px;
                            color: #28a745;
                            font-weight: bold;
                        }
                        @media print {
                            body { background: white; }
                            .certificate { box-shadow: none; border: 8px solid #28a745; }
                        }
                    </style>
                </head>
                <body>
                    <div class="certificate">
                        <div class="seal">🏆</div>
                        <div class="header">
                            <h1 class="title">Certificate of Completion</h1>
                            <p class="subtitle">This certifies that</p>
                        </div>
                        
                        <div class="student-name">\${userName}</div>
                        
                        <div class="body-text">
                            has successfully completed the course
                        </div>
                        
                        <div class="course-title">"\${courseTitle}"</div>
                        
                        <div class="date">
                            on \${completionDate}
                        </div>
                        
                        <div class="signatures">
                            <div class="signature">
                                <div class="signature-line"></div>
                                <div class="signature-name">\${instructorName}</div>
                                <div class="signature-title">Instructor</div>
                            </div>
                            <div class="signature">
                                <div class="signature-image" style="margin-bottom: 10px;">
                                    <img src="../assets/images/omateq-signature.png" alt="Omateq Digitals Signature" style="max-height: 50px; max-width: 120px;" onerror="this.src='../assets/images/logo-full.svg'; this.style.maxHeight='40px'; this.style.maxWidth='100px';">
                                    <div class="signature-line" style="border-top: 2px solid #333; width: 150px; margin: 0 auto 10px; display: none;"></div>
                                </div>
                                <div class="signature-name">Omateq Digitals</div>
                                <div class="signature-title">Training Organization</div>
                            </div>
                        </div>
                        
                        <div class="organization">
                            <p>Issued by Omateq Digitals</p>
                        </div>
                    </div>
                    
                    <script>
                        // Auto-print after a short delay
                        setTimeout(() => {
                            window.print();
                        }, 1000);
                    <\/script>
                </body>
                </html>
            `;
            
            certificateWindow.document.write(certificateHTML);
            certificateWindow.document.close();
        }
    </script>
</body>
</html>
<?php
// End of PHP processing
?>