<?php
session_start();
require_once __DIR__ . '/config/database.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: authentication/login.php');
    exit();
}

$product_id = $_GET['product_id'] ?? '';

if (empty($product_id)) {
    header('Location: products.php');
    exit();
}

try {
    $pdo = getConnection();
    
    // Check if user has purchased this product and get product download info from transactions
    $stmt = $pdo->prepare("SELECT t.*, p.product_name, p.is_downloadable, p.download_file_url, p.download_file_name, p.download_file_size, p.download_file_type, p.download_access_duration FROM transactions t 
                          JOIN products p ON t.product_id = p.id 
                          WHERE t.user_id = ? AND t.product_id = ? AND t.status = 'completed'");
    $stmt->execute([$_SESSION['user_id'], $product_id]);
    $order = $stmt->fetch();
    
    if (!$order) {
        $_SESSION['download_error'] = "You haven't purchased this product yet.";
        header('Location: download.php?product_id=' . $product_id);
        exit();
    }
    
    // Check if product is downloadable
    if ($order['is_downloadable'] != 1) {
        $_SESSION['download_error'] = "This product is not available for download.";
        header('Location: download.php?product_id=' . $product_id);
        exit();
    }
    
    // Check if download file exists
    if (!$order['download_file_url'] || !$order['download_file_name']) {
        $_SESSION['download_error'] = "Download file is not available for this product. Please contact support.";
        header('Location: download.php?product_id=' . $product_id);
        exit();
    }
    
    // Check download limit
    $download_count = $order['download_count'] ?? 0;
    $max_downloads = $order['download_limit'] ?? 3;
    if ($download_count >= $max_downloads) {
        $_SESSION['download_error'] = "You have reached the maximum download limit ($max_downloads downloads) for this product. Please repurchase to download again.";
        header('Location: download.php?product_id=' . $product_id);
        exit();
    }
    
    // Check download access duration (if order is older than allowed duration)
    $access_duration_days = $order['download_access_duration'] ?? 365;
    $order_date = strtotime($order['created_at']);
    $expiry_date = $order_date + ($access_duration_days * 24 * 60 * 60);
    
    if (time() > $expiry_date) {
        $_SESSION['download_error'] = "Your download access has expired. Please repurchase to download again.";
        header('Location: download.php?product_id=' . $product_id);
        exit();
    }
    
    // Check if file exists
    $file_path = '../' . $order['download_file_url']; // Add ../ to go up from users/ directory
    if (!file_exists($file_path)) {
        $_SESSION['download_error'] = "Product file not found. Please contact support.";
        header('Location: download.php?product_id=' . $product_id);
        exit();
    }
    
    // Update download count in transactions table
    $stmt = $pdo->prepare("UPDATE transactions SET download_count = download_count + 1, updated_at = NOW() WHERE id = ?");
    $stmt->execute([$order['id']]);
    
    // Get file info
    $file_name = $order['download_file_name'] ?: basename($file_path);
    $file_size = filesize($file_path);
    $file_extension = pathinfo($file_path, PATHINFO_EXTENSION);
    
    // Set appropriate headers for download
    header('Content-Type: application/octet-stream');
    header('Content-Disposition: attachment; filename="' . $file_name . '"');
    header('Content-Length: ' . $file_size);
    header('Cache-Control: no-cache, must-revalidate');
    header('Pragma: no-cache');
    header('Expires: 0');
    
    // Clear any output buffering
    if (ob_get_level()) {
        ob_end_clean();
    }
    
    // Stream the file
    $file = fopen($file_path, 'rb');
    if ($file) {
        while (!feof($file)) {
            echo fread($file, 8192);
            flush();
        }
        fclose($file);
    }
    
    // Log download activity
    $stmt = $pdo->prepare("INSERT INTO download_logs (user_id, product_id, order_id, downloaded_at) VALUES (?, ?, ?, NOW())");
    $stmt->execute([$_SESSION['user_id'], $product_id, $order['id']]);
    
    exit();
    
} catch (Exception $e) {
    $_SESSION['download_error'] = "Download failed. Please try again or contact support.";
    header('Location: download.php?product_id=' . $product_id);
    exit();
}
?>
