<?php
require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/../admin-dashboard/config/settings.php';
require_once __DIR__ . '/../admin-dashboard/config/branding.php';
require_once __DIR__ . '/config/session_manager.php';

// Initialize session and check login with timeout
UserSessionManager::initSession();
UserSessionManager::requireLogin('authentication/login.php');

// Get user data
try {
    $pdo = getConnection();
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $user = $stmt->fetch();
    
    if (!$user) {
        session_destroy();
        header('Location: authentication/login.php');
        exit();
    }
} catch (Exception $e) {
    session_destroy();
    header('Location: authentication/login.php');
    exit();
}

// Check for verification success message
$verification_success = '';
if (isset($_SESSION['verification_success'])) {
    $verification_success = $_SESSION['verification_success'];
    unset($_SESSION['verification_success']);
}

// Function to format time ago
function time_ago($datetime) {
    $time = time() - strtotime($datetime);
    if ($time < 60) return 'just now';
    if ($time < 3600) return floor($time/60) . ' min ago';
    if ($time < 86400) return floor($time/3600) . ' hr ago';
    if ($time < 2592000) return floor($time/86400) . ' days ago';
    return date('M j', strtotime($datetime));
}

// Get script request replies
$script_replies = [];
$script_replies_count = 0;
try {
    $stmt = $pdo->prepare("SELECT * FROM script_requests WHERE user_id = ? AND admin_reply IS NOT NULL ORDER BY created_at DESC LIMIT 5");
    $stmt->execute([$_SESSION['user_id']]);
    $script_replies = $stmt->fetchAll();
    $script_replies_count = count($script_replies);
} catch (Exception $e) {
    // Handle error silently
}

// Get dashboard statistics
$stats = [
    'total_products' => 0,
    'total_courses' => 0,
    'completed_courses' => 0,
    'ongoing_courses' => 0,
    'total_purchases' => 0,
    'total_spent' => 0
];

try {
    // Get total products available
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM products WHERE status = 'active'");
    $stmt->execute();
    $stats['total_products'] = $stmt->fetchColumn() ?: 0;

    // Get total courses available
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM courses WHERE status = 'active'");
    $stmt->execute();
    $stats['total_courses'] = $stmt->fetchColumn() ?: 0;

    // Get user's course enrollments
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM course_enrollments WHERE user_id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $stats['ongoing_courses'] = $stmt->fetchColumn() ?: 0;

    // Get completed courses (assuming completed means enrolled for now)
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM course_enrollments WHERE user_id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $stats['completed_courses'] = $stmt->fetchColumn() ?: 0;

    // Get total purchases and amount spent
    $stmt = $pdo->prepare("SELECT COUNT(*) as total_purchases, COALESCE(SUM(total_amount), 0) as total_spent FROM orders WHERE user_id = ? AND status = 'delivered'");
    $stmt->execute([$_SESSION['user_id']]);
    $purchase_data = $stmt->fetch();
    
    if ($purchase_data && is_array($purchase_data)) {
        $stats['total_purchases'] = $purchase_data['total_purchases'] ?? 0;
        $stats['total_spent'] = $purchase_data['total_spent'] ?? 0;
    } else {
        $stats['total_purchases'] = 0;
        $stats['total_spent'] = 0;
    }

} catch (Exception $e) {
    error_log("Dashboard stats error: " . $e->getMessage());
    // Ensure stats array has default values even if there's an error
    $stats['total_products'] = $stats['total_products'] ?? 0;
    $stats['total_courses'] = $stats['total_courses'] ?? 0;
    $stats['completed_courses'] = $stats['completed_courses'] ?? 0;
    $stats['ongoing_courses'] = $stats['ongoing_courses'] ?? 0;
    $stats['total_purchases'] = $stats['total_purchases'] ?? 0;
    $stats['total_spent'] = $stats['total_spent'] ?? 0;
}

// Get product transactions
$product_transactions = [];
try {
    $stmt = $pdo->prepare("SELECT o.*, p.product_name, p.thumbnail_url FROM orders o 
                          LEFT JOIN products p ON o.product_id = p.id 
                          WHERE o.user_id = ? AND o.product_id IS NOT NULL 
                          ORDER BY o.created_at DESC LIMIT 10");
    $stmt->execute([$_SESSION['user_id']]);
    $product_transactions = $stmt->fetchAll();
} catch (Exception $e) {
    error_log("Product transactions error: " . $e->getMessage());
}

// Get course enrollments
$course_enrollments = [];
try {
    $stmt = $pdo->prepare("SELECT ce.*, c.title as course_title, c.thumbnail, c.price, c.duration, c.level 
                          FROM course_enrollments ce 
                          JOIN courses c ON ce.course_id = c.id 
                          WHERE ce.user_id = ? 
                          ORDER BY ce.enrolled_at DESC LIMIT 10");
    $stmt->execute([$_SESSION['user_id']]);
    $course_enrollments = $stmt->fetchAll();
} catch (Exception $e) {
    error_log("Course enrollments error: " . $e->getMessage());
}

// Get recent transactions for header notifications
$transactions = [];
$transactions_count = 0;
try {
    $stmt = $pdo->prepare("SELECT 'purchase' as type, product_name as item_name, total_amount as amount, status, created_at FROM orders WHERE user_id = ? 
                          UNION ALL 
                          SELECT 'course' as type, title as item_name, price as amount, 'completed' as status, created_at FROM course_enrollments ce 
                          JOIN courses c ON ce.course_id = c.id WHERE ce.user_id = ? 
                          ORDER BY created_at DESC LIMIT 5");
    $stmt->execute([$_SESSION['user_id'], $_SESSION['user_id']]);
    $transactions = $stmt->fetchAll();
    $transactions_count = count($transactions);
} catch (Exception $e) {
    // Handle error silently
}
?>
<!DOCTYPE html>
<html lang="en">

<head>


  <!-- begin::Edura Meta Basic -->
  <meta charset="utf-8">
  <meta name="theme-color" content="#0D9488">
  <meta name="robots" content="index, follow">
  <meta name="author" content="LayoutDrop">
  <meta name="format-detection" content="telephone=no">
  <meta name="keywords" content="School Management, Education Admin Dashboard, Student Information System, University Dashboard, College Admin Template, Learning Management System, Education CRM, Teacher Management, Student Attendance Dashboard, Online Education Admin, Responsive Admin Template, Bootstrap 5 Education Admin, eLearning Dashboard, School ERP, Class Management Dashboard, Academic Dashboard, Education Web Application, Modern Admin Template, Admin UI Kit, ThemeForest Admin Template">
  <meta name="description" content="Edura is a professional and modern School Management & Education Admin Dashboard Template built with Bootstrap. It includes light and dark modes, and is perfect for managing students, teachers, courses, attendance, fees, and academic activities — ideal for schools, colleges, universities, and eLearning platforms.">
  <!-- end::Edura Meta Basic -->

  <!-- begin::Edura Meta Social -->
  <meta property="og:url" content="https://demos.layoutdrop.com/edura/demo/">
  <meta property="og:site_name" content="Dashboard | Edura School Management & Education Admin Dashboard Template">
  <meta property="og:type" content="website">
  <meta property="og:locale" content="en_us">
  <meta property="og:title" content="Dashboard | Edura School Management & Education Admin Dashboard Template">
  <meta property="og:description" content="Edura is a professional and modern School Management & Education Admin Dashboard Template built with Bootstrap. It includes light and dark modes, and is perfect for managing students, teachers, courses, attendance, fees, and academic activities — ideal for schools, colleges, universities, and eLearning platforms.">
  <meta property="og:image" content="https://demos.layoutdrop.com/edura/demo/preview.png">
  <!-- end::Edura Meta Social -->

  <!-- begin::Edura Meta Twitter -->
  <meta name="twitter:card" content="summary">
  <meta name="twitter:url" content="https://demos.layoutdrop.com/edura/demo/">
  <meta name="twitter:creator" content="@layoutdrop">
  <meta name="twitter:title" content="Dashboard | Edura School Management & Education Admin Dashboard Template">
  <meta name="twitter:description" content="Edura is a professional and modern School Management & Education Admin Dashboard Template built with Bootstrap. It includes light and dark modes, and is perfect for managing students, teachers, courses, attendance, fees, and academic activities — ideal for schools, colleges, universities, and eLearning platforms.">
  <!-- end::Edura Meta Twitter -->

  <!-- begin::Edura Website Page Title -->
  <title>Dashboard | Edura School Management & Education Admin Dashboard Template</title>
  <!-- end::Edura Website Page Title -->

  <!-- begin::Edura Mobile Specific -->
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <!-- end::Edura Mobile Specific -->

  <!-- begin::Edura Favicon Tags -->
  <?php echo getFaviconHtml(); ?>
  <!-- end::Edura Favicon Tags -->

  <!-- begin::Edura Google Fonts -->
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Plus+Jakarta+Sans:ital,wght@0,200..800;1,200..800&display=swap" rel="stylesheet">
  <!-- end::Edura Google Fonts -->

  <!-- begin::Edura Required Stylesheet -->
  <link rel="stylesheet" href="assets/libs/flaticon/css/all/all.css">
  <link rel="stylesheet" href="assets/libs/lucide/lucide.css">
  <link rel="stylesheet" href="assets/libs/fontawesome/css/all.min.css">
  <link rel="stylesheet" href="assets/libs/simplebar/simplebar.css">
  <link rel="stylesheet" href="assets/libs/node-waves/waves.css">
  <link rel="stylesheet" href="assets/libs/bootstrap-select/css/bootstrap-select.min.css">
  <!-- end::Edura Required Stylesheet -->

  <!-- begin::Edura CSS Stylesheet -->
  <link rel="stylesheet" href="assets/libs/flatpickr/flatpickr.min.css">
  <link rel="stylesheet" href="assets/libs/datatables/datatables.min.css">
  <link rel="stylesheet" href="assets/css/styles.css">
  <!-- end::Edura CSS Stylesheet -->
  
  <style>
    /* Logo sizing for better fit */
    .app-navbar-brand .navbar-brand-logo img {
      max-height: 50px;
      max-width: 180px;
      width: auto;
      height: auto;
      object-fit: contain;
    }
    
    .app-navbar-brand .navbar-brand-mini img {
      max-height: 35px;
      max-width: 35px;
      width: auto;
      height: auto;
      object-fit: contain;
    }
  </style>

  <!-- begin::Edura Googletagmanager -->
  <script async src="https://www.googletagmanager.com/gtag/js?id=G-XWVQM68HHQ"></script>
  <script>
		window.dataLayer = window.dataLayer || [];
		function gtag(){dataLayer.push(arguments);}
		gtag('js', new Date());

		gtag('config', 'G-XWVQM68HHQ', {
			'cookie_flags': 'SameSite=None;Secure',
			'send_page_view': true
		});
	</script>
  <!-- end::Edura Googletagmanager -->

</head>

<body>
  <div class="page-layout">

    <!-- begin::Edura Page Header -->
    <header class="app-header">
      <div class="app-header-inner">
        <button class="app-toggler" type="button">
          <span></span>
          <span></span>
          <span></span>
        </button>
        <div class="app-header-start">
          <form class="d-none d-md-flex align-items-center h-100 w-lg-250px w-xxl-300px position-relative" action="index.html#">
            <button type="button" class="btn btn-sm border-0 position-absolute start-0 ms-3 p-0">
              <i class="fi fi-rr-search"></i>
            </button>
            <input type="text" class="form-control ps-5" placeholder="Search anything's">
          </form>
        </div>
        <div class="app-header-end">
          <div class="px-lg-3 px-2 ps-0 d-flex align-items-center">
            <a href="javascript:void(0);" class="theme-btn">
              <i class="fi fi-rr-brightness icon-light"></i>
              <div class="theme-toggle"></div>
              <i class="fi fi-rr-moon icon-dark"></i>
            </a>
          </div>
          <div class="vr my-3"></div>
          <div class="d-flex align-items-center gap-sm-2 gap-0 px-lg-4 px-sm-2 px-1">
            <div class="dropdown text-end">
              <a href="javascript:void(0);" class="btn btn-icon btn-action-gray rounded-circle waves-effect waves-light position-relative" data-bs-toggle="dropdown" data-bs-auto-close="outside" aria-expanded="true">
                <i class="fi fi-rr-envelope"></i>
                <span class="position-absolute top-0 end-0 p-1 mt-1 me-1 bg-danger border border-3 border-light rounded-circle">
                  <span class="visually-hidden">New messages</span>
                </span>
              </a>
              <div class="dropdown-menu dropdown-menu-lg-end p-0 w-300px mt-2">
                <div class="px-3 py-3 border-bottom d-flex justify-content-between align-items-center">
                  <h6 class="mb-0">Script Request Replies <span class="badge badge-sm rounded-pill bg-primary ms-2"><?php echo $script_replies_count ?? 0; ?></span></h6>
                  <i class="bi bi-x-lg cursor-pointer"></i>
                </div>
                <div class="p-2" style="height: 300px;" data-simplebar>
                  <ul class="list-group list-group-hover list-group-smooth list-group-unlined">
                    <?php if (!empty($script_replies)): ?>
                      <?php foreach ($script_replies as $reply): ?>
                        <li class="list-group-item d-flex justify-content-between align-items-center">
                          <div class="avatar avatar-xs bg-info rounded-circle text-white">
                            <i class="fi fi-rr-user"></i>
                          </div>
                          <div class="ms-2 me-auto">
                            <h6 class="mb-0">Admin Reply</h6>
                            <small class="text-body d-block"><?php echo htmlspecialchars(substr($reply['admin_reply'], 0, 50)) . '...'; ?></small>
                            <small class="text-muted position-absolute end-0 top-0 mt-2 me-3"><?php echo time_ago($reply['created_at']); ?></small>
                          </div>
                        </li>
                      <?php endforeach; ?>
                    <?php else: ?>
                      <li class="list-group-item text-center py-4">
                        <i class="fi fi-rr-envelope text-muted" style="font-size: 2rem;"></i>
                        <p class="text-muted mt-2 mb-0">No script request replies yet</p>
                      </li>
                    <?php endif; ?>
                  </ul>
                </div>
                <div class="p-2">
                  <a href="script-request.php" class="btn w-100 btn-primary waves-effect waves-light">View all replies</a>
                </div>
              </div>
            </div>
            <div class="dropdown text-end">
              <a href="javascript:void(0);" class="btn btn-icon btn-action-gray rounded-circle waves-effect waves-light" data-bs-toggle="dropdown" data-bs-auto-close="outside" aria-expanded="true">
                <i class="fi fi-rr-bell"></i>
              </a>
              <div class="dropdown-menu dropdown-menu-lg-end p-0 w-300px mt-2">
                <div class="px-3 py-3 border-bottom d-flex justify-content-between align-items-center">
                  <h6 class="mb-0">Transactions <span class="badge badge-sm rounded-pill bg-primary ms-2"><?php echo $transactions_count ?? 0; ?></span></h6>
                  <i class="bi bi-x-lg cursor-pointer"></i>
                </div>
                <div class="p-2" style="height: 300px;" data-simplebar>
                  <ul class="list-group list-group-hover list-group-smooth list-group-unlined">
                    <?php if (!empty($transactions)): ?>
                      <?php foreach ($transactions as $transaction): ?>
                        <li class="list-group-item d-flex justify-content-between align-items-center">
                          <div class="avatar avatar-xs bg-<?php echo $transaction['type'] === 'purchase' ? 'success' : 'info'; ?> rounded-circle text-white">
                            <i class="fi fi-rr-<?php echo $transaction['type'] === 'purchase' ? 'shopping-cart' : 'graduation-cap'; ?>"></i>
                          </div>
                          <div class="ms-2 me-auto">
                            <h6 class="mb-0"><?php echo ucfirst($transaction['type']); ?> - <?php echo htmlspecialchars($transaction['item_name']); ?></h6>
                            <small class="text-body d-block"><?php echo $transaction['status'] === 'completed' ? 'Payment successful' : 'Payment pending'; ?> - $<?php echo number_format($transaction['amount'], 2); ?></small>
                            <small class="text-muted position-absolute end-0 top-0 mt-2 me-3"><?php echo time_ago($transaction['created_at']); ?></small>
                          </div>
                        </li>
                      <?php endforeach; ?>
                    <?php else: ?>
                      <li class="list-group-item text-center py-4">
                        <i class="fi fi-rr-shopping-cart text-muted" style="font-size: 2rem;"></i>
                        <p class="text-muted mt-2 mb-0">No transactions yet</p>
                      </li>
                    <?php endif; ?>
                  </ul>
                </div>
                <div class="p-2">
                  <a href="products.php" class="btn w-100 btn-primary waves-effect waves-light">View all transactions</a>
                </div>
              </div>
            </div>
          </div>
          <div class="vr my-3"></div>
          <div class="dropdown text-end ms-2">
            <a href="javascript:void(0);" class="d-flex align-items-center py-2" data-bs-toggle="dropdown" data-bs-auto-close="outside" aria-expanded="true">
              <div class="avatar avatar-sm rounded-circle avatar-status-success">
                <?php if ($user['avatar'] && file_exists($user['avatar'])): ?>
                  <img src="<?php echo htmlspecialchars($user['avatar']); ?>" alt="<?php echo htmlspecialchars($user['full_name']); ?>">
                <?php else: ?>
                  <img src="assets/images/avatar/avatar1.webp" alt="<?php echo htmlspecialchars($user['full_name']); ?>">
                <?php endif; ?>
              </div>
            </a>
            <ul class="dropdown-menu dropdown-menu-end w-225px mt-1">
              <li class="d-flex align-items-center p-2">
                <div class="avatar avatar-sm rounded-circle">
                  <?php if ($user['avatar'] && file_exists($user['avatar'])): ?>
                    <img src="<?php echo htmlspecialchars($user['avatar']); ?>" alt="<?php echo htmlspecialchars($user['full_name']); ?>">
                  <?php else: ?>
                    <img src="assets/images/avatar/avatar1.webp" alt="<?php echo htmlspecialchars($user['full_name']); ?>">
                  <?php endif; ?>
                </div>
                <div class="ms-2">
                  <div class="fw-bold text-dark"><?php echo htmlspecialchars($user['full_name']); ?></div>
                  <small class="text-body d-block lh-sm"><?php echo htmlspecialchars($user['email']); ?></small>
                </div>
              </li>
              <li>
                <div class="dropdown-divider my-1"></div>
              </li>
              <li>
                <a class="dropdown-item d-flex align-items-center gap-2" href="profile.php">
                  <i class="fi fi-rr-user scale-1x"></i> Profile
                </a>
              </li>
              <li>
                <a class="dropdown-item d-flex align-items-center gap-2" href="help-center.php">
                  <i class="fi fi-rs-interrogation scale-1x"></i> Help Center
                </a>
              </li>
              <li>
                <div class="dropdown-divider my-1"></div>
              </li>
              <li>
                <a class="dropdown-item d-flex align-items-center gap-2 text-danger" href="authentication/logout.php">
                  <i class="fi fi-sr-exit scale-1x"></i> Logout
                </a>
              </li>
            </ul>
          </div>
        </div>
      </div>
    </header>
    <!-- end::Edura Page Header -->

    <!-- begin::Edura Sidebar Menu -->
    <aside class="app-menubar" id="menubar">
      <div class="app-navbar-brand">
        <a class="navbar-brand-logo" href="index.php">
          <?php echo '<img src="' . '../' . getSetting('main_logo', 'assets/images/logo.svg') . '" alt="logo">'; ?>
        </a>
        <a class="navbar-brand-mini" href="index.php">
          <?php echo '<img src="' . '../' . getSetting('mini_logo', 'assets/images/logo-text-white.svg') . '" alt="logo">'; ?>
        </a>
      </div>
      <nav class="app-navbar" data-simplebar>
        <ul class="menubar">
              <li class="menu-item">
                <a class="menu-link" href="index.php">
              <i class="fi fi-rr-apps"></i>
                  <span class="menu-label">Dashboard</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="../users/products.php">
              <i class="fi fi-rr-users"></i>
              <span class="menu-label">Products</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="../users/course.php">
              <i class="fi fi-rr-graduation-cap"></i>
              <span class="menu-label">Course</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="../users/my_courses.php">
              <i class="fi fi-rr-book"></i>
              <span class="menu-label">My Courses</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="../users/profile.php">
              <i class="fi fi-rr-receipt"></i>
              <span class="menu-label">Profile</span>
                </a>
              </li>
              
        </ul>
      </nav>
      <div class="app-footer">
        <div class="text-center py-2">
          <small class="text-white-50">
            <i class="fi fi-rr-copyright me-1"></i>
            2025 Omateq Digitals
          </small>
        </div>
        <a href="mailto:info@omateqdigitals.com" class="btn btn-outline-white waves-effect btn-app-nav w-100">
          <i class="fi fi-rs-interrogation"></i>
          <span class="nav-text">Help and Support</span>
        </a>
      </div>
    </aside>
    <!-- end::Edura Sidebar Menu -->

    <main class="app-wrapper">

      <div class="container-fluid">

        <div class="app-page-head d-flex flex-wrap gap-3 align-items-center justify-content-between">
          <div class="clearfix">
            <h1 class="app-page-title">Welcome, <?php echo htmlspecialchars($user['full_name']); ?>!</h1>
            <span>Your Learning Dashboard</span>
          </div>
        </div>

        <?php if (!empty($verification_success)): ?>
          <div class="alert alert-success alert-dismissible fade show" role="alert">
            <div class="d-flex align-items-center">
              <i class="fi fi-rr-check-circle me-3" style="font-size: 2rem;"></i>
              <div>
                <h5 class="alert-heading mb-1">🎉 Congratulations!</h5>
                <p class="mb-0"><?php echo htmlspecialchars($verification_success); ?></p>
              </div>
            </div>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
          </div>
        <?php endif; ?>

        <!-- Dashboard Statistics Widgets -->
        <div class="row mb-4">
          <div class="col-xxl-3 col-md-6">
            <div class="card">
              <div class="card-body">
                <div class="d-flex flex-column mb-3">
                  <h6 class="text-body text-uppercase text-2xs mb-1">Total Products</h6>
                  <h2 class="mb-0"><?php echo $stats['total_products']; ?></h2>
                </div>
                <div class="avatar bg-primary bg-opacity-05 shadow-info rounded-circle avatar-xl text-primary mb-0 mx-auto">
                  <i class="fi fi-rr-shopping-cart"></i>
                </div>
              </div>
              <div class="card-footer border-0">
                <span class="text-primary me-1">
                  <i class="fi fi-rr-box me-1"></i>
                </span>
                <span>Available for purchase</span>
              </div>
            </div>
          </div>

          <div class="col-xxl-3 col-md-6">
            <div class="card">
              <div class="card-body">
                <div class="d-flex flex-column mb-3">
                  <h6 class="text-body text-uppercase text-2xs mb-1">Total Courses</h6>
                  <h2 class="mb-0"><?php echo $stats['total_courses']; ?></h2>
                </div>
                <div class="avatar bg-success bg-opacity-05 shadow-info rounded-circle avatar-xl text-success mb-0 mx-auto">
                  <i class="fi fi-rr-graduation-cap"></i>
                </div>
              </div>
              <div class="card-footer border-0">
                <span class="text-success me-1">
                  <i class="fi fi-rr-book me-1"></i>
                </span>
                <span>Available courses</span>
              </div>
            </div>
          </div>

          <div class="col-xxl-3 col-md-6">
            <div class="card">
              <div class="card-body">
                <div class="d-flex flex-column mb-3">
                  <h6 class="text-body text-uppercase text-2xs mb-1">Completed Courses</h6>
                  <h2 class="mb-0"><?php echo $stats['completed_courses']; ?></h2>
                </div>
                <div class="avatar bg-info bg-opacity-05 shadow-info rounded-circle avatar-xl text-info mb-0 mx-auto">
                  <i class="fi fi-rr-check-circle"></i>
                </div>
              </div>
              <div class="card-footer border-0">
                <span class="text-info me-1">
                  <i class="fi fi-rr-trophy me-1"></i>
                </span>
                <span>Courses completed</span>
              </div>
            </div>
          </div>

          <div class="col-xxl-3 col-md-6">
            <div class="card">
              <div class="card-body">
                <div class="d-flex flex-column mb-3">
                  <h6 class="text-body text-uppercase text-2xs mb-1">Ongoing Courses</h6>
                  <h2 class="mb-0"><?php echo $stats['ongoing_courses']; ?></h2>
                </div>
                <div class="avatar bg-warning bg-opacity-05 shadow-info rounded-circle avatar-xl text-warning mb-0 mx-auto">
                  <i class="fi fi-rr-play"></i>
                </div>
              </div>
              <div class="card-footer border-0">
                <span class="text-warning me-1">
                  <i class="fi fi-rr-clock me-1"></i>
                </span>
                <span>Currently enrolled</span>
              </div>
            </div>
          </div>
        </div>

        <!-- Product Transactions Table -->
        <div class="row mb-4">
          <div class="col-12">
            <div class="card">
              <div class="card-header d-flex align-items-center justify-content-between border-0 pb-0">
                <h6 class="card-title mb-0">
                  <i class="fi fi-rr-shopping-cart me-2"></i>
                  Product Transactions
                </h6>
                <a href="products.php" class="btn btn-sm btn-outline-primary waves-effect btn-shadow">
                  <i class="fi fi-rr-eye me-1"></i> View All Products
                </a>
              </div>
              <div class="card-body">
                <?php if (!empty($product_transactions)): ?>
                  <div class="table-responsive">
                    <table class="table table-hover">
                      <thead>
                        <tr>
                          <th>Product</th>
                          <th>Order Number</th>
                          <th>Amount</th>
                          <th>Status</th>
                          <th>Date</th>
                          <th>Action</th>
                        </tr>
                      </thead>
                      <tbody>
                        <?php foreach ($product_transactions as $transaction): ?>
                          <tr>
                            <td>
                              <div class="d-flex align-items-center">
                                <?php if ($transaction['thumbnail_url']): ?>
                                  <img src="../<?php echo htmlspecialchars($transaction['thumbnail_url']); ?>" class="avatar avatar-sm rounded me-2" alt="Product">
                                <?php else: ?>
                                  <div class="avatar avatar-sm bg-light rounded me-2 d-flex align-items-center justify-content-center">
                                    <i class="fi fi-rr-shopping-cart text-muted"></i>
                                  </div>
                                <?php endif; ?>
                                <div>
                                  <h6 class="mb-0"><?php echo htmlspecialchars($transaction['product_name']); ?></h6>
                                  <small class="text-muted">Product Purchase</small>
                                </div>
                              </div>
                            </td>
                            <td>
                              <span class="badge bg-light text-dark"><?php echo htmlspecialchars($transaction['order_number']); ?></span>
                            </td>
                            <td>
                              <span class="fw-bold text-success">$<?php echo number_format($transaction['total_amount'], 2); ?></span>
                            </td>
                            <td>
                              <?php if ($transaction['status'] === 'delivered'): ?>
                                <span class="badge bg-success">Completed</span>
                              <?php elseif ($transaction['status'] === 'pending'): ?>
                                <span class="badge bg-warning">Pending</span>
                              <?php else: ?>
                                <span class="badge bg-secondary"><?php echo ucfirst($transaction['status']); ?></span>
                              <?php endif; ?>
                            </td>
                            <td>
                              <small class="text-muted"><?php echo time_ago($transaction['created_at']); ?></small>
                            </td>
                            <td>
                              <?php if ($transaction['status'] === 'delivered'): ?>
                                <a href="download.php?product_id=<?php echo $transaction['product_id']; ?>" class="btn btn-sm btn-outline-primary">
                                  <i class="fi fi-rr-download me-1"></i> Download
                                </a>
                              <?php else: ?>
                                <span class="text-muted">Awaiting approval</span>
                              <?php endif; ?>
                            </td>
                          </tr>
                        <?php endforeach; ?>
                      </tbody>
                    </table>
                  </div>
                <?php else: ?>
                  <div class="text-center py-5">
                    <i class="fi fi-rr-shopping-cart text-muted" style="font-size: 3rem;"></i>
                    <h5 class="mt-3 text-muted">No Product Transactions</h5>
                    <p class="text-muted">You haven't purchased any products yet.</p>
                    <a href="products.php" class="btn btn-primary">
                      <i class="fi fi-rr-shopping-cart me-2"></i> Browse Products
                    </a>
                  </div>
                <?php endif; ?>
              </div>
            </div>
          </div>
        </div>

        <!-- Course Learning Table -->
        <div class="row">
          <div class="col-12">
            <div class="card">
              <div class="card-header d-flex align-items-center justify-content-between border-0 pb-0">
                <h6 class="card-title mb-0">
                  <i class="fi fi-rr-graduation-cap me-2"></i>
                  Course Learning Progress
                </h6>
                <a href="course.php" class="btn btn-sm btn-outline-primary waves-effect btn-shadow">
                  <i class="fi fi-rr-eye me-1"></i> View All Courses
                </a>
              </div>
              <div class="card-body">
                <?php if (!empty($course_enrollments)): ?>
                  <div class="table-responsive">
                    <table class="table table-hover">
                      <thead>
                        <tr>
                          <th>Course</th>
                          <th>Level</th>
                          <th>Duration</th>
                          <th>Price</th>
                          <th>Status</th>
                          <th>Enrolled</th>
                          <th>Action</th>
                        </tr>
                      </thead>
                      <tbody>
                        <?php foreach ($course_enrollments as $enrollment): ?>
                          <tr>
                            <td>
                              <div class="d-flex align-items-center">
                                <?php if ($enrollment['thumbnail']): ?>
                                  <img src="../<?php echo htmlspecialchars($enrollment['thumbnail']); ?>" class="avatar avatar-sm rounded me-2" alt="Course">
                                <?php else: ?>
                                  <div class="avatar avatar-sm bg-light rounded me-2 d-flex align-items-center justify-content-center">
                                    <i class="fi fi-rr-graduation-cap text-muted"></i>
                                  </div>
                                <?php endif; ?>
                                <div>
                                  <h6 class="mb-0"><?php echo htmlspecialchars($enrollment['course_title']); ?></h6>
                                  <small class="text-muted">Course Enrollment</small>
                                </div>
                              </div>
                            </td>
                            <td>
                              <span class="badge bg-<?php echo $enrollment['level'] === 'beginner' ? 'success' : ($enrollment['level'] === 'intermediate' ? 'warning' : 'danger'); ?>">
                                <?php echo ucfirst($enrollment['level']); ?>
                              </span>
                            </td>
                            <td>
                              <small class="text-muted"><?php echo htmlspecialchars($enrollment['duration'] ?? 'Not specified'); ?></small>
                            </td>
                            <td>
                              <span class="fw-bold text-primary">$<?php echo number_format($enrollment['price'], 2); ?></span>
                            </td>
                            <td>
                              <?php if ($enrollment['status'] === 'enrolled'): ?>
                                <span class="badge bg-warning">In Progress</span>
                              <?php elseif ($enrollment['status'] === 'completed'): ?>
                                <span class="badge bg-success">Completed</span>
                              <?php else: ?>
                                <span class="badge bg-secondary"><?php echo ucfirst($enrollment['status']); ?></span>
                              <?php endif; ?>
                            </td>
                            <td>
                              <small class="text-muted"><?php echo time_ago($enrollment['enrolled_at']); ?></small>
                            </td>
                            <td>
                              <a href="course_learning.php?course_id=<?php echo $enrollment['course_id']; ?>" class="btn btn-sm btn-primary">
                                <i class="fi fi-rr-play me-1"></i> Continue Learning
                              </a>
                            </td>
                          </tr>
                        <?php endforeach; ?>
                      </tbody>
                    </table>
                  </div>
                <?php else: ?>
                  <div class="text-center py-5">
                    <i class="fi fi-rr-graduation-cap text-muted" style="font-size: 3rem;"></i>
                    <h5 class="mt-3 text-muted">No Course Enrollments</h5>
                    <p class="text-muted">You haven't enrolled in any courses yet.</p>
                    <a href="course.php" class="btn btn-primary">
                      <i class="fi fi-rr-graduation-cap me-2"></i> Browse Courses
                    </a>
                  </div>
                <?php endif; ?>
              </div>
            </div>
          </div>
        </div>

      </div>

    </main>

    <!-- begin::Edura Footer -->
    <footer class="footer-wrapper bg-body text-center">
      <div class="container-fluid">
        <p class="mb-0">© <span class="currentYear">2025</span> Omateq Digitals. Proudly powered by Omateq Digitals Ltd.</p>
      </div>
    </footer>
    <!-- end::Edura Footer -->

    <!-- WhatsApp Chat Button -->
    <div class="whatsapp-chat-button">
      <a href="https://wa.me/2347069378861?text=Hello%20Omateq%20Digitals!%20I%20would%20like%20to%20know%20more%20about%20your%20services." 
         target="_blank" 
         rel="noopener noreferrer"
         class="whatsapp-link"
         title="Chat with us on WhatsApp">
        <i class="fab fa-whatsapp"></i>
        <span class="whatsapp-text">Chat with us</span>
      </a>
    </div>
    
    <style>
    /* WhatsApp Chat Button Styles */
    .whatsapp-chat-button {
      position: fixed;
      bottom: 20px;
      right: 20px;
      z-index: 1000;
      animation: pulse 2s infinite;
    }
    
    .whatsapp-link {
      display: flex;
      align-items: center;
      background: #25D366;
      color: white;
      text-decoration: none;
      padding: 12px 16px;
      border-radius: 50px;
      box-shadow: 0 4px 12px rgba(37, 211, 102, 0.4);
      transition: all 0.3s ease;
      font-family: 'Poppins', sans-serif;
      font-weight: 500;
      font-size: 14px;
    }
    
    .whatsapp-link:hover {
      background: #128C7E;
      color: white;
      text-decoration: none;
      transform: translateY(-2px);
      box-shadow: 0 6px 20px rgba(37, 211, 102, 0.6);
    }
    
    .whatsapp-link i {
      font-size: 24px;
      margin-right: 8px;
    }
    
    .whatsapp-text {
      white-space: nowrap;
    }
    
    @keyframes pulse {
      0% {
        box-shadow: 0 0 0 0 rgba(37, 211, 102, 0.7);
      }
      70% {
        box-shadow: 0 0 0 10px rgba(37, 211, 102, 0);
      }
      100% {
        box-shadow: 0 0 0 0 rgba(37, 211, 102, 0);
      }
    }
    
    /* Mobile responsiveness */
    @media (max-width: 768px) {
      .whatsapp-chat-button {
        bottom: 15px;
        right: 15px;
      }
      
      .whatsapp-link {
        padding: 10px 14px;
        font-size: 13px;
      }
      
      .whatsapp-link i {
        font-size: 22px;
        margin-right: 6px;
      }
      
      .whatsapp-text {
        display: none;
      }
    }
    </style>

  </div>
  <!-- begin::Edura Page Scripts -->
  <script src="assets/libs/global/global.min.js"></script>
  <script src="assets/libs/sortable/Sortable.min.js"></script>
  <script src="assets/libs/chartjs/chart.js"></script>
  <script src="assets/libs/flatpickr/flatpickr.min.js"></script>
  <script src="assets/libs/apexcharts/apexcharts.min.js"></script>
  <script src="assets/libs/datatables/datatables.min.js"></script>
  <script src="assets/js/dashboard.js"></script>
  <script src="assets/js/appSettings.js"></script>
  <script src="assets/js/main.js"></script>
  <!-- end::Edura Page Scripts -->
  
  <!-- Session Monitor -->
  <div data-session-monitor 
       data-timeout-duration="60" 
       data-warning-duration="30" 
       data-check-interval="5000" 
       data-login-url="authentication/login.php">
  </div>
  <script src="assets/js/session-monitor.js"></script>
</body>

</html>
<?php
// End of PHP processing
?>