<?php
// Start output buffering to prevent warnings from breaking JSON
ob_start();

session_start();
require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/config/payment_config.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['status' => 'error', 'message' => 'User not logged in']);
    exit();
}

// Get POST data
$input = json_decode(file_get_contents('php://input'), true);
$order_id = $input['order_id'] ?? null;
$course_id = $input['course_id'] ?? null;
$reference = $input['reference'] ?? null;
$amount = $input['amount'] ?? null;

if (!$reference) {
    http_response_code(400);
    echo json_encode(['status' => 'error', 'message' => 'Missing reference']);
    exit();
}

try {
    $pdo = getConnection();
    
    // Handle course payment
    if ($course_id && $amount) {
        // Get course details
        $stmt = $pdo->prepare("SELECT * FROM courses WHERE id = ? AND status = 'active'");
        $stmt->execute([$course_id]);
        $course = $stmt->fetch();
        
        if (!$course) {
            http_response_code(404);
            echo json_encode(['status' => 'error', 'message' => 'Course not found']);
            exit();
        }
        
        // Check if user is already enrolled
        $stmt = $pdo->prepare("SELECT id FROM course_enrollments WHERE user_id = ? AND course_id = ?");
        $stmt->execute([$_SESSION['user_id'], $course_id]);
        if ($stmt->fetch()) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'You are already enrolled in this course']);
            exit();
        }
        
        // Create order-like structure for course
        $order = [
            'id' => 'course_' . $course_id,
            'total_amount' => $amount,
            'product_name' => $course['title'],
            'user_id' => $_SESSION['user_id']
        ];
    } else {
        // Handle product order
        if (!$order_id) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'Missing order ID or course ID']);
            exit();
        }
        
        $stmt = $pdo->prepare("SELECT o.*, p.product_name, p.product_amount FROM orders o 
                              JOIN products p ON o.product_id = p.id 
                              WHERE o.id = ? AND o.user_id = ?");
        $stmt->execute([$order_id, $_SESSION['user_id']]);
        $order = $stmt->fetch();
        
        if (!$order) {
            http_response_code(404);
            echo json_encode(['status' => 'error', 'message' => 'Order not found']);
            exit();
        }
    }
    
    // Load payment configuration
    $payment_config = include __DIR__ . '/config/payment_config.php';
    
    // Debug: Log payment config
    error_log("Payment Config Loaded: " . json_encode($payment_config));
    
    // Ensure payment config has the expected structure
    if (!isset($payment_config['paystack'])) {
        error_log("Payment config missing paystack section: " . json_encode($payment_config));
        echo json_encode(['status' => 'error', 'message' => 'Payment configuration error. Please contact support.']);
        exit();
    }
    
    // Get live exchange rate with detailed information (consistent with payment_handler.php)
    require_once __DIR__ . '/config/exchange_rate.php';
    try {
        $exchange_rate_info = getExchangeRateWithInfo();
        $exchange_rate = $exchange_rate_info['rate'] ?? 1500; // Fallback rate
    } catch (Exception $e) {
        error_log("Exchange rate error: " . $e->getMessage());
        $exchange_rate = 1500;
        $exchange_rate_info = [
            'rate' => 1500,
            'source' => 'fallback',
            'age_minutes' => 0
        ];
    }
    
    // Check if using test or live keys
    $secret_key = $payment_config['paystack']['secret_key'] ?? '';
    if (empty($secret_key)) {
        error_log("Paystack secret key is empty");
        echo json_encode(['status' => 'error', 'message' => 'Payment configuration incomplete. Please contact support.']);
        exit();
    }
    $is_test_key = strpos($secret_key, 'sk_test_') === 0;
    
    // Calculate amount in kobo (Paystack uses kobo for NGN)
    $amount_usd = $order['total_amount'];
    
    if ($is_test_key) {
        // For test keys, use a lower exchange rate to stay within test limits
        $test_exchange_rate = 100; // 1 USD = 100 NGN for testing
        $amount_ngn = $amount_usd * $test_exchange_rate;
        error_log("Using test exchange rate ($test_exchange_rate) for test API key");
    } else {
        // For live keys, use the same exchange rate as payment_handler.php
        $amount_ngn = $amount_usd * $exchange_rate;
        error_log("Using live exchange rate ($exchange_rate) for live API key from source: " . ($exchange_rate_info['source'] ?? 'unknown'));
    }
    
    $amount_in_kobo = $amount_ngn * 100;
    
    // Log amount calculation for debugging
    error_log("Paystack Amount Calculation: USD: $amount_usd, Exchange Rate: $exchange_rate, NGN: $amount_ngn, Kobo: $amount_in_kobo");
    
    // Get user email from database if not in session
    $user_email = $_SESSION['user_email'] ?? null;
    if (!$user_email) {
        $stmt = $pdo->prepare("SELECT email FROM users WHERE id = ?");
        $stmt->execute([$_SESSION['user_id']]);
        $user = $stmt->fetch();
        $user_email = $user['email'] ?? 'customer@example.com';
    }
    
    // Ensure amount is a valid integer
    $amount_in_kobo = (int) round($amount_in_kobo);
    
    // Validate amount limits based on key type
    if ($is_test_key) {
        // Test key limit: 1,000,000 kobo = 10,000 NGN
        $max_kobo = 1000000;
        if ($amount_in_kobo > $max_kobo) {
            echo json_encode(['status' => 'error', 'message' => 'Amount too large for test transaction. Maximum allowed: ₦10,000']);
            exit();
        }
    } else {
        // Live key limit: 100,000,000 kobo = 1,000,000 NGN
        $max_kobo = 100000000;
        if ($amount_in_kobo > $max_kobo) {
            echo json_encode(['status' => 'error', 'message' => 'Amount too large. Maximum allowed: ₦1,000,000']);
            exit();
        }
    }
    
    if ($amount_in_kobo < 100) { // Minimum 1 NGN
        echo json_encode(['status' => 'error', 'message' => 'Amount too small. Minimum allowed: ₦1']);
        exit();
    }
    
    // Generate a unique reference quickly
    $reference = 'PAYSTACK_' . time() . '_' . uniqid() . '_' . rand(1000, 9999);
    
    // Prepare Paystack initialization data
    $paystack_data = [
        'email' => $user_email,
        'amount' => $amount_in_kobo,
        'currency' => 'NGN',
        'reference' => $reference,
        'callback_url' => $payment_config['paystack']['callback_url'] ?? '',
        'metadata' => [
            'order_id' => $order_id,
            'user_id' => $_SESSION['user_id'],
            'product_name' => $order['product_name']
        ]
    ];
    
    // Log the data being sent to Paystack
    error_log("Paystack Request Data: " . json_encode($paystack_data));
    
    // Initialize Paystack transaction
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, 'https://api.paystack.co/transaction/initialize');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($paystack_data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Bearer ' . $secret_key,
        'Content-Type: application/json'
    ]);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10); // 10 second timeout
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5); // 5 second connection timeout
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);
    
    // Log the response for debugging
    error_log("Paystack Response - HTTP Code: $http_code, Response: $response, CURL Error: $curl_error");
    
    if ($curl_error) {
        error_log("Paystack CURL Error: $curl_error");
        error_log("CURL Info: " . json_encode($curl_info));
        echo json_encode([
            'status' => 'error', 
            'message' => 'Network error. Please check your connection and try again.',
            'debug' => [
                'curl_error' => $curl_error,
                'http_code' => $http_code,
                'total_time' => $curl_info['total_time'] ?? 'unknown'
            ]
        ]);
        exit();
    }
    
    if ($http_code === 200) {
        $result = json_decode($response, true);
        
        if ($result && $result['status'] === true) {
            // Update order with Paystack reference
            $stmt = $pdo->prepare("UPDATE orders SET payment_reference = ? WHERE id = ? AND user_id = ?");
            $stmt->execute([$reference, $order_id, $_SESSION['user_id']]);
            
            // Clean any warnings before output
            ob_clean();
            
            echo json_encode([
                'status' => 'success',
                'authorization_url' => $result['data']['authorization_url'],
                'access_code' => $result['data']['access_code'],
                'amount_details' => [
                    'usd' => $amount_usd,
                    'exchange_rate' => $exchange_rate,
                    'ngn' => $amount_ngn,
                    'kobo' => $amount_in_kobo,
                    'rate_info' => $exchange_rate_info
                ]
            ]);
        } else {
            $error_message = 'Failed to initialize payment';
            if ($result && isset($result['message'])) {
                $error_message .= ': ' . $result['message'];
                
                // Check if it's a duplicate reference error
                if (strpos($result['message'], 'Duplicate') !== false || strpos($result['message'], 'duplicate') !== false) {
                    error_log("Duplicate reference detected: $reference");
                    $error_message = 'Duplicate transaction reference detected. Please try again.';
                }
            } elseif ($result && isset($result['data']['message'])) {
                $error_message .= ': ' . $result['data']['message'];
            }
            // Clean any warnings before output
            ob_clean();
            echo json_encode(['status' => 'error', 'message' => $error_message]);
        }
    } else {
        $error_message = 'Paystack API error: HTTP ' . $http_code;
        if ($response) {
            $result = json_decode($response, true);
            if ($result && isset($result['message'])) {
                $error_message .= ' - ' . $result['message'];
                
                // Check if it's a duplicate reference error
                if (strpos($result['message'], 'Duplicate') !== false || strpos($result['message'], 'duplicate') !== false) {
                    error_log("Duplicate reference detected in HTTP $http_code: $reference");
                    $error_message = 'Duplicate transaction reference detected. Please try again.';
                }
            }
        }
        // Clean any warnings before output
        ob_clean();
        echo json_encode(['status' => 'error', 'message' => $error_message]);
    }
    
} catch (Exception $e) {
    http_response_code(500);
    // Clean any warnings before output
    ob_clean();
    echo json_encode(['status' => 'error', 'message' => 'Server error: ' . $e->getMessage()]);
}
?>
