<?php
session_start();
require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/config/payment_config.php';
require_once __DIR__ . '/../admin-dashboard/config/settings.php';
require_once __DIR__ . '/../admin-dashboard/config/branding.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: authentication/login.php');
    exit();
}

// Get user data
try {
    $pdo = getConnection();
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $user = $stmt->fetch();
    
    if (!$user) {
        session_destroy();
        header('Location: authentication/login.php');
        exit();
    }
} catch (Exception $e) {
    session_destroy();
    header('Location: authentication/login.php');
    exit();
}

$payment_config = require __DIR__ . '/config/payment_config.php';
$error = '';
$success = '';

// Handle manual payment verification
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'verify_payment') {
    $reference = trim($_POST['reference'] ?? '');
    $order_id = $_POST['order_id'] ?? '';
    
    if (empty($reference) || empty($order_id)) {
        $error = "Please provide both reference and order ID.";
    } else {
        try {
            // Get order details
            $stmt = $pdo->prepare("SELECT * FROM orders WHERE id = ? AND user_id = ?");
            $stmt->execute([$order_id, $_SESSION['user_id']]);
            $order = $stmt->fetch();
            
            if (!$order) {
                $error = "Order not found.";
            } else {
                // Verify payment with Paystack
                $payment_verified = verifyPaystackPayment($reference, $payment_config['paystack']['secret_key']);
                
                if ($payment_verified) {
                    // Update order status
                    $stmt = $pdo->prepare("UPDATE orders SET status = 'delivered', payment_status = 'paid', payment_reference = ?, updated_at = NOW() WHERE id = ?");
                    $stmt->execute([$reference, $order_id]);
                    
                    // Update transaction status
                    $stmt = $pdo->prepare("UPDATE transactions SET status = 'completed', verification_date = NOW(), updated_at = NOW() WHERE order_number = ?");
                    $stmt->execute([$order['order_number']]);
                    
                    $_SESSION['payment_success'] = "Payment verified successfully! You can now download your product.";
                    header('Location: download.php?product_id=' . $order['product_id']);
                    exit();
                } else {
                    $error = "Payment verification failed. Please check your reference and try again.";
                }
            }
        } catch (Exception $e) {
            $error = "Verification failed: " . $e->getMessage();
        }
    }
}

// Function to verify Paystack payment
function verifyPaystackPayment($reference, $secret_key) {
    try {
        $url = "https://api.paystack.co/transaction/verify/" . $reference;
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            "Authorization: Bearer " . $secret_key
        ]);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code === 200) {
            $result = json_decode($response, true);
            return $result['status'] === true && $result['data']['status'] === 'success';
        }
        
        return false;
    } catch (Exception $e) {
        return false;
    }
}

// Get script request replies
$script_replies = [];
$script_replies_count = 0;
try {
    $stmt = $pdo->prepare("SELECT * FROM script_requests WHERE user_id = ? AND admin_reply IS NOT NULL ORDER BY created_at DESC LIMIT 5");
    $stmt->execute([$_SESSION['user_id']]);
    $script_replies = $stmt->fetchAll();
    $script_replies_count = count($script_replies);
} catch (Exception $e) {
    // Handle error silently
}

// Get transactions (purchases and course enrollments)
$transactions = [];
$transactions_count = 0;
try {
    $stmt = $pdo->prepare("SELECT 'purchase' as type, product_name as item_name, amount, status, created_at FROM orders WHERE user_id = ? 
                          UNION ALL 
                          SELECT 'course' as type, course_title as item_name, price as amount, 'completed' as status, created_at FROM course_enrollments ce 
                          JOIN courses c ON ce.course_id = c.id WHERE ce.user_id = ? 
                          ORDER BY created_at DESC LIMIT 5");
    $stmt->execute([$_SESSION['user_id'], $_SESSION['user_id']]);
    $transactions = $stmt->fetchAll();
    $transactions_count = count($transactions);
} catch (Exception $e) {
    // Handle error silently
}

// Function to format time ago
function time_ago($datetime) {
    $time = time() - strtotime($datetime);
    if ($time < 60) return 'just now';
    if ($time < 3600) return floor($time/60) . ' min ago';
    if ($time < 86400) return floor($time/3600) . ' hr ago';
    if ($time < 2592000) return floor($time/86400) . ' days ago';
    return date('M j', strtotime($datetime));
}
?>

<!DOCTYPE html>
<html lang="en">

<head>
  <!-- begin::Edura Meta Basic -->
  <meta charset="utf-8">
  <meta name="theme-color" content="#0D9488">
  <meta name="robots" content="index, follow">
  <meta name="author" content="LayoutDrop">
  <meta name="format-detection" content="telephone=no">
  <meta name="keywords" content="School Management, Education Admin Dashboard, Student Information System, University Dashboard, College Admin Template, Learning Management System, Education CRM, Teacher Management, Student Attendance Dashboard, Online Education Admin, Responsive Admin Template, Bootstrap 5 Education Admin, eLearning Dashboard, School ERP, Class Management Dashboard, Academic Dashboard, Education Web Application, Modern Admin Template, Admin UI Kit, ThemeForest Admin Template">
  <meta name="description" content="Edura is a professional and modern School Management & Education Admin Dashboard Template built with Bootstrap. It includes light and dark modes, and is perfect for managing students, teachers, courses, attendance, fees, and academic activities — ideal for schools, colleges, universities, and eLearning platforms.">
  <!-- end::Edura Meta Basic -->

  <!-- begin::Edura Meta Social -->
  <meta property="og:url" content="https://demos.layoutdrop.com/edura/demo/">
  <meta property="og:site_name" content="Dashboard | Edura School Management & Education Admin Dashboard Template">
  <meta property="og:type" content="website">
  <meta property="og:locale" content="en_us">
  <meta property="og:title" content="Dashboard | Edura School Management & Education Admin Dashboard Template">
  <meta property="og:description" content="Edura is a professional and modern School Management & Education Admin Dashboard Template built with Bootstrap. It includes light and dark modes, and is perfect for managing students, teachers, courses, attendance, fees, and academic activities — ideal for schools, colleges, universities, and eLearning platforms.">
  <meta property="og:image" content="https://demos.layoutdrop.com/edura/demo/preview.png">
  <!-- end::Edura Meta Social -->

  <!-- begin::Edura Meta Twitter -->
  <meta name="twitter:card" content="summary_large_image">
  <meta name="twitter:site" content="@LayoutDrop">
  <meta name="twitter:creator" content="@LayoutDrop">
  <meta name="twitter:title" content="Dashboard | Edura School Management & Education Admin Dashboard Template">
  <meta name="twitter:description" content="Edura is a professional and modern School Management & Education Admin Dashboard Template built with Bootstrap. It includes light and dark modes, and is perfect for managing students, teachers, courses, attendance, fees, and academic activities — ideal for schools, colleges, universities, and eLearning platforms.">
  <meta name="twitter:image" content="https://demos.layoutdrop.com/edura/demo/preview.png">
  <!-- end::Edura Meta Twitter -->

  <!-- begin::Edura Website Page Title -->
  <title>Manual Payment Verification | Omateq Digitals</title>
  <!-- end::Edura Website Page Title -->

  <!-- begin::Edura Mobile Specific -->
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <!-- end::Edura Mobile Specific -->

  <!-- begin::Edura Favicon Tags -->
  <?php echo getFaviconHtml(); ?>
  <!-- end::Edura Favicon Tags -->

  <!-- begin::Edura Google Fonts -->
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Plus+Jakarta+Sans:ital,wght@0,200..800;1,200..800&display=swap" rel="stylesheet">
  <!-- end::Edura Google Fonts -->

  <!-- begin::Edura Required Stylesheet -->
  <link rel="stylesheet" href="assets/libs/flaticon/css/all/all.css">
  <link rel="stylesheet" href="assets/libs/lucide/lucide.css">
  <link rel="stylesheet" href="assets/libs/fontawesome/css/all.min.css">
  <link rel="stylesheet" href="assets/libs/simplebar/simplebar.css">
  <link rel="stylesheet" href="assets/libs/node-waves/waves.css">
  <link rel="stylesheet" href="assets/libs/bootstrap-select/css/bootstrap-select.min.css">
  <!-- end::Edura Required Stylesheet -->

  <!-- begin::Edura CSS Stylesheet -->
  <link rel="stylesheet" href="assets/css/styles.css">
  <!-- end::Edura CSS Stylesheet -->
</head>

<body>
  <div class="page-layout">

    <!-- begin::Edura Page Header -->
    <header class="app-header">
      <div class="app-header-inner">
        <button class="app-toggler" type="button">
          <span></span>
          <span></span>
          <span></span>
        </button>
        <div class="app-header-start">
          <form class="d-none d-md-flex align-items-center h-100 w-lg-250px w-xxl-300px position-relative" action="index.html#">
            <button type="button" class="btn btn-sm border-0 position-absolute start-0 ms-3 p-0">
              <i class="fi fi-rr-search"></i>
            </button>
            <input type="text" class="form-control ps-5" placeholder="Search...">
          </form>
        </div>
        <div class="app-header-end">
          <div class="px-lg-3 px-2 ps-0 d-flex align-items-center">
            <a href="javascript:void(0);" class="theme-btn">
              <i class="fi fi-rr-brightness icon-light"></i>
              <div class="theme-toggle"></div>
              <i class="fi fi-rr-moon icon-dark"></i>
            </a>
          </div>
          <div class="vr my-3"></div>
          <div class="d-flex align-items-center gap-sm-2 gap-0 px-lg-4 px-sm-2 px-1">
            <div class="dropdown text-end">
              <a href="javascript:void(0);" class="btn btn-icon btn-action-gray rounded-circle waves-effect waves-light" data-bs-toggle="dropdown" data-bs-auto-close="outside" aria-expanded="true">
                <i class="fi fi-rr-envelope"></i>
              </a>
              <div class="dropdown-menu dropdown-menu-lg-end p-0 w-300px mt-2">
                <div class="px-3 py-3 border-bottom d-flex justify-content-between align-items-center">
                  <h6 class="mb-0">Messages <span class="badge badge-sm rounded-pill bg-primary ms-2"><?php echo $script_replies_count ?? 0; ?></span></h6>
                  <i class="bi bi-x-lg cursor-pointer"></i>
                </div>
                <div class="p-2" style="height: 300px;" data-simplebar>
                  <ul class="list-group list-group-hover list-group-smooth list-group-unlined">
                    <?php if (!empty($script_replies)): ?>
                      <?php foreach ($script_replies as $reply): ?>
                        <li class="list-group-item d-flex justify-content-between align-items-center">
                          <div class="avatar avatar-xs bg-info rounded-circle text-white">
                            <i class="fi fi-rr-user"></i>
                          </div>
                          <div class="ms-2 me-auto">
                            <h6 class="mb-0">Admin Reply</h6>
                            <small class="text-body d-block"><?php echo htmlspecialchars(substr($reply['admin_reply'], 0, 50)) . '...'; ?></small>
                            <small class="text-muted position-absolute end-0 top-0 mt-2 me-3"><?php echo time_ago($reply['created_at']); ?></small>
                          </div>
                        </li>
                      <?php endforeach; ?>
                    <?php else: ?>
                      <li class="list-group-item text-center py-4">
                        <i class="fi fi-rr-envelope text-muted" style="font-size: 2rem;"></i>
                        <p class="text-muted mt-2 mb-0">No script request replies yet</p>
                      </li>
                    <?php endif; ?>
                  </ul>
                </div>
                <div class="p-2">
                  <a href="script-request.php" class="btn w-100 btn-primary waves-effect waves-light">View all replies</a>
                </div>
              </div>
            </div>
            <div class="dropdown text-end">
              <a href="javascript:void(0);" class="btn btn-icon btn-action-gray rounded-circle waves-effect waves-light" data-bs-toggle="dropdown" data-bs-auto-close="outside" aria-expanded="true">
                <i class="fi fi-rr-bell"></i>
              </a>
              <div class="dropdown-menu dropdown-menu-lg-end p-0 w-300px mt-2">
                <div class="px-3 py-3 border-bottom d-flex justify-content-between align-items-center">
                  <h6 class="mb-0">Transactions <span class="badge badge-sm rounded-pill bg-primary ms-2"><?php echo $transactions_count ?? 0; ?></span></h6>
                  <i class="bi bi-x-lg cursor-pointer"></i>
                </div>
                <div class="p-2" style="height: 300px;" data-simplebar>
                  <ul class="list-group list-group-hover list-group-smooth list-group-unlined">
                    <?php if (!empty($transactions)): ?>
                      <?php foreach ($transactions as $transaction): ?>
                        <li class="list-group-item d-flex justify-content-between align-items-center">
                          <div class="avatar avatar-xs bg-<?php echo $transaction['type'] === 'purchase' ? 'success' : 'info'; ?> rounded-circle text-white">
                            <i class="fi fi-rr-<?php echo $transaction['type'] === 'purchase' ? 'shopping-cart' : 'graduation-cap'; ?>"></i>
                          </div>
                          <div class="ms-2 me-auto">
                            <h6 class="mb-0"><?php echo ucfirst($transaction['type']); ?> - <?php echo htmlspecialchars($transaction['item_name']); ?></h6>
                            <small class="text-body d-block"><?php echo $transaction['status'] === 'completed' ? 'Payment successful' : 'Payment pending'; ?> - $<?php echo number_format($transaction['amount'], 2); ?></small>
                            <small class="text-muted position-absolute end-0 top-0 mt-2 me-3"><?php echo time_ago($transaction['created_at']); ?></small>
                          </div>
                        </li>
                      <?php endforeach; ?>
                    <?php else: ?>
                      <li class="list-group-item text-center py-4">
                        <i class="fi fi-rr-shopping-cart text-muted" style="font-size: 2rem;"></i>
                        <p class="text-muted mt-2 mb-0">No transactions yet</p>
                      </li>
                    <?php endif; ?>
                  </ul>
                </div>
                <div class="p-2">
                  <a href="products.php" class="btn w-100 btn-primary waves-effect waves-light">View all transactions</a>
                </div>
              </div>
            </div>
          </div>
          <div class="vr my-3"></div>
          <div class="dropdown text-end ms-2">
            <a href="javascript:void(0);" class="d-flex align-items-center py-2" data-bs-toggle="dropdown" data-bs-auto-close="outside" aria-expanded="true">
              <div class="avatar avatar-sm rounded-circle avatar-status-success">
                <?php if ($user['avatar'] && file_exists($user['avatar'])): ?>
                  <img src="<?php echo htmlspecialchars($user['avatar']); ?>" alt="<?php echo htmlspecialchars($user['full_name']); ?>">
                <?php else: ?>
                  <img src="assets/images/avatar/avatar1.webp" alt="<?php echo htmlspecialchars($user['full_name']); ?>">
                <?php endif; ?>
              </div>
            </a>
            <ul class="dropdown-menu dropdown-menu-end w-225px mt-1">
              <li class="d-flex align-items-center p-2">
                <div class="avatar avatar-sm rounded-circle">
                  <?php if ($user['avatar'] && file_exists($user['avatar'])): ?>
                    <img src="<?php echo htmlspecialchars($user['avatar']); ?>" alt="<?php echo htmlspecialchars($user['full_name']); ?>">
                  <?php else: ?>
                    <img src="assets/images/avatar/avatar1.webp" alt="<?php echo htmlspecialchars($user['full_name']); ?>">
                  <?php endif; ?>
                </div>
                <div class="ms-2">
                  <div class="fw-bold text-dark"><?php echo htmlspecialchars($user['full_name']); ?></div>
                  <small class="text-body d-block lh-sm"><?php echo htmlspecialchars($user['email']); ?></small>
                </div>
              </li>
              <li>
                <div class="dropdown-divider my-1"></div>
              </li>
              <li>
                <a class="dropdown-item d-flex align-items-center gap-2" href="profile.php">
                  <i class="fi fi-rr-user scale-1x"></i> Profile
                </a>
              </li>
              <li>
                <a class="dropdown-item d-flex align-items-center gap-2" href="help-center.php">
                  <i class="fi fi-rs-interrogation scale-1x"></i> Help Center
                </a>
              </li>
              <li>
                <div class="dropdown-divider my-1"></div>
              </li>
              <li>
                <a class="dropdown-item d-flex align-items-center gap-2 text-danger" href="authentication/logout.php">
                  <i class="fi fi-sr-exit scale-1x"></i> Logout
                </a>
              </li>
            </ul>
          </div>
        </div>
      </div>
    </header>
    <!-- end::Edura Page Header -->

    <!-- begin::Edura Sidebar Menu -->
    <aside class="app-menubar" id="menubar">
      <div class="app-navbar-brand">
        <a class="navbar-brand-logo" href="index.php">
          <img src="assets/images/logo.svg" alt="logo">
        </a>
        <a class="navbar-brand-mini" href="index.php">
          <img src="assets/images/logo-text-white.svg" alt="logo">
        </a>
      </div>
      <nav class="app-navbar" data-simplebar>
        <ul class="menubar">
              <li class="menu-item">
                <a class="menu-link" href="index.php">
              <i class="fi fi-rr-apps"></i>
                  <span class="menu-label">Dashboard</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="products.php">
              <i class="fi fi-rr-shopping-cart"></i>
              <span class="menu-label">Products</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="course.php">
              <i class="fi fi-rr-graduation-cap"></i>
              <span class="menu-label">Course</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="profile.php">
              <i class="fi fi-rr-user"></i>
              <span class="menu-label">Profile</span>
                </a>
              </li>
              
        </ul>
      </nav>
      <div class="app-footer">
        <div class="text-center py-2">
          <small class="text-white-50">
            <i class="fi fi-rr-copyright me-1"></i>
            2025 Omateq Digitals
          </small>
        </div>
        <a href="mailto:info@omateqdigitals.com" class="btn btn-outline-white waves-effect btn-app-nav w-100">
          <i class="fi fi-rs-interrogation"></i>
          <span class="nav-text">Help and Support</span>
        </a>
      </div>
    </aside>
    <!-- end::Edura Sidebar Menu -->

    <main class="app-wrapper">
      <div class="container-fluid">
        <div class="app-page-head d-flex flex-wrap gap-3 align-items-center justify-content-between">
          <div class="clearfix">
            <h1 class="app-page-title">Manual Payment Verification</h1>
            <span>Verify your payment if automatic callback failed</span>
          </div>
        </div>

        <!-- Content -->
        <div class="row justify-content-center">
          <div class="col-lg-8 col-xl-6">
            <div class="card">
              <div class="card-header">
                <h5 class="card-title mb-0">
                  <i class="fi fi-rr-credit-card me-2"></i>
                  Verify Payment
                </h5>
              </div>
              <div class="card-body">
                <?php if (!empty($success)): ?>
                  <div class="alert alert-success alert-dismissible fade show" role="alert">
                    <i class="fi fi-rr-check-circle me-2"></i>
                    <?php echo htmlspecialchars($success); ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                  </div>
                <?php endif; ?>

                <?php if (!empty($error)): ?>
                  <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <i class="fi fi-rr-exclamation-triangle me-2"></i>
                    <?php echo htmlspecialchars($error); ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                  </div>
                <?php endif; ?>

                <form method="POST">
                  <input type="hidden" name="action" value="verify_payment">
                  
                  <div class="mb-3">
                    <label for="order_id" class="form-label">Order ID</label>
                    <input type="text" class="form-control" id="order_id" name="order_id" 
                           value="<?php echo htmlspecialchars($_GET['order_id'] ?? ''); ?>" required>
                    <div class="form-text">Enter the order ID from your payment</div>
                  </div>
                  
                  <div class="mb-3">
                    <label for="reference" class="form-label">Payment Reference</label>
                    <input type="text" class="form-control" id="reference" name="reference" 
                           placeholder="Enter your Paystack reference" required>
                    <div class="form-text">Enter the reference number from your Paystack payment</div>
                  </div>
                  
                  <div class="d-grid">
                    <button type="submit" class="btn btn-primary">
                      <i class="fi fi-rr-check me-2"></i>
                      Verify Payment
                    </button>
                  </div>
                </form>
                
                <div class="mt-4">
                  <div class="card border-info">
                    <div class="card-header bg-info-subtle">
                      <h6 class="mb-0">
                        <i class="fi fi-rr-info me-2"></i>
                        How to find your payment reference
                      </h6>
                    </div>
                    <div class="card-body">
                      <ol class="mb-0">
                        <li>Check your email for Paystack payment confirmation</li>
                        <li>Look for the reference number in the email</li>
                        <li>Or check your Paystack dashboard</li>
                      </ol>
                    </div>
                  </div>
                </div>
                
                <div class="mt-3 d-flex gap-2">
                  <a href="products.php" class="btn btn-outline-secondary">
                    <i class="fi fi-rr-arrow-left me-2"></i>
                    Back to Products
                  </a>
                  <a href="index.php" class="btn btn-outline-primary">
                    <i class="fi fi-rr-home me-2"></i>
                    Dashboard
                  </a>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </main>

  </div>

  <!-- begin::Edura Scripts -->
  <script src="assets/libs/global/global.min.js"></script>
  <script src="assets/js/appSettings.js"></script>
  <script src="assets/js/main.js"></script>
  <!-- end::Edura Scripts -->

  <!-- WhatsApp Chat Button -->
  <div class="whatsapp-chat-button">
    <a href="https://wa.me/2347069378861?text=Hello%20Omateq%20Digitals!%20I%20would%20like%20to%20know%20more%20about%20your%20services." 
       target="_blank" 
       rel="noopener noreferrer"
       class="whatsapp-link"
       title="Chat with us on WhatsApp">
      <i class="fab fa-whatsapp"></i>
      <span class="whatsapp-text">Chat with us</span>
    </a>
  </div>
  
  <style>
  /* WhatsApp Chat Button Styles */
  .whatsapp-chat-button {
    position: fixed;
    bottom: 20px;
    right: 20px;
    z-index: 1000;
    animation: pulse 2s infinite;
  }
  
  .whatsapp-link {
    display: flex;
    align-items: center;
    background: #25D366;
    color: white;
    text-decoration: none;
    padding: 12px 16px;
    border-radius: 50px;
    box-shadow: 0 4px 12px rgba(37, 211, 102, 0.4);
    transition: all 0.3s ease;
    font-family: 'Poppins', sans-serif;
    font-weight: 500;
    font-size: 14px;
  }
  
  .whatsapp-link:hover {
    background: #128C7E;
    color: white;
    text-decoration: none;
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(37, 211, 102, 0.6);
  }
  
  .whatsapp-link i {
    font-size: 24px;
    margin-right: 8px;
  }
  
  .whatsapp-text {
    white-space: nowrap;
  }
  
  @keyframes pulse {
    0% {
      box-shadow: 0 0 0 0 rgba(37, 211, 102, 0.7);
    }
    70% {
      box-shadow: 0 0 0 10px rgba(37, 211, 102, 0);
    }
    100% {
      box-shadow: 0 0 0 0 rgba(37, 211, 102, 0);
    }
  }
  
  /* Mobile responsiveness */
  @media (max-width: 768px) {
    .whatsapp-chat-button {
      bottom: 15px;
      right: 15px;
    }
    
    .whatsapp-link {
      padding: 10px 14px;
      font-size: 13px;
    }
    
    .whatsapp-link i {
      font-size: 22px;
      margin-right: 6px;
    }
    
    .whatsapp-text {
      display: none;
    }
  }
  </style>

</body>

</html>
