<?php
session_start();
require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/../admin-dashboard/config/settings.php';
require_once __DIR__ . '/../admin-dashboard/config/branding.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: authentication/login.php');
    exit();
}

// Get user data
try {
    $pdo = getConnection();
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $user = $stmt->fetch();
    
    if (!$user) {
        session_destroy();
        header('Location: authentication/login.php');
        exit();
    }
} catch (Exception $e) {
    session_destroy();
    header('Location: authentication/login.php');
    exit();
}

$error = '';
$success = '';

// Get user's enrolled courses
$enrolled_courses = [];
try {
    $pdo = getConnection();
    $stmt = $pdo->prepare("
        SELECT ce.*, c.title, c.description, c.thumbnail, c.price, c.duration, c.level, c.instructor, c.curriculum, c.status as course_status
        FROM course_enrollments ce 
        JOIN courses c ON ce.course_id = c.id 
        WHERE ce.user_id = ? 
        ORDER BY ce.enrolled_at DESC
    ");
    $stmt->execute([$_SESSION['user_id']]);
    $enrolled_courses = $stmt->fetchAll();
} catch (Exception $e) {
    $error = "Error loading courses: " . $e->getMessage();
}

// Get script request replies
$script_replies = [];
$script_replies_count = 0;
try {
    $stmt = $pdo->prepare("SELECT * FROM script_requests WHERE user_id = ? AND admin_reply IS NOT NULL ORDER BY created_at DESC LIMIT 5");
    $stmt->execute([$_SESSION['user_id']]);
    $script_replies = $stmt->fetchAll();
    $script_replies_count = count($script_replies);
} catch (Exception $e) {
    // Handle error silently
}

// Get transactions (purchases and course enrollments)
$transactions = [];
$transactions_count = 0;
try {
    $stmt = $pdo->prepare("SELECT 'purchase' as type, product_name as item_name, total_amount as amount, status, created_at FROM orders WHERE user_id = ? 
                          UNION ALL 
                          SELECT 'course' as type, title as item_name, price as amount, 'completed' as status, created_at FROM course_enrollments ce 
                          JOIN courses c ON ce.course_id = c.id WHERE ce.user_id = ? 
                          ORDER BY created_at DESC LIMIT 5");
    $stmt->execute([$_SESSION['user_id'], $_SESSION['user_id']]);
    $transactions = $stmt->fetchAll();
    $transactions_count = count($transactions);
} catch (Exception $e) {
    // Handle error silently
}

// Function to calculate progress based on completed lessons
function calculateProgress($enrollment) {
    global $pdo;
    
    try {
        // Get total lessons for this course
        $stmt = $pdo->prepare("SELECT COUNT(*) as total_lessons FROM course_lessons WHERE course_id = ? AND is_active = 'yes'");
        $stmt->execute([$enrollment['course_id']]);
        $total_lessons = $stmt->fetch()['total_lessons'];
        
        if ($total_lessons == 0) {
            return 0;
        }
        
        // Get completed lessons for this user and course
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as completed_lessons 
            FROM video_progress 
            WHERE user_id = ? AND course_id = ? AND is_completed = 1
        ");
        $stmt->execute([$enrollment['user_id'], $enrollment['course_id']]);
        $completed_lessons = $stmt->fetch()['completed_lessons'];
        
        // Calculate percentage
        $progress = ($completed_lessons / $total_lessons) * 100;
        return round($progress, 1);
        
    } catch (Exception $e) {
        error_log("Progress calculation error: " . $e->getMessage());
        // Fallback to database progress_percentage if available
        if (isset($enrollment['progress_percentage']) && $enrollment['progress_percentage'] !== null) {
            return round($enrollment['progress_percentage'], 1);
        }
        return 0;
    }
}

// Function to get course status based on lesson completion
function getCourseStatus($enrollment) {
    $progress = calculateProgress($enrollment);
    
    // Course is completed only when all lessons are completed (100% progress)
    if ($progress >= 100) {
        // Auto-complete course if all lessons are done
        autoCompleteCourse($enrollment);
        return 'completed';
    } elseif ($progress > 80) {
        return 'almost_complete';
    } elseif ($progress > 0) {
        return 'in_progress';
    } else {
        return 'not_started';
    }
}

// Function to automatically mark course as completed when all lessons are done
function autoCompleteCourse($enrollment) {
    global $pdo;
    
    try {
        // Only update if not already completed
        if ($enrollment['status'] !== 'completed') {
            $stmt = $pdo->prepare("
                UPDATE course_enrollments 
                SET status = 'completed', completed_at = NOW() 
                WHERE user_id = ? AND course_id = ? AND status != 'completed'
            ");
            $stmt->execute([$enrollment['user_id'], $enrollment['course_id']]);
        }
    } catch (Exception $e) {
        error_log("Auto-complete course error: " . $e->getMessage());
    }
}

// Function to format time ago
function time_ago($datetime) {
    $time = time() - strtotime($datetime);
    if ($time < 60) return 'just now';
    if ($time < 3600) return floor($time/60) . ' min ago';
    if ($time < 86400) return floor($time/3600) . ' hr ago';
    if ($time < 2592000) return floor($time/86400) . ' days ago';
    return date('M j', strtotime($datetime));
}
?>
<!DOCTYPE html>
<html lang="en">

<head>
  <!-- begin::Edura Meta Basic -->
  <meta charset="utf-8">
  <meta name="theme-color" content="#0D9488">
  <meta name="robots" content="index, follow">
  <meta name="author" content="LayoutDrop">
  <meta name="format-detection" content="telephone=no">
  <meta name="keywords" content="School Management, Education Admin Dashboard, Student Information System, University Dashboard, College Admin Template, Learning Management System, Education CRM, Teacher Management, Student Attendance Dashboard, Online Education Admin, Responsive Admin Template, Bootstrap 5 Education Admin, eLearning Dashboard, School ERP, Class Management Dashboard, Academic Dashboard, Education Web Application, Modern Admin Template, Admin UI Kit, ThemeForest Admin Template">
  <meta name="description" content="Edura is a professional and modern School Management & Education Admin Dashboard Template built with Bootstrap. It includes light and dark modes, and is perfect for managing students, teachers, courses, attendance, fees, and academic activities — ideal for schools, colleges, universities, and eLearning platforms.">
  <!-- end::Edura Meta Basic -->

  <!-- begin::Edura Meta Social -->
  <meta property="og:url" content="https://demos.layoutdrop.com/edura/demo/">
  <meta property="og:site_name" content="Dashboard | Edura School Management & Education Admin Dashboard Template">
  <meta property="og:type" content="website">
  <meta property="og:locale" content="en_us">
  <meta property="og:title" content="Dashboard | Edura School Management & Education Admin Dashboard Template">
  <meta property="og:description" content="Edura is a professional and modern School Management & Education Admin Dashboard Template built with Bootstrap. It includes light and dark modes, and is perfect for managing students, teachers, courses, attendance, fees, and academic activities — ideal for schools, colleges, universities, and eLearning platforms.">
  <meta property="og:image" content="https://demos.layoutdrop.com/edura/demo/preview.png">
  <!-- end::Edura Meta Social -->

  <!-- begin::Edura Meta Twitter -->
  <meta name="twitter:card" content="summary_large_image">
  <meta name="twitter:site" content="@LayoutDrop">
  <meta name="twitter:creator" content="@LayoutDrop">
  <meta name="twitter:title" content="Dashboard | Edura School Management & Education Admin Dashboard Template">
  <meta name="twitter:description" content="Edura is a professional and modern School Management & Education Admin Dashboard Template built with Bootstrap. It includes light and dark modes, and is perfect for managing students, teachers, courses, attendance, fees, and academic activities — ideal for schools, colleges, universities, and eLearning platforms.">
  <meta name="twitter:image" content="https://demos.layoutdrop.com/edura/demo/preview.png">
  <!-- end::Edura Meta Twitter -->

  <!-- begin::Edura Website Page Title -->
  <title>My Courses | Omateq Digitals</title>
  <!-- end::Edura Website Page Title -->

  <!-- begin::Edura Mobile Specific -->
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <!-- end::Edura Mobile Specific -->

  <!-- begin::Edura Favicon Tags -->
  <?php echo getFaviconHtml(); ?>
  <!-- end::Edura Favicon Tags -->

  <!-- begin::Edura Google Fonts -->
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Plus+Jakarta+Sans:ital,wght@0,200..800;1,200..800&display=swap" rel="stylesheet">
  <!-- end::Edura Google Fonts -->

  <!-- begin::Edura Required Stylesheet -->
  <link rel="stylesheet" href="assets/libs/flaticon/css/all/all.css">
  <link rel="stylesheet" href="assets/libs/lucide/lucide.css">
  <link rel="stylesheet" href="assets/libs/fontawesome/css/all.min.css">
  <link rel="stylesheet" href="assets/libs/simplebar/simplebar.css">
  <link rel="stylesheet" href="assets/libs/node-waves/waves.css">
  <link rel="stylesheet" href="assets/libs/bootstrap-select/css/bootstrap-select.min.css">
  <!-- end::Edura Required Stylesheet -->

  <!-- begin::Edura CSS Stylesheet -->
  <link rel="stylesheet" href="assets/css/styles.css">
  <!-- end::Edura CSS Stylesheet -->
  <style>
    /* Logo sizing for better fit */
    .app-navbar-brand .navbar-brand-logo img {
      max-height: 50px;
      max-width: 180px;
      width: auto;
      height: auto;
      object-fit: contain;
    }
    
    .app-navbar-brand .navbar-brand-mini img {
      max-height: 35px;
      max-width: 35px;
      width: auto;
      height: auto;
      object-fit: contain;
    }
  </style>
  
  <!-- Custom CSS for course layout -->
  <style>
    .course-card {
      transition: all 0.3s ease;
      border: none;
      box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    }
    .course-card:hover {
      transform: translateY(-5px);
      box-shadow: 0 5px 20px rgba(0,0,0,0.15);
    }
    .course-thumbnail {
      height: 200px;
      object-fit: cover;
      border-radius: 10px 10px 0 0;
    }
    .enrolled-badge {
      position: absolute;
      top: 10px;
      right: 10px;
      z-index: 2;
    }
    .course-level {
      font-size: 0.8rem;
      padding: 4px 8px;
      border-radius: 12px;
    }
    .level-beginner { background-color: #d4edda; color: #155724; }
    .level-intermediate { background-color: #fff3cd; color: #856404; }
    .level-advanced { background-color: #f8d7da; color: #721c24; }
    .stats-card {
      background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
      color: white;
      border-radius: 15px;
    }
    .progress-bar {
      height: 6px;
      border-radius: 3px;
    }
  </style>
</head>

<body>
  <div class="page-layout">

    <!-- begin::Edura Page Header -->
    <header class="app-header">
      <div class="app-header-inner">
        <button class="app-toggler" type="button">
          <span></span>
          <span></span>
          <span></span>
        </button>
        <div class="app-header-start">
          <form class="d-none d-md-flex align-items-center h-100 w-lg-250px w-xxl-300px position-relative" action="index.html#">
            <button type="button" class="btn btn-sm border-0 position-absolute start-0 ms-3 p-0">
              <i class="fi fi-rr-search"></i>
            </button>
            <input type="text" class="form-control ps-5" placeholder="Search courses...">
          </form>
        </div>
        <div class="app-header-end">
          <div class="px-lg-3 px-2 ps-0 d-flex align-items-center">
            <a href="javascript:void(0);" class="theme-btn">
              <i class="fi fi-rr-brightness icon-light"></i>
              <div class="theme-toggle"></div>
              <i class="fi fi-rr-moon icon-dark"></i>
            </a>
          </div>
          <div class="vr my-3"></div>
          <div class="d-flex align-items-center gap-sm-2 gap-0 px-lg-4 px-sm-2 px-1">
            <div class="dropdown text-end">
              <a href="javascript:void(0);" class="btn btn-icon btn-action-gray rounded-circle waves-effect waves-light" data-bs-toggle="dropdown" data-bs-auto-close="outside" aria-expanded="true">
                <i class="fi fi-rr-envelope"></i>
              </a>
              <div class="dropdown-menu dropdown-menu-lg-end p-0 w-300px mt-2">
                <div class="px-3 py-3 border-bottom d-flex justify-content-between align-items-center">
                  <h6 class="mb-0">Messages <span class="badge badge-sm rounded-pill bg-primary ms-2"><?php echo $script_replies_count ?? 0; ?></span></h6>
                  <i class="bi bi-x-lg cursor-pointer"></i>
                </div>
                <div class="p-2" style="height: 300px;" data-simplebar>
                  <ul class="list-group list-group-hover list-group-smooth list-group-unlined">
                    <?php if (!empty($script_replies)): ?>
                      <?php foreach ($script_replies as $reply): ?>
                        <li class="list-group-item d-flex justify-content-between align-items-center">
                          <div class="avatar avatar-xs bg-info rounded-circle text-white">
                            <i class="fi fi-rr-user"></i>
                          </div>
                          <div class="ms-2 me-auto">
                            <h6 class="mb-0">Admin Reply</h6>
                            <small class="text-body d-block"><?php echo htmlspecialchars(substr($reply['admin_reply'], 0, 50)) . '...'; ?></small>
                            <small class="text-muted position-absolute end-0 top-0 mt-2 me-3"><?php echo time_ago($reply['created_at']); ?></small>
                          </div>
                        </li>
                      <?php endforeach; ?>
                    <?php else: ?>
                      <li class="list-group-item text-center py-4">
                        <i class="fi fi-rr-envelope text-muted" style="font-size: 2rem;"></i>
                        <p class="text-muted mt-2 mb-0">No script request replies yet</p>
                      </li>
                    <?php endif; ?>
                  </ul>
                </div>
                <div class="p-2">
                  <a href="script-request.php" class="btn w-100 btn-primary waves-effect waves-light">View all replies</a>
                </div>
              </div>
            </div>
            <div class="dropdown text-end">
              <a href="javascript:void(0);" class="btn btn-icon btn-action-gray rounded-circle waves-effect waves-light" data-bs-toggle="dropdown" data-bs-auto-close="outside" aria-expanded="true">
                <i class="fi fi-rr-bell"></i>
              </a>
              <div class="dropdown-menu dropdown-menu-lg-end p-0 w-300px mt-2">
                <div class="px-3 py-3 border-bottom d-flex justify-content-between align-items-center">
                  <h6 class="mb-0">Transactions <span class="badge badge-sm rounded-pill bg-primary ms-2"><?php echo $transactions_count ?? 0; ?></span></h6>
                  <i class="bi bi-x-lg cursor-pointer"></i>
                </div>
                <div class="p-2" style="height: 300px;" data-simplebar>
                  <ul class="list-group list-group-hover list-group-smooth list-group-unlined">
                    <?php if (!empty($transactions)): ?>
                      <?php foreach ($transactions as $transaction): ?>
                        <li class="list-group-item d-flex justify-content-between align-items-center">
                          <div class="avatar avatar-xs bg-<?php echo $transaction['type'] === 'purchase' ? 'success' : 'info'; ?> rounded-circle text-white">
                            <i class="fi fi-rr-<?php echo $transaction['type'] === 'purchase' ? 'shopping-cart' : 'graduation-cap'; ?>"></i>
                          </div>
                          <div class="ms-2 me-auto">
                            <h6 class="mb-0"><?php echo ucfirst($transaction['type']); ?> - <?php echo htmlspecialchars($transaction['item_name']); ?></h6>
                            <small class="text-body d-block"><?php echo $transaction['status'] === 'completed' ? 'Payment successful' : 'Payment pending'; ?> - $<?php echo number_format($transaction['amount'], 2); ?></small>
                            <small class="text-muted position-absolute end-0 top-0 mt-2 me-3"><?php echo time_ago($transaction['created_at']); ?></small>
                          </div>
                        </li>
                      <?php endforeach; ?>
                    <?php else: ?>
                      <li class="list-group-item text-center py-4">
                        <i class="fi fi-rr-shopping-cart text-muted" style="font-size: 2rem;"></i>
                        <p class="text-muted mt-2 mb-0">No transactions yet</p>
                      </li>
                    <?php endif; ?>
                  </ul>
                </div>
                <div class="p-2">
                  <a href="products.php" class="btn w-100 btn-primary waves-effect waves-light">View all transactions</a>
                </div>
              </div>
            </div>
          </div>
          <div class="vr my-3"></div>
          <div class="dropdown text-end ms-2">
            <a href="javascript:void(0);" class="d-flex align-items-center py-2" data-bs-toggle="dropdown" data-bs-auto-close="outside" aria-expanded="true">
              <div class="avatar avatar-sm rounded-circle avatar-status-success">
                <?php if ($user && $user['avatar'] && file_exists($user['avatar'])): ?>
                  <img src="<?php echo htmlspecialchars($user['avatar']); ?>" alt="<?php echo htmlspecialchars($user['full_name']); ?>">
                <?php else: ?>
                  <img src="assets/images/avatar/avatar1.webp" alt="<?php echo htmlspecialchars($user['full_name']); ?>">
                <?php endif; ?>
              </div>
            </a>
            <ul class="dropdown-menu dropdown-menu-end w-225px mt-1">
              <li class="d-flex align-items-center p-2">
                <div class="avatar avatar-sm rounded-circle">
                  <?php if ($user && $user['avatar'] && file_exists($user['avatar'])): ?>
                    <img src="<?php echo htmlspecialchars($user['avatar']); ?>" alt="<?php echo htmlspecialchars($user['full_name']); ?>">
                  <?php else: ?>
                    <img src="assets/images/avatar/avatar1.webp" alt="<?php echo htmlspecialchars($user['full_name']); ?>">
                  <?php endif; ?>
                </div>
                <div class="ms-2">
                  <div class="fw-bold text-dark"><?php echo htmlspecialchars($user['full_name']); ?></div>
                  <small class="text-body d-block lh-sm"><?php echo htmlspecialchars($user['email']); ?></small>
                </div>
              </li>
              <li>
                <div class="dropdown-divider my-1"></div>
              </li>
              <li>
                <a class="dropdown-item d-flex align-items-center gap-2" href="profile.php">
                  <i class="fi fi-rr-user scale-1x"></i> Profile
                </a>
              </li>
              <li>
                <a class="dropdown-item d-flex align-items-center gap-2" href="help-center.php">
                  <i class="fi fi-rs-interrogation scale-1x"></i> Help Center
                </a>
              </li>
              <li>
                <div class="dropdown-divider my-1"></div>
              </li>
              <li>
                <a class="dropdown-item d-flex align-items-center gap-2 text-danger" href="authentication/logout.php">
                  <i class="fi fi-sr-exit scale-1x"></i> Logout
                </a>
              </li>
            </ul>
          </div>
        </div>
      </div>
    </header>
    <!-- end::Edura Page Header -->

    <!-- begin::Edura Sidebar Menu -->
    <aside class="app-menubar" id="menubar">
      <div class="app-navbar-brand">
        <a class="navbar-brand-logo" href="index.php">
          <?php echo '<img src="' . '../' . getSetting('main_logo', 'assets/images/logo.svg') . '" alt="logo">'; ?>
        </a>
        <a class="navbar-brand-mini" href="index.php">
          <?php echo '<img src="' . '../' . getSetting('mini_logo', 'assets/images/logo-text-white.svg') . '" alt="logo">'; ?>
        </a>
      </div>
      <nav class="app-navbar" data-simplebar>
        <ul class="menubar">
              <li class="menu-item">
                <a class="menu-link" href="index.php">
              <i class="fi fi-rr-apps"></i>
                  <span class="menu-label">Dashboard</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="products.php">
              <i class="fi fi-rr-shopping-cart"></i>
              <span class="menu-label">Products</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="course.php">
              <i class="fi fi-rr-graduation-cap"></i>
              <span class="menu-label">Course</span>
                </a>
              </li>
              <li class="menu-item active">
            <a class="menu-link" href="my_courses.php">
              <i class="fi fi-rr-book"></i>
              <span class="menu-label">My Courses</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="profile.php">
              <i class="fi fi-rr-user"></i>
              <span class="menu-label">Profile</span>
                </a>
              </li>
              
        </ul>
      </nav>
      <div class="app-footer">
        <div class="text-center py-2">
          <small class="text-white-50">
            <i class="fi fi-rr-copyright me-1"></i>
            2025 Omateq Digitals
          </small>
        </div>
        <a href="mailto:info@omateqdigitals.com" class="btn btn-outline-white waves-effect btn-app-nav w-100">
          <i class="fi fi-rs-interrogation"></i>
          <span class="nav-text">Help and Support</span>
        </a>
      </div>
    </aside>
    <!-- end::Edura Sidebar Menu -->

    <main class="app-wrapper">
      <div class="container-fluid">
        <div class="app-page-head d-flex flex-wrap gap-3 align-items-center justify-content-between">
          <div class="clearfix">
            <h1 class="app-page-title">My Courses</h1>
            <span>View and manage your enrolled courses</span>
          </div>
          <div class="app-page-head-actions">
            <a href="course.php" class="btn btn-primary">
              <i class="fi fi-rr-plus me-2"></i> Browse More Courses
            </a>
          </div>
        </div>

        <!-- Success/Error Messages -->
        <?php if ($error): ?>
          <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fi fi-rr-exclamation-triangle me-2"></i>
            <?php echo htmlspecialchars($error); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
          </div>
        <?php endif; ?>
        <?php if ($success): ?>
          <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fi fi-rr-check-circle me-2"></i>
            <?php echo htmlspecialchars($success); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
          </div>
        <?php endif; ?>

        <!-- Course Statistics -->
        <div class="row mb-4">
          <div class="col-md-3 mb-3">
            <div class="card stats-card p-3">
              <div class="d-flex align-items-center">
                <div class="icon-square bg-white-opacity-20 rounded-3 me-3">
                  <i class="fi fi-rr-graduation-cap text-white fs-4"></i>
                </div>
                <div>
                  <h5 class="mb-0 text-white"><?php echo count($enrolled_courses); ?></h5>
                  <p class="text-white-50 mb-0">Enrolled Courses</p>
                </div>
              </div>
            </div>
          </div>
          <div class="col-md-3 mb-3">
            <div class="card stats-card p-3" style="background: linear-gradient(135deg, #f6d365 0%, #fda085 100%);">
              <div class="d-flex align-items-center">
                <div class="icon-square bg-white-opacity-20 rounded-3 me-3">
                  <i class="fi fi-rr-check-circle text-white fs-4"></i>
                </div>
                <div>
                  <h5 class="mb-0 text-white"><?php echo count(array_filter($enrolled_courses, function($course) { return $course['status'] === 'completed'; })); ?></h5>
                  <p class="text-white-50 mb-0">Completed</p>
                </div>
              </div>
            </div>
          </div>
          <div class="col-md-3 mb-3">
            <div class="card stats-card p-3" style="background: linear-gradient(135deg, #a18cd1 0%, #fbc2eb 100%);">
              <div class="d-flex align-items-center">
                <div class="icon-square bg-white-opacity-20 rounded-3 me-3">
                  <i class="fi fi-rr-clock text-white fs-4"></i>
                </div>
                <div>
                  <h5 class="mb-0 text-white"><?php echo count(array_filter($enrolled_courses, function($course) { return $course['status'] === 'enrolled'; })); ?></h5>
                  <p class="text-white-50 mb-0">In Progress</p>
                </div>
              </div>
            </div>
          </div>
          <div class="col-md-3 mb-3">
            <div class="card stats-card p-3" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);">
              <div class="d-flex align-items-center">
                <div class="icon-square bg-white-opacity-20 rounded-3 me-3">
                  <i class="fi fi-rr-book text-white fs-4"></i>
                </div>
                <div>
                  <h5 class="mb-0 text-white"><?php echo count(array_filter($enrolled_courses, function($course) { return $course['status'] === 'not_started'; })); ?></h5>
                  <p class="text-white-50 mb-0">Not Started</p>
                </div>
              </div>
            </div>
          </div>
        </div>

        <!-- Enrolled Courses -->
        <?php if (!empty($enrolled_courses)): ?>
          <div class="row">
            <?php foreach ($enrolled_courses as $course): ?>
              <?php 
              $progress = calculateProgress($course);
              $status = getCourseStatus($course);
              ?>
              <div class="col-lg-4 col-md-6 mb-4">
                <div class="card course-card h-100">
                  <div class="position-relative">
                    <?php if (isset($course['thumbnail']) && !empty($course['thumbnail'])): ?>
                      <img src="../<?php echo htmlspecialchars($course['thumbnail']); ?>" class="card-img-top course-thumbnail" alt="<?php echo htmlspecialchars($course['title'] ?? 'Course'); ?>">
                    <?php else: ?>
                      <div class="card-img-top course-thumbnail bg-light d-flex align-items-center justify-content-center">
                        <i class="fi fi-rr-graduation-cap text-muted" style="font-size: 3rem;"></i>
                      </div>
                    <?php endif; ?>
                    <div class="enrolled-badge">
                      <?php if ($status === 'completed'): ?>
                        <span class="badge bg-success">
                          <i class="fi fi-rr-check me-1"></i> Completed
                        </span>
                      <?php elseif ($status === 'almost_complete'): ?>
                        <span class="badge bg-warning">
                          <i class="fi fi-rr-clock me-1"></i> Almost Complete
                        </span>
                      <?php elseif ($status === 'in_progress'): ?>
                        <span class="badge bg-info">
                          <i class="fi fi-rr-play me-1"></i> In Progress
                        </span>
                      <?php else: ?>
                        <span class="badge bg-secondary">
                          <i class="fi fi-rr-book me-1"></i> Not Started
                        </span>
                      <?php endif; ?>
                    </div>
                  </div>
                  <div class="card-body d-flex flex-column">
                    <h5 class="card-title"><?php echo htmlspecialchars($course['title'] ?? 'Course'); ?></h5>
                    <p class="text-muted small mb-3"><?php echo htmlspecialchars(substr($course['description'] ?? '', 0, 100)) . (strlen($course['description'] ?? '') > 100 ? '...' : ''); ?></p>
                    
                    <!-- Course Details -->
                    <div class="mb-3">
                      <div class="d-flex align-items-center mb-1">
                        <i class="fi fi-rr-user me-2 text-muted"></i>
                        <small class="text-muted"><?php echo htmlspecialchars($course['instructor'] ?? 'Instructor'); ?></small>
                      </div>
                      <div class="d-flex align-items-center mb-1">
                        <i class="fi fi-rr-clock me-2 text-muted"></i>
                        <small class="text-muted"><?php echo htmlspecialchars($course['duration'] ?? 'Duration not specified'); ?></small>
                      </div>
                      <div class="d-flex align-items-center mb-1">
                        <i class="fi fi-rr-signal me-2 text-muted"></i>
                        <small class="text-muted"><?php echo ucfirst($course['level'] ?? 'beginner'); ?> Level</small>
                      </div>
                    </div>

                    <!-- Progress Bar -->
                    <div class="mb-3">
                      <div class="d-flex justify-content-between align-items-center mb-1">
                        <small class="text-muted">Progress</small>
                        <small class="text-muted"><?php echo $progress; ?>%</small>
                      </div>
                      <div class="progress">
                        <div class="progress-bar <?php echo $status === 'completed' ? 'bg-success' : ($status === 'almost_complete' ? 'bg-warning' : 'bg-primary'); ?>" 
                             style="width: <?php echo $progress; ?>%"
                             role="progressbar">
                        </div>
                      </div>
                    </div>

                    <!-- Price -->
                    <div class="mb-3">
                      <span class="h5 text-primary">$<?php echo number_format($course['price'], 2); ?></span>
                    </div>

                    <!-- Action Buttons -->
                    <div class="mt-auto">
                      <?php if ($status === 'completed'): ?>
                        <div class="d-grid gap-2">
                          <a href="course_learning.php?course_id=<?php echo $course['course_id']; ?>" 
                             class="btn btn-success w-100">
                            <i class="fi fi-rr-trophy me-1"></i>
                            View Certificate
                          </a>
                          <a href="course_learning.php?course_id=<?php echo $course['course_id']; ?>" 
                             class="btn btn-outline-primary w-100">
                            <i class="fi fi-rr-refresh me-1"></i>
                            Review Course
                          </a>
                        </div>
                      <?php else: ?>
                        <div class="d-grid gap-2">
                          <a href="course_learning.php?course_id=<?php echo $course['course_id']; ?>" 
                             class="btn btn-primary w-100">
                            <i class="fi fi-rr-play me-1"></i>
                            <?php echo $status === 'not_started' ? 'Start Course' : 'Continue Course'; ?>
                          </a>
                          <?php if ($progress > 0 && $progress < 100): ?>
                            <button class="btn btn-outline-warning btn-sm w-100" 
                                    onclick="confirmManualComplete(<?php echo $course['course_id']; ?>, '<?php echo htmlspecialchars($course['title']); ?>', <?php echo $progress; ?>)">
                              <i class="fi fi-rr-check me-1"></i>
                              Mark as Complete
                            </button>
                          <?php endif; ?>
                        </div>
                      <?php endif; ?>
                    </div>
                  </div>
                  
                  <!-- Card Footer -->
                  <div class="card-footer bg-transparent border-0 pt-0">
                    <div class="d-flex justify-content-between align-items-center">
                      <small class="text-muted">
                        <i class="fi fi-rr-calendar me-1"></i>
                        Enrolled <?php echo date('M d, Y', strtotime($course['enrolled_at'])); ?>
                      </small>
                    </div>
                  </div>
                </div>
              </div>
            <?php endforeach; ?>
          </div>
        <?php else: ?>
          <!-- Empty State -->
          <div class="row">
            <div class="col-12">
              <div class="card">
                <div class="card-body text-center py-5">
                  <div class="mb-4">
                    <i class="fi fi-rr-graduation-cap text-muted" style="font-size: 4rem;"></i>
                  </div>
                  <h3 class="text-muted mb-3">No Courses Enrolled</h3>
                  <p class="text-muted mb-4">
                    You haven't enrolled in any courses yet. Start your learning journey today!
                  </p>
                  <a href="course.php" class="btn btn-primary btn-lg">
                    <i class="fi fi-rr-graduation-cap me-2"></i> Browse Courses
                  </a>
                </div>
              </div>
            </div>
          </div>
        <?php endif; ?>
      </div>
    </main>
  </div>
  <!-- end::Edura Page Layout -->

  <!-- begin::Edura Required Scripts -->
  <script src="assets/libs/global/global.min.js"></script>
  <script src="assets/js/main.js"></script>
  <!-- end::Edura Required Scripts -->

  <script>
    // Function to confirm manual course completion
    function confirmManualComplete(courseId, courseTitle, currentProgress) {
      const message = `Are you sure you want to mark "${courseTitle}" as complete?\n\n` +
                     `Current Progress: ${currentProgress}%\n\n` +
                     `Note: This will mark the course as completed even though not all lessons are finished. ` +
                     `It's recommended to complete all lessons for the best learning experience.`;
      
      if (confirm(message)) {
        markCourseComplete(courseId);
      }
    }
    
    // Function to mark course as complete
    function markCourseComplete(courseId) {
      fetch('mark_course_complete.php', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: `course_id=${courseId}`
      })
      .then(response => response.json())
      .then(data => {
        if (data.success) {
          alert('Course marked as complete successfully!');
          location.reload();
        } else {
          alert('Error: ' + data.message);
        }
      })
      .catch(error => {
        console.error('Error:', error);
        alert('An error occurred while marking the course as complete.');
      });
    }
  </script>

  <!-- WhatsApp Chat Button -->
  <div class="whatsapp-chat-button">
    <a href="https://wa.me/2347069378861?text=Hello%20Omateq%20Digitals!%20I%20would%20like%20to%20know%20more%20about%20your%20services." 
       target="_blank" 
       rel="noopener noreferrer"
       class="whatsapp-link"
       title="Chat with us on WhatsApp">
      <i class="fab fa-whatsapp"></i>
      <span class="whatsapp-text">Chat with us</span>
    </a>
  </div>
  
  <style>
  /* WhatsApp Chat Button Styles */
  .whatsapp-chat-button {
    position: fixed;
    bottom: 20px;
    right: 20px;
    z-index: 1000;
    animation: pulse 2s infinite;
  }
  
  .whatsapp-link {
    display: flex;
    align-items: center;
    background: #25D366;
    color: white;
    text-decoration: none;
    padding: 12px 16px;
    border-radius: 50px;
    box-shadow: 0 4px 12px rgba(37, 211, 102, 0.4);
    transition: all 0.3s ease;
    font-family: 'Poppins', sans-serif;
    font-weight: 500;
    font-size: 14px;
  }
  
  .whatsapp-link:hover {
    background: #128C7E;
    color: white;
    text-decoration: none;
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(37, 211, 102, 0.6);
  }
  
  .whatsapp-link i {
    font-size: 24px;
    margin-right: 8px;
  }
  
  .whatsapp-text {
    white-space: nowrap;
  }
  
  @keyframes pulse {
    0% {
      box-shadow: 0 0 0 0 rgba(37, 211, 102, 0.7);
    }
    70% {
      box-shadow: 0 0 0 10px rgba(37, 211, 102, 0);
    }
    100% {
      box-shadow: 0 0 0 0 rgba(37, 211, 102, 0);
    }
  }
  
  /* Mobile responsiveness */
  @media (max-width: 768px) {
    .whatsapp-chat-button {
      bottom: 15px;
      right: 15px;
    }
    
    .whatsapp-link {
      padding: 10px 14px;
      font-size: 13px;
    }
    
    .whatsapp-link i {
      font-size: 22px;
      margin-right: 6px;
    }
    
    .whatsapp-text {
      display: none;
    }
  }
  </style>
</body>
</html>
<?php
// End of PHP processing
?>