<?php
session_start();
require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/config/payment_config.php';

// Load payment configuration
$payment_config = require __DIR__ . '/config/payment_config.php';

// Log all callback data for debugging
error_log("Payment Callback - Full GET: " . print_r($_GET, true));
error_log("Payment Callback - Session ID: " . session_id());
error_log("Payment Callback - User ID in session: " . ($_SESSION['user_id'] ?? 'Not set'));

// Handle Paystack callback
if (isset($_GET['reference'])) {
    $reference = $_GET['reference'];
    $status = $_GET['status'] ?? 'failed';
    
    // Log callback parameters for debugging
    error_log("Paystack Callback - Reference: $reference, Status: $status, Trxref: " . ($_GET['trxref'] ?? 'Not set'));
    
    try {
        $pdo = getConnection();
        
        // First, try to find the order by reference without user_id constraint
        $stmt = $pdo->prepare("SELECT * FROM orders WHERE payment_reference = ?");
        $stmt->execute([$reference]);
        $order = $stmt->fetch();
        
        if (!$order) {
            error_log("No order found for reference: $reference");
            // Try to find by trxref as well
            if (isset($_GET['trxref'])) {
                $stmt = $pdo->prepare("SELECT * FROM orders WHERE payment_reference = ?");
                $stmt->execute([$_GET['trxref']]);
                $order = $stmt->fetch();
                error_log("Order found by trxref: " . ($order ? 'Yes' : 'No'));
            }
        }
        
        if ($order) {
            // Always verify payment with Paystack API since status parameter might be missing
            error_log("Order found for reference: $reference, Order ID: " . $order['id']);
            $payment_verified = verifyPaystackPayment($reference, $payment_config['paystack']['secret_key']);
            error_log("Payment verification result: " . ($payment_verified ? 'SUCCESS' : 'FAILED'));
            
            if ($payment_verified) {
                // Update order status to delivered
                $stmt = $pdo->prepare("UPDATE orders SET status = 'delivered', payment_status = 'paid', updated_at = NOW() WHERE id = ?");
                $stmt->execute([$order['id']]);
                error_log("Order status updated to delivered for order ID: " . $order['id']);
                
                // Update transaction status to completed
                $stmt = $pdo->prepare("UPDATE transactions SET status = 'completed', verification_date = NOW(), updated_at = NOW() WHERE order_number = ?");
                $stmt->execute([$order['order_number']]);
                error_log("Transaction status updated to completed for order: " . $order['order_number']);
                
                // Set success message in session if available
                if (isset($_SESSION['user_id'])) {
                    $_SESSION['payment_success'] = "Payment successful! You can now download your product.";
                    header('Location: download.php?product_id=' . $order['product_id']);
                } else {
                    // If no session, redirect to login with success message
                    header('Location: authentication/login.php?payment_success=1&product_id=' . $order['product_id']);
                }
                exit();
            } else {
                error_log("Payment verification failed for reference: $reference");
                if (isset($_SESSION['user_id'])) {
                    $_SESSION['payment_error'] = "Payment verification failed. Please contact support.";
                    header('Location: products.php');
                } else {
                    header('Location: authentication/login.php?payment_error=1');
                }
                exit();
            }
        } else {
            error_log("No order found for reference: $reference");
            if (isset($_SESSION['user_id'])) {
                $_SESSION['payment_error'] = "Order not found. Please contact support.";
                header('Location: products.php');
            } else {
                header('Location: authentication/login.php?payment_error=2');
            }
            exit();
        }
        
    } catch (Exception $e) {
        error_log("Exception in payment callback: " . $e->getMessage());
        if (isset($_SESSION['user_id'])) {
            $_SESSION['payment_error'] = "Payment processing error. Please contact support.";
            header('Location: products.php');
        } else {
            header('Location: authentication/login.php?payment_error=3');
        }
        exit();
    }
} else {
    // No reference provided
    error_log("No reference provided in callback");
    if (isset($_SESSION['user_id'])) {
        $_SESSION['payment_error'] = "Invalid payment reference.";
        header('Location: products.php');
    } else {
        header('Location: authentication/login.php?payment_error=4');
    }
    exit();
}

// Function to verify Paystack payment
function verifyPaystackPayment($reference, $secret_key) {
    try {
        $url = "https://api.paystack.co/transaction/verify/" . $reference;
        error_log("Verifying payment with Paystack API: $url");
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            "Authorization: Bearer " . $secret_key
        ]);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        error_log("Paystack API Response - HTTP Code: $http_code, Response: " . substr($response, 0, 200));
        
        if ($http_code === 200) {
            $result = json_decode($response, true);
            $is_success = $result['status'] === true && $result['data']['status'] === 'success';
            error_log("Payment verification - API Status: " . ($result['status'] ? 'true' : 'false') . ", Transaction Status: " . ($result['data']['status'] ?? 'unknown') . ", Final Result: " . ($is_success ? 'SUCCESS' : 'FAILED'));
            return $is_success;
        }
        
        error_log("Paystack API returned non-200 status: $http_code");
        return false;
    } catch (Exception $e) {
        error_log("Exception in verifyPaystackPayment: " . $e->getMessage());
        return false;
    }
}
?>
