<?php
// Start output buffering to catch any errors
ob_start();

session_start();
require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/config/payment_config.php';
require_once __DIR__ . '/config/exchange_rate.php';
require_once __DIR__ . '/../admin-dashboard/config/settings.php';
require_once __DIR__ . '/../admin-dashboard/config/branding.php';

// Load payment configuration
$payment_config = include __DIR__ . '/config/payment_config.php';

// Get live exchange rate with detailed information
try {
    $exchange_rate_info = getExchangeRateWithInfo();
    $exchange_rate = $exchange_rate_info['rate'] ?? 1500; // Fallback rate
} catch (Exception $e) {
    error_log("Exchange rate error: " . $e->getMessage());
    $exchange_rate = 1500;
    $exchange_rate_info = [
        'rate' => 1500,
        'source' => 'fallback',
        'age_minutes' => 0
    ];
}

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: authentication/login.php');
    exit();
}

// Get user data
try {
    $pdo = getConnection();
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $user = $stmt->fetch();
    
    if (!$user) {
        session_destroy();
        header('Location: authentication/login.php');
        exit();
    }
} catch (Exception $e) {
    session_destroy();
    header('Location: authentication/login.php');
    exit();
}

// Get script request replies
$script_replies = [];
$script_replies_count = 0;
try {
    $stmt = $pdo->prepare("SELECT * FROM script_requests WHERE user_id = ? AND admin_reply IS NOT NULL ORDER BY created_at DESC LIMIT 5");
    $stmt->execute([$_SESSION['user_id']]);
    $script_replies = $stmt->fetchAll();
    $script_replies_count = count($script_replies);
} catch (Exception $e) {
    // Handle error silently
}

// Get transactions (purchases and course enrollments)
$transactions = [];
$transactions_count = 0;
try {
    $stmt = $pdo->prepare("SELECT 'purchase' as type, product_name as item_name, total_amount as amount, status, created_at FROM orders WHERE user_id = ? 
                          UNION ALL 
                          SELECT 'course' as type, title as item_name, price as amount, 'completed' as status, created_at FROM course_enrollments ce 
                          JOIN courses c ON ce.course_id = c.id WHERE ce.user_id = ? 
                          ORDER BY created_at DESC LIMIT 5");
    $stmt->execute([$_SESSION['user_id'], $_SESSION['user_id']]);
    $transactions = $stmt->fetchAll();
    $transactions_count = count($transactions);
} catch (Exception $e) {
    // Handle error silently
}

// Function to get time ago
function time_ago($datetime) {
    $time = time() - strtotime($datetime);
    
    if ($time < 60) return 'just now';
    if ($time < 3600) return floor($time/60) . ' min ago';
    if ($time < 86400) return floor($time/3600) . ' hr ago';
    if ($time < 2592000) return floor($time/86400) . ' days ago';
    return date('M j', strtotime($datetime));
}

$order_id = $_GET['order_id'] ?? '';
$method = $_GET['method'] ?? '';
$course_id = $_GET['course_id'] ?? '';
$type = $_GET['type'] ?? '';

// Handle course enrollment payment - redirect to course page for modal payment
if ($type === 'course' && !empty($course_id)) {
    // Redirect to course page where payment will be handled via modal
    header('Location: course.php');
    exit();
}

if (empty($order_id) || empty($method)) {
    header('Location: index.php');
    exit();
}

// Get order details
try {
    // Clear any output that might have been generated
    ob_clean();
    
    $pdo = getConnection();
    
    if ($type === 'course') {
        // For courses, get course order details
        $stmt = $pdo->prepare("SELECT co.*, co.course_name as product_name, co.course_price as product_amount FROM course_orders co 
                              WHERE co.id = ? AND co.user_id = ?");
    } else {
        // For products, get product details
        $stmt = $pdo->prepare("SELECT o.*, p.product_name, p.product_amount FROM orders o 
                              JOIN products p ON o.product_id = p.id 
                              WHERE o.id = ? AND o.user_id = ?");
    }
    
    $stmt->execute([$order_id, $_SESSION['user_id']]);
    $order = $stmt->fetch();
    
    if (!$order || !isset($order['total_amount'])) {
        header('Location: index.php');
        exit();
    }
} catch (Exception $e) {
    // Clear any output and redirect
    ob_clean();
    error_log("Payment handler error: " . $e->getMessage());
    header('Location: index.php');
    exit();
}

// Check if this is a Paystack callback redirect
if (isset($_GET['reference']) && isset($_GET['status'])) {
    $reference = $_GET['reference'];
    $status = $_GET['status'];
    
    if ($status === 'success') {
        // This is a successful Paystack callback
        try {
            // Verify payment with Paystack API
            $payment_verified = verifyPaystackPayment($reference, $payment_config['paystack']['secret_key']);
            
            if ($payment_verified) {
                // Handle course enrollment payment
                if ($type === 'course' && isset($_SESSION['course_enrollment'])) {
                    $enrollment = $_SESSION['course_enrollment'];
                    
                    // Update course enrollment status
                    $stmt = $pdo->prepare("UPDATE course_enrollments SET payment_status = 'paid', payment_reference = ?, transaction_id = ?, payment_date = NOW() WHERE id = ?");
                    $stmt->execute([$reference, $reference, $enrollment['enrollment_id']]);
                    
                    // Create transaction record for course
                    $transaction_id = 'TXN-COURSE-' . time() . '-' . str_pad(rand(1, 9999), 4, '0', STR_PAD_LEFT);
                    $stmt = $pdo->prepare("INSERT INTO transactions (user_id, product_id, order_number, transaction_id, amount, currency, payment_method, payment_provider, status, payment_reference, created_at) VALUES (?, ?, ?, ?, ?, 'USD', 'paystack', 'Paystack', 'completed', ?, NOW())");
                    $stmt->execute([
                        $_SESSION['user_id'],
                        $enrollment['course_id'],
                        'COURSE-' . $enrollment['enrollment_id'],
                        $transaction_id,
                        $enrollment['amount'],
                        $reference
                    ]);
                    
                    // Clear session
                    unset($_SESSION['course_enrollment']);
                    
                    $_SESSION['payment_success'] = "Payment successful! You are now enrolled in the course.";
                    header('Location: course_learning.php?course_id=' . $enrollment['course_id']);
                } else {
                    // Handle product purchase
                    $_SESSION['payment_success'] = "Payment successful! You can now download your product.";
                    header('Location: download.php?product_id=' . $order['product_id']);
                }
                exit();
            } else {
                error_log("Payment verification failed for reference: $reference");
                $_SESSION['payment_error'] = "Payment verification failed. Please contact support.";
                if ($type === 'course') {
                    header('Location: course.php');
                } else {
                    header('Location: products.php');
                }
                exit();
            }
        } catch (Exception $e) {
            error_log("Error processing Paystack callback: " . $e->getMessage());
            $_SESSION['payment_error'] = "Payment processing error. Please contact support.";
            if ($type === 'course') {
                header('Location: course.php');
            } else {
                header('Location: products.php');
            }
            exit();
        }
    }
}

// Handle payment completion
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'complete_payment') {
    $payment_reference = $_POST['payment_reference'] ?? '';
    $payment_status = $_POST['payment_status'] ?? 'pending';
    $payment_proof = null;
    
    // Handle file upload for payment proof
    if (isset($_FILES['payment_proof']) && $_FILES['payment_proof']['error'] === UPLOAD_ERR_OK) {
        $upload_dir = __DIR__ . '/uploads/payment_proofs/';
        if (!is_dir($upload_dir)) {
            mkdir($upload_dir, 0777, true);
        }
        
        $file_extension = pathinfo($_FILES['payment_proof']['name'], PATHINFO_EXTENSION);
        $file_name = 'proof_' . $order_id . '_' . time() . '.' . $file_extension;
        $file_path = $upload_dir . $file_name;
        
        if (move_uploaded_file($_FILES['payment_proof']['tmp_name'], $file_path)) {
            $payment_proof = 'uploads/payment_proofs/' . $file_name;
        }
    }
    
    if (!empty($payment_reference)) {
        try {
            // Update order status
            $stmt = $pdo->prepare("UPDATE orders SET status = ?, payment_reference = ?, payment_proof = ?, updated_at = NOW() WHERE id = ?");
            $stmt->execute([$payment_status, $payment_reference, $payment_proof, $order_id]);
            
            if ($payment_status === 'completed') {
                $_SESSION['payment_success'] = "Payment successful! You can now download your product.";
                header('Location: index.php');
                exit();
            } else {
                $_SESSION['payment_success'] = "Payment proof submitted! We will verify and confirm your payment shortly.";
                header('Location: index.php');
                exit();
            }
        } catch (Exception $e) {
            $error = "Payment processing failed. Please try again.";
        }
    } else {
        $error = "Please provide a payment reference.";
    }
}

// Function to verify Paystack payment
function verifyPaystackPayment($reference, $secret_key) {
    try {
        $url = "https://api.paystack.co/transaction/verify/" . $reference;
        error_log("Verifying payment with Paystack API: $url");
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            "Authorization: Bearer " . $secret_key
        ]);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        error_log("Paystack API Response - HTTP Code: $http_code, Response: " . substr($response, 0, 200));
        
        if ($http_code === 200) {
            $result = json_decode($response, true);
            $is_success = $result['status'] === true && $result['data']['status'] === 'success';
            error_log("Payment verification - API Status: " . ($result['status'] ? 'true' : 'false') . ", Transaction Status: " . ($result['data']['status'] ?? 'unknown') . ", Final Result: " . ($is_success ? 'SUCCESS' : 'FAILED'));
            return $is_success;
        }
        
        error_log("Paystack API returned non-200 status: $http_code");
        return false;
    } catch (Exception $e) {
        error_log("Exception in verifyPaystackPayment: " . $e->getMessage());
        return false;
    }
}
?>
<!DOCTYPE html>
<html lang="en">

<head>
  <!-- begin::Edura Meta Basic -->
  <meta charset="utf-8">
  <meta name="theme-color" content="#0D9488">
  <meta name="robots" content="index, follow">
  <meta name="author" content="LayoutDrop">
  <meta name="format-detection" content="telephone=no">
  <meta name="keywords" content="Omateq Digitals, Payment, Secure Payment, Paystack, Online Payment, Digital Services, Web Development, Mobile App Development, Software Solutions, IT Services, Technology Solutions, Digital Transformation">
  <meta name="description" content="Secure payment processing for Omateq Digitals services. Complete your payment safely and securely with our integrated Paystack payment gateway. Professional digital services and technology solutions.">
  <!-- end::Edura Meta Basic -->

  <!-- begin::Edura Meta Social -->
  <meta property="og:url" content="https://omateqdigitals.com/users/payment_handler.php">
  <meta property="og:site_name" content="Omateq Digitals">
  <meta property="og:type" content="website">
  <meta property="og:locale" content="en_us">
  <meta property="og:title" content="Secure Payment | Omateq Digitals">
  <meta property="og:description" content="Secure payment processing for Omateq Digitals services. Complete your payment safely and securely with our integrated Paystack payment gateway. Professional digital services and technology solutions.">
  <meta property="og:image" content="https://omateqdigitals.com/assets/img/logo.png">
  <!-- end::Edura Meta Social -->

  <!-- begin::Edura Meta Twitter -->
  <meta name="twitter:card" content="summary_large_image">
  <meta name="twitter:site" content="@OmateqDigitals">
  <meta name="twitter:creator" content="@OmateqDigitals">
  <meta name="twitter:title" content="Secure Payment | Omateq Digitals">
  <meta name="twitter:description" content="Secure payment processing for Omateq Digitals services. Complete your payment safely and securely with our integrated Paystack payment gateway. Professional digital services and technology solutions.">
  <meta name="twitter:image" content="https://omateqdigitals.com/assets/img/logo.png">
  <!-- end::Edura Meta Twitter -->

  <!-- begin::Edura Website Page Title -->
  <title>Secure Payment | Omateq Digitals</title>
  <!-- end::Edura Website Page Title -->

  <!-- begin::Edura Mobile Specific -->
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <!-- end::Edura Mobile Specific -->

  <!-- begin::Edura Favicon Tags -->
  <?php echo getFaviconHtml(); ?>
  <!-- end::Edura Favicon Tags -->

  <!-- begin::Edura Google Fonts -->
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Plus+Jakarta+Sans:ital,wght@0,200..800;1,200..800&display=swap" rel="stylesheet">
  <!-- end::Edura Google Fonts -->

  <!-- begin::Edura Required Stylesheet -->
  <link rel="stylesheet" href="assets/libs/flaticon/css/all/all.css">
  <link rel="stylesheet" href="assets/libs/lucide/lucide.css">
  <link rel="stylesheet" href="assets/libs/fontawesome/css/all.min.css">
  <link rel="stylesheet" href="assets/libs/simplebar/simplebar.css">
  <link rel="stylesheet" href="assets/libs/node-waves/waves.css">
  <link rel="stylesheet" href="assets/libs/bootstrap-select/css/bootstrap-select.min.css">
  <!-- end::Edura Required Stylesheet -->

  <!-- begin::Edura CSS Stylesheet -->
  <link rel="stylesheet" href="assets/css/styles.css">
  <!-- end::Edura CSS Stylesheet -->
  <style>
    /* Logo sizing for better fit */
    .app-navbar-brand .navbar-brand-logo img {
      max-height: 50px;
      max-width: 180px;
      width: auto;
      height: auto;
      object-fit: contain;
    }
    
    .app-navbar-brand .navbar-brand-mini img {
      max-height: 35px;
      max-width: 35px;
      width: auto;
      height: auto;
      object-fit: contain;
    }
  </style>
  <!-- Paystack Script with fallback -->
  <script src="https://js.paystack.co/v1/inline.js" onerror="console.log('Paystack script blocked, using fallback')"></script>
  <script>
    // Fallback if Paystack script is blocked
    if (typeof PaystackPop === 'undefined') {
      window.PaystackPop = {
        setup: function(options) {
          return {
            openIframe: function() {
              // Fallback: redirect to Paystack
              const form = document.createElement('form');
              form.method = 'POST';
              form.action = 'https://api.paystack.co/transaction/initialize';
              form.target = '_blank';
              
              const fields = {
                email: options.email,
                amount: options.amount,
                currency: options.currency,
                reference: options.ref,
                key: options.key,
                callback_url: '<?php echo $payment_config['paystack']['callback_url']; ?>'
              };
              
              for (const key in fields) {
                const input = document.createElement('input');
                input.type = 'hidden';
                input.name = key;
                input.value = fields[key];
                form.appendChild(input);
              }
              
              document.body.appendChild(form);
              form.submit();
              document.body.removeChild(form);
            }
          };
        }
      };
    }
  </script>
  <style>
    /* Paystack Popup Modal Styling */
    .paystack-popup {
      position: fixed !important;
      top: 0 !important;
      left: 0 !important;
      width: 100% !important;
      height: 100% !important;
      background: rgba(0, 0, 0, 0.8) !important;
      z-index: 9999 !important;
      display: flex !important;
      align-items: center !important;
      justify-content: center !important;
    }
    
    .paystack-popup .popup-content {
      background: white !important;
      border-radius: 15px !important;
      box-shadow: 0 20px 40px rgba(0, 0, 0, 0.3) !important;
      width: 90% !important;
      max-width: 450px !important;
      max-height: 90vh !important;
      overflow: hidden !important;
      position: relative !important;
    }
    
    .paystack-popup .close-btn {
      position: absolute !important;
      top: 15px !important;
      right: 15px !important;
      background: #ff6b6b !important;
      border: none !important;
      border-radius: 50% !important;
      width: 40px !important;
      height: 40px !important;
      display: flex !important;
      align-items: center !important;
      justify-content: center !important;
      cursor: pointer !important;
      font-size: 24px !important;
      font-weight: bold !important;
      color: white !important;
      z-index: 10000 !important;
      transition: all 0.3s ease !important;
      box-shadow: 0 2px 8px rgba(255, 107, 107, 0.3) !important;
    }
    
    .paystack-popup .close-btn:hover {
      background: #ff5252 !important;
      transform: scale(1.1) !important;
      box-shadow: 0 4px 12px rgba(255, 107, 107, 0.4) !important;
    }
    
    .paystack-popup .close-btn:active {
      transform: scale(0.95) !important;
    }
    
    .paystack-popup .popup-header {
      padding: 30px 30px 20px 30px !important;
      text-align: center !important;
      border-bottom: 1px solid #e9ecef !important;
    }
    
    .paystack-popup .popup-header h5 {
      margin: 0 0 10px 0 !important;
      font-weight: 600 !important;
      color: #333 !important;
    }
    
    .paystack-popup .popup-header p {
      margin: 0 !important;
      color: #6c757d !important;
      font-size: 14px !important;
    }
    
    .paystack-popup .popup-body {
      padding: 20px 30px 30px 30px !important;
    }
    
    .paystack-popup .payment-details {
      background: #f8f9fa !important;
      border-radius: 10px !important;
      padding: 20px !important;
      margin-bottom: 25px !important;
    }
    
    .paystack-popup .detail-row {
      display: flex !important;
      justify-content: space-between !important;
      align-items: center !important;
      margin-bottom: 10px !important;
    }
    
    .paystack-popup .detail-row:last-child {
      margin-bottom: 0 !important;
    }
    
    .paystack-popup .detail-row span:first-child {
      font-weight: 500 !important;
      color: #6c757d !important;
    }
    
    .paystack-popup .detail-row span:last-child {
      font-weight: 600 !important;
      color: #333 !important;
    }
    
    .paystack-popup .popup-actions {
      display: flex !important;
      gap: 15px !important;
    }
    
    .paystack-popup .popup-actions .btn {
      flex: 1 !important;
      padding: 12px 20px !important;
      border-radius: 8px !important;
      font-weight: 500 !important;
      transition: all 0.3s ease !important;
    }
    
    /* Mobile responsiveness */
    @media (max-width: 768px) {
      .paystack-popup .popup-content {
        width: 95% !important;
        margin: 20px !important;
      }
      
      .paystack-popup .popup-header {
        padding: 20px 20px 15px 20px !important;
      }
      
      .paystack-popup .popup-body {
        padding: 15px 20px 20px 20px !important;
      }
      
      .paystack-popup .payment-details {
        padding: 15px !important;
      }
      
      .paystack-popup .close-btn {
        width: 35px !important;
        height: 35px !important;
        font-size: 20px !important;
        top: 10px !important;
        right: 10px !important;
      }
      
      .paystack-popup .popup-actions {
        flex-direction: column !important;
        gap: 10px !important;
      }
    }
    
    @media (max-width: 480px) {
      .paystack-popup .popup-content {
        width: 98% !important;
        margin: 10px !important;
      }
      
      .paystack-popup .popup-header h5 {
        font-size: 1.1rem !important;
      }
      
      .paystack-popup .popup-header p {
        font-size: 13px !important;
      }
    }
    
    /* Loading button state */
    .btn:disabled {
      opacity: 0.7;
      cursor: not-allowed;
    }
    
    /* Notification positioning */
    .notification-container {
      position: fixed;
      top: 20px;
      right: 20px;
      z-index: 10000;
    }
    
    /* Loading overlay */
    .loading-overlay {
      position: fixed;
      top: 0;
      left: 0;
      width: 100%;
      height: 100%;
      background: rgba(0, 0, 0, 0.7);
      z-index: 9998;
      display: flex;
      align-items: center;
      justify-content: center;
      flex-direction: column;
      color: white;
    }
    
    .loading-spinner {
      width: 50px;
      height: 50px;
      border: 4px solid rgba(255, 255, 255, 0.3);
      border-top: 4px solid #007bff;
      border-radius: 50%;
      animation: spin 1s linear infinite;
      margin-bottom: 20px;
    }
    
    @keyframes spin {
      0% { transform: rotate(0deg); }
      100% { transform: rotate(360deg); }
    }
    
    /* Paystack Iframe Container - Now using inline styles for better control */
  </style>
</head>

<body>
  <div class="page-layout">

    <!-- begin::Edura Page Header -->
    <header class="app-header">
      <div class="app-header-inner">
        <button class="app-toggler" type="button">
          <span></span>
          <span></span>
          <span></span>
        </button>
        <div class="app-header-start">
          <form class="d-none d-md-flex align-items-center h-100 w-lg-250px w-xxl-300px position-relative" action="index.html#">
            <button type="button" class="btn btn-sm border-0 position-absolute start-0 ms-3 p-0">
              <i class="fi fi-rr-search"></i>
            </button>
            <input type="text" class="form-control ps-5" placeholder="Search courses...">
          </form>
        </div>
        <div class="app-header-end">
          <div class="px-lg-3 px-2 ps-0 d-flex align-items-center">
            <a href="javascript:void(0);" class="theme-btn">
              <i class="fi fi-rr-brightness icon-light"></i>
              <div class="theme-toggle"></div>
              <i class="fi fi-rr-moon icon-dark"></i>
            </a>
          </div>
          <div class="vr my-3"></div>
          <div class="d-flex align-items-center gap-sm-2 gap-0 px-lg-4 px-sm-2 px-1">
            <div class="dropdown text-end">
              <a href="javascript:void(0);" class="btn btn-icon btn-action-gray rounded-circle waves-effect waves-light" data-bs-toggle="dropdown" data-bs-auto-close="outside" aria-expanded="true">
                <i class="fi fi-rr-envelope"></i>
              </a>
              <div class="dropdown-menu dropdown-menu-lg-end p-0 w-300px mt-2">
                <div class="px-3 py-3 border-bottom d-flex justify-content-between align-items-center">
                  <h6 class="mb-0">Messages <span class="badge badge-sm rounded-pill bg-primary ms-2"><?php echo $script_replies_count ?? 0; ?></span></h6>
                  <i class="bi bi-x-lg cursor-pointer"></i>
                </div>
                <div class="p-2" style="height: 300px;" data-simplebar>
                  <ul class="list-group list-group-hover list-group-smooth list-group-unlined">
                    <?php if (!empty($script_replies)): ?>
                      <?php foreach ($script_replies as $reply): ?>
                        <li class="list-group-item d-flex justify-content-between align-items-center">
                          <div class="avatar avatar-xs bg-info rounded-circle text-white">
                            <i class="fi fi-rr-user"></i>
                          </div>
                          <div class="ms-2 me-auto">
                            <h6 class="mb-0">Admin Reply</h6>
                            <small class="text-body d-block"><?php echo htmlspecialchars(substr($reply['admin_reply'], 0, 50)) . '...'; ?></small>
                            <small class="text-muted position-absolute end-0 top-0 mt-2 me-3"><?php echo time_ago($reply['created_at']); ?></small>
                          </div>
                        </li>
                      <?php endforeach; ?>
                    <?php else: ?>
                      <li class="list-group-item text-center py-4">
                        <i class="fi fi-rr-envelope text-muted" style="font-size: 2rem;"></i>
                        <p class="text-muted mt-2 mb-0">No script request replies yet</p>
                      </li>
                    <?php endif; ?>
                  </ul>
                </div>
                <div class="p-2">
                  <a href="script-request.php" class="btn w-100 btn-primary waves-effect waves-light">View all replies</a>
                </div>
              </div>
            </div>
            <div class="dropdown text-end">
              <a href="javascript:void(0);" class="btn btn-icon btn-action-gray rounded-circle waves-effect waves-light" data-bs-toggle="dropdown" data-bs-auto-close="outside" aria-expanded="true">
                <i class="fi fi-rr-bell"></i>
              </a>
              <div class="dropdown-menu dropdown-menu-lg-end p-0 w-300px mt-2">
                <div class="px-3 py-3 border-bottom d-flex justify-content-between align-items-center">
                  <h6 class="mb-0">Notifications <span class="badge badge-sm rounded-pill bg-primary ms-2"><?php echo $transactions_count ?? 0; ?></span></h6>
                  <i class="bi bi-x-lg cursor-pointer"></i>
                </div>
                <div class="p-2" style="height: 300px;" data-simplebar>
                  <ul class="list-group list-group-hover list-group-smooth list-group-unlined">
                    <?php if (!empty($transactions)): ?>
                      <?php foreach ($transactions as $transaction): ?>
                        <li class="list-group-item d-flex justify-content-between align-items-center">
                          <div class="avatar avatar-xs bg-success rounded-circle text-white">
                            <i class="fi fi-rr-<?php echo $transaction['type'] === 'course' ? 'graduation-cap' : 'shopping-cart'; ?>"></i>
                          </div>
                          <div class="ms-2 me-auto">
                            <h6 class="mb-0"><?php echo ucfirst($transaction['type']); ?> - <?php echo htmlspecialchars($transaction['item_name']); ?></h6>
                            <small class="text-body d-block">$<?php echo number_format($transaction['amount'], 2); ?> - <?php echo ucfirst($transaction['status']); ?></small>
                            <small class="text-muted position-absolute end-0 top-0 mt-2 me-3"><?php echo time_ago($transaction['created_at']); ?></small>
                          </div>
                        </li>
                      <?php endforeach; ?>
                    <?php else: ?>
                      <li class="list-group-item text-center py-4">
                        <i class="fi fi-rr-bell text-muted" style="font-size: 2rem;"></i>
                        <p class="text-muted mt-2 mb-0">No notifications yet</p>
                      </li>
                    <?php endif; ?>
                  </ul>
                </div>
                <div class="p-2">
                  <a href="my_courses.php" class="btn w-100 btn-primary waves-effect waves-light">View all notifications</a>
                </div>
              </div>
            </div>
            <div class="dropdown text-end">
              <a href="javascript:void(0);" class="btn btn-icon btn-action-gray rounded-circle waves-effect waves-light" data-bs-toggle="dropdown" data-bs-auto-close="outside" aria-expanded="true">
                <i class="fi fi-rr-user"></i>
              </a>
              <div class="dropdown-menu dropdown-menu-lg-end p-0 w-300px mt-2">
                <div class="px-3 py-3 border-bottom d-flex justify-content-between align-items-center">
                  <h6 class="mb-0"><?php echo htmlspecialchars($user['full_name']); ?></h6>
                  <i class="bi bi-x-lg cursor-pointer"></i>
                </div>
                <ul class="list-unstyled p-2">
                  <li>
                    <div class="d-flex align-items-center gap-2 px-2 py-2">
                      <div class="avatar avatar-sm bg-primary rounded-circle text-white">
                        <i class="fi fi-rr-user"></i>
                      </div>
                      <div class="ms-2">
                        <div class="fw-bold text-dark"><?php echo htmlspecialchars($user['full_name']); ?></div>
                        <small class="text-body d-block lh-sm"><?php echo htmlspecialchars($user['email']); ?></small>
                      </div>
                    </div>
                  </li>
                  <li>
                    <div class="dropdown-divider my-1"></div>
                  </li>
                  <li>
                    <a class="dropdown-item d-flex align-items-center gap-2" href="profile.php">
                      <i class="fi fi-rr-user scale-1x"></i> Profile
                    </a>
                  </li>
                  <li>
                    <a class="dropdown-item d-flex align-items-center gap-2" href="help-center.php">
                      <i class="fi fi-rs-interrogation scale-1x"></i> Help Center
                    </a>
                  </li>
                  <li>
                    <div class="dropdown-divider my-1"></div>
                  </li>
                  <li>
                    <a class="dropdown-item d-flex align-items-center gap-2 text-danger" href="authentication/logout.php">
                      <i class="fi fi-sr-exit scale-1x"></i> Logout
                    </a>
                  </li>
                </ul>
              </div>
            </div>
          </div>
        </div>
      </div>
    </header>
    <!-- end::Edura Page Header -->

    <!-- begin::Edura Sidebar Menu -->
    <aside class="app-menubar" id="menubar">
      <div class="app-navbar-brand">
        <a class="navbar-brand-logo" href="index.php">
          <?php 
          $main_logo = SettingsManager::get('main_logo', 'assets/img/logo.png');
          if (strpos($main_logo, 'uploads/') === 0) {
            $main_logo = '../' . $main_logo;
          } elseif (strpos($main_logo, 'assets/') === 0) {
            $main_logo = '../' . $main_logo;
          }
          echo '<img src="' . $main_logo . '" alt="Omateq Digitals Logo">'; 
          ?>
        </a>
        <a class="navbar-brand-mini" href="index.php">
          <?php 
          $mini_logo = SettingsManager::get('mini_logo', 'assets/img/logo.png');
          if (strpos($mini_logo, 'uploads/') === 0) {
            $mini_logo = '../' . $mini_logo;
          } elseif (strpos($mini_logo, 'assets/') === 0) {
            $mini_logo = '../' . $mini_logo;
          }
          echo '<img src="' . $mini_logo . '" alt="Omateq Digitals Logo">'; 
          ?>
        </a>
      </div>
      <nav class="app-navbar" data-simplebar>
        <ul class="menubar">
          <li class="menu-item">
            <a class="menu-link" href="index.php">
              <i class="fi fi-rr-apps"></i>
              <span class="menu-label">Dashboard</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link active" href="products.php">
              <i class="fi fi-rr-users"></i>
              <span class="menu-label">Products</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="course.php">
              <i class="fi fi-rr-graduation-cap"></i>
              <span class="menu-label">Course</span>
            </a>
          </li>
          <li class="menu-item">
            <a class="menu-link" href="profile.php">
              <i class="fi fi-rr-receipt"></i>
              <span class="menu-label">Profile</span>
            </a>
          </li>
        </ul>
      </nav>
      <div class="app-footer">
        <div class="text-center py-2">
          <small class="text-white-50">
            <i class="fi fi-rr-copyright me-1"></i>
            2025 Omateq Digitals
          </small>
        </div>
        <a href="mailto:info@omateqdigitals.com" class="btn btn-outline-white waves-effect btn-app-nav w-100">
          <i class="fi fi-rs-interrogation"></i>
          <span class="nav-text">Help and Support</span>
        </a>
      </div>
    </aside>
    <!-- end::Edura Sidebar Menu -->

    <main class="app-wrapper">
      <div class="container-fluid">
        
        <div class="app-page-head d-flex flex-wrap gap-3 align-items-center justify-content-between">
          <div class="clearfix">
            <h1 class="app-page-title">Secure Payment</h1>
            <span>Complete your payment securely with Paystack</span>
          </div>
        </div>

        <div class="row justify-content-center">
          <div class="col-lg-8 col-md-10">
            <div class="card">
              <div class="card-header">
                <h4 class="mb-0">Complete Your Payment</h4>
                <p class="mb-0 text-muted">Secure payment processing powered by Paystack</p>
              </div>
              <div class="card-body">
                <?php if (isset($error)): ?>
                  <div class="alert alert-danger">
                    <i class="fi fi-rr-exclamation-triangle me-2"></i>
                    <?php echo htmlspecialchars($error); ?>
                  </div>
                <?php endif; ?>

                <div class="row">
                  <div class="col-md-6">
                    <h6>Order Details</h6>
                    <div class="border rounded p-3 mb-4">
                      <div class="d-flex justify-content-between mb-2">
                        <span>Product:</span>
                        <strong><?php echo htmlspecialchars($order['product_name']); ?></strong>
                      </div>
                      <div class="d-flex justify-content-between mb-2">
                        <span>Amount:</span>
                        <strong>$<?php echo number_format($order['total_amount'] ?? 0 ?? 0, 2); ?></strong>
                      </div>
                      <div class="d-flex justify-content-between mb-2">
                        <span>Order ID:</span>
                        <span class="text-muted">#<?php echo $order_id; ?></span>
                      </div>
                      <div class="d-flex justify-content-between">
                        <span>Payment Method:</span>
                        <span class="badge bg-primary">
                          Paystack
                        </span>
                      </div>
                    </div>
                  </div>
                  
                  <div class="col-md-6">
                    <!-- Paystack Payment Gateway - Only Option -->
                    <h6>Secure Payment Gateway</h6>
                    <div class="alert alert-success">
                      <i class="fi fi-rr-shield-check me-2"></i>
                      Complete your payment securely using our trusted payment processor.
                    </div>
                    
                    <div class="border rounded p-3 mb-3">
                      <div class="mb-2">
                        <label class="form-label small">Amount (USD):</label>
                        <input type="text" class="form-control" value="$<?php echo number_format($order['total_amount'] ?? 0, 2); ?>" readonly>
                      </div>
                      <div class="mb-2">
                        <label class="form-label small">Amount (NGN):</label>
                        <input type="text" class="form-control" value="₦<?php echo number_format(($order['total_amount'] ?? 0) * $exchange_rate, 2); ?>" readonly>
                      </div>
                      <div class="mb-2">
                        <label class="form-label small">Exchange Rate:</label>
                        <input type="text" class="form-control" value="1 USD = ₦<?php echo number_format($exchange_rate, 2); ?>" readonly>
                      </div>
                      <div class="mb-2">
                        <label class="form-label small">Rate Info:</label>
                        <input type="text" class="form-control" value="Source: <?php echo ucfirst($exchange_rate_info['source']); ?> | Updated: <?php echo $exchange_rate_info['age_minutes']; ?> min ago" readonly>
                      </div>
                      <div class="mb-2">
                        <label class="form-label small">Payment Method:</label>
                        <input type="text" class="form-control" value="Paystack (Cards, Bank Transfer, USSD)" readonly>
                      </div>
                    </div>
                    
                    <div class="d-grid gap-2">
                      <button class="btn btn-outline-info btn-sm" onclick="refreshExchangeRate()" id="refreshRateBtn">
                        <i class="fi fi-rr-refresh me-1"></i>
                        Refresh Exchange Rate
                      </button>
                      <button class="btn btn-primary btn-lg" onclick="initiatePaystack()" id="paystackBtn">
                        <i class="fi fi-rr-credit-card me-2"></i>
                        Pay Securely with Paystack
                      </button>
                    </div>
                    
                    <div class="mt-3 text-center">
                      <small class="text-muted">
                        <i class="fi fi-rr-lock me-1"></i>
                        Your payment is secured with 256-bit SSL encryption
                      </small>
                    </div>
                  </div>
                </div>
                
                <div class="text-center mt-4">
                  <a href="index.php" class="btn btn-outline-secondary">
                    <i class="fi fi-rr-arrow-left me-2"></i>
                    Back to Dashboard
                  </a>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
      
      <!-- Payment Support Section -->
      <div class="row mt-4">
        <div class="col-12">
          <div class="card border-info">
            <div class="card-header bg-info-subtle">
              <h6 class="mb-0">
                <i class="fi fi-rr-question-circle me-2"></i>
                Need Help with Payment?
              </h6>
            </div>
            <div class="card-body">
              <p class="mb-3">
                If you encounter any issues with your payment or need assistance, 
                please contact our support team.
              </p>
              <div class="d-flex gap-2">
                <a href="mailto:support@omateqdigitals.com" class="btn btn-outline-info">
                  <i class="fi fi-rr-envelope me-2"></i>
                  Email Support
                </a>
                <a href="tel:+2347069378861" class="btn btn-outline-info">
                  <i class="fi fi-rr-phone me-2"></i>
                  Call Support
                </a>
              </div>
            </div>
          </div>
        </div>
      </div>
    </main>
  </div>

  <script src="assets/libs/global/global.min.js"></script>
  <script src="assets/js/appSettings.js"></script>
  <script src="assets/js/main.js"></script>
  
  <script>
    // Define global functions first to ensure they're available
    window.closePaystackPopup = function() {
      console.log('Closing Paystack popup...');
      
      // Close the popup container
      const popup = document.getElementById('paystack-popup-container');
      if (popup) {
        popup.style.display = 'none';
        popup.remove(); // Remove from DOM completely
      }
      
      // Close any iframe containers
      const iframeContainer = document.getElementById('paystack-iframe-container');
      if (iframeContainer) {
        iframeContainer.style.display = 'none';
        iframeContainer.remove();
      }
      
      // Reset payment button
      const button = window.currentPaymentButton;
      const originalText = window.currentPaymentOriginalText;
      if (button && originalText) {
        button.innerHTML = originalText;
        button.disabled = false;
        button.classList.remove('loading');
      }
      
      // Clear stored references
      window.currentPaymentButton = null;
      window.currentPaymentOriginalText = null;
      window.currentPaymentReference = null;
      
      console.log('Paystack popup closed successfully');
    };
    
    window.openPaystackWindow = function() {
      const reference = window.currentPaymentReference;
      const orderId = <?php echo $order_id; ?>;
      
      // Show payment form immediately - no loading delay
      showPaystackIframe('', orderId, true); // Show loading state in iframe
      
      // Initialize Paystack transaction in background
      const paymentData = {
        reference: reference
      };
      
      // Add order-specific data
      paymentData.order_id = orderId;
      
      // Make API call in background with timeout
      const timeoutId = setTimeout(() => {
        showNotification('Payment initialization is taking longer than expected. Please try again.', 'warning');
        closePaystackIframe();
      }, 15000); // 15 second timeout
      
      fetch('initiate_paystack.php', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(paymentData)
      })
      .then(response => {
        if (!response.ok) {
          throw new Error(`HTTP error! status: ${response.status}`);
        }
        return response.text(); // Get as text first
      })
      .then(text => {
        try {
          return JSON.parse(text);
        } catch (e) {
          console.error('Invalid JSON response:', text);
          throw new Error('Invalid response from server. Please try again.');
        }
      })
      .then(data => {
        clearTimeout(timeoutId); // Clear timeout on success
        if (data.status === 'success') {
          // Log amount details for debugging
          console.log('Payment Amount Details:', data.amount_details);
          
          // Update iframe with actual payment URL
          updatePaystackIframe(data.authorization_url);
        } else {
          // Error initializing payment
          showNotification('Failed to initialize payment: ' + data.message, 'error');
          closePaystackIframe();
        }
      })
      .catch(error => {
        clearTimeout(timeoutId); // Clear timeout on error
        console.error('Payment initialization error:', error);
        
        // Show more specific error message
        let errorMessage = 'Payment initialization failed. Please try again.';
        if (error.message.includes('Invalid response')) {
          errorMessage = 'Server error. Please refresh the page and try again.';
        } else if (error.message.includes('HTTP error')) {
          errorMessage = 'Network error. Please check your connection and try again.';
        }
        
        showNotification(errorMessage, 'error');
        closePaystackIframe();
      });
    };

    
    function initiatePaystack() {
      // Show loading state
      const button = document.getElementById('paystackBtn');
      const originalText = button.innerHTML;
      button.innerHTML = '<i class="fi fi-rr-spinner me-2"></i>Initializing Payment...';
      button.disabled = true;
      
      // Generate a unique reference with better uniqueness using microtime equivalent
      const reference = 'PAYSTACK_' + (Date.now() * 1000 + Math.floor(Math.random() * 1000)) + '_' + Math.random().toString(36).substr(2, 9) + '_' + Math.floor(Math.random() * 10000);
      const amount_usd = <?php echo json_encode(isset($order['total_amount']) ? $order['total_amount'] : 0); ?>;
      const exchange_rate = <?php echo json_encode(isset($exchange_rate) ? $exchange_rate : 1500); ?>;
      const amount = <?php echo json_encode((isset($order['total_amount']) ? $order['total_amount'] : 0) * (isset($exchange_rate) ? $exchange_rate : 1500) * 100); ?>; // Convert to kobo
      const rate_info = <?php echo json_encode(isset($exchange_rate_info) ? $exchange_rate_info : ['rate' => 1500, 'source' => 'fallback', 'age_minutes' => 0]); ?>;
      const email = <?php echo json_encode(isset($_SESSION['user_email']) ? $_SESSION['user_email'] : 'customer@example.com'); ?>;
      
      // Show loading overlay
      showLoadingOverlay('Initializing Paystack Payment...');
      
      // Update order with reference before redirecting
      fetch('update_order_reference.php', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          order_id: <?php echo $order_id; ?>,
          reference: reference
        })
      }).then(() => {
        // Hide loading overlay
        hideLoadingOverlay();
        // Create popup container if it doesn't exist
        let popupContainer = document.getElementById('paystack-popup-container');
        if (!popupContainer) {
          popupContainer = document.createElement('div');
          popupContainer.id = 'paystack-popup-container';
          popupContainer.className = 'paystack-popup';
          document.body.appendChild(popupContainer);
        }
        
        // Create popup content
        popupContainer.innerHTML = `
          <div class="popup-content">
            <button class="close-btn" onclick="closePaystackPopup();" title="Close payment modal (ESC)">×</button>
            <div class="popup-header">
              <h5>Complete Your Payment</h5>
              <p>You will be redirected to Paystack to complete your payment securely.</p>
            </div>
            <div class="popup-body">
              <div class="payment-details">
                <div class="detail-row">
                  <span>Amount (USD):</span>
                  <span>$${amount_usd}</span>
                </div>
                <div class="detail-row">
                  <span>Exchange Rate:</span>
                  <span>1 USD = ₦${exchange_rate.toFixed(2)}</span>
                </div>
                <div class="detail-row">
                  <span>Amount (NGN):</span>
                  <span>₦${(amount / 100).toLocaleString()}</span>
                </div>
                <div class="detail-row">
                  <span>Rate Source:</span>
                  <span>${rate_info.source} (${rate_info.age_minutes} min ago)</span>
                </div>
                <div class="detail-row">
                  <span>Reference:</span>
                  <span>${reference}</span>
                </div>
              </div>
              <div class="popup-actions">
                <button class="btn btn-secondary" onclick="closePaystackPopup();" title="Cancel payment and close modal">
                  <i class="fi fi-rr-cross me-2"></i>Cancel
                </button>
                <button class="btn btn-primary" onclick="openPaystackWindow();" title="Continue to Paystack payment">
                  <i class="fi fi-rr-credit-card me-2"></i>Proceed to Payment
                </button>
              </div>
            </div>
          </div>
        `;
        
        // Show popup container
        popupContainer.style.display = 'flex';
        
        // Store reference for popup functions
        window.currentPaymentReference = reference;
        window.currentPaymentButton = button;
        window.currentPaymentOriginalText = originalText;
        
        // Add keyboard support (ESC key to close)
        const handleKeyPress = (e) => {
          if (e.key === 'Escape') {
            closePaystackPopup();
            document.removeEventListener('keydown', handleKeyPress);
          }
        };
        document.addEventListener('keydown', handleKeyPress);
        
        // Add click outside to close functionality
        const handleClickOutside = (e) => {
          if (e.target === popupContainer) {
            closePaystackPopup();
            popupContainer.removeEventListener('click', handleClickOutside);
          }
        };
        popupContainer.addEventListener('click', handleClickOutside);
        
        // Event listeners removed to prevent duplicate calls
        
        // Reset button after a delay
        setTimeout(() => {
          if (!button.innerHTML.includes('Payment Successful')) {
            button.innerHTML = originalText;
            button.disabled = false;
          }
        }, 5000);
        
      }).catch((error) => {
        // Hide loading overlay and reset button on error
        hideLoadingOverlay();
        button.innerHTML = originalText;
        button.disabled = false;
        showNotification('Error initiating payment. Please try again.', 'error');
        console.error('Payment initiation error:', error);
      });
    }
    
    // Loading overlay functions
    function showLoadingOverlay(message = 'Loading...') {
      const overlay = document.createElement('div');
      overlay.className = 'loading-overlay';
      overlay.id = 'loadingOverlay';
      overlay.innerHTML = `
        <div class="loading-spinner"></div>
        <div>${message}</div>
      `;
      document.body.appendChild(overlay);
    }
    
    function hideLoadingOverlay() {
      const overlay = document.getElementById('loadingOverlay');
      if (overlay) {
        overlay.remove();
      }
    }
    
    // Notification function
    function showNotification(message, type = 'info') {
      const notification = document.createElement('div');
      notification.className = `alert alert-${type === 'error' ? 'danger' : type === 'warning' ? 'warning' : 'info'} alert-dismissible fade show position-fixed`;
      notification.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
      notification.innerHTML = `
        <i class="fi fi-rr-${type === 'error' ? 'exclamation-triangle' : type === 'warning' ? 'exclamation' : 'info'} me-2"></i>
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
      `;
      document.body.appendChild(notification);
      
      // Auto remove after 5 seconds
      setTimeout(() => {
        if (notification.parentNode) {
          notification.remove();
        }
      }, 5000);
    }
    
    // Function to refresh exchange rate
    function refreshExchangeRate() {
      const refreshBtn = document.getElementById('refreshRateBtn');
      const originalText = refreshBtn.innerHTML;
      
      // Show loading state
      refreshBtn.innerHTML = '<i class="fi fi-rr-spinner me-1"></i>Refreshing...';
      refreshBtn.disabled = true;
      
      // Fetch new exchange rate
      fetch('refresh_exchange_rate.php', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        }
      })
      .then(response => response.json())
      .then(data => {
        if (data.success) {
          // Update the displayed rate
          const rateInput = document.querySelector('input[value*="1 USD = ₦"]');
          if (rateInput) {
            rateInput.value = `1 USD = ₦${data.rate.toFixed(2)}`;
          }
          
          // Update the NGN amount
          const ngnInput = document.querySelector('input[value*="₦"]:not([value*="1 USD"])');
          if (ngnInput) {
            const usdAmount = <?php echo $order['total_amount'] ?? 0; ?>;
            ngnInput.value = `₦${(usdAmount * data.rate).toLocaleString()}`;
          }
          
          // Update rate info
          const rateInfoInput = document.querySelector('input[value*="Source:"]');
          if (rateInfoInput) {
            rateInfoInput.value = `Source: ${data.source} | Updated: 0 min ago`;
          }
          
          // Update global variables
          window.currentExchangeRate = data.rate;
          
          // Show success message
          showNotification('Exchange rate updated successfully!', 'success');
        } else {
          showNotification('Failed to refresh exchange rate: ' + data.message, 'error');
        }
      })
      .catch(error => {
        console.error('Error refreshing exchange rate:', error);
        showNotification('Error refreshing exchange rate. Please try again.', 'error');
      })
      .finally(() => {
        // Reset button
        refreshBtn.innerHTML = originalText;
        refreshBtn.disabled = false;
      });
    }
    
    function showPaystackIframe(authorizationUrl, orderId, showLoading = false) {
      console.log('showPaystackIframe called with URL:', authorizationUrl, 'showLoading:', showLoading);
      
      // Check if we're on localhost - use new window instead of iframe
      const isLocalhost = window.location.hostname === 'localhost' || window.location.hostname === '127.0.0.1';
      
      if (isLocalhost && authorizationUrl) {
        console.log('Localhost detected - opening payment in new window');
        openPaystackInNewWindow(authorizationUrl, orderId);
        return;
      }
      
      // Create simple iframe container without modal wrapper
      const iframeContainer = document.createElement('div');
      iframeContainer.id = 'paystack-iframe-container';
      iframeContainer.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0, 0, 0, 0.8);
        z-index: 10000;
        display: flex;
        align-items: center;
        justify-content: center;
        padding: 20px;
      `;
      
      // Determine iframe content based on whether we have URL or showing loading
      let iframeContent = '';
      if (showLoading || !authorizationUrl) {
        iframeContent = `
          <div style="
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            height: 600px;
            background: #f8f9fa;
            color: #6c757d;
          ">
            <div style="
              width: 60px;
              height: 60px;
              border: 4px solid #e3e3e3;
              border-top: 4px solid #007bff;
              border-radius: 50%;
              animation: spin 1s linear infinite;
              margin-bottom: 20px;
            "></div>
            <h5 style="margin: 0 0 10px 0; color: #333;">Initializing Payment...</h5>
            <p style="margin: 0; font-size: 14px;">Please wait while we prepare your payment form</p>
          </div>
          <style>
            @keyframes spin {
              0% { transform: rotate(0deg); }
              100% { transform: rotate(360deg); }
            }
          </style>
        `;
      } else {
        iframeContent = `
          <iframe 
            id="paystack-iframe" 
            src="${authorizationUrl}" 
            frameborder="0"
            allowtransparency="true"
            allow="payment"
            style="width: 100%; height: 600px; border: none; display: block;"
            onload="handleIframeLoad()"
          ></iframe>
        `;
      }
      
      iframeContainer.innerHTML = `
        <div style="
          background: white;
          border-radius: 15px;
          box-shadow: 0 20px 40px rgba(0, 0, 0, 0.3);
          width: 100%;
          max-width: 800px;
          max-height: 90vh;
          overflow: hidden;
          position: relative;
          display: flex;
          flex-direction: column;
        ">
          <div style="
            padding: 20px 25px;
            border-bottom: 1px solid #e9ecef;
            display: flex;
            justify-content: space-between;
            align-items: center;
            background: #f8f9fa;
          ">
            <h5 style="margin: 0; font-weight: 600; color: #333; font-size: 1.2rem;">
              <i class="fi fi-rr-credit-card me-2"></i>Complete Payment
            </h5>
            <button type="button" onclick="closePaystackIframe()" style="
              background: #ff6b6b;
              border: none;
              font-size: 1.2rem;
              color: white;
              cursor: pointer;
              padding: 0;
              width: 35px;
              height: 35px;
              display: flex;
              align-items: center;
              justify-content: center;
              border-radius: 50%;
              transition: all 0.3s ease;
              font-weight: bold;
            " onmouseover="this.style.background='#ff5252'; this.style.transform='scale(1.1)';" onmouseout="this.style.background='#ff6b6b'; this.style.transform='scale(1)';">
              ×
            </button>
          </div>
          <div style="flex: 1; padding: 0; overflow: hidden;">
            ${iframeContent}
          </div>
        </div>
      `;
      
      // Add to body
      document.body.appendChild(iframeContainer);
      console.log('Paystack iframe container added to body');
      
      // Listen for payment completion messages from iframe
      window.addEventListener('message', function(event) {
        if (event.origin !== 'https://checkout.paystack.com') return;
          
          if (event.data === 'payment_completed' || event.data === 'payment_successful') {
            closePaystackIframe();
            checkPaymentStatus(orderId);
          }
        });
        
        // Check payment status periodically
        const statusCheckInterval = setInterval(() => {
          checkPaymentStatus(orderId, (success) => {
            if (success) {
              clearInterval(statusCheckInterval);
              closePaystackIframe();
            }
          });
        }, 3000);
        
        // Store interval ID for cleanup
        iframeContainer.dataset.statusCheckInterval = statusCheckInterval;
    }
    
    // Function to open Paystack in new window (for localhost)
    function openPaystackInNewWindow(authorizationUrl, orderId) {
      console.log('Opening Paystack in new window:', authorizationUrl);
      
      // Open payment in new window
      const paymentWindow = window.open(
        authorizationUrl, 
        'paystack_payment', 
        'width=800,height=600,scrollbars=yes,resizable=yes,status=yes,toolbar=no,menubar=no,location=no'
      );
      
      if (!paymentWindow) {
        alert('Please allow popups for this site to complete your payment.');
        return;
      }
      
      // Focus the new window
      paymentWindow.focus();
      
      // Check if window is closed
      const checkClosed = setInterval(() => {
        if (paymentWindow.closed) {
          clearInterval(checkClosed);
          console.log('Payment window closed - checking payment status');
          checkPaymentStatusAfterWindowClose(orderId);
        }
      }, 1000);
      
      // Check payment status periodically
      const statusCheckInterval = setInterval(() => {
        checkPaymentStatus(orderId, (success) => {
          if (success) {
            clearInterval(statusCheckInterval);
            clearInterval(checkClosed);
            paymentWindow.close();
            showNotification('Payment successful!', 'success');
            setTimeout(() => {
              window.location.reload();
            }, 2000);
          }
        });
      }, 3000);
      
      // Store intervals for cleanup
      window.paymentIntervals = {
        statusCheck: statusCheckInterval,
        windowCheck: checkClosed
      };
    }
    
    // Function to check payment status after window closes
    function checkPaymentStatusAfterWindowClose(orderId) {
      console.log('Checking payment status after window close');
      checkPaymentStatus(orderId, (success) => {
        if (success) {
          showNotification('Payment successful!', 'success');
          setTimeout(() => {
            window.location.reload();
          }, 2000);
        } else {
          showNotification('Payment not completed. Please try again.', 'warning');
        }
      });
    }
    
    // Function to update iframe with actual payment URL
    function updatePaystackIframe(authorizationUrl) {
      console.log('Updating iframe with URL:', authorizationUrl);
      
      // Check if we're on localhost - use new window instead
      const isLocalhost = window.location.hostname === 'localhost' || window.location.hostname === '127.0.0.1';
      
      if (isLocalhost) {
        console.log('Localhost detected - opening payment in new window');
        openPaystackInNewWindow(authorizationUrl, <?php echo $order_id; ?>);
        return;
      }
      
      const iframeContainer = document.getElementById('paystack-iframe-container');
      if (iframeContainer) {
        const iframeDiv = iframeContainer.querySelector('div > div:last-child');
        if (iframeDiv) {
          iframeDiv.innerHTML = `
            <iframe 
              id="paystack-iframe" 
              src="${authorizationUrl}" 
              frameborder="0"
              allowtransparency="true"
              allow="payment"
              style="width: 100%; height: 600px; border: none; display: block;"
              onload="handleIframeLoad()"
            ></iframe>
          `;
          console.log('Iframe updated with payment URL');
        }
      }
    }
    
    function closePaystackIframe() {
      const iframeContainer = document.getElementById('paystack-iframe-container');
      if (iframeContainer) {
        // Clear status check interval
        const intervalId = iframeContainer.dataset.statusCheckInterval;
        if (intervalId) {
          clearInterval(intervalId);
        }
        
        // Remove container immediately
        iframeContainer.remove();
        console.log('Paystack iframe container removed');
      }
    }
    
    function handleIframeLoad() {
      console.log('Paystack iframe loaded');
    }
    
    function checkPaymentStatus(orderId, callback) {
      fetch(`check_payment_status.php?order_id=${orderId}`)
        .then(response => response.json())
        .then(data => {
          if (data.status === 'completed') {
            showNotification('Payment successful! Redirecting to download...', 'success');
            setTimeout(() => {
              // Get product_id from the current page or fetch it
              const urlParams = new URLSearchParams(window.location.search);
              const orderId = urlParams.get('order_id');
              
              // Fetch product_id from the order
              fetch(`get_order_product.php?order_id=${orderId}`)
                .then(response => response.json())
                .then(data => {
                  if (data.success && data.product_id) {
                    window.location.href = `download.php?product_id=${data.product_id}`;
                  } else {
                    // Fallback to products page
                    window.location.href = 'products.php';
                  }
                })
                .catch(error => {
                  console.error('Error fetching product ID:', error);
                  window.location.href = 'products.php';
                });
            }, 2000);
            if (callback) callback(true);
          } else if (callback) {
            callback(false);
          }
        })
        .catch(error => {
          console.error('Error checking payment status:', error);
          if (callback) callback(false);
        });
    }
  </script>

    <!-- begin::Edura Footer -->
    <footer class="footer-wrapper bg-body text-center">
      <div class="container-fluid">
        <p class="mb-0">© <span class="currentYear">2025</span> Omateq Digitals. Proudly powered by Omateq Digitals Ltd.</p>
      </div>
    </footer>
    <!-- end::Edura Footer -->

    <!-- WhatsApp Chat Button -->
    <div class="whatsapp-chat-button">
      <a href="https://wa.me/2347069378861?text=Hello%20Omateq%20Digitals!%20I%20would%20like%20to%20know%20more%20about%20your%20services." 
         target="_blank" 
         rel="noopener noreferrer"
         class="whatsapp-link"
         title="Chat with us on WhatsApp">
        <i class="fab fa-whatsapp"></i>
        <span class="whatsapp-text">Chat with us</span>
      </a>
    </div>
    
    <style>
    /* WhatsApp Chat Button Styles */
    .whatsapp-chat-button {
      position: fixed;
      bottom: 20px;
      right: 20px;
      z-index: 1000;
      animation: pulse 2s infinite;
    }
    
    .whatsapp-link {
      display: flex;
      align-items: center;
      background: #25D366;
      color: white;
      text-decoration: none;
      padding: 12px 16px;
      border-radius: 50px;
      box-shadow: 0 4px 12px rgba(37, 211, 102, 0.4);
      transition: all 0.3s ease;
      font-family: 'Poppins', sans-serif;
      font-weight: 500;
      font-size: 14px;
    }
    
    .whatsapp-link:hover {
      background: #128C7E;
      color: white;
      text-decoration: none;
      transform: translateY(-2px);
      box-shadow: 0 6px 20px rgba(37, 211, 102, 0.6);
    }
    
    .whatsapp-link i {
      font-size: 24px;
      margin-right: 8px;
    }
    
    .whatsapp-text {
      white-space: nowrap;
    }
    
    @keyframes pulse {
      0% {
        box-shadow: 0 0 0 0 rgba(37, 211, 102, 0.7);
      }
      70% {
        box-shadow: 0 0 0 10px rgba(37, 211, 102, 0);
      }
      100% {
        box-shadow: 0 0 0 0 rgba(37, 211, 102, 0);
      }
    }
    
    /* Mobile responsiveness */
    @media (max-width: 768px) {
      .whatsapp-chat-button {
        bottom: 15px;
        right: 15px;
      }
      
      .whatsapp-link {
        padding: 10px 14px;
        font-size: 13px;
      }
      
      .whatsapp-link i {
        font-size: 22px;
        margin-right: 6px;
      }
      
      .whatsapp-text {
        display: none;
      }
    }
    </style>

  </div>
</body>

</html>
