<?php
session_start();
require_once __DIR__ . '/config/database.php';

header('Content-Type: application/json');

// Load payment configuration
$payment_config = require __DIR__ . '/config/payment_config.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'User not logged in']);
    exit();
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

if (!$input || !isset($input['course_id']) || !isset($input['reference']) || !isset($input['amount'])) {
    echo json_encode(['success' => false, 'message' => 'Invalid input data']);
    exit();
}

$course_id = $input['course_id'];
$reference = $input['reference'];
$amount = $input['amount'];

try {
    error_log("Starting course payment processing for course ID: " . $course_id);
    error_log("Payment reference: " . $reference);
    error_log("Payment amount: " . $amount);
    
    $pdo = getConnection();
    error_log("Database connection established successfully");
    
    // Verify payment with Paystack
    error_log("Starting Paystack payment verification...");
    error_log("Payment config loaded: " . (isset($payment_config) ? 'Yes' : 'No'));
    error_log("Paystack config exists: " . (isset($payment_config['paystack']) ? 'Yes' : 'No'));
    error_log("Secret key exists: " . (isset($payment_config['paystack']['secret_key']) ? 'Yes' : 'No'));
    if (isset($payment_config['paystack']['secret_key'])) {
        error_log("Secret key value: " . substr($payment_config['paystack']['secret_key'], 0, 20) . "...");
    }
    $payment_verified = verifyPaystackPayment($reference, $payment_config['paystack']['secret_key']);
    
    if (!$payment_verified) {
        error_log("Payment verification failed for reference: " . $reference);
        echo json_encode(['success' => false, 'message' => 'Payment verification failed']);
        exit();
    }
    
    error_log("Payment verification successful");
    
    // Get course details
    error_log("Fetching course details for ID: " . $course_id);
    $stmt = $pdo->prepare("SELECT * FROM courses WHERE id = ? AND status = 'active'");
    $stmt->execute([$course_id]);
    $course = $stmt->fetch();
    
    if (!$course) {
        error_log("Course not found or not active: " . $course_id);
        echo json_encode(['success' => false, 'message' => 'Course not found']);
        exit();
    }
    
    error_log("Course found: " . $course['title'] . " - Price: " . $course['price']);
    
    // Check if user is already enrolled
    error_log("Checking if user " . $_SESSION['user_id'] . " is already enrolled in course " . $course_id);
    $stmt = $pdo->prepare("SELECT id FROM course_enrollments WHERE user_id = ? AND course_id = ?");
    $stmt->execute([$_SESSION['user_id'], $course_id]);
    
    if ($stmt->fetch()) {
        error_log("User is already enrolled in this course");
        echo json_encode(['success' => false, 'message' => 'You are already enrolled in this course']);
        exit();
    }
    
    error_log("User is not enrolled, proceeding with enrollment");
    
    // Enroll user in course with payment details
    error_log("Creating course enrollment record...");
    $stmt = $pdo->prepare("INSERT INTO course_enrollments (user_id, course_id, status, payment_status, payment_reference, payment_amount, payment_method, enrolled_at) VALUES (?, ?, 'enrolled', 'paid', ?, ?, 'paystack', NOW())");
    $stmt->execute([$_SESSION['user_id'], $course_id, $reference, $course['price']]);
    error_log("Course enrollment record created successfully");
    
    // Create order record for tracking (courses don't have product_id)
    $order_number = 'COURSE-' . date('Y-m-d') . '-' . str_pad(rand(1, 9999), 4, '0', STR_PAD_LEFT);
    error_log("Creating order record with number: " . $order_number);
    $stmt = $pdo->prepare("INSERT INTO orders (order_number, user_id, product_id, product_name, product_price, total_amount, payment_method, payment_reference, status, created_at) VALUES (?, ?, NULL, ?, ?, ?, 'paystack', ?, 'delivered', NOW())");
    $stmt->execute([
        $order_number,
        $_SESSION['user_id'],
        $course['title'],
        $course['price'],
        $course['price'],
        $reference
    ]);
    error_log("Order record created successfully");
    
    // Create transaction record for detailed tracking (courses don't have product_id)
    $transaction_id = 'TXN-COURSE-' . time() . '-' . str_pad(rand(1, 9999), 4, '0', STR_PAD_LEFT);
    error_log("Creating transaction record with ID: " . $transaction_id);
    $stmt = $pdo->prepare("INSERT INTO transactions (user_id, product_id, order_number, transaction_id, amount, currency, payment_method, payment_provider, status, payment_reference, created_at) VALUES (?, NULL, ?, ?, ?, 'USD', 'paystack', 'Paystack', 'completed', ?, NOW())");
    $stmt->execute([
        $_SESSION['user_id'],
        $order_number,
        $transaction_id,
        $course['price'],
        $reference
    ]);
    error_log("Transaction record created successfully");
    
    echo json_encode([
        'success' => true, 
        'message' => 'Payment successful and course enrollment completed',
        'course_id' => $course_id,
        'transaction_id' => $transaction_id
    ]);
    
} catch (Exception $e) {
    error_log("Course payment error: " . $e->getMessage());
    error_log("Course payment error trace: " . $e->getTraceAsString());
    echo json_encode([
        'success' => false, 
        'message' => 'Payment processing error: ' . $e->getMessage(),
        'debug' => [
            'error' => $e->getMessage(),
            'file' => $e->getFile(),
            'line' => $e->getLine()
        ]
    ]);
}

// Function to verify Paystack payment
function verifyPaystackPayment($reference, $secret_key) {
    try {
        $url = "https://api.paystack.co/transaction/verify/" . $reference;
        
        error_log("Verifying Paystack payment with reference: " . $reference);
        error_log("Using secret key: " . substr($secret_key, 0, 10) . "...");
        error_log("Full secret key length: " . strlen($secret_key));
        error_log("Secret key starts with: " . substr($secret_key, 0, 20));
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            "Authorization: Bearer " . $secret_key
        ]);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curl_error = curl_error($ch);
        curl_close($ch);
        
        error_log("Paystack API Response - HTTP Code: " . $http_code);
        error_log("Paystack API Response: " . $response);
        
        if ($curl_error) {
            error_log("cURL Error: " . $curl_error);
            return false;
        }
        
        if ($http_code === 200) {
            $result = json_decode($response, true);
            error_log("Paystack verification result: " . json_encode($result));
            
            if ($result && isset($result['status']) && isset($result['data'])) {
                $is_success = $result['status'] === true && $result['data']['status'] === 'success';
                error_log("Payment verification success: " . ($is_success ? 'true' : 'false'));
                return $is_success;
            } else {
                error_log("Invalid Paystack response format");
                return false;
            }
        }
        
        error_log("Paystack API returned non-200 status: " . $http_code);
        return false;
    } catch (Exception $e) {
        error_log("Paystack verification error: " . $e->getMessage());
        error_log("Paystack verification error trace: " . $e->getTraceAsString());
        return false;
    }
}
?>
