<?php
session_start();
require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/../admin-dashboard/config/settings.php';
require_once __DIR__ . '/../admin-dashboard/config/branding.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: authentication/login.php');
    exit();
}

// Get user data
try {
    $pdo = getConnection();
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $user = $stmt->fetch();
    
    if (!$user) {
        session_destroy();
        header('Location: authentication/login.php');
        exit();
    }
} catch (Exception $e) {
    session_destroy();
    header('Location: authentication/login.php');
    exit();
}

// Check for success/error messages
$payment_success = '';
if (isset($_SESSION['payment_success'])) {
    $payment_success = $_SESSION['payment_success'];
    unset($_SESSION['payment_success']);
}

$download_error = '';
if (isset($_SESSION['download_error'])) {
    $download_error = $_SESSION['download_error'];
    unset($_SESSION['download_error']);
}

// Function to format time ago
function time_ago($datetime) {
    $time = time() - strtotime($datetime);
    if ($time < 60) return 'just now';
    if ($time < 3600) return floor($time/60) . ' min ago';
    if ($time < 86400) return floor($time/3600) . ' hr ago';
    if ($time < 2592000) return floor($time/86400) . ' days ago';
    return date('M j', strtotime($datetime));
}

// Get script request replies
$script_replies = [];
$script_replies_count = 0;
try {
    $stmt = $pdo->prepare("SELECT * FROM script_requests WHERE user_id = ? AND admin_reply IS NOT NULL ORDER BY created_at DESC LIMIT 5");
    $stmt->execute([$_SESSION['user_id']]);
    $script_replies = $stmt->fetchAll();
    $script_replies_count = count($script_replies);
} catch (Exception $e) {
    // Handle error silently
}

// Get transactions (purchases and course enrollments)
$transactions = [];
$transactions_count = 0;
try {
    $stmt = $pdo->prepare("SELECT 'purchase' as type, product_name as item_name, total_amount as amount, status, created_at FROM orders WHERE user_id = ? 
                          UNION ALL 
                          SELECT 'course' as type, course_title as item_name, price as amount, 'completed' as status, created_at FROM course_enrollments ce 
                          JOIN courses c ON ce.course_id = c.id WHERE ce.user_id = ? 
                          ORDER BY created_at DESC LIMIT 5");
    $stmt->execute([$_SESSION['user_id'], $_SESSION['user_id']]);
    $transactions = $stmt->fetchAll();
    $transactions_count = count($transactions);
} catch (Exception $e) {
    // Handle error silently
}

// Get all products
$products = [];
try {
    $stmt = $pdo->prepare("SELECT id, product_name, product_description, thumbnail_url, category, product_features, demo_link, demo_login_details, product_specifications, product_amount, status, sort_order, is_downloadable, download_file_url, download_file_name, download_file_size, download_file_type, download_access_duration, download_limit, download_instructions, created_at FROM products WHERE status = 'active' ORDER BY created_at DESC");
    $stmt->execute();
    $products = $stmt->fetchAll();
} catch (Exception $e) {
    // Handle error silently
    error_log("Products query error: " . $e->getMessage());
}

// Get user's purchased products and download counts from transactions
$user_purchases = [];
$download_counts = [];
$transaction_status = [];
try {
    $stmt = $pdo->prepare("SELECT product_id, download_count, max_downloads, status FROM transactions WHERE user_id = ? AND status = 'completed'");
    $stmt->execute([$_SESSION['user_id']]);
    $purchases = $stmt->fetchAll();
    
    foreach ($purchases as $purchase) {
        $user_purchases[$purchase['product_id']] = true;
        $download_counts[$purchase['product_id']] = $purchase['download_count'] ?? 0;
        $transaction_status[$purchase['product_id']] = [
            'status' => $purchase['status'],
            'download_count' => $purchase['download_count'] ?? 0,
            'max_downloads' => $purchase['max_downloads'] ?? 3
        ];
    }
} catch (Exception $e) {
    // Handle error silently
    error_log("User purchases query error: " . $e->getMessage());
}

// Handle purchase action
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'purchase') {
        $product_id = $_POST['product_id'];
        $payment_method = 'paystack'; // Always use Paystack
        
        // Get product details
        $stmt = $pdo->prepare("SELECT * FROM products WHERE id = ? AND status = 'active'");
        $stmt->execute([$product_id]);
        $product = $stmt->fetch();
        
        if ($product) {
            // Generate unique order number
            do {
                $order_number = 'ORD-' . date('Y-m-d') . '-' . str_pad(rand(1, 9999), 4, '0', STR_PAD_LEFT);
                $check_stmt = $pdo->prepare("SELECT id FROM transactions WHERE order_number = ?");
                $check_stmt->execute([$order_number]);
            } while ($check_stmt->fetch());
            
            // Generate unique transaction ID
            $transaction_id = 'TXN-' . time() . '-' . str_pad(rand(1, 9999), 4, '0', STR_PAD_LEFT);
            
            // Paystack payments are processed immediately
            $initial_status = 'completed';
            
            // Create transaction
            $stmt = $pdo->prepare("INSERT INTO transactions (user_id, product_id, order_number, transaction_id, amount, currency, payment_method, payment_provider, status, max_downloads, created_at) VALUES (?, ?, ?, ?, ?, 'USD', ?, ?, ?, ?, NOW())");
            $stmt->execute([
                $_SESSION['user_id'], 
                $product_id, 
                $order_number, 
                $transaction_id, 
                $product['product_amount'], 
                $payment_method, 
                $payment_method, 
                $initial_status,
                $product['download_limit'] ?? 3
            ]);
            
            $transaction_id_db = $pdo->lastInsertId();
            
            // Also create order for backward compatibility
            $stmt = $pdo->prepare("INSERT INTO orders (order_number, user_id, product_id, product_name, product_price, total_amount, payment_method, status, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())");
            $stmt->execute([$order_number, $_SESSION['user_id'], $product_id, $product['product_name'], $product['product_amount'], $product['product_amount'], $payment_method, $initial_status]);
            
            $order_id = $pdo->lastInsertId();
            
            // Redirect to Paystack payment
            header("Location: payment_handler.php?order_id=$order_id&method=paystack");
            exit();
        }
    }
}

// Handle download action - redirect to download handler
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'download') {
    $product_id = $_POST['product_id'];
    header("Location: download_handler.php?product_id=$product_id");
    exit();
}
?>
<!DOCTYPE html>
<html lang="en">

<head>
  <!-- begin::Edura Meta Basic -->
  <meta charset="utf-8">
  <meta name="theme-color" content="#0D9488">
  <meta name="robots" content="index, follow">
  <meta name="author" content="LayoutDrop">
  <meta name="format-detection" content="telephone=no">
  <meta name="keywords" content="Products, Digital Products, Downloads, Purchase">
  <meta name="description" content="Browse and purchase digital products from Omateq Digitals">
  <!-- end::Edura Meta Basic -->

  <!-- begin::Edura Website Page Title -->
  <title>Products | Omateq Digitals</title>
  <!-- end::Edura Website Page Title -->

  <!-- begin::Edura Mobile Specific -->
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <!-- end::Edura Mobile Specific -->

  <!-- begin::Edura Favicon Tags -->
  <?php echo getFaviconHtml(); ?>
  <!-- end::Edura Favicon Tags -->

  <!-- begin::Edura Google Fonts -->
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Plus+Jakarta+Sans:ital,wght@0,200..800;1,200..800&display=swap" rel="stylesheet">
  <!-- end::Edura Google Fonts -->

  <!-- begin::Edura Required Stylesheet -->
  <link rel="stylesheet" href="assets/libs/flaticon/css/all/all.css">
  <link rel="stylesheet" href="assets/libs/lucide/lucide.css">
  <link rel="stylesheet" href="assets/libs/fontawesome/css/all.min.css">
  <link rel="stylesheet" href="assets/libs/simplebar/simplebar.css">
  <link rel="stylesheet" href="assets/libs/node-waves/waves.css">
  <link rel="stylesheet" href="assets/libs/bootstrap-select/css/bootstrap-select.min.css">
  <!-- end::Edura Required Stylesheet -->

  <!-- begin::Edura CSS Stylesheet -->
  <link rel="stylesheet" href="assets/css/styles.css">
  <!-- end::Edura CSS Stylesheet -->
  <style>
    /* Logo sizing for better fit */
    .app-navbar-brand .navbar-brand-logo img {
      max-height: 50px;
      max-width: 180px;
      width: auto;
      height: auto;
      object-fit: contain;
    }
    
    .app-navbar-brand .navbar-brand-mini img {
      max-height: 35px;
      max-width: 35px;
      width: auto;
      height: auto;
      object-fit: contain;
    }
  </style>
  
  <!-- Custom CSS for 3-column product layout -->
  <style>
    .products-grid {
      display: grid;
      grid-template-columns: repeat(3, 1fr);
      gap: 1.5rem;
    }
    
    @media (max-width: 991.98px) {
      .products-grid {
        grid-template-columns: repeat(2, 1fr);
        gap: 1rem;
      }
    }
    
    @media (max-width: 575.98px) {
      .products-grid {
        grid-template-columns: 1fr;
        gap: 1rem;
      }
      
      
      .product-price-badge {
        font-size: 1rem !important;
        padding: 6px 10px !important;
      }
    }
    
    .product-card {
      transition: transform 0.3s ease, box-shadow 0.3s ease;
      border: 1px solid #e9ecef;
    }
    
    .product-card:hover {
      transform: translateY(-5px);
      box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
    }
    
    .product-card:hover .product-price-badge {
      transform: scale(1.1);
      box-shadow: 0 6px 20px rgba(40, 167, 69, 0.4) !important;
    }
    
    
    .product-price-badge {
      transition: all 0.3s ease;
    }
    
    .product-image {
      height: 200px;
      object-fit: cover;
      width: 100%;
    }
    
    .product-price {
      font-size: 1.5rem;
      font-weight: 700;
      color: #28a745;
      text-shadow: 1px 1px 2px rgba(0, 0, 0, 0.1);
    }
    
    .product-price-badge {
      background: linear-gradient(135deg, #28a745, #20c997) !important;
      color: white !important;
      font-size: 1.1rem !important;
      font-weight: 700 !important;
      padding: 8px 12px !important;
      border-radius: 20px !important;
      box-shadow: 0 4px 15px rgba(40, 167, 69, 0.3) !important;
      border: 2px solid rgba(255, 255, 255, 0.2) !important;
      backdrop-filter: blur(10px) !important;
    }
    
    
    .product-title {
      font-size: 1.1rem;
      font-weight: 600;
      margin-bottom: 0.5rem;
      color: white;
      text-shadow: 1px 1px 2px rgba(0, 0, 0, 0.5);
    }
    
    .product-description {
      color: #6c757d;
      font-size: 0.9rem;
      line-height: 1.4;
    }
  </style>
</head>

<body>
  <div class="page-layout">

    <!-- begin::Edura Page Header -->
    <header class="app-header">
      <div class="app-header-inner">
        <button class="app-toggler" type="button">
          <span></span>
          <span></span>
          <span></span>
        </button>
        <div class="app-header-start">
          <form class="d-none d-md-flex align-items-center h-100 w-lg-250px w-xxl-300px position-relative" action="index.html#">
            <button type="button" class="btn btn-sm border-0 position-absolute start-0 ms-3 p-0">
              <i class="fi fi-rr-search"></i>
            </button>
            <input type="text" class="form-control ps-5" placeholder="Search products...">
          </form>
        </div>
        <div class="app-header-end">
          <div class="px-lg-3 px-2 ps-0 d-flex align-items-center">
            <a href="javascript:void(0);" class="theme-btn">
              <i class="fi fi-rr-brightness icon-light"></i>
              <div class="theme-toggle"></div>
              <i class="fi fi-rr-moon icon-dark"></i>
            </a>
          </div>
          <div class="vr my-3"></div>
          <div class="d-flex align-items-center gap-sm-2 gap-0 px-lg-4 px-sm-2 px-1">
            <div class="dropdown text-end">
              <a href="javascript:void(0);" class="btn btn-icon btn-action-gray rounded-circle waves-effect waves-light position-relative" data-bs-toggle="dropdown" data-bs-auto-close="outside" aria-expanded="true">
                <i class="fi fi-rr-envelope"></i>
                <span class="position-absolute top-0 end-0 p-1 mt-1 me-1 bg-danger border border-3 border-light rounded-circle">
                  <span class="visually-hidden">New messages</span>
                </span>
              </a>
              <div class="dropdown-menu dropdown-menu-lg-end p-0 w-300px mt-2">
                <div class="px-3 py-3 border-bottom d-flex justify-content-between align-items-center">
                  <h6 class="mb-0">Script Request Replies <span class="badge badge-sm rounded-pill bg-primary ms-2"><?php echo $script_replies_count ?? 0; ?></span></h6>
                  <i class="bi bi-x-lg cursor-pointer"></i>
                </div>
                <div class="p-2" style="height: 300px;" data-simplebar>
                  <ul class="list-group list-group-hover list-group-smooth list-group-unlined">
                    <?php if (!empty($script_replies)): ?>
                      <?php foreach ($script_replies as $reply): ?>
                        <li class="list-group-item d-flex justify-content-between align-items-center">
                          <div class="avatar avatar-xs bg-info rounded-circle text-white">
                            <i class="fi fi-rr-user"></i>
                          </div>
                          <div class="ms-2 me-auto">
                            <h6 class="mb-0">Admin Reply</h6>
                            <small class="text-body d-block"><?php echo htmlspecialchars(substr($reply['admin_reply'], 0, 50)) . '...'; ?></small>
                            <small class="text-muted position-absolute end-0 top-0 mt-2 me-3"><?php echo time_ago($reply['created_at']); ?></small>
                          </div>
                        </li>
                      <?php endforeach; ?>
                    <?php else: ?>
                      <li class="list-group-item text-center py-4">
                        <i class="fi fi-rr-envelope text-muted" style="font-size: 2rem;"></i>
                        <p class="text-muted mt-2 mb-0">No script request replies yet</p>
                      </li>
                    <?php endif; ?>
                  </ul>
                </div>
                <div class="p-2">
                  <a href="script-request.php" class="btn w-100 btn-primary waves-effect waves-light">View all replies</a>
                </div>
              </div>
            </div>
            <div class="dropdown text-end">
              <a href="javascript:void(0);" class="btn btn-icon btn-action-gray rounded-circle waves-effect waves-light" data-bs-toggle="dropdown" data-bs-auto-close="outside" aria-expanded="true">
                <i class="fi fi-rr-bell"></i>
              </a>
              <div class="dropdown-menu dropdown-menu-lg-end p-0 w-300px mt-2">
                <div class="px-3 py-3 border-bottom d-flex justify-content-between align-items-center">
                  <h6 class="mb-0">Transactions <span class="badge badge-sm rounded-pill bg-primary ms-2"><?php echo $transactions_count ?? 0; ?></span></h6>
                  <i class="bi bi-x-lg cursor-pointer"></i>
                </div>
                <div class="p-2" style="height: 300px;" data-simplebar>
                  <ul class="list-group list-group-hover list-group-smooth list-group-unlined">
                    <?php if (!empty($transactions)): ?>
                      <?php foreach ($transactions as $transaction): ?>
                        <li class="list-group-item d-flex justify-content-between align-items-center">
                          <div class="avatar avatar-xs bg-<?php echo $transaction['type'] === 'purchase' ? 'success' : 'info'; ?> rounded-circle text-white">
                            <i class="fi fi-rr-<?php echo $transaction['type'] === 'purchase' ? 'shopping-cart' : 'graduation-cap'; ?>"></i>
                          </div>
                          <div class="ms-2 me-auto">
                            <h6 class="mb-0"><?php echo ucfirst($transaction['type']); ?> - <?php echo htmlspecialchars($transaction['item_name']); ?></h6>
                            <small class="text-body d-block"><?php echo $transaction['status'] === 'completed' ? 'Payment successful' : 'Payment pending'; ?> - $<?php echo number_format($transaction['amount'], 2); ?></small>
                            <small class="text-muted position-absolute end-0 top-0 mt-2 me-3"><?php echo time_ago($transaction['created_at']); ?></small>
                          </div>
                        </li>
                      <?php endforeach; ?>
                    <?php else: ?>
                      <li class="list-group-item text-center py-4">
                        <i class="fi fi-rr-shopping-cart text-muted" style="font-size: 2rem;"></i>
                        <p class="text-muted mt-2 mb-0">No transactions yet</p>
                      </li>
                    <?php endif; ?>
                  </ul>
                </div>
                <div class="p-2">
                  <a href="products.php" class="btn w-100 btn-primary waves-effect waves-light">View all transactions</a>
                </div>
              </div>
            </div>
          </div>
          <div class="vr my-3"></div>
          <div class="dropdown text-end ms-2">
            <a href="javascript:void(0);" class="d-flex align-items-center py-2" data-bs-toggle="dropdown" data-bs-auto-close="outside" aria-expanded="true">
              <div class="avatar avatar-sm rounded-circle avatar-status-success">
                <?php if ($user['avatar'] && file_exists($user['avatar'])): ?>
                  <img src="<?php echo htmlspecialchars($user['avatar']); ?>" alt="<?php echo htmlspecialchars($user['full_name']); ?>">
                <?php else: ?>
                  <img src="assets/images/avatar/avatar1.webp" alt="<?php echo htmlspecialchars($user['full_name']); ?>">
                <?php endif; ?>
              </div>
            </a>
            <ul class="dropdown-menu dropdown-menu-end w-225px mt-1">
              <li class="d-flex align-items-center p-2">
                <div class="avatar avatar-sm rounded-circle">
                  <?php if ($user['avatar'] && file_exists($user['avatar'])): ?>
                    <img src="<?php echo htmlspecialchars($user['avatar']); ?>" alt="<?php echo htmlspecialchars($user['full_name']); ?>">
                  <?php else: ?>
                    <img src="assets/images/avatar/avatar1.webp" alt="<?php echo htmlspecialchars($user['full_name']); ?>">
                  <?php endif; ?>
                </div>
                <div class="ms-2">
                  <div class="fw-bold text-dark"><?php echo htmlspecialchars($user['full_name']); ?></div>
                  <small class="text-body d-block lh-sm"><?php echo htmlspecialchars($user['email']); ?></small>
                </div>
              </li>
              <li>
                <div class="dropdown-divider my-1"></div>
              </li>
              <li>
                <a class="dropdown-item d-flex align-items-center gap-2" href="profile.php">
                  <i class="fi fi-rr-user scale-1x"></i> Profile
                </a>
              </li>
              <li>
                <a class="dropdown-item d-flex align-items-center gap-2" href="help-center.php">
                  <i class="fi fi-rs-interrogation scale-1x"></i> Help Center
                </a>
              </li>
              <li>
                <div class="dropdown-divider my-1"></div>
              </li>
              <li>
                <a class="dropdown-item d-flex align-items-center gap-2 text-danger" href="authentication/logout.php">
                  <i class="fi fi-sr-exit scale-1x"></i> Logout
                </a>
              </li>
            </ul>
          </div>
        </div>
      </div>
    </header>
    <!-- end::Edura Page Header -->

    <!-- begin::Edura Sidebar Menu -->
    <aside class="app-menubar" id="menubar">
      <div class="app-navbar-brand">
        <a class="navbar-brand-logo" href="index.php">
          <?php echo '<img src="' . '../' . getSetting('main_logo', 'assets/images/logo.svg') . '" alt="logo">'; ?>
        </a>
        <a class="navbar-brand-mini" href="index.php">
          <?php echo '<img src="' . '../' . getSetting('mini_logo', 'assets/images/logo-text-white.svg') . '" alt="logo">'; ?>
        </a>
      </div>
      <nav class="app-navbar" data-simplebar>
        <ul class="menubar">
              <li class="menu-item">
                <a class="menu-link" href="index.php">
              <i class="fi fi-rr-apps"></i>
                  <span class="menu-label">Dashboard</span>
                </a>
              </li>
              <li class="menu-item active">
            <a class="menu-link" href="products.php">
              <i class="fi fi-rr-shopping-cart"></i>
              <span class="menu-label">Products</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="course.php">
              <i class="fi fi-rr-graduation-cap"></i>
              <span class="menu-label">Course</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="my_courses.php">
              <i class="fi fi-rr-book"></i>
              <span class="menu-label">My Courses</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="profile.php">
              <i class="fi fi-rr-user"></i>
              <span class="menu-label">Profile</span>
                </a>
              </li>
              
        </ul>
      </nav>
      <div class="app-footer">
        <div class="text-center py-2">
          <small class="text-white-50">
            <i class="fi fi-rr-copyright me-1"></i>
            2025 Omateq Digitals
          </small>
        </div>
        <a href="mailto:info@omateqdigitals.com" class="btn btn-outline-white waves-effect btn-app-nav w-100">
          <i class="fi fi-rs-interrogation"></i>
          <span class="nav-text">Help and Support</span>
        </a>
      </div>
    </aside>
    <!-- end::Edura Sidebar Menu -->

    <main class="app-wrapper">
      <div class="container-fluid">
        <div class="app-page-head d-flex flex-wrap gap-3 align-items-center justify-content-between">
          <div class="clearfix">
            <h1 class="app-page-title">Digital Products</h1>
            <span>Browse and purchase our digital products</span>
          </div>
        </div>

        <?php if (!empty($payment_success)): ?>
          <div class="alert alert-success alert-dismissible fade show" role="alert">
            <div class="d-flex align-items-center">
              <i class="fi fi-rr-check-circle me-3" style="font-size: 2rem;"></i>
              <div>
                <h5 class="alert-heading mb-1">🎉 Payment Successful!</h5>
                <p class="mb-0"><?php echo htmlspecialchars($payment_success); ?></p>
              </div>
            </div>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
          </div>
        <?php endif; ?>

        <?php if (!empty($download_error)): ?>
          <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <div class="d-flex align-items-center">
              <i class="fi fi-rr-exclamation-triangle me-3" style="font-size: 2rem;"></i>
              <div>
                <h5 class="alert-heading mb-1">Download Error</h5>
                <p class="mb-0"><?php echo htmlspecialchars($download_error); ?></p>
              </div>
            </div>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
          </div>
        <?php endif; ?>

        <!-- Products Grid - 3 columns per row -->
        <div class="products-grid">
          <?php if (!empty($products)): ?>
            <?php foreach ($products as $product): ?>
              <div class="product-card">
                <div class="card h-100">
                  <div class="card-header border-0 p-0 m-0 position-relative overflow-hidden">
                    <?php if (!empty($product['thumbnail_url'])): ?>
                      <?php 
                        $image_path = $product['thumbnail_url'];
                        // Add ../ prefix if the path doesn't start with http or /
                        if (!str_starts_with($image_path, 'http') && !str_starts_with($image_path, '/')) {
                          $image_path = '../' . $image_path;
                        }
                      ?>
                      <img src="<?php echo htmlspecialchars($image_path); ?>" alt="<?php echo htmlspecialchars($product['product_name']); ?>" class="img-fluid rounded-top product-image">
                    <?php else: ?>
                      <div class="bg-light d-flex align-items-center justify-content-center" style="height: 200px;">
                        <i class="fi fi-rr-image text-muted" style="font-size: 3rem;"></i>
                      </div>
                    <?php endif; ?>
                    <div class="position-absolute top-0 end-0 m-3">
                      <span class="product-price-badge">$<?php echo number_format($product['product_amount'], 2); ?></span>
                    </div>
                  </div>
                  <div class="card-body">
                    <h5 class="card-title product-title mb-3"><?php echo htmlspecialchars($product['product_name']); ?></h5>
                    <p class="text-muted product-description mb-3"><?php echo htmlspecialchars(substr($product['product_description'], 0, 100)) . '...'; ?></p>
                    
                    <?php 
                    // Check if user has any transaction for this product
                    $has_transaction = false;
                    $transaction_info = null;
                    $pending_transaction = false;
                    
                    // Check for completed transactions
                    if (isset($transaction_status[$product['id']])) {
                        $has_transaction = true;
                        $transaction_info = $transaction_status[$product['id']];
                    }
                    
                    // Check for pending transactions
                    try {
                        $stmt = $pdo->prepare("SELECT status, payment_method FROM transactions WHERE user_id = ? AND product_id = ? AND status = 'pending'");
                        $stmt->execute([$_SESSION['user_id'], $product['id']]);
                        $pending_txn = $stmt->fetch();
                        if ($pending_txn) {
                            $pending_transaction = true;
                        }
                    } catch (Exception $e) {
                        // Handle error silently
                    }
                    ?>
                    
                    <?php if ($product['is_downloadable'] == 1): ?>
                      <div class="d-flex align-items-center gap-2 mb-3">
                        <i class="fi fi-rr-download text-primary"></i>
                        <small class="text-muted">Digital Download</small>
                      </div>
                      
                      <?php if ($has_transaction && $transaction_info['status'] === 'completed'): ?>
                        <?php 
                        $max_downloads = $transaction_info['max_downloads'];
                        $downloads_left = $max_downloads - $transaction_info['download_count'];
                        ?>
                        <?php if ($downloads_left > 0 && $product['download_file_url']): ?>
                          <div class="alert alert-success alert-sm mb-3">
                            <i class="fi fi-rr-check-circle me-2"></i>
                            <strong>Purchased!</strong> <?php echo $downloads_left; ?> of <?php echo $max_downloads; ?> downloads remaining
                          </div>
                          <form method="POST" class="d-inline">
                            <input type="hidden" name="action" value="download">
                            <input type="hidden" name="product_id" value="<?php echo $product['id']; ?>">
                            <button type="submit" class="btn btn-success btn-sm">
                              <i class="fi fi-rr-download me-1"></i> Download File
                            </button>
                          </form>
                        <?php elseif ($downloads_left > 0 && !$product['download_file_url']): ?>
                          <div class="alert alert-warning alert-sm mb-3">
                            <i class="fi fi-rr-exclamation-triangle me-2"></i>
                            Download file not available. Contact support.
                          </div>
                        <?php else: ?>
                          <div class="alert alert-warning alert-sm mb-3">
                            <i class="fi fi-rr-exclamation-triangle me-2"></i>
                            Download limit reached! Repurchase to download again.
                          </div>
                          <button type="button" class="btn btn-warning btn-sm" data-bs-toggle="modal" data-bs-target="#purchaseModal" data-product-id="<?php echo $product['id']; ?>" data-product-name="<?php echo htmlspecialchars($product['product_name']); ?>" data-product-price="<?php echo $product['product_amount']; ?>">
                            <i class="fi fi-rr-shopping-cart me-1"></i> Repurchase
                          </button>
                        <?php endif; ?>
                      <?php elseif ($pending_transaction): ?>
                        <div class="alert alert-warning alert-sm mb-3">
                          <i class="fi fi-rr-clock me-2"></i>
                          <strong>Payment Pending</strong> - Awaiting admin approval
                        </div>
                        <button class="btn btn-warning btn-sm" disabled>
                          <i class="fi fi-rr-hourglass me-1"></i> Awaiting Approval
                        </button>
                      <?php else: ?>
                        <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#purchaseModal" data-product-id="<?php echo $product['id']; ?>" data-product-name="<?php echo htmlspecialchars($product['product_name']); ?>" data-product-price="<?php echo $product['product_amount']; ?>">
                          <i class="fi fi-rr-shopping-cart me-1"></i> Purchase
                        </button>
                      <?php endif; ?>
                    <?php else: ?>
                      <div class="d-flex align-items-center gap-2 mb-3">
                        <i class="fi fi-rr-eye text-primary"></i>
                        <small class="text-muted">View Only</small>
                      </div>
                      <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#purchaseModal" data-product-id="<?php echo $product['id']; ?>" data-product-name="<?php echo htmlspecialchars($product['product_name']); ?>" data-product-price="<?php echo $product['product_amount']; ?>">
                        <i class="fi fi-rr-shopping-cart me-1"></i> Purchase
                      </button>
                    <?php endif; ?>
                    
                    <button type="button" class="btn btn-outline-secondary btn-sm ms-2" data-bs-toggle="modal" data-bs-target="#productDetailsModal" data-product='<?php echo json_encode($product); ?>'>
                      <i class="fi fi-rr-eye me-1"></i> View Details
                    </button>
                  </div>
                </div>
              </div>
            <?php endforeach; ?>
          <?php else: ?>
            <div class="col-12">
              <div class="card">
                <div class="card-body text-center py-5">
                  <i class="fi fi-rr-shopping-cart text-muted" style="font-size: 4rem;"></i>
                  <h4 class="mt-3 mb-2">No Products Available</h4>
                  <p class="text-muted">Check back later for new digital products!</p>
                </div>
              </div>
            </div>
          <?php endif; ?>
        </div>
      </div>
    </main>

    <!-- Product Details Modal -->
    <div class="modal fade" id="productDetailsModal" tabindex="-1" aria-labelledby="productDetailsModalLabel" aria-hidden="true">
      <div class="modal-dialog modal-lg">
        <div class="modal-content">
          <div class="modal-header">
            <h5 class="modal-title" id="productDetailsModalLabel">Product Details</h5>
            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
          </div>
          <div class="modal-body" id="productDetailsContent">
            <!-- Content will be loaded via JavaScript -->
          </div>
          <div class="modal-footer">
            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
          </div>
        </div>
      </div>
    </div>

    <!-- Purchase Modal -->
    <div class="modal fade" id="purchaseModal" tabindex="-1" aria-labelledby="purchaseModalLabel" aria-hidden="true">
      <div class="modal-dialog">
        <div class="modal-content">
          <div class="modal-header">
            <h5 class="modal-title" id="purchaseModalLabel">Purchase Product</h5>
            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
          </div>
          <form method="POST" id="purchaseForm">
            <input type="hidden" name="action" value="purchase">
            <input type="hidden" name="product_id" id="purchaseProductId">
            <input type="hidden" name="payment_region" value="nigeria">
            <input type="hidden" name="payment_method" value="paystack">
            <div class="modal-body">
              <div class="mb-4">
                <h6 id="purchaseProductName"></h6>
                <div class="payment-details">
                  <div class="d-flex justify-content-between mb-2">
                    <span>Price (USD):</span>
                    <span class="fw-bold">$<span id="purchaseProductPrice"></span></span>
                  </div>
                  <div class="d-flex justify-content-between mb-2">
                    <span>Exchange Rate:</span>
                    <span class="fw-bold" id="purchaseExchangeRate">1 USD = ₦<?php 
                      require_once __DIR__ . '/config/exchange_rate.php';
                      try {
                        $exchange_rate_info = getExchangeRateWithInfo();
                        $exchange_rate = $exchange_rate_info['rate'] ?? 1500;
                        echo number_format($exchange_rate, 2);
                      } catch (Exception $e) {
                        echo number_format(1500, 2);
                      }
                    ?></span>
                  </div>
                  <div class="d-flex justify-content-between mb-2">
                    <span>Amount (NGN):</span>
                    <span class="fw-bold text-primary" id="purchaseAmountNGN">₦0.00</span>
                  </div>
                </div>
              </div>
              
              <!-- Payment Method - Paystack Only -->
              <div class="mb-4">
                <label class="form-label">Payment Method</label>
                <div class="card border-primary">
                  <div class="card-body text-center">
                    <div class="d-flex align-items-center justify-content-center">
                      <i class="fi fi-rr-credit-card me-3 text-primary" style="font-size: 2rem;"></i>
                      <div>
                        <h6 class="mb-1">Paystack</h6>
                        <small class="text-muted">Card, Bank Transfer & USSD</small>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
              
              <!-- Security Notice -->
              <div class="alert alert-info">
                <i class="fi fi-rr-shield-check me-2"></i>
                <strong>Secure Payment:</strong> Your payment is protected with 256-bit SSL encryption and processed securely by Paystack.
              </div>
              
              <!-- Payment Features -->
              <div class="row text-center">
                <div class="col-4">
                  <i class="fi fi-rr-credit-card text-primary mb-2" style="font-size: 1.5rem;"></i>
                  <small class="d-block text-muted">Cards</small>
                </div>
                <div class="col-4">
                  <i class="fi fi-rr-bank text-primary mb-2" style="font-size: 1.5rem;"></i>
                  <small class="d-block text-muted">Bank Transfer</small>
                </div>
                <div class="col-4">
                  <i class="fi fi-rr-mobile text-primary mb-2" style="font-size: 1.5rem;"></i>
                  <small class="d-block text-muted">USSD</small>
                </div>
              </div>
            </div>
            <div class="modal-footer">
              <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
              <button type="submit" class="btn btn-primary">
                <i class="fi fi-rr-credit-card me-2"></i>
                Pay Securely
              </button>
            </div>
          </form>
        </div>
      </div>
    </div>

  </div>

  <!-- begin::Edura Page Scripts -->
  <script src="assets/libs/global/global.min.js"></script>
  <script src="assets/js/appSettings.js"></script>
  <script src="assets/js/main.js"></script>
  
  <script>
    // Product Details Modal
    document.getElementById('productDetailsModal').addEventListener('show.bs.modal', function (event) {
      const button = event.relatedTarget;
      const product = JSON.parse(button.getAttribute('data-product'));
      
      const modalBody = document.getElementById('productDetailsContent');
      modalBody.innerHTML = `
        <div class="row">
          <div class="col-md-4">
            ${product.thumbnail_url ? `<img src="${product.thumbnail_url.startsWith('http') || product.thumbnail_url.startsWith('/') ? product.thumbnail_url : '../' + product.thumbnail_url}" alt="${product.product_name}" class="img-fluid rounded">` : ''}
            <div class="bg-light d-flex align-items-center justify-content-center rounded" style="height: 200px; ${product.thumbnail_url ? 'display: none;' : ''}">
              <i class="fi fi-rr-image text-muted" style="font-size: 3rem;"></i>
            </div>
          </div>
          <div class="col-md-8">
            <h4>${product.product_name}</h4>
            <p class="text-muted">${product.product_description}</p>
            <div class="text-center mb-4 p-3" style="background: linear-gradient(135deg, #f8f9fa, #e9ecef); border-radius: 15px; border: 3px solid #28a745;">
              <div style="font-size: 2.5rem; font-weight: 800; color: #28a745; text-shadow: 2px 2px 4px rgba(0, 0, 0, 0.1); margin: 0; line-height: 1;">$${parseFloat(product.product_amount).toFixed(2)}</div>
              <div style="font-size: 1rem; color: #6c757d; font-weight: 600; margin: 0;">USD</div>
              ${product.is_downloadable == 1 ? '<span class="badge bg-primary mt-2" style="font-size: 0.9rem; padding: 6px 12px;">Digital Download</span>' : '<span class="badge bg-secondary mt-2" style="font-size: 0.9rem; padding: 6px 12px;">View Only</span>'}
            </div>
            ${product.is_downloadable == 1 ? `
              <div class="d-flex align-items-center gap-2 mb-3">
                <i class="fi fi-rr-download text-primary"></i>
                <small class="text-muted">${product.download_limit || 3} Downloads Included</small>
              </div>
              <div class="d-flex align-items-center gap-2">
                <i class="fi fi-rr-clock text-primary"></i>
                <small class="text-muted">Instant Access After Purchase</small>
              </div>
              ${product.download_instructions ? `<div class="alert alert-info mt-3"><h6><i class="fi fi-rr-info me-2"></i>Download Instructions:</h6><p class="mb-0 small">${product.download_instructions}</p></div>` : ''}
            ` : `
              <div class="d-flex align-items-center gap-2 mb-3">
                <i class="fi fi-rr-eye text-primary"></i>
                <small class="text-muted">This product is for viewing only</small>
              </div>
            `}
            ${product.product_features ? `<div class="mt-3"><h6>Features:</h6><p class="text-muted small">${product.product_features}</p></div>` : ''}
            ${product.demo_link ? `<div class="mt-2"><a href="${product.demo_link}" target="_blank" class="btn btn-outline-primary btn-sm"><i class="fi fi-rr-external-link me-1"></i> View Demo</a></div>` : ''}
          </div>
        </div>
      `;
    });

    // Purchase Modal
    document.getElementById('purchaseModal').addEventListener('show.bs.modal', function (event) {
      const button = event.relatedTarget;
      const productId = button.getAttribute('data-product-id');
      const productName = button.getAttribute('data-product-name');
      const productPrice = button.getAttribute('data-product-price');
      
      document.getElementById('purchaseProductId').value = productId;
      document.getElementById('purchaseProductName').textContent = productName;
      document.getElementById('purchaseProductPrice').textContent = parseFloat(productPrice).toFixed(2);
      
      // Calculate and display NGN amount using the same exchange rate as payment_handler.php
      const exchangeRate = <?php 
        require_once __DIR__ . '/config/exchange_rate.php';
        try {
          $exchange_rate_info = getExchangeRateWithInfo();
          $exchange_rate = $exchange_rate_info['rate'] ?? 1500;
          echo json_encode($exchange_rate);
        } catch (Exception $e) {
          echo json_encode(1500);
        }
      ?>;
      const priceUSD = parseFloat(productPrice);
      const amountNGN = priceUSD * exchangeRate;
      document.getElementById('purchaseAmountNGN').textContent = `₦${amountNGN.toLocaleString()}`;
      
      // Payment method is pre-selected as Paystack
      // No need to reset payment options since we only have one option
    });
    
    // Payment method is pre-selected as Paystack - no region selection needed
    
    // Handle form submission validation
    document.getElementById('purchaseForm').addEventListener('submit', function(e) {
      // Payment method is pre-selected as Paystack, no validation needed
      // Form will submit directly to Paystack payment
    });
  </script>

  <!-- WhatsApp Chat Button -->
  <div class="whatsapp-chat-button">
    <a href="https://wa.me/2347069378861?text=Hello%20Omateq%20Digitals!%20I%20would%20like%20to%20know%20more%20about%20your%20services." 
       target="_blank" 
       rel="noopener noreferrer"
       class="whatsapp-link"
       title="Chat with us on WhatsApp">
      <i class="fab fa-whatsapp"></i>
      <span class="whatsapp-text">Chat with us</span>
    </a>
  </div>
  
  <style>
  /* WhatsApp Chat Button Styles */
  .whatsapp-chat-button {
    position: fixed;
    bottom: 20px;
    right: 20px;
    z-index: 1000;
    animation: pulse 2s infinite;
  }
  
  .whatsapp-link {
    display: flex;
    align-items: center;
    background: #25D366;
    color: white;
    text-decoration: none;
    padding: 12px 16px;
    border-radius: 50px;
    box-shadow: 0 4px 12px rgba(37, 211, 102, 0.4);
    transition: all 0.3s ease;
    font-family: 'Poppins', sans-serif;
    font-weight: 500;
    font-size: 14px;
  }
  
  .whatsapp-link:hover {
    background: #128C7E;
    color: white;
    text-decoration: none;
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(37, 211, 102, 0.6);
  }
  
  .whatsapp-link i {
    font-size: 24px;
    margin-right: 8px;
  }
  
  .whatsapp-text {
    white-space: nowrap;
  }
  
  @keyframes pulse {
    0% {
      box-shadow: 0 0 0 0 rgba(37, 211, 102, 0.7);
    }
    70% {
      box-shadow: 0 0 0 10px rgba(37, 211, 102, 0);
    }
    100% {
      box-shadow: 0 0 0 0 rgba(37, 211, 102, 0);
    }
  }
  
  /* Mobile responsiveness */
  @media (max-width: 768px) {
    .whatsapp-chat-button {
      bottom: 15px;
      right: 15px;
    }
    
    .whatsapp-link {
      padding: 10px 14px;
      font-size: 13px;
    }
    
    .whatsapp-link i {
      font-size: 22px;
      margin-right: 6px;
    }
    
    .whatsapp-text {
      display: none;
    }
  }
  </style>

  <!-- end::Edura Page Scripts -->
</body>

</html>
<?php
// End of PHP processing
?>