<?php
require_once 'config/database.php';
require_once __DIR__ . '/../admin-dashboard/config/settings.php';
require_once __DIR__ . '/../admin-dashboard/config/branding.php';
require_once __DIR__ . '/config/session_manager.php';

// Initialize session and check login with timeout
UserSessionManager::initSession();
UserSessionManager::requireLogin('authentication/login.php');

$user_id = $_SESSION['user_id'];
$success_message = '';
$error_message = '';

try {
    $pdo = getConnection();
    
    // Handle form submissions
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        if (isset($_POST['update_profile'])) {
            // Update profile information
            $full_name = trim($_POST['full_name']);
            $email = trim($_POST['email']);
            $phone = trim($_POST['phone']);
            $bio = trim($_POST['bio']);
            
            // Validate email
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $error_message = 'Please enter a valid email address.';
            } else {
                // Check if email is already taken by another user
                $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ? AND id != ?");
                $stmt->execute([$email, $user_id]);
                if ($stmt->fetch()) {
                    $error_message = 'This email address is already in use.';
                } else {
                    // Update user profile
                    $stmt = $pdo->prepare("UPDATE users SET full_name = ?, email = ?, phone = ?, updated_at = NOW() WHERE id = ?");
                    if ($stmt->execute([$full_name, $email, $phone, $user_id])) {
                        $success_message = 'Profile updated successfully!';
                    } else {
                        $error_message = 'Failed to update profile. Please try again.';
                    }
                }
            }
        }
        
        if (isset($_POST['change_password'])) {
            // Change password
            $current_password = $_POST['current_password'];
            $new_password = $_POST['new_password'];
            $confirm_password = $_POST['confirm_password'];
            
            // Get current password hash
            $stmt = $pdo->prepare("SELECT password FROM users WHERE id = ?");
            $stmt->execute([$user_id]);
            $user = $stmt->fetch();
            
            if (!password_verify($current_password, $user['password'])) {
                $error_message = 'Current password is incorrect.';
            } elseif ($new_password !== $confirm_password) {
                $error_message = 'New passwords do not match.';
            } elseif (strlen($new_password) < 6) {
                $error_message = 'New password must be at least 6 characters long.';
            } else {
                $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
                $stmt = $pdo->prepare("UPDATE users SET password = ?, updated_at = NOW() WHERE id = ?");
                if ($stmt->execute([$hashed_password, $user_id])) {
                    $success_message = 'Password changed successfully!';
                } else {
                    $error_message = 'Failed to change password. Please try again.';
                }
            }
        }
        
        if (isset($_POST['upload_avatar'])) {
            // Handle avatar upload
            if (isset($_FILES['avatar']) && $_FILES['avatar']['error'] === UPLOAD_ERR_OK) {
                $upload_dir = 'assets/uploads/avatars/';
                if (!file_exists($upload_dir)) {
                    mkdir($upload_dir, 0777, true);
                }
                
                $file_extension = strtolower(pathinfo($_FILES['avatar']['name'], PATHINFO_EXTENSION));
                $allowed_extensions = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
                
                if (in_array($file_extension, $allowed_extensions)) {
                    $file_name = 'avatar_' . $user_id . '_' . time() . '.' . $file_extension;
                    $file_path = $upload_dir . $file_name;
                    
                    if (move_uploaded_file($_FILES['avatar']['tmp_name'], $file_path)) {
                        // Delete old avatar if exists
                        $stmt = $pdo->prepare("SELECT avatar FROM users WHERE id = ?");
                        $stmt->execute([$user_id]);
                        $old_avatar = $stmt->fetchColumn();
                        if ($old_avatar && file_exists($old_avatar)) {
                            unlink($old_avatar);
                        }
                        
                        // Update database
                        $stmt = $pdo->prepare("UPDATE users SET avatar = ?, updated_at = NOW() WHERE id = ?");
                        if ($stmt->execute([$file_path, $user_id])) {
                            $success_message = 'Avatar updated successfully!';
                        } else {
                            $error_message = 'Failed to update avatar in database.';
                        }
                    } else {
                        $error_message = 'Failed to upload avatar. Please try again.';
                    }
                } else {
                    $error_message = 'Invalid file type. Please upload JPG, PNG, GIF, or WebP images only.';
                }
            } else {
                $error_message = 'Please select a valid image file.';
            }
        }
    }
    
    // Get user data
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch();
    
    if (!$user) {
        header('Location: authentication/login.php');
        exit();
    }
    
    // Get user's course enrollment count
    $stmt = $pdo->prepare("SELECT COUNT(*) as course_count FROM course_enrollments WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $course_stats = $stmt->fetch();
    
} catch (Exception $e) {
    $error_message = 'An error occurred. Please try again later.';
    error_log("Profile page error: " . $e->getMessage());
}
?>
<!DOCTYPE html>
<html lang="en">

<head>
  <!-- begin::Edura Meta Basic -->
  <meta charset="utf-8">
  <meta name="theme-color" content="#0D9488">
  <meta name="robots" content="index, follow">
  <meta name="author" content="LayoutDrop">
  <meta name="format-detection" content="telephone=no">
  <meta name="keywords" content="Profile, User Account, Account Settings">
  <meta name="description" content="Manage your profile and account settings">
  <!-- end::Edura Meta Basic -->

  <!-- begin::Edura Website Page Title -->
  <title>Profile | Omateq Digitals</title>
  <!-- end::Edura Website Page Title -->

  <!-- begin::Edura Mobile Specific -->
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <!-- end::Edura Mobile Specific -->

  <!-- begin::Edura Favicon Tags -->
  <?php echo getFaviconHtml(); ?>
  <!-- end::Edura Favicon Tags -->

  <!-- begin::Edura Google Fonts -->
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Plus+Jakarta+Sans:ital,wght@0,200..800;1,200..800&display=swap" rel="stylesheet">
  <!-- end::Edura Google Fonts -->

  <!-- begin::Edura Required Stylesheet -->
  <link rel="stylesheet" href="assets/libs/flaticon/css/all/all.css">
  <link rel="stylesheet" href="assets/libs/lucide/lucide.css">
  <link rel="stylesheet" href="assets/libs/fontawesome/css/all.min.css">
  <link rel="stylesheet" href="assets/libs/simplebar/simplebar.css">
  <link rel="stylesheet" href="assets/libs/node-waves/waves.css">
  <link rel="stylesheet" href="assets/libs/bootstrap-select/css/bootstrap-select.min.css">
  <!-- end::Edura Required Stylesheet -->

  <!-- begin::Edura CSS Stylesheet -->
  <link rel="stylesheet" href="assets/css/styles.css">
  <!-- end::Edura CSS Stylesheet -->
  <style>
    /* Logo sizing for better fit */
    .app-navbar-brand .navbar-brand-logo img {
      max-height: 50px;
      max-width: 180px;
      width: auto;
      height: auto;
      object-fit: contain;
    }
    
    .app-navbar-brand .navbar-brand-mini img {
      max-height: 35px;
      max-width: 35px;
      width: auto;
      height: auto;
      object-fit: contain;
    }
  </style>
</head>

<body>
  <div class="page-layout">

    <!-- begin::Edura Page Header -->
    <header class="app-header">
      <div class="app-header-inner">
        <button class="app-toggler" type="button">
          <span></span>
          <span></span>
          <span></span>
        </button>
        <div class="app-header-start">
          <form class="d-none d-md-flex align-items-center h-100 w-lg-250px w-xxl-300px position-relative" action="profile.php#">
            <button type="button" class="btn btn-sm border-0 position-absolute start-0 ms-3 p-0">
              <i class="fi fi-rr-search"></i>
            </button>
            <input type="text" class="form-control ps-5" placeholder="Search anything's">
          </form>
        </div>
        <div class="app-header-end">
          <div class="px-lg-3 px-2 ps-0 d-flex align-items-center">
            <a href="javascript:void(0);" class="theme-btn">
              <i class="fi fi-rr-brightness icon-light"></i>
              <div class="theme-toggle"></div>
              <i class="fi fi-rr-moon icon-dark"></i>
            </a>
          </div>
          <div class="vr my-3"></div>
          <div class="d-flex align-items-center gap-sm-2 gap-0 px-lg-4 px-sm-2 px-1">
            <a href="javascript:void(0);" class="btn btn-icon btn-action-gray rounded-circle waves-effect waves-light position-relative">
              <i class="fi fi-rr-envelope"></i>
            </a>
            <a href="javascript:void(0);" class="btn btn-icon btn-action-gray rounded-circle waves-effect waves-light">
              <i class="fi fi-rr-bell"></i>
            </a>
          </div>
          <div class="vr my-3"></div>
          <div class="dropdown text-end ms-2">
            <a href="javascript:void(0);" class="d-flex align-items-center py-2" data-bs-toggle="dropdown" data-bs-auto-close="outside" aria-expanded="true">
              <div class="avatar avatar-sm rounded-circle avatar-status-success">
                <?php if ($user['avatar'] && file_exists($user['avatar'])): ?>
                  <img src="<?php echo htmlspecialchars($user['avatar']); ?>" alt="Profile">
                <?php else: ?>
                  <img src="assets/images/avatar/avatar1.webp" alt="Profile">
                <?php endif; ?>
              </div>
            </a>
            <ul class="dropdown-menu dropdown-menu-end w-225px mt-1">
              <li class="d-flex align-items-center p-2">
                <div class="avatar avatar-sm rounded-circle">
                  <?php if ($user['avatar'] && file_exists($user['avatar'])): ?>
                    <img src="<?php echo htmlspecialchars($user['avatar']); ?>" alt="Profile">
                  <?php else: ?>
                    <img src="assets/images/avatar/avatar1.webp" alt="Profile">
                  <?php endif; ?>
                </div>
                <div class="ms-2">
                  <div class="fw-bold text-dark"><?php echo htmlspecialchars($user['full_name']); ?></div>
                  <small class="text-body d-block lh-sm"><?php echo htmlspecialchars($user['email']); ?></small>
                </div>
              </li>
              <li>
                <div class="dropdown-divider my-1"></div>
              </li>
              <li>
                <a class="dropdown-item d-flex align-items-center gap-2" href="profile.php">
                  <i class="fi fi-rr-user scale-1x"></i> View Profile
                </a>
              </li>
              <li>
                <div class="dropdown-divider my-1"></div>
              </li>
              <li>
                <a class="dropdown-item d-flex align-items-center gap-2 text-danger" href="authentication/logout.php">
                  <i class="fi fi-sr-exit scale-1x"></i> Log Out
                </a>
              </li>
            </ul>
          </div>
        </div>
      </div>
    </header>
    <!-- end::Edura Page Header -->

    <!-- begin::Edura Sidebar Menu -->
    <aside class="app-menubar" id="menubar">
      <div class="app-navbar-brand">
        <a class="navbar-brand-logo" href="index.php">
          <?php echo '<img src="' . '../' . getSetting('main_logo', 'assets/images/logo.svg') . '" alt="logo">'; ?>
        </a>
        <a class="navbar-brand-mini" href="index.php">
          <?php echo '<img src="' . '../' . getSetting('mini_logo', 'assets/images/logo-text-white.svg') . '" alt="logo">'; ?>
        </a>
      </div>
      <nav class="app-navbar" data-simplebar>
        <ul class="menubar">
          <li class="menu-item">
            <a class="menu-link" href="index.php">
              <i class="fi fi-rr-apps"></i>
              <span class="menu-label">Dashboard</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="products.php">
              <i class="fi fi-rr-users"></i>
              <span class="menu-label">Products</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="course.php">
              <i class="fi fi-rr-graduation-cap"></i>
              <span class="menu-label">Course</span>
                </a>
              </li>
              <li class="menu-item">
            <a class="menu-link" href="my_courses.php">
              <i class="fi fi-rr-book"></i>
              <span class="menu-label">My Courses</span>
                </a>
              </li>
              <li class="menu-item active">
            <a class="menu-link" href="profile.php">
              <i class="fi fi-rr-user"></i>
              <span class="menu-label">Profile</span>
                </a>
              </li>
              
        </ul>
      </nav>
      <div class="app-footer">
        <div class="text-center py-2">
          <small class="text-white-50">
            <i class="fi fi-rr-copyright me-1"></i>
            2025 Omateq Digitals
          </small>
        </div>
        <a href="mailto:info@omateqdigitals.com" class="btn btn-outline-white waves-effect btn-app-nav w-100">
          <i class="fi fi-rs-interrogation"></i>
          <span class="nav-text">Help and Support</span>
        </a>
      </div>
    </aside>
    <!-- end::Edura Sidebar Menu -->

    <main class="app-wrapper">

      <div class="container-fluid">

        <div class="app-page-head d-flex flex-wrap gap-3 align-items-center justify-content-between">
          <div class="clearfix">
            <h1 class="app-page-title">Profile</h1>
            <nav aria-label="breadcrumb">
              <ol class="breadcrumb mb-0">
                <li class="breadcrumb-item">
                  <a href="index.php">Dashboard</a>
                </li>
                <li class="breadcrumb-item active" aria-current="page">Profile</li>
              </ol>
            </nav>
          </div>
        </div>

        <?php if ($success_message): ?>
          <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fi fi-rr-check-circle me-2"></i>
            <?php echo htmlspecialchars($success_message); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
          </div>
        <?php endif; ?>

        <?php if ($error_message): ?>
          <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fi fi-rr-exclamation-triangle me-2"></i>
            <?php echo htmlspecialchars($error_message); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
          </div>
        <?php endif; ?>

        <div class="row">

          <div class="col-lg-12">
            <div class="card">
              <div class="card-body">
                <div class="d-flex flex-wrap gap-4 align-items-center">
                  <div class="d-flex align-items-center">
                    <div class="position-relative">
                      <div class="avatar avatar-xxl rounded-circle">
                        <?php if ($user['avatar'] && file_exists($user['avatar'])): ?>
                          <img src="<?php echo htmlspecialchars($user['avatar']); ?>" alt="Profile">
                        <?php else: ?>
                          <img src="assets/images/avatar/avatar1.webp" alt="Profile">
                        <?php endif; ?>
                      </div>
                      <button type="button" class="avatar avatar-xxs bg-primary rounded-circle text-white position-absolute top-0 end-0" data-bs-toggle="modal" data-bs-target="#avatarModal">
                        <i class="fi fi-rr-camera"></i>
                      </button>
                    </div>
                    <div class="ms-3">
                      <h4 class="fw-bold mb-0"><?php echo htmlspecialchars($user['full_name']); ?></h4>
                      <small class="mb-2">Student</small>
                      <div class="d-flex flex-wrap gap-1 mt-2">
                        <span class="badge badge-sm px-2 rounded-pill text-bg-primary"><?php echo ucfirst($user['status']); ?></span>
                        <span class="badge badge-sm px-2 rounded-pill text-bg-success">Verified</span>
                      </div>
                    </div>
                  </div>
                  <div class="d-flex gap-2 ms-md-auto">
                    <a href="my_courses.php" class="btn btn-primary waves-effect waves-light">My Courses</a>
                    <button type="button" class="btn btn-outline-secondary waves-effect waves-light" data-bs-toggle="modal" data-bs-target="#avatarModal">Change Avatar</button>
                  </div>
                </div>
              </div>
            </div>
          </div>

          <div class="col-lg-4 col-sm-12">
            <div class="row">
              <div class="col-12">
                <div class="card">
                  <div class="card-header d-flex justify-content-between align-items-center">
                    <h4 class="card-title mb-0">Basic Information</h4>
                    <button class="btn btn-white btn-sm btn-shadow btn-icon waves-effect" type="button" data-bs-toggle="collapse" data-bs-target="#basicInfoCollapse">
                      <i class="fi fi-rr-pencil"></i>
                    </button>
                  </div>
                  <div class="card-body">
                    <div class="mb-3">
                      <span class="mb-1">Full Name</span>
                      <p class="text-dark fw-semibold mb-0"><?php echo htmlspecialchars($user['full_name']); ?></p>
                    </div>
                    <div class="mb-3">
                      <span class="mb-1">Email</span>
                      <p class="text-dark fw-semibold mb-0"><?php echo htmlspecialchars($user['email']); ?></p>
                    </div>
                    <div class="mb-3">
                      <span class="mb-1">Phone</span>
                      <p class="text-dark fw-semibold mb-0"><?php echo htmlspecialchars($user['phone'] ?: 'Not provided'); ?></p>
                    </div>
                    <div class="mb-3">
                      <span class="mb-1">Username</span>
                      <p class="text-dark fw-semibold mb-0"><?php echo htmlspecialchars($user['username']); ?></p>
                    </div>
                    <div class="mb-2">
                      <span class="mb-1">Joined Date</span>
                      <p class="text-dark fw-semibold mb-0"><?php echo date('F j, Y', strtotime($user['created_at'])); ?></p>
                    </div>
                  </div>
                </div>
              </div>
              <div class="col-12">
                <div class="card">
                  <div class="card-header">
                    <h4 class="card-title">Statistics</h4>
                  </div>
                  <div class="card-body">
                    <div class="row text-center">
                      <div class="col-6">
                        <h3 class="text-primary mb-1"><?php echo $course_stats['course_count']; ?></h3>
                        <small class="text-muted">Enrolled Courses</small>
                      </div>
                      <div class="col-6">
                        <h3 class="text-success mb-1"><?php echo $user['email_verified'] ? 'Yes' : 'No'; ?></h3>
                        <small class="text-muted">Email Verified</small>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </div>

          <div class="col-lg-8 col-sm-12">
            <div class="row">
              <div class="col-12">
                <div class="card">
                  <div class="card-header">
                    <h4 class="card-title">Account Settings</h4>
                  </div>
                  <div class="card-body">
                    <form method="POST" action="">
                      <input type="hidden" name="update_profile" value="1">
                      <div class="row mb-3">
                        <div class="col-md-6">
                          <label class="form-label">Full Name</label>
                          <input type="text" class="form-control" name="full_name" value="<?php echo htmlspecialchars($user['full_name']); ?>" required>
                        </div>
                        <div class="col-md-6">
                          <label class="form-label">Email</label>
                          <input type="email" class="form-control" name="email" value="<?php echo htmlspecialchars($user['email']); ?>" required>
                        </div>
                      </div>
                      <div class="row mb-3">
                        <div class="col-md-6">
                          <label class="form-label">Phone</label>
                          <input type="tel" class="form-control" name="phone" value="<?php echo htmlspecialchars($user['phone'] ?: ''); ?>">
                        </div>
                        <div class="col-md-6">
                          <label class="form-label">Username</label>
                          <input type="text" class="form-control" value="<?php echo htmlspecialchars($user['username']); ?>" readonly>
                          <small class="text-muted">Username cannot be changed</small>
                        </div>
                      </div>
                      <div class="mb-3">
                        <label class="form-label">Bio</label>
                        <textarea class="form-control" name="bio" rows="3" placeholder="Tell us about yourself..."><?php echo htmlspecialchars($user['bio'] ?? ''); ?></textarea>
                      </div>
                      <div class="text-end">
                        <button type="submit" class="btn btn-success waves-effect waves-light">Save Changes</button>
                      </div>
                    </form>
                  </div>
                </div>
              </div>
              
              <div class="col-12">
                <div class="card">
                  <div class="card-header">
                    <h4 class="card-title">Change Password</h4>
                  </div>
                  <div class="card-body">
                    <form method="POST" action="">
                      <input type="hidden" name="change_password" value="1">
                      <div class="row mb-3">
                        <div class="col-md-12">
                          <label class="form-label">Current Password</label>
                          <input type="password" class="form-control" name="current_password" required>
                        </div>
                      </div>
                      <div class="row mb-3">
                        <div class="col-md-6">
                          <label class="form-label">New Password</label>
                          <input type="password" class="form-control" name="new_password" required minlength="6">
                        </div>
                        <div class="col-md-6">
                          <label class="form-label">Confirm New Password</label>
                          <input type="password" class="form-control" name="confirm_password" required minlength="6">
                        </div>
                      </div>
                      <div class="text-end">
                        <button type="submit" class="btn btn-warning waves-effect waves-light">Change Password</button>
                      </div>
                    </form>
                  </div>
                </div>
              </div>
            </div>
          </div>

        </div>

      </div>

    </main>

    <!-- Avatar Upload Modal -->
    <div class="modal fade" id="avatarModal" tabindex="-1" aria-labelledby="avatarModalLabel" aria-hidden="true">
      <div class="modal-dialog">
        <div class="modal-content">
          <div class="modal-header">
            <h5 class="modal-title" id="avatarModalLabel">Change Avatar</h5>
            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
          </div>
          <form method="POST" action="" enctype="multipart/form-data">
            <input type="hidden" name="upload_avatar" value="1">
            <div class="modal-body">
              <div class="text-center mb-3">
                <div class="avatar avatar-xxl rounded-circle mx-auto mb-3">
                  <?php if ($user['avatar'] && file_exists($user['avatar'])): ?>
                    <img src="<?php echo htmlspecialchars($user['avatar']); ?>" alt="Current Avatar" id="avatarPreview">
                  <?php else: ?>
                    <img src="assets/images/avatar/avatar1.webp" alt="Current Avatar" id="avatarPreview">
                  <?php endif; ?>
                </div>
                <p class="text-muted">Upload a new profile picture</p>
              </div>
              <div class="mb-3">
                <label for="avatar" class="form-label">Choose Image</label>
                <input type="file" class="form-control" id="avatar" name="avatar" accept="image/*" required>
                <small class="text-muted">Supported formats: JPG, PNG, GIF, WebP. Max size: 5MB</small>
              </div>
            </div>
            <div class="modal-footer">
              <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
              <button type="submit" class="btn btn-primary">Upload Avatar</button>
            </div>
          </form>
        </div>
      </div>
    </div>

    <!-- begin::Edura Footer -->
    <footer class="footer-wrapper bg-body text-center">
      <div class="container-fluid">
        <p class="mb-0">© <span class="currentYear">2025</span> Omateq Digitals. Proudly powered by Omateq Digitals Ltd.</p>
      </div>
    </footer>
    <!-- end::Edura Footer -->

    <!-- WhatsApp Chat Button -->
    <div class="whatsapp-chat-button">
      <a href="https://wa.me/2347069378861?text=Hello%20Omateq%20Digitals!%20I%20would%20like%20to%20know%20more%20about%20your%20services." 
         target="_blank" 
         rel="noopener noreferrer"
         class="whatsapp-link"
         title="Chat with us on WhatsApp">
        <i class="fab fa-whatsapp"></i>
        <span class="whatsapp-text">Chat with us</span>
      </a>
    </div>
    
    <style>
    /* WhatsApp Chat Button Styles */
    .whatsapp-chat-button {
      position: fixed;
      bottom: 20px;
      right: 20px;
      z-index: 1000;
      animation: pulse 2s infinite;
    }
    
    .whatsapp-link {
      display: flex;
      align-items: center;
      background: #25D366;
      color: white;
      text-decoration: none;
      padding: 12px 16px;
      border-radius: 50px;
      box-shadow: 0 4px 12px rgba(37, 211, 102, 0.4);
      transition: all 0.3s ease;
      font-family: 'Poppins', sans-serif;
      font-weight: 500;
      font-size: 14px;
    }
    
    .whatsapp-link:hover {
      background: #128C7E;
      color: white;
      text-decoration: none;
      transform: translateY(-2px);
      box-shadow: 0 6px 20px rgba(37, 211, 102, 0.6);
    }
    
    .whatsapp-link i {
      font-size: 24px;
      margin-right: 8px;
    }
    
    .whatsapp-text {
      white-space: nowrap;
    }
    
    @keyframes pulse {
      0% {
        box-shadow: 0 0 0 0 rgba(37, 211, 102, 0.7);
      }
      70% {
        box-shadow: 0 0 0 10px rgba(37, 211, 102, 0);
      }
      100% {
        box-shadow: 0 0 0 0 rgba(37, 211, 102, 0);
      }
    }
    
    /* Mobile responsiveness */
    @media (max-width: 768px) {
      .whatsapp-chat-button {
        bottom: 15px;
        right: 15px;
      }
      
      .whatsapp-link {
        padding: 10px 14px;
        font-size: 13px;
      }
      
      .whatsapp-link i {
        font-size: 22px;
        margin-right: 6px;
      }
      
      .whatsapp-text {
        display: none;
      }
    }
    </style>

  </div>

  <!-- begin::Edura Page Scripts -->
  <script src="assets/libs/global/global.min.js"></script>
  <script src="assets/js/appSettings.js"></script>
  <script src="assets/js/main.js"></script>
  
  <script>
    // Avatar preview functionality
    document.getElementById('avatar').addEventListener('change', function(e) {
      const file = e.target.files[0];
      if (file) {
        const reader = new FileReader();
        reader.onload = function(e) {
          document.getElementById('avatarPreview').src = e.target.result;
        };
        reader.readAsDataURL(file);
      }
    });
  </script>
  <!-- end::Edura Page Scripts -->
</body>

</html>
<?php
// End of PHP processing
?>